<?php
/**
 * Plugin Name: Affiliate Hub
 * Plugin URI: https://frumbik.com/AffiliateHub
 * Description: A powerful, modular affiliate link management plugin for WordPress. Easily create, manage, track, and optimize your affiliate links with advanced features and multilingual support.
 * Version: 1.0.0
 * Requires PHP: 7.4
 * Requires at least: 6.0
 * Tested up to: 6.8
 * Author: Mariusz Kobak
 * Author URI: https://frumbik.com
 * Text Domain: affiliate-hub
 * Domain Path: /languages/
 * License: GPL v2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 *
 *
 * @package AffiliateHub
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

// Check for minimum requirements
if (version_compare(PHP_VERSION, '7.4', '<')) {
    add_action('admin_notices', function() {
    echo '<div class="notice notice-error"><p>' . 
         sprintf(
             // translators: %s: current PHP version
             esc_html__('Affiliate Hub requires PHP 7.4 or higher. Current version: %s', 'affiliate-hub'),
             esc_html(PHP_VERSION)
         ) . '</p></div>';
    });
    return;
}

// Check WordPress version
if (version_compare($GLOBALS['wp_version'], '6.0', '<')) {
    add_action('admin_notices', function() {
    echo '<div class="notice notice-error"><p>' . 
         esc_html__('Affiliate Hub requires WordPress 6.0 or higher.', 'affiliate-hub') . 
             '</p></div>';
    });
    return;
}

// Memory limit check
$memory_limit = ini_get('memory_limit');
$memory_limit_bytes = wp_convert_hr_to_bytes($memory_limit);
if ($memory_limit_bytes < 134217728) { // 128MB
    add_action('admin_notices', function() {
    echo '<div class="notice notice-warning"><p>' . 
         sprintf(
             // translators: %s: current PHP memory_limit value (e.g. 128M)
             esc_html__('Affiliate Hub recommends at least 128MB of PHP memory. Current limit: %s', 'affiliate-hub'),
             esc_html(ini_get('memory_limit'))
         ) . '</p></div>';
    });
}

// Suppress WordPress.org connection warnings for our plugin
add_action('init', function() {
    // Remove WordPress.org connection error notices on our plugin pages
    if (isset($_GET['page']) && strpos($_GET['page'], 'affiliate-hub') !== false) {
        // Temporarily suppress the specific WordPress.org warning
        add_filter('pre_site_transient_update_plugins', '__return_null');
        
        // Custom notice about WordPress.org connection (informational only)
        add_action('admin_notices', function() {
            if (get_transient('affiliate_hub_wp_org_notice_dismissed')) {
                return;
            }
            
            echo '<div class="notice notice-info is-dismissible" data-dismiss-key="affiliate_hub_wp_org_notice">';
            echo '<p><strong>Affiliate Hub:</strong> ';
            echo esc_html__('If you see WordPress.org connection warnings, they don\'t affect this plugin\'s functionality. All features work normally.', 'affiliate-hub');
            echo '</p></div>';
            
            // Auto-dismiss after showing once
            set_transient('affiliate_hub_wp_org_notice_dismissed', true, WEEK_IN_SECONDS);
        });
    }
});

// Define plugin constants
define('AFFILIATE_HUB_VERSION', '1.0.0');
define('AFFILIATE_HUB_FILE', __FILE__);
define('AFFILIATE_HUB_PATH', plugin_dir_path(__FILE__));
define('AFFILIATE_HUB_URL', plugin_dir_url(__FILE__));
define('AFFILIATE_HUB_BASENAME', plugin_basename(__FILE__));

// Emergency memory handler - only for actual memory errors
register_shutdown_function(function() {
    $error = error_get_last();
    if ($error && 
        $error['type'] === E_ERROR && 
        (strpos($error['message'], 'memory') !== false || 
         strpos($error['message'], 'Allowed memory size') !== false)) {
        // Enable safe mode only for real memory errors
        update_option('affiliate_hub_safe_mode', true);
        error_log('Affiliate Hub: Memory error detected, safe mode enabled automatically');
    }
});

// Increase memory limit if possible
if (ini_get('memory_limit') !== '-1') {
    $current_limit = wp_convert_hr_to_bytes(ini_get('memory_limit'));
    if ($current_limit < 268435456) { // Less than 256MB
        ini_set('memory_limit', '256M');
    }
}

// Optimized Autoloader with caching and security improvements
spl_autoload_register(function ($class_name) {
    // Only handle our namespace
    if (strpos($class_name, 'AffiliateHub\\') !== 0) {
        return;
    }
    
    // Prevent infinite loops and cache loaded files
    static $loading = array();
    static $loaded_files = array(); // Performance cache
    
    if (isset($loading[$class_name])) {
        return;
    }
    
    // Check if already loaded
    if (isset($loaded_files[$class_name])) {
        return;
    }
    
    $loading[$class_name] = true;
    
    // Remove namespace prefix and convert to file path
    $class_file = str_replace('AffiliateHub\\', '', $class_name);
    $class_file = str_replace('\\', DIRECTORY_SEPARATOR, $class_file);
    $class_file .= '.php';
    
    $file_path = AFFILIATE_HUB_PATH . 'includes' . DIRECTORY_SEPARATOR . $class_file;
    
    // Security check: prevent directory traversal
    $real_includes_path = realpath(AFFILIATE_HUB_PATH . 'includes');
    $real_file_path = realpath($file_path);
    
    if ($real_file_path && 
        strpos($real_file_path, $real_includes_path) === 0 && 
        file_exists($file_path)) {
        require_once $file_path;
        $loaded_files[$class_name] = true; // Cache success
    }
    
    unset($loading[$class_name]);
});

// Initialize the plugin with error handling
function affiliate_hub_init() {
    // Check emergency disable flag
    if (get_option('affiliate_hub_emergency_disabled')) {
        return;
    }
    
    // Force disable safe mode if memory is sufficient
    if (get_option('affiliate_hub_safe_mode', false)) {
        $memory_limit = wp_convert_hr_to_bytes(ini_get('memory_limit'));
        $memory_usage = memory_get_usage(true);
        $available_memory = $memory_limit - $memory_usage;
        
        // If we have more than 64MB available, disable safe mode
        if ($available_memory > 67108864) { // 64MB
            delete_option('affiliate_hub_safe_mode');
            update_option('affiliate_hub_enable_stats', true);
        }
    }
    
    try {
        if (class_exists('AffiliateHub\\Core\\Plugin')) {
            AffiliateHub\Core\Plugin::get_instance();
            
            // Register error handler for critical errors
            register_shutdown_function(function() {
                $error = error_get_last();
                if ($error && in_array($error['type'], [E_ERROR, E_PARSE, E_CORE_ERROR, E_COMPILE_ERROR])) {
                    if (strpos($error['file'], 'affiliate-hub') !== false) {
                        error_log('Affiliate Hub fatal error: ' . $error['message'] . ' in ' . $error['file'] . ':' . $error['line']);
                        
                        // Disable plugin temporarily
                        update_option('affiliate_hub_emergency_disabled', true);
                    }
                }
            });
        }
    } catch (\Exception $e) {
        error_log('Affiliate Hub initialization error: ' . $e->getMessage());
        update_option('affiliate_hub_emergency_disabled', true);
    }
}
add_action('plugins_loaded', 'affiliate_hub_init');

// Ensure DB tables exist on admin init to avoid missing table errors on upgrades
add_action('admin_init', function() {
    if (class_exists('AffiliateHub\\Core\\Activator')) {
        try {
            \AffiliateHub\Core\Activator::ensure_tables();
        } catch (\Exception $e) {
            // swallow - Activator will log if needed
        }
    }
});

// Check emergency disable flag and show admin notice
add_action('admin_init', function() {
    if (get_option('affiliate_hub_emergency_disabled')) {
        add_action('admin_notices', function() {
            echo '<div class="notice notice-error is-dismissible"><p>' . 
              esc_html__('Affiliate Hub was automatically disabled due to a critical error. Please check your error logs.', 'affiliate-hub') . 
              '</p><p><a href="' . esc_url(add_query_arg('affiliate_hub_reset', 'true')) . '" class="button">' .
              esc_html__('Try to re-enable', 'affiliate-hub') . '</a></p></div>';
        });
        
        if (isset($_GET['affiliate_hub_reset'])) {
            delete_option('affiliate_hub_emergency_disabled');
          wp_safe_redirect(admin_url('plugins.php'));
            exit;
        }
    }
});

// Activation hook
function affiliate_hub_activate() {
    if (class_exists('AffiliateHub\\Core\\Activator')) {
        AffiliateHub\Core\Activator::activate();
    }
}
register_activation_hook(__FILE__, 'affiliate_hub_activate');

// Deactivation hook
function affiliate_hub_deactivate() {
    if (class_exists('AffiliateHub\\Core\\Deactivator')) {
        AffiliateHub\Core\Deactivator::deactivate();
    }
}
register_deactivation_hook(__FILE__, 'affiliate_hub_deactivate');

// Uninstall hook
function affiliate_hub_uninstall() {
    if (class_exists('AffiliateHub\\Core\\Uninstaller')) {
        AffiliateHub\Core\Uninstaller::uninstall();
    }
}
register_uninstall_hook(__FILE__, 'affiliate_hub_uninstall');
