/**
 * ApexCharts Migration Test Suite - JavaScript
 * Comprehensive testing for WordPress Affiliate Hub Plugin
 */

// Global variables for testing
let testCharts = {};
let performanceMetrics = {};
let testLog = [];

// Initialize test environment
document.addEventListener('DOMContentLoaded', function() {
    initializeTestSuite();
});

function initializeTestSuite() {
    log('🎯 Initializing ApexCharts Test Suite...');
    document.getElementById('session-time').textContent = new Date().toLocaleString();
    detectBrowserInfo();
    log('✅ Test Suite Ready for Testing');
}

function log(message) {
    const timestamp = new Date().toLocaleTimeString();
    const logEntry = `[${timestamp}] ${message}`;
    testLog.push(logEntry);
    
    const logElement = document.getElementById('test-log');
    const div = document.createElement('div');
    div.textContent = logEntry;
    logElement.appendChild(div);
    logElement.scrollTop = logElement.scrollHeight;
}

function updateStatus(elementId, status, success = true) {
    const element = document.getElementById(elementId);
    element.textContent = status;
    element.className = 'status ' + (success ? 'status-success' : 'status-error');
}

// Performance Testing Functions
function runPerformanceTest() {
    log('🔥 Starting Performance Benchmark...');
    updateStatus('line-chart-status', 'Testing...', false);
    
    const startTime = performance.now();
    let renderStartTime;
    
    // Measure library load time
    const loadEndTime = performance.now();
    const loadTime = loadEndTime - startTime;
    document.getElementById('load-time').textContent = Math.round(loadTime);
    
    // Test chart rendering performance
    renderStartTime = performance.now();
    
    const sampleData = generateSampleLineData();
    const options = {
        series: [{
            name: 'Performance Test',
            data: sampleData.data
        }],
        chart: {
            type: 'line',
            height: 300,
            animations: {
                enabled: true,
                speed: 1000
            },
            events: {
                mounted: function() {
                    const renderEndTime = performance.now();
                    const renderTime = renderEndTime - renderStartTime;
                    document.getElementById('render-time').textContent = Math.round(renderTime);
                    
                    // Measure memory usage
                    if (performance.memory) {
                        const memoryMB = Math.round(performance.memory.usedJSHeapSize / 1024 / 1024);
                        document.getElementById('memory-usage').textContent = memoryMB;
                    } else {
                        document.getElementById('memory-usage').textContent = 'N/A';
                    }
                    
                    log(`⚡ Performance Results: Load=${Math.round(loadTime)}ms, Render=${Math.round(renderTime)}ms`);
                }
            }
        },
        xaxis: {
            categories: sampleData.categories
        },
        colors: ['#0073aa']
    };
    
    // Create performance test chart
    if (testCharts.performance) {
        testCharts.performance.destroy();
    }
    
    const chartElement = document.createElement('div');
    chartElement.style.height = '200px';
    document.querySelector('.performance-metrics').appendChild(chartElement);
    
    testCharts.performance = new ApexCharts(chartElement, options);
    testCharts.performance.render();
    
    // Measure FPS during animation
    measureFPS();
}

function measureFPS() {
    let frameCount = 0;
    const startTime = performance.now();
    const duration = 2000; // 2 seconds
    
    function countFrame() {
        frameCount++;
        const elapsed = performance.now() - startTime;
        
        if (elapsed < duration) {
            requestAnimationFrame(countFrame);
        } else {
            const fps = Math.round(frameCount / (elapsed / 1000));
            document.getElementById('fps').textContent = fps;
            log(`🎬 Animation FPS: ${fps}`);
        }
    }
    
    requestAnimationFrame(countFrame);
}

// Chart Testing Functions
function testLineChart() {
    log('📈 Testing Line Chart (Core Statistics)...');
    
    const data = generateSampleLineData();
    const options = {
        series: [{
            name: 'Total Clicks',
            data: data.data.map(d => d + Math.random() * 10)
        }, {
            name: 'Unique Clicks',
            data: data.data.map(d => d * 0.7 + Math.random() * 5)
        }],
        chart: {
            type: 'line',
            height: 350,
            animations: {
                enabled: true,
                easing: 'easeinout',
                speed: 600
            },
            toolbar: {
                show: true,
                tools: {
                    download: true,
                    zoom: true,
                    pan: true,
                    reset: true
                }
            }
        },
        xaxis: {
            categories: data.categories,
            title: { text: 'Date' }
        },
        yaxis: {
            title: { text: 'Clicks' },
            min: 0
        },
        colors: ['#0073aa', '#00a32a'],
        stroke: {
            width: 3,
            curve: 'smooth'
        },
        fill: {
            type: 'gradient',
            gradient: {
                shade: 'light',
                type: 'vertical',
                shadeIntensity: 0.25,
                opacityFrom: 0.85,
                opacityTo: 0.1
            }
        },
        legend: {
            position: 'top'
        },
        tooltip: {
            theme: 'light',
            y: {
                formatter: function(value) {
                    return value + ' clicks';
                }
            }
        }
    };
    
    if (testCharts.line) {
        testCharts.line.destroy();
    }
    
    testCharts.line = new ApexCharts(document.getElementById('line-chart-test'), options);
    testCharts.line.render().then(() => {
        updateStatus('line-chart-status', 'Success ✅');
        log('✅ Line Chart test completed successfully');
    }).catch(error => {
        updateStatus('line-chart-status', 'Failed ❌', false);
        log(`❌ Line Chart test failed: ${error.message}`);
    });
}

function testDonutChart() {
    log('🍩 Testing Donut Chart (Link Scanner)...');
    
    const options = {
        series: [44, 55, 13, 43],
        labels: ['Active', 'Issues', 'Errors', 'Pending'],
        colors: ['#28a745', '#ffc107', '#dc3545', '#6c757d'],
        chart: {
            type: 'donut',
            height: 350,
            animations: {
                enabled: true,
                easing: 'easeinout',
                speed: 600
            }
        },
        plotOptions: {
            pie: {
                donut: {
                    size: '45%',
                    labels: {
                        show: true,
                        total: {
                            show: true,
                            label: 'Total',
                            formatter: function(w) {
                                return w.globals.seriesTotals.reduce((a, b) => a + b, 0);
                            }
                        }
                    }
                }
            }
        },
        dataLabels: {
            enabled: true,
            formatter: function(val) {
                return Math.round(val) + '%';
            }
        },
        legend: {
            position: 'bottom'
        },
        tooltip: {
            theme: 'light'
        }
    };
    
    if (testCharts.donut) {
        testCharts.donut.destroy();
    }
    
    testCharts.donut = new ApexCharts(document.getElementById('donut-chart-test'), options);
    testCharts.donut.render().then(() => {
        updateStatus('donut-chart-status', 'Success ✅');
        log('✅ Donut Chart test completed successfully');
    }).catch(error => {
        updateStatus('donut-chart-status', 'Failed ❌', false);
        log(`❌ Donut Chart test failed: ${error.message}`);
    });
}

function testBarChart() {
    log('📊 Testing Bar Chart (Browser Stats)...');
    
    const options = {
        series: [{
            name: 'Clicks',
            data: [30, 40, 45, 50, 49, 60, 70]
        }],
        chart: {
            type: 'bar',
            height: 350,
            animations: {
                enabled: true,
                easing: 'easeinout',
                speed: 600
            }
        },
        xaxis: {
            categories: ['Chrome', 'Firefox', 'Safari', 'Edge', 'Opera', 'Mobile', 'Other']
        },
        yaxis: {
            title: { text: 'Clicks' }
        },
        colors: ['#0073aa', '#00a32a', '#d63638', '#dba617', '#9b51e0', '#17a2b8', '#6c757d'],
        plotOptions: {
            bar: {
                borderRadius: 4,
                horizontal: false,
                columnWidth: '60%',
                distributed: true
            }
        },
        dataLabels: {
            enabled: true,
            style: {
                colors: ['#fff'],
                fontWeight: 'bold'
            }
        },
        legend: {
            show: false
        }
    };
    
    if (testCharts.bar) {
        testCharts.bar.destroy();
    }
    
    testCharts.bar = new ApexCharts(document.getElementById('bar-chart-test'), options);
    testCharts.bar.render().then(() => {
        updateStatus('bar-chart-status', 'Success ✅');
        log('✅ Bar Chart test completed successfully');
    }).catch(error => {
        updateStatus('bar-chart-status', 'Failed ❌', false);
        log(`❌ Bar Chart test failed: ${error.message}`);
    });
}

function testPieChart() {
    log('🥧 Testing Pie Chart (OS Distribution)...');
    
    const options = {
        series: [44, 33, 23],
        labels: ['Windows', 'macOS', 'Linux'],
        colors: ['#0073aa', '#00a32a', '#d63638'],
        chart: {
            type: 'pie',
            height: 350,
            animations: {
                enabled: true,
                easing: 'easeinout',
                speed: 600
            }
        },
        plotOptions: {
            pie: {
                expandOnClick: true
            }
        },
        dataLabels: {
            enabled: true,
            formatter: function(val, opts) {
                return opts.w.config.labels[opts.seriesIndex] + ": " + Math.round(val) + "%";
            }
        },
        legend: {
            position: 'bottom'
        },
        tooltip: {
            theme: 'light'
        }
    };
    
    if (testCharts.pie) {
        testCharts.pie.destroy();
    }
    
    testCharts.pie = new ApexCharts(document.getElementById('pie-chart-test'), options);
    testCharts.pie.render().then(() => {
        updateStatus('pie-chart-status', 'Success ✅');
        log('✅ Pie Chart test completed successfully');
    }).catch(error => {
        updateStatus('pie-chart-status', 'Failed ❌', false);
        log(`❌ Pie Chart test failed: ${error.message}`);
    });
}

function runAllChartTests() {
    log('🚀 Running All Chart Tests...');
    testLineChart();
    setTimeout(() => testDonutChart(), 500);
    setTimeout(() => testBarChart(), 1000);
    setTimeout(() => testPieChart(), 1500);
    setTimeout(() => log('✅ All Chart Tests Completed'), 2000);
}

// Data Update Testing
function updateLineChart() {
    if (!testCharts.line) return;
    const newData = generateSampleLineData();
    testCharts.line.updateSeries([{
        name: 'Total Clicks',
        data: newData.data
    }, {
        name: 'Unique Clicks', 
        data: newData.data.map(d => d * 0.8)
    }]);
    log('🔄 Line Chart data updated');
}

function updateDonutChart() {
    if (!testCharts.donut) return;
    const newData = [
        Math.floor(Math.random() * 50) + 20,
        Math.floor(Math.random() * 30) + 10,
        Math.floor(Math.random() * 20) + 5,
        Math.floor(Math.random() * 25) + 15
    ];
    testCharts.donut.updateSeries(newData);
    log('🔄 Donut Chart data updated');
}

function updateBarChart() {
    if (!testCharts.bar) return;
    const newData = Array.from({length: 7}, () => Math.floor(Math.random() * 80) + 20);
    testCharts.bar.updateSeries([{
        name: 'Clicks',
        data: newData
    }]);
    log('🔄 Bar Chart data updated');
}

function updatePieChart() {
    if (!testCharts.pie) return;
    const newData = [
        Math.floor(Math.random() * 40) + 30,
        Math.floor(Math.random() * 30) + 20,
        Math.floor(Math.random() * 25) + 15
    ];
    testCharts.pie.updateSeries(newData);
    log('🔄 Pie Chart data updated');
}

// Responsive Design Testing
function testResponsiveDesign() {
    log('📱 Testing Responsive Design...');
    
    // Mobile chart
    createResponsiveChart('mobile-chart', 300, 200);
    
    // Tablet chart
    createResponsiveChart('tablet-chart', 600, 250);
    
    // Desktop chart
    createResponsiveChart('desktop-chart', 900, 300);
    
    updateStatus('responsive-status', 'Success ✅');
    log('✅ Responsive design tests completed');
}

function createResponsiveChart(elementId, width, height) {
    const data = generateSampleLineData();
    const options = {
        series: [{
            name: 'Sample Data',
            data: data.data
        }],
        chart: {
            type: 'line',
            height: height,
            width: width
        },
        xaxis: {
            categories: data.categories
        },
        colors: ['#0073aa'],
        stroke: {
            curve: 'smooth'
        }
    };
    
    const chart = new ApexCharts(document.getElementById(elementId), options);
    chart.render();
}

// Feature Testing Functions
function testZoomPan() {
    log('🔍 Testing Zoom/Pan functionality...');
    if (testCharts.line) {
        // Enable zoom/pan if not already enabled
        testCharts.line.updateOptions({
            chart: {
                zoom: {
                    enabled: true,
                    type: 'x'
                },
                pan: {
                    enabled: true
                }
            }
        });
        log('✅ Zoom/Pan enabled on line chart');
    } else {
        log('❌ No line chart available for zoom/pan test');
    }
}

function testExport() {
    log('📥 Testing Export functionality...');
    if (testCharts.line) {
        try {
            testCharts.line.dataURI().then(({imgURI, blob}) => {
                log('✅ Chart export successful');
            });
        } catch (error) {
            log(`❌ Export failed: ${error.message}`);
        }
    } else {
        log('❌ No chart available for export test');
    }
}

function testAnimations() {
    log('🎬 Testing Animation performance...');
    runAllChartTests(); // This will test animations as charts render
}

function testTooltips() {
    log('💬 Testing Tooltips... (Hover over charts to test)');
}

function testRealTimeUpdates() {
    log('📡 Testing Real-time Updates...');
    let updateCount = 0;
    const maxUpdates = 5;
    
    const interval = setInterval(() => {
        updateLineChart();
        updateDonutChart();
        updateCount++;
        
        if (updateCount >= maxUpdates) {
            clearInterval(interval);
            log('✅ Real-time updates test completed');
        }
    }, 1000);
}

function testDataValidation() {
    log('✅ Testing Data Validation...');
    
    // Test empty data
    try {
        const emptyOptions = {
            series: [],
            chart: { type: 'line', height: 100 }
        };
        const tempChart = new ApexCharts(document.createElement('div'), emptyOptions);
        tempChart.render();
        tempChart.destroy();
        log('✅ Empty data handling: OK');
    } catch (error) {
        log(`❌ Empty data handling failed: ${error.message}`);
    }
    
    // Test invalid data
    try {
        const invalidOptions = {
            series: [{ name: 'Test', data: ['invalid', null, undefined] }],
            chart: { type: 'line', height: 100 }
        };
        const tempChart = new ApexCharts(document.createElement('div'), invalidOptions);
        tempChart.render();
        tempChart.destroy();
        log('✅ Invalid data handling: OK');
    } catch (error) {
        log(`❌ Invalid data handling failed: ${error.message}`);
    }
}

function testErrorHandling() {
    log('❌ Testing Error Handling...');
    
    try {
        // Test invalid chart type
        const invalidOptions = {
            series: [{ name: 'Test', data: [1, 2, 3] }],
            chart: { type: 'invalidtype', height: 100 }
        };
        const tempChart = new ApexCharts(document.createElement('div'), invalidOptions);
        tempChart.render();
        log('❌ Invalid chart type should have thrown error');
    } catch (error) {
        log('✅ Error handling for invalid chart type: OK');
    }
}

// Browser Compatibility Testing
function testBrowserCompatibility() {
    log('🌐 Testing Browser Compatibility...');
    
    // WebGL support
    const canvas = document.createElement('canvas');
    const webglSupport = !!(canvas.getContext('webgl') || canvas.getContext('experimental-webgl'));
    document.getElementById('webgl-support').textContent = webglSupport ? 'Yes' : 'No';
    
    // Canvas support
    const canvasSupport = !!(canvas.getContext && canvas.getContext('2d'));
    document.getElementById('canvas-support').textContent = canvasSupport ? 'Yes' : 'No';
    
    // SVG support
    const svgSupport = !!(document.createElementNS && document.createElementNS('http://www.w3.org/2000/svg', 'svg').createSVGRect);
    document.getElementById('svg-support').textContent = svgSupport ? 'Yes' : 'No';
    
    // CSS3 support
    const css3Support = 'transform' in document.body.style || 'webkitTransform' in document.body.style;
    document.getElementById('css3-support').textContent = css3Support ? 'Yes' : 'No';
    
    log('✅ Browser compatibility check completed');
}

function detectBrowserInfo() {
    const userAgent = navigator.userAgent;
    let browserInfo = 'Unknown';
    
    if (userAgent.includes('Chrome')) browserInfo = 'Chrome';
    else if (userAgent.includes('Firefox')) browserInfo = 'Firefox';
    else if (userAgent.includes('Safari')) browserInfo = 'Safari';
    else if (userAgent.includes('Edge')) browserInfo = 'Edge';
    else if (userAgent.includes('Opera')) browserInfo = 'Opera';
    
    document.getElementById('browser-info').textContent = browserInfo;
}

// Utility Functions
function generateSampleLineData() {
    const categories = [];
    const data = [];
    const now = new Date();
    
    for (let i = 29; i >= 0; i--) {
        const date = new Date(now);
        date.setDate(date.getDate() - i);
        categories.push(date.toLocaleDateString());
        data.push(Math.floor(Math.random() * 100) + 20);
    }
    
    return { categories, data };
}

function clearResults() {
    document.getElementById('load-time').textContent = '--';
    document.getElementById('render-time').textContent = '--';
    document.getElementById('memory-usage').textContent = '--';
    document.getElementById('fps').textContent = '--';
    log('🗑️ Performance results cleared');
}

function clearLog() {
    testLog = [];
    document.getElementById('test-log').innerHTML = `
        <div>🚀 ApexCharts Migration Test Suite Initialized</div>
        <div>📅 Test Session Started: ${new Date().toLocaleString()}</div>
        <div>---</div>
    `;
}

function exportTestResults() {
    const results = {
        timestamp: new Date().toISOString(),
        browserInfo: document.getElementById('browser-info').textContent,
        performance: {
            loadTime: document.getElementById('load-time').textContent,
            renderTime: document.getElementById('render-time').textContent,
            memoryUsage: document.getElementById('memory-usage').textContent,
            fps: document.getElementById('fps').textContent
        },
        compatibility: {
            webgl: document.getElementById('webgl-support').textContent,
            canvas: document.getElementById('canvas-support').textContent,
            svg: document.getElementById('svg-support').textContent,
            css3: document.getElementById('css3-support').textContent
        },
        testLog: testLog
    };
    
    const blob = new Blob([JSON.stringify(results, null, 2)], { type: 'application/json' });
    const url = URL.createObjectURL(blob);
    const a = document.createElement('a');
    a.href = url;
    a.download = `apexcharts-test-results-${Date.now()}.json`;
    document.body.appendChild(a);
    a.click();
    document.body.removeChild(a);
    URL.revokeObjectURL(url);
    
    log('📤 Test results exported');
}
