/**
 * Admin JavaScript for Affiliate Hub
 */
(function($) {
    'use strict';

    var AffiliateHubAdmin = {
        init: function() {
            console.log('AffiliateHubAdmin initializing...');
            this.bindEvents();
            this.initCharts();
            this.initLinkPrefixPreview();
            this.initAffiliateLinksTable();
            console.log('AffiliateHubAdmin initialized');
        },

        bindEvents: function() {
            var self = this;
            
            // Link management (for list pages)
            $(document).on('click', '.test-link', function(e) { self.testLink.call(this, e); });
            $(document).on('click', '.copy-link', function(e) { self.copyLink.call(this, e); });
            
            // MetaBox buttons (edit post page) - use only delegation to avoid conflicts
            $(document).on('click', '#test-link-btn', function(e) { 
                e.stopPropagation(); // Prevent multiple triggers
                self.testLink.call(this, e); 
            });
            $(document).on('click', '#copy-url-btn', function(e) { 
                e.stopPropagation(); // Prevent multiple triggers
                self.copyLink.call(this, e); 
            });
            
            // Bulk actions
            $('#doaction, #doaction2').on('click', this.handleBulkAction);
            
            // Settings
            $(document).on('change', '#affiliate_hub_enable_click_tracking', this.toggleTrackingOptions);
        },

        initLinkPrefixPreview: function() {
            var prefixInput = $('#affiliate_hub_link_prefix');
            var linkPreview = $('#link-preview');
            
            if (prefixInput.length && linkPreview.length) {
                var baseUrl = window.location.protocol + '//' + window.location.hostname + '/';
                
                function updatePreview() {
                    var prefix = prefixInput.val().trim();
                    if (prefix) {
                        linkPreview.text(baseUrl + prefix + '/example-link/');
                    } else {
                        linkPreview.text(baseUrl + 'example-link/');
                    }
                }
                
                prefixInput.on('input keyup change paste', updatePreview);
                updatePreview();
            }
        },

        initCharts: function() {
            var ctx = document.getElementById('affiliate-hub-chart');
            if (!ctx) return;

            try {
                var data = JSON.parse(ctx.dataset.chartData || '[]');
                
                if (data.length === 0) {
                    ctx.parentElement.innerHTML = '<p>No data available</p>';
                    return;
                }

                if (window.affiliateHubChart) {
                    window.affiliateHubChart.destroy();
                }

                window.affiliateHubChart = new Chart(ctx, {
                    type: 'line',
                    data: {
                        labels: data.map(function(item) { return item.date; }),
                        datasets: [{
                            label: 'Clicks',
                            data: data.map(function(item) { return item.clicks; }),
                            borderColor: '#0073aa',
                            backgroundColor: 'rgba(0, 115, 170, 0.1)',
                            tension: 0.1
                        }]
                    },
                    options: {
                        responsive: true,
                        plugins: {
                            legend: { display: false }
                        },
                        scales: {
                            y: {
                                beginAtZero: true,
                                ticks: { stepSize: 1 }
                            }
                        }
                    }
                });
            } catch (error) {
                ctx.parentElement.innerHTML = '<p>Error loading chart data.</p>';
            }
        },

        testLink: function(e) {
            e.preventDefault();
            var url = $(this).data('url');
            
            console.log('Test Link clicked - URL:', url, 'Element:', this);
            
            if (url) {
                window.open(url, '_blank');
            } else {
                // Fallback - try to get URL from preview field
                var previewUrl = $('#cloaked_url_preview').val();
                if (previewUrl) {
                    console.log('Using preview URL:', previewUrl);
                    window.open(previewUrl, '_blank');
                } else {
                    alert('URL not available yet. Please make sure you have entered a link slug.');
                }
            }
        },

        copyLink: function(e) {
            e.preventDefault();
            var url = $(this).data('url');
            
            console.log('Copy Link clicked - URL:', url, 'Element:', this);
            
            if (!url) {
                // Fallback - try to get URL from preview field
                url = $('#cloaked_url_preview').val();
                console.log('Using preview URL for copy:', url);
            }
            
            if (url) {
                if (navigator.clipboard) {
                    navigator.clipboard.writeText(url).then(function() {
                        $('#copy-success').show().delay(2000).fadeOut();
                    }).catch(function(err) {
                        console.error('Clipboard API failed:', err);
                        AffiliateHubAdmin.fallbackCopyToClipboard(url);
                    });
                } else {
                    AffiliateHubAdmin.fallbackCopyToClipboard(url);
                }
            } else {
                alert('URL not available yet. Please make sure you have entered a link slug.');
            }
        },

        fallbackCopyToClipboard: function(text) {
            var temp = $('<input>');
            $('body').append(temp);
            temp.val(text).select();
            document.execCommand('copy');
            temp.remove();
            $('#copy-success').show().delay(2000).fadeOut();
        },

        handleBulkAction: function(e) {
            var action = $(this).prev('select').val();
            if (action === 'delete' || action === 'trash') {
                if (!confirm('Are you sure you want to perform this bulk action?')) {
                    e.preventDefault();
                    return false;
                }
            }
        },

        toggleTrackingOptions: function() {
            var isEnabled = $(this).is(':checked');
            $('.tracking-dependent').toggle(isEnabled);
        },

        initAffiliateLinksTable: function() {
            // Check if we're on the affiliate links page
            if (!$('body.post-type-affiliate_link').length) {
                return;
            }

            // Simple table layout fix - just ensure consistent column behavior
            var $table = $('.wp-list-table');
            if ($table.length) {
                // Remove any table-layout restrictions to let it flow naturally
                $table.css('table-layout', 'auto');
            }
        }
    };

    // Initialize when document is ready
    jQuery(document).ready(function($) {
        AffiliateHubAdmin.init();
    });

})(jQuery);
