/**
 * Advanced Multi-Select Component for Analytics Filters
 */
class AdvancedMultiSelect {
    constructor(containerId, type, options = {}) {
        this.containerId = containerId;
        this.type = type; // 'category' or 'tag'
        this.options = {
            placeholder: options.placeholder || 'Search...',
            maxSelections: options.maxSelections || null,
            onChange: options.onChange || (() => {}),
            ...options
        };
        
        this.selectedItems = new Map(); // Map<id, {id, name}>
        this.allOptions = new Map(); // Map<id, {id, name, count}>
        this.filteredOptions = new Map();
        this.isOpen = false;
        this.isInitialized = false;
        
        // Safely initialize
        this.safeInit();
    }
    
    safeInit() {
        try {
            this.container = document.getElementById(this.containerId);
            if (!this.container) {
                console.warn(`Multi-select container ${this.containerId} not found`);
                return;
            }
            
            // Find elements safely
            this.input = this.container.querySelector('.multi-select-input input');
            this.selectedContainer = this.container.querySelector('.selected-tags');
            this.dropdown = this.container.querySelector('.multi-select-dropdown');
            this.optionsContainer = this.container.querySelector('.dropdown-options');
            this.multiSelectInput = this.container.querySelector('.multi-select-input');
            
            // Check if all required elements exist
            if (!this.input || !this.selectedContainer || !this.dropdown || !this.optionsContainer || !this.multiSelectInput) {
                console.warn(`Missing required elements in ${this.containerId}`);
                return;
            }
            
            this.loadOptions();
            this.bindEvents();
            this.isInitialized = true;
            
            console.log(`✅ ${this.type} multi-select initialized successfully`);
        } catch (error) {
            console.error(`❌ Error initializing ${this.type} multi-select:`, error);
        }
    }
    
    loadOptions() {
        try {
            // Load options from DOM
            const options = this.container.querySelectorAll('.dropdown-option');
            options.forEach(option => {
                const id = option.dataset.value;
                const name = option.dataset.text;
                const countEl = option.querySelector('.option-count');
                const count = countEl ? parseInt(countEl.textContent.replace(/[()]/g, '')) : 0;
                
                if (id && name) {
                    this.allOptions.set(id, { id, name, count });
                }
            });
            
            this.filteredOptions = new Map(this.allOptions);
            console.log(`${this.type} options loaded:`, this.allOptions.size);
        } catch (error) {
            console.error(`Error loading ${this.type} options:`, error);
        }
    }
    
    bindEvents() {
        try {
            // Input focus/click
            this.input.addEventListener('focus', () => this.open());
            this.multiSelectInput.addEventListener('click', (e) => {
                if (e.target === this.input) return;
                this.open();
                this.input.focus();
            });
            
            // Input typing
            this.input.addEventListener('input', (e) => this.handleSearch(e.target.value));
            
            // Quick filter buttons
            const quickBtns = this.container.querySelectorAll('.quick-filter-btn');
            quickBtns.forEach(btn => {
                btn.addEventListener('click', (e) => {
                    e.stopPropagation();
                    this.toggleSelection(btn.dataset.value, btn.textContent.trim());
                });
            });
            
            // Option clicks
            this.optionsContainer.addEventListener('click', (e) => {
                const option = e.target.closest('.dropdown-option');
                if (option) {
                    e.stopPropagation();
                    this.toggleSelection(option.dataset.value, option.dataset.text);
                }
            });
            
            // Close on outside click
            document.addEventListener('click', (e) => {
                if (!this.container.contains(e.target)) {
                    this.close();
                }
            });
            
            // Keyboard navigation
            this.input.addEventListener('keydown', (e) => this.handleKeydown(e));
            
        } catch (error) {
            console.error(`Error binding ${this.type} events:`, error);
        }
    }
    
    handleSearch(query) {
        try {
            const lowerQuery = query.toLowerCase();
            this.filteredOptions.clear();
            
            if (!query.trim()) {
                this.filteredOptions = new Map(this.allOptions);
            } else {
                for (const [id, item] of this.allOptions) {
                    if (item.name.toLowerCase().includes(lowerQuery)) {
                        this.filteredOptions.set(id, item);
                    }
                }
            }
            
            this.renderOptions();
        } catch (error) {
            console.error(`Error in ${this.type} search:`, error);
        }
    }
    
    renderOptions() {
        try {
            this.optionsContainer.innerHTML = '';
            
            if (this.filteredOptions.size === 0) {
                this.optionsContainer.innerHTML = '<div class="dropdown-empty">No options found</div>';
                return;
            }
            
            for (const [id, item] of this.filteredOptions) {
                const isSelected = this.selectedItems.has(id);
                const option = document.createElement('div');
                option.className = `dropdown-option ${isSelected ? 'selected' : ''}`;
                option.dataset.value = id;
                option.dataset.text = item.name;
                
                option.innerHTML = `
                    <span class="option-name">${this.escapeHtml(item.name)}</span>
                    <span class="option-count">(${item.count})</span>
                `;
                
                this.optionsContainer.appendChild(option);
            }
        } catch (error) {
            console.error(`Error rendering ${this.type} options:`, error);
        }
    }
    
    toggleSelection(id, name) {
        try {
            if (!id || !name) return;
            
            if (this.selectedItems.has(id)) {
                this.removeSelection(id);
            } else {
                if (this.options.maxSelections && this.selectedItems.size >= this.options.maxSelections) {
                    return; // Max selections reached
                }
                this.addSelection(id, name);
            }
            
            this.input.value = '';
            this.handleSearch('');
            this.options.onChange(this.getSelectedIds());
        } catch (error) {
            console.error(`Error toggling ${this.type} selection:`, error);
        }
    }
    
    addSelection(id, name) {
        this.selectedItems.set(id, { id, name });
        this.renderSelectedTags();
        this.updateQuickFilters();
    }
    
    removeSelection(id) {
        this.selectedItems.delete(id);
        this.renderSelectedTags();
        this.updateQuickFilters();
    }
    
    renderSelectedTags() {
        try {
            this.selectedContainer.innerHTML = '';
            
            for (const [id, item] of this.selectedItems) {
                const tag = document.createElement('div');
                tag.className = 'selected-tag';
                tag.innerHTML = `
                    <span>${this.escapeHtml(item.name)}</span>
                    <span class="remove-tag" data-id="${id}">×</span>
                `;
                
                const removeBtn = tag.querySelector('.remove-tag');
                removeBtn.addEventListener('click', (e) => {
                    e.stopPropagation();
                    this.removeSelection(id);
                    this.options.onChange(this.getSelectedIds());
                });
                
                this.selectedContainer.appendChild(tag);
            }
            
            // Update placeholder
            this.input.placeholder = this.selectedItems.size > 0 ? 
                `${this.selectedItems.size} selected` : 
                this.options.placeholder;
        } catch (error) {
            console.error(`Error rendering ${this.type} selected tags:`, error);
        }
    }
    
    updateQuickFilters() {
        try {
            const quickBtns = this.container.querySelectorAll('.quick-filter-btn');
            quickBtns.forEach(btn => {
                const isSelected = this.selectedItems.has(btn.dataset.value);
                btn.style.background = isSelected ? '#0073aa' : '#e7f3ff';
                btn.style.color = isSelected ? 'white' : '#0056b3';
                btn.style.borderColor = isSelected ? '#0073aa' : '#b3d7ff';
            });
        } catch (error) {
            console.error(`Error updating ${this.type} quick filters:`, error);
        }
    }
    
    open() {
        if (this.isOpen || !this.isInitialized) return;
        
        try {
            this.isOpen = true;
            this.dropdown.style.display = 'block';
            this.multiSelectInput.classList.add('active');
            this.renderOptions();
            
            console.log(`${this.type} dropdown opened`);
        } catch (error) {
            console.error(`Error opening ${this.type} dropdown:`, error);
        }
    }
    
    close() {
        if (!this.isOpen || !this.isInitialized) return;
        
        try {
            this.isOpen = false;
            this.dropdown.style.display = 'none';
            this.multiSelectInput.classList.remove('active');
            this.input.value = '';
            this.handleSearch('');
            
            console.log(`${this.type} dropdown closed`);
        } catch (error) {
            console.error(`Error closing ${this.type} dropdown:`, error);
        }
    }
    
    handleKeydown(e) {
        try {
            switch (e.key) {
                case 'Escape':
                    this.close();
                    break;
                case 'Enter':
                    e.preventDefault();
                    const firstOption = this.optionsContainer.querySelector('.dropdown-option:not(.selected)');
                    if (firstOption) {
                        this.toggleSelection(firstOption.dataset.value, firstOption.dataset.text);
                    }
                    break;
            }
        } catch (error) {
            console.error(`Error handling ${this.type} keydown:`, error);
        }
    }
    
    getSelectedIds() {
        return Array.from(this.selectedItems.keys());
    }
    
    setSelectedItems(ids) {
        try {
            this.selectedItems.clear();
            
            ids.forEach(id => {
                const item = this.allOptions.get(id.toString());
                if (item) {
                    this.selectedItems.set(id.toString(), item);
                }
            });
            
            this.renderSelectedTags();
            this.updateQuickFilters();
            this.renderOptions();
        } catch (error) {
            console.error(`Error setting ${this.type} selected items:`, error);
        }
    }
    
    clear() {
        try {
            this.selectedItems.clear();
            this.renderSelectedTags();
            this.updateQuickFilters();
            this.renderOptions();
            this.options.onChange([]);
        } catch (error) {
            console.error(`Error clearing ${this.type} selections:`, error);
        }
    }
    
    escapeHtml(unsafe) {
        return unsafe
            .replace(/&/g, "&amp;")
            .replace(/</g, "&lt;")
            .replace(/>/g, "&gt;")
            .replace(/"/g, "&quot;")
            .replace(/'/g, "&#039;");
    }
}

// Global variables for multi-selects
window.categoryMultiSelect = null;
window.tagMultiSelect = null;

// Initialize multi-selects when DOM is ready
jQuery(document).ready(function($) {
    // Small delay to ensure all elements are rendered
    setTimeout(function() {
        try {
            // Initialize category multi-select
            if (document.getElementById('category-multi-select')) {
                window.categoryMultiSelect = new AdvancedMultiSelect('category-multi-select', 'category', {
                    placeholder: 'Search categories...',
                    onChange: (selectedIds) => {
                        console.log('📂 Categories changed:', selectedIds);
                        if (window.analyticsDashboard) {
                            window.analyticsDashboard.loadDashboardData();
                        }
                    }
                });
            } else {
                console.warn('Category multi-select container not found');
            }
            
            // Initialize tag multi-select
            if (document.getElementById('tag-multi-select')) {
                window.tagMultiSelect = new AdvancedMultiSelect('tag-multi-select', 'tag', {
                    placeholder: 'Search tags...',
                    onChange: (selectedIds) => {
                        console.log('🏷️ Tags changed:', selectedIds);
                        if (window.analyticsDashboard) {
                            window.analyticsDashboard.loadDashboardData();
                        }
                    }
                });
            } else {
                console.warn('Tag multi-select container not found');
            }
            
            console.log('🎯 Advanced Multi-Select initialization complete');
        } catch (error) {
            console.error('❌ Error initializing multi-selects:', error);
        }
    }, 500);
});

console.log('🎯 Advanced Multi-Select script loaded');
