/**
 * Enhanced Analytics Dashboard JavaScript - ApexCharts Version
 * Provides comprehensive analytics with modern ApexCharts visualization
 */

class AnalyticsDashboard {
    constructor() {
        this.charts = {};
        this.currentFilters = {};
        this.isLoading = false;
        this.apexchartsReady = false;
        
        this.init();
    }
    
    init() {
        // Wait for DOM and ApexCharts to be ready
        if (document.readyState === 'loading') {
            document.addEventListener('DOMContentLoaded', () => this.initializeWhenReady());
        } else {
            this.initializeWhenReady();
        }
    }
    
    initializeWhenReady() {
        // Check if ApexCharts is available
        if (typeof ApexCharts !== 'undefined') {
            this.apexchartsReady = true;
            console.log('ApexCharts loaded successfully');
            this.bindEvents();
        } else {
            // Wait for ApexCharts to load
            let attempts = 0;
            const checkApex = () => {
                attempts++;
                if (typeof ApexCharts !== 'undefined') {
                    this.apexchartsReady = true;
                    console.log('ApexCharts loaded after', attempts, 'attempts');
                    this.bindEvents();
                } else if (attempts < 20) {
                    setTimeout(checkApex, 100);
                } else {
                    console.error('ApexCharts failed to load after 2 seconds');
                    this.showError('Chart library failed to load. Please refresh the page.');
                }
            };
            checkApex();
        }
    }
    
    bindEvents() {
        console.log('Binding events for Analytics Dashboard');
        
        // Show content and hide loading
        this.showContent();
        
        // Filter controls
        const dateRange = document.getElementById('date-range');
        if (dateRange) {
            dateRange.addEventListener('change', (e) => {
                this.toggleCustomDateRange(e.target.value === 'custom');
            });
        }
        
        const applyFilters = document.getElementById('apply-filters');
        if (applyFilters) {
            applyFilters.addEventListener('click', () => {
                this.applyFilters();
            });
        }
        
        const exportData = document.getElementById('export-data');
        if (exportData) {
            exportData.addEventListener('click', () => {
                this.exportData();
            });
        }
        
        // Load initial data
        this.loadAnalyticsData();
    }
    
    showContent() {
        const loading = document.getElementById('analytics-loading');
        const content = document.getElementById('analytics-content');
        
        if (loading) loading.style.display = 'none';
        if (content) content.style.display = 'block';
    }
    
    showError(message) {
        console.error('Analytics Error:', message);
        
        // Show error in loading area
        const loading = document.getElementById('analytics-loading');
        if (loading) {
            loading.innerHTML = `<div class="error"><p><strong>Error:</strong> ${this.escapeHtml(message)}</p></div>`;
            loading.style.display = 'block';
        }
        
        // Also show in content area if it exists
        const content = document.getElementById('analytics-content');
        if (content) {
            const errorDiv = document.createElement('div');
            errorDiv.className = 'analytics-error';
            errorDiv.innerHTML = `<p><strong>Error:</strong> ${this.escapeHtml(message)}</p>`;
            content.insertBefore(errorDiv, content.firstChild);
        }
    }
    
    // Simple method for escaping HTML
    escapeHtml(text) {
        if (!text) return '';
        const div = document.createElement('div');
        div.textContent = text;
        return div.innerHTML;
    }
    
    // Placeholder methods for functionality
    toggleCustomDateRange(show) {
        const customRange = document.querySelector('.custom-date-range');
        if (customRange) {
            customRange.style.display = show ? 'block' : 'none';
        }
    }
    
    applyFilters() {
        console.log('Applying filters...');
        this.loadAnalyticsData();
    }
    
    exportData() {
        console.log('Exporting data...');
        // Create form and submit for export
        const form = document.createElement('form');
        form.method = 'POST';
        form.action = affiliateHubAnalytics.ajax_url;
        
        // Add action field
        const actionField = document.createElement('input');
        actionField.type = 'hidden';
        actionField.name = 'action';
        actionField.value = 'affiliate_hub_export_analytics';
        form.appendChild(actionField);
        
        // Add nonce field
        const nonceField = document.createElement('input');
        nonceField.type = 'hidden';
        nonceField.name = 'nonce';
        nonceField.value = affiliateHubAnalytics.nonce;
        form.appendChild(nonceField);
        
        // Submit form
        document.body.appendChild(form);
        form.submit();
        document.body.removeChild(form);
    }
    
    loadAnalyticsData() {
        console.log('Loading analytics data...');
        
        if (this.isLoading) {
            console.log('Already loading, skipping...');
            return;
        }
        
        this.isLoading = true;
        
        // Show loading indicator
        const loadingIndicator = document.getElementById('loading-indicator');
        if (loadingIndicator) {
            loadingIndicator.style.display = 'block';
        }
        
        // Collect filter data
        const formData = new FormData();
        formData.append('action', 'affiliate_hub_analytics_data');
        formData.append('nonce', affiliateHubAnalytics.nonce);
        
        // Add filter values
        const dateRange = document.getElementById('date-range');
        if (dateRange) {
            formData.append('date_range', dateRange.value);
        }
        
        const dateFrom = document.getElementById('date-from');
        const dateTo = document.getElementById('date-to');
        if (dateFrom && dateTo && dateRange && dateRange.value === 'custom') {
            formData.append('date_from', dateFrom.value);
            formData.append('date_to', dateTo.value);
        }
        
        const linkFilter = document.getElementById('link-filter');
        if (linkFilter) {
            formData.append('link_id', linkFilter.value);
        }
        
        const uniqueOnly = document.getElementById('unique-only');
        if (uniqueOnly) {
            formData.append('unique_only', uniqueOnly.checked ? '1' : '0');
        }
        
        // Send AJAX request
        fetch(affiliateHubAnalytics.ajax_url, {
            method: 'POST',
            body: formData
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                this.updateDashboard(data.data);
            } else {
                throw new Error(data.data || 'Unknown error');
            }
        })
        .catch(error => {
            console.error('Analytics loading error:', error);
            this.showError('Failed to load analytics data: ' + error.message);
        })
        .finally(() => {
            this.isLoading = false;
            
            // Hide loading indicator
            if (loadingIndicator) {
                loadingIndicator.style.display = 'none';
            }
        });
    }
    
    updateDashboard(data) {
        console.log('Updating dashboard with data:', data);
        
        // Update summary cards
        this.updateSummaryCards(data.summary || {});
        
        // Update charts
        this.updateCharts(data);
        
        // Update tables
        this.updateTables(data);
    }
    
    updateSummaryCards(summary) {
        // Update total clicks
        const totalClicks = document.getElementById('total-clicks');
        if (totalClicks) {
            totalClicks.textContent = summary.total_clicks || '0';
        }
        
        // Update unique clicks
        const uniqueClicks = document.getElementById('unique-clicks');
        if (uniqueClicks) {
            uniqueClicks.textContent = summary.unique_clicks || '0';
        }
        
        // Update other summary values as needed
        const avgProcessingTime = document.getElementById('avg-processing-time');
        if (avgProcessingTime) {
            avgProcessingTime.textContent = (summary.avg_processing_time || '0') + ' ms';
        }
        
        const geoCacheRate = document.getElementById('geo-cache-rate');
        if (geoCacheRate) {
            geoCacheRate.textContent = (summary.geo_cache_rate || '0') + '%';
        }
    }
    
    updateCharts(data) {
        // Simple chart updates - can be expanded based on actual data structure
        console.log('Updating charts...');
        
        // Create or update clicks chart
        if (data.clicks_chart) {
            this.createClicksChart(data.clicks_chart);
        }
        
        // Create or update top links chart
        if (data.top_links_chart) {
            this.createTopLinksChart(data.top_links_chart);
        }
    }
    
    createClicksChart(chartData) {
        const chartElement = document.getElementById('clicks-chart');
        if (!chartElement || !this.apexchartsReady) return;
        
        const options = {
            series: [{
                name: 'Clicks',
                data: chartData.data || []
            }],
            chart: {
                type: 'line',
                height: 350
            },
            xaxis: {
                categories: chartData.categories || []
            },
            title: {
                text: 'Clicks Over Time'
            }
        };
        
        if (this.charts.clicksChart) {
            this.charts.clicksChart.updateOptions(options);
        } else {
            this.charts.clicksChart = new ApexCharts(chartElement, options);
            this.charts.clicksChart.render();
        }
    }
    
    createTopLinksChart(chartData) {
        const chartElement = document.getElementById('top-links-chart');
        if (!chartElement || !this.apexchartsReady) return;
        
        const options = {
            series: chartData.data || [],
            chart: {
                type: 'bar',
                height: 350
            },
            xaxis: {
                categories: chartData.categories || []
            },
            title: {
                text: 'Top Performing Links'
            }
        };
        
        if (this.charts.topLinksChart) {
            this.charts.topLinksChart.updateOptions(options);
        } else {
            this.charts.topLinksChart = new ApexCharts(chartElement, options);
            this.charts.topLinksChart.render();
        }
    }
    
    updateTables(data) {
        // Update top links table
        if (data.top_links) {
            this.renderTopLinksTable(data.top_links);
        }
        
        // Update recent activity table
        if (data.recent_activity) {
            this.renderRecentActivityTable(data.recent_activity);
        }
    }
    
    renderTopLinksTable(topLinks) {
        const tableBody = document.querySelector('#top-links-table tbody');
        if (!tableBody) return;
        
        tableBody.innerHTML = '';
        
        topLinks.forEach(link => {
            const row = document.createElement('tr');
            
            // Create clickable link
            const linkTitle = link.link_title || 'Untitled';
            const editUrl = `/wp-admin/post.php?post=${link.link_id}&action=edit`;
            
            row.innerHTML = `
                <td>
                    <a href="${this.escapeHtml(editUrl)}" target="_blank" class="affiliate-link-title">
                        ${this.escapeHtml(linkTitle)} 
                        <span class="dashicons dashicons-external" style="font-size: 12px; margin-left: 5px;"></span>
                    </a>
                    ${link.destination_url ? `<br><small>${this.escapeHtml(link.destination_url)}</small>` : ''}
                </td>
                <td>${parseInt(link.total_clicks) || 0}</td>
                <td>${parseInt(link.unique_clicks) || 0}</td>
                <td>${parseFloat(link.ctr_rate) || 0}%</td>
            `;
            
            tableBody.appendChild(row);
        });
    }
    
    renderRecentActivityTable(recentActivity) {
        const tableBody = document.querySelector('#recent-activity-table tbody');
        if (!tableBody) return;
        
        tableBody.innerHTML = '';
        
        recentActivity.forEach(activity => {
            const row = document.createElement('tr');
            
            // Format date
            const date = new Date(activity.clicked_at);
            const formattedDate = date.toLocaleDateString() + ' ' + date.toLocaleTimeString();
            
            // Create clickable link
            const linkTitle = activity.link_title || 'Untitled';
            const editUrl = `/wp-admin/post.php?post=${activity.link_id}&action=edit`;
            
            row.innerHTML = `
                <td>${this.escapeHtml(formattedDate)}</td>
                <td>
                    <a href="${this.escapeHtml(editUrl)}" target="_blank" class="affiliate-link-title">
                        ${this.escapeHtml(linkTitle)}
                        <span class="dashicons dashicons-external" style="font-size: 12px; margin-left: 5px;"></span>
                    </a>
                </td>
                <td>${this.escapeHtml(activity.ip_address || 'Unknown')}</td>
                <td>${this.escapeHtml(activity.country || 'Unknown')}</td>
                <td>${this.escapeHtml(activity.browser || 'Unknown')}</td>
                <td>${this.escapeHtml(activity.device_type || 'Unknown')}</td>
            `;
            
            tableBody.appendChild(row);
        });
    }
}

// Initialize when DOM is ready
document.addEventListener('DOMContentLoaded', () => {
    console.log('Initializing Analytics Dashboard');
    console.log('affiliateHubAnalytics available:', typeof affiliateHubAnalytics !== 'undefined');
    if (typeof affiliateHubAnalytics !== 'undefined') {
        console.log('affiliateHubAnalytics data:', affiliateHubAnalytics);
    }
    new AnalyticsDashboard();
});

// Also initialize immediately if DOM is already loaded
if (document.readyState !== 'loading') {
    console.log('DOM already loaded, initializing Analytics Dashboard immediately');
    if (typeof affiliateHubAnalytics !== 'undefined') {
        new AnalyticsDashboard();
    } else {
        console.error('affiliateHubAnalytics not available in global init');
    }
}
