/**
 * Analytics Dashboard JavaScript - Clean Working Version
 */

console.log('🚀 ANALYTICS DASHBOARD SCRIPT LOADING');

// Wait for jQuery and ensure everything is available
jQuery(document).ready(function($) {
    console.log('✅ jQuery ready - Starting Analytics Dashboard');
    
    // Verify dependencies
    if (typeof affiliateHubAnalytics === 'undefined') {
        console.error('❌ affiliateHubAnalytics not available');
        return;
    }
    
    if (typeof ApexCharts === 'undefined') {
        console.error('❌ ApexCharts not available');
        return;
    }
    
    console.log('✅ All dependencies ready');
    
    // Visual confirmation that script is working
    $('body').append('<div id="analytics-working" style="position:fixed;top:60px;right:10px;background:blue;color:white;padding:5px;z-index:9999;font-size:10px;">Analytics Loading...</div>');
    
    // Initialize Analytics Dashboard
    window.analyticsDashboard = new AnalyticsDashboard();
    
    setTimeout(() => {
        $('#analytics-working').remove();
    }, 3000);
});

class AnalyticsDashboard {
    constructor() {
        console.log('📊 AnalyticsDashboard constructor started');
        this.isLoading = false;
        this.isLoadingActivity = false;
        this.charts = {};
        this.currentActivityPage = 1;
        this.currentActivitySort = { column: 'click_time', order: 'DESC' };
        this.initDashboard();
    }
    
    initDashboard() {
        console.log('🔄 Initializing dashboard...');
        
        // Hide loading message and show content
        const loadingEl = document.getElementById('analytics-loading');
        const contentEl = document.getElementById('analytics-content');
        
        if (loadingEl) {
            loadingEl.style.display = 'none';
            console.log('✅ Loading message hidden');
        }
        
        if (contentEl) {
            contentEl.style.display = 'block';
            console.log('✅ Content shown');
        }
        
        // Initialize activity controls
        this.initActivityControls();
        
        // Load dashboard data
        this.loadDashboardData();
    }
    
    loadDashboardData() {
        console.log('📡 Loading dashboard data...');
        
        if (this.isLoading) {
            console.log('⏳ Already loading, skipping...');
            return;
        }
        
        this.isLoading = true;
        this.showLoadingState();
        
        const filterData = this.getFilters();
        
        // Check if filters are valid (getFilters returns null for invalid custom range)
        if (filterData === null) {
            console.log('⚠️ Invalid filters - aborting request');
            this.isLoading = false;
            this.hideLoadingState();
            return;
        }
        
        const ajaxData = {
            action: 'affiliate_hub_analytics_data',
            nonce: affiliateHubAnalytics.nonce,
            ...filterData
        };
        
        console.log('📡 Sending AJAX data:', ajaxData);
        
        jQuery.ajax({
            url: affiliateHubAnalytics.ajax_url,
            type: 'POST',
            data: ajaxData,
            success: (response) => {
                console.log('✅ Dashboard data loaded:', response);
                
                if (response.success) {
                    this.renderDashboard(response.data);
                } else {
                    console.error('❌ Dashboard data error:', response.data);
                    this.showError('Failed to load dashboard data: ' + (response.data || 'Unknown error'));
                }
                
                this.isLoading = false;
                this.hideLoadingState();
            },
            error: (xhr, status, error) => {
                console.error('❌ AJAX error:', {xhr, status, error});
                this.showError('Network error loading dashboard data');
                this.isLoading = false;
                this.hideLoadingState();
            }
        });
    }
    
    renderDashboard(data) {
        console.log('🎨 Rendering dashboard with data:', data);
        console.log('🔍 Data structure check:');
        console.log('  - Summary:', data.summary);
        console.log('  - Trends count:', (data.trends || []).length);
        console.log('  - Top links count:', (data.top_links || []).length);
        console.log('  - Recent activity count:', (data.recent_activity || []).length);
        
        try {
            // Render summary stats first
            this.renderSummaryStats(data.summary || {});
            
            // Render charts
            this.renderClicksChart(data.trends || []);
            this.renderTopLinksChart(data.top_links || []);
            
            // Render tables
            this.renderTopLinksTable(data.top_links || []);
            this.renderRecentActivityTable(data.recent_activity || []);
            
            console.log('✅ Dashboard rendered successfully');
        } catch (error) {
            console.error('❌ Error rendering dashboard:', error);
            this.showError('Error rendering dashboard: ' + error.message);
        }
    }
    
    renderSummaryStats(summary) {
        console.log('📊 Rendering summary stats:', summary);
        
        // Update summary cards using ID selectors
        const stats = [
            { id: 'total-clicks', value: summary.total_clicks || 0 },
            { id: 'unique-clicks', value: summary.unique_clicks || 0 },
            { id: 'unique-sessions', value: summary.unique_sessions || 0 },
            { id: 'avg-processing-time', value: Math.round(summary.avg_processing_time || 0) },
            { id: 'geo-cache-rate', value: Math.round(summary.geo_cache_hit_rate || 0) }
        ];
        
        stats.forEach(stat => {
            const element = document.getElementById(stat.id);
            if (element) {
                element.textContent = stat.value;
                console.log(`✅ Updated ${stat.id}: ${stat.value}`);
            } else {
                console.warn(`❌ Element #${stat.id} not found`);
            }
        });
        
        console.log('✅ Summary stats rendered');
    }
    
    renderClicksChart(data) {
        console.log('📈 Rendering clicks chart with data:', data);
        
        const chartElement = document.getElementById('clicks-chart');
        if (!chartElement) {
            console.error('❌ Clicks chart element not found');
            return;
        }
        
        const options = {
            series: [{
                name: 'Total Clicks',
                data: data.map(item => ({
                    x: item.date,
                    y: parseInt(item.total_clicks) || 0
                }))
            }, {
                name: 'Unique Clicks',
                data: data.map(item => ({
                    x: item.date,
                    y: parseInt(item.unique_clicks) || 0
                }))
            }],
            chart: {
                type: 'area',
                height: 350,
                toolbar: { show: false }
            },
            colors: ['#007cba', '#00a0d2'],
            dataLabels: { enabled: false },
            stroke: { curve: 'smooth', width: 2 },
            xaxis: {
                type: 'datetime',
                labels: { format: 'MMM dd' }
            },
            yaxis: {
                title: { text: 'Clicks' }
            }
        };
        
        if (this.charts.clicks) {
            this.charts.clicks.destroy();
        }
        
        this.charts.clicks = new ApexCharts(chartElement, options);
        this.charts.clicks.render();
        console.log('✅ Clicks chart rendered');
    }
    
    renderTopLinksChart(data) {
        console.log('📊 Rendering top links chart with data:', data);
        
        const chartElement = document.getElementById('top-links-chart');
        if (!chartElement) {
            console.error('❌ Top links chart element not found');
            return;
        }
        
        const options = {
            series: data.map(item => parseInt(item.total_clicks) || 0),
            chart: {
                type: 'donut',
                height: 350
            },
            labels: data.map(item => item.link_title || 'Unknown'),
            colors: ['#007cba', '#00a0d2', '#0073aa', '#005177', '#003f5c']
        };
        
        if (this.charts.topLinks) {
            this.charts.topLinks.destroy();
        }
        
        this.charts.topLinks = new ApexCharts(chartElement, options);
        this.charts.topLinks.render();
        console.log('✅ Top links chart rendered');
    }
    
    renderTopLinksTable(data) {
        console.log('📋 Rendering top links table with data:', data);
        
        const tableBody = document.querySelector('#top-links-table tbody');
        if (!tableBody) {
            console.error('❌ Top links table body not found');
            return;
        }
        
        tableBody.innerHTML = '';
        
        if (data.length === 0) {
            tableBody.innerHTML = '<tr><td colspan="6">No data available</td></tr>';
            return;
        }
        
        data.forEach(link => {
            const row = document.createElement('tr');
            const editUrl = `${window.location.origin}/wp-admin/post.php?post=${link.link_id}&action=edit`;
            const destinationUrl = link.destination_url || 'Not set';
            const lastClick = link.last_click ? new Date(link.last_click).toLocaleString() : 'Never';
            
            row.innerHTML = `
                <td class="link-title"><a href="${editUrl}" target="_blank" title="Edit link">${link.link_title || 'Unknown'}</a></td>
                <td class="url-cell"><span title="${destinationUrl}">${destinationUrl.length > 50 ? destinationUrl.substring(0, 50) + '...' : destinationUrl}</span></td>
                <td class="number">${parseInt(link.total_clicks) || 0}</td>
                <td class="number">${parseInt(link.unique_clicks) || 0}</td>
                <td class="number">${parseFloat(link.ctr_rate) || 0}%</td>
                <td class="date-cell"><span title="${link.last_click || 'Never'}">${lastClick}</span></td>
            `;
            tableBody.appendChild(row);
        });
        
        console.log('✅ Top links table rendered with', data.length, 'rows');
    }
    
    renderRecentActivityTable(data) {
        console.log('📋 Initial recent activity render - loading paginated data');
        console.log('📋 Data passed to renderRecentActivityTable:', data);
        
        // For initial load, trigger the new paginated table load
        console.log('🔄 Calling loadRecentActivity...');
        this.loadRecentActivity();
    }
    
    getDateRange() {
        const select = document.getElementById('date-range');
        return select ? select.value : '30';
    }
    
    getFilters() {
        console.log('🔍 Getting filters...');
        
        const filters = {
            date_range: this.getDateRange()
        };
        
        console.log('📅 Date range:', filters.date_range);
        
        // Add link filter
        const linkFilter = document.getElementById('link-filter');
        if (linkFilter) {
            if (linkFilter.value) {
                filters.link_id = linkFilter.value;
                console.log('🔗 Link filter:', filters.link_id);
            }
        } else {
            console.warn('⚠️ Link filter element not found');
        }
        
        // Add category filter (multi-select)
        if (window.categoryMultiSelect && window.categoryMultiSelect.isInitialized) {
            const selectedCategoryIds = window.categoryMultiSelect.getSelectedIds();
            if (selectedCategoryIds.length > 0) {
                filters.category_ids = selectedCategoryIds;
                console.log('🏷️ Category filter:', filters.category_ids);
            }
        } else {
            console.log('ℹ️ Category multi-select not available or not initialized');
            // Check if there's a simple category select fallback
            const categorySelect = document.getElementById('category-filter');
            if (categorySelect && categorySelect.value) {
                filters.category_ids = [categorySelect.value];
                console.log('🏷️ Category filter (fallback):', filters.category_ids);
            }
        }
        
        // Add tag filter (multi-select)
        if (window.tagMultiSelect && window.tagMultiSelect.isInitialized) {
            const selectedTagIds = window.tagMultiSelect.getSelectedIds();
            if (selectedTagIds.length > 0) {
                filters.tag_ids = selectedTagIds;
                console.log('🏷️ Tag filter:', filters.tag_ids);
            }
        } else {
            console.log('ℹ️ Tag multi-select not available or not initialized');
            // Check if there's a simple tag select fallback
            const tagSelect = document.getElementById('tag-filter');
            if (tagSelect && tagSelect.value) {
                filters.tag_ids = [tagSelect.value];
                console.log('🏷️ Tag filter (fallback):', filters.tag_ids);
            }
        }
        
        // Add unique only filter
        const uniqueOnly = document.getElementById('unique-only');
        if (uniqueOnly) {
            if (uniqueOnly.checked) {
                filters.unique_only = '1';
                console.log('✅ Unique only filter:', filters.unique_only);
            }
        } else {
            console.warn('⚠️ Unique only checkbox not found');
        }
        
        // Add custom date range if selected
        if (filters.date_range === 'custom') {
            const dateFrom = document.getElementById('date-from');
            const dateTo = document.getElementById('date-to');
            
            if (dateFrom && dateFrom.value) {
                filters.date_from = dateFrom.value;
                console.log('📅 Custom date from:', filters.date_from);
            } else {
                console.warn('⚠️ Date from missing or empty');
            }
            
            if (dateTo && dateTo.value) {
                filters.date_to = dateTo.value;
                console.log('📅 Custom date to:', filters.date_to);
            } else {
                console.warn('⚠️ Date to missing or empty');
            }
            
            // If custom is selected but dates are missing, don't proceed
            if (!filters.date_from || !filters.date_to) {
                console.log('⚠️ Custom range selected but dates missing - skipping request');
                return null; // Signal that we shouldn't proceed with the request
            }
        }
        
        console.log('📅 Final filters:', filters);
        return filters;
    }
    
    showLoadingState() {
        const charts = document.querySelectorAll('.chart-container');
        const tables = document.querySelectorAll('.table-container');
        const summaryCards = document.querySelectorAll('.summary-card');
        
        [...charts, ...tables, ...summaryCards].forEach(container => {
            container.style.opacity = '0.5';
        });
        
        // Show loading indicator on Apply button
        const applyBtn = document.getElementById('apply-filters');
        if (applyBtn) {
            applyBtn.textContent = 'Loading...';
            applyBtn.disabled = true;
        }
    }
    
    hideLoadingState() {
        const charts = document.querySelectorAll('.chart-container');
        const tables = document.querySelectorAll('.table-container');
        const summaryCards = document.querySelectorAll('.summary-card');
        
        [...charts, ...tables, ...summaryCards].forEach(container => {
            container.style.opacity = '1';
        });
        
        // Restore Apply button
        const applyBtn = document.getElementById('apply-filters');
        if (applyBtn) {
            applyBtn.textContent = 'Apply Filters';
            applyBtn.disabled = false;
        }
    }
    
    showError(message) {
        console.error('🚨 Dashboard Error:', message);
        
        const errorContainer = document.createElement('div');
        errorContainer.className = 'notice notice-error';
        errorContainer.innerHTML = `<p><strong>Analytics Error:</strong> ${message}</p>`;
        
        const contentEl = document.getElementById('analytics-content');
        if (contentEl) {
            contentEl.insertBefore(errorContainer, contentEl.firstChild);
        }
    }
}

// Setup event handlers
jQuery(document).ready(function($) {
    console.log('🔧 Setting up event handlers...');
    
    // Apply Filters button
    $('#apply-filters').on('click', function() {
        console.log('🔄 Apply filters clicked');
        if (window.analyticsDashboard) {
            try {
                window.analyticsDashboard.loadDashboardData();
            } catch (error) {
                console.error('❌ Error in loadDashboardData:', error);
            }
        } else {
            console.error('❌ analyticsDashboard not available');
        }
    });
    
    // Date range change handler
    $('#date-range').on('change', function() {
        const value = this.value;
        console.log('📅 Date range changed to:', value);
        
        try {
            // Show/hide custom date range inputs
            const customRange = $('.custom-date-range');
            if (value === 'custom') {
                customRange.show();
                console.log('✅ Custom date range shown');
                // Don't auto-reload for custom, wait for dates to be set
            } else {
                customRange.hide();
                console.log('✅ Custom date range hidden');
                // Auto-reload when not custom
                if (window.analyticsDashboard) {
                    window.analyticsDashboard.loadDashboardData();
                }
            }
        } catch (error) {
            console.error('❌ Error in date range handler:', error);
        }
    });
    
    // Link filter change
    $('#link-filter').on('change', function() {
        console.log('🔗 Link filter changed to:', this.value);
        try {
            if (window.analyticsDashboard) {
                window.analyticsDashboard.loadDashboardData();
            }
        } catch (error) {
            console.error('❌ Error in link filter handler:', error);
        }
    });
    
    // Unique only checkbox
    $('#unique-only').on('change', function() {
        console.log('✅ Unique only changed to:', this.checked);
        try {
            if (window.analyticsDashboard) {
                window.analyticsDashboard.loadDashboardData();
            }
        } catch (error) {
            console.error('❌ Error in unique only handler:', error);
        }
    });
    
    // Custom date inputs
    $('#date-from, #date-to').on('change', function() {
        console.log('📅 Custom date changed:', this.id, this.value);
        try {
            // Only auto-reload if both dates are set
            const dateFrom = $('#date-from').val();
            const dateTo = $('#date-to').val();
            console.log('📅 Both dates check:', {dateFrom, dateTo});
            
            if (dateFrom && dateTo && window.analyticsDashboard) {
                console.log('✅ Both dates set, reloading...');
                window.analyticsDashboard.loadDashboardData();
            } else {
                console.log('⏳ Waiting for both dates to be set');
            }
        } catch (error) {
            console.error('❌ Error in custom date handler:', error);
        }
    });
    
    // Initialize activity controls when dashboard is ready
    if (window.analyticsDashboard) {
        window.analyticsDashboard.initActivityControls();
        
        // Add test button for debugging
        const testBtn = $('<button type="button" class="button" style="margin-left: 10px;">Test Table</button>');
        $('#activity-per-page').parent().append(testBtn);
        
        testBtn.on('click', function() {
            console.log('🧪 Testing table...');
            
            jQuery.ajax({
                url: affiliateHubAnalytics.ajax_url,
                type: 'POST',
                data: {
                    action: 'affiliate_hub_test_table',
                    nonce: affiliateHubAnalytics.nonce
                },
                success: function(response) {
                    console.log('🧪 Table test result:', response);
                    
                    if (response.success) {
                        const data = response.data;
                        alert(`Table: ${data.table_name}\nExists: ${data.table_exists}\nRecords: ${data.record_count}\n${data.message || ''}`);
                        
                        // If we added data, reload activity
                        if (data.record_count > 0 && window.analyticsDashboard) {
                            window.analyticsDashboard.loadRecentActivity();
                        }
                    } else {
                        alert('Test failed: ' + (response.data?.message || 'Unknown error'));
                    }
                },
                error: function(xhr, status, error) {
                    console.error('🧪 Table test error:', {xhr, status, error});
                    alert('Test request failed: ' + error);
                }
            });
        });
    }
});

// Recent Activity Pagination Methods
AnalyticsDashboard.prototype.loadRecentActivity = function(page = 1, perPage = null, sortBy = 'click_time', sortOrder = 'DESC') {
    console.log('🔄 loadRecentActivity called with params:', {page, perPage, sortBy, sortOrder});
    
    if (this.isLoadingActivity) {
        console.log('⏳ Already loading activity, skipping...');
        return;
    }
    
    console.log('✅ Starting to load recent activity...');
    this.isLoadingActivity = true;
    
    const currentPerPage = perPage || document.getElementById('activity-per-page')?.value || 25;
    
        console.log('🔍 Checking affiliateHubAnalytics:', typeof affiliateHubAnalytics);
        console.log('🔍 affiliateHubAnalytics object:', affiliateHubAnalytics);
        
        if (typeof affiliateHubAnalytics === 'undefined') {
            console.error('❌ affiliateHubAnalytics is not defined!');
            
            // Fallback - try to get from WordPress
            if (typeof wp !== 'undefined' && wp.ajax && wp.ajax.settings) {
                console.log('🔄 Using WordPress AJAX fallback...');
                var fallbackData = {
                    action: 'affiliate_hub_recent_activity',
                    nonce: wp.ajax.settings.nonce || jQuery('#_wpnonce').val() || '',
                    page: page,
                    per_page: parseInt(currentPerPage),
                    sort_by: sortBy,
                    sort_order: sortOrder,
                    ...this.getFilters()
                };
                
                console.log('🔄 Fallback AJAX data:', fallbackData);
                
                jQuery.ajax({
                    url: wp.ajax.settings.url || '/wp-admin/admin-ajax.php',
                    type: 'POST',
                    data: fallbackData,
                    success: (response) => {
                        console.log('✅ Fallback Recent activity AJAX response:', response);
                        
                        if (response.success) {
                            console.log('📊 Fallback Activity data received:', response.data);
                            this.renderPaginatedActivity(response.data);
                            this.currentActivityPage = page;
                            this.currentActivitySort = { column: sortBy, order: sortOrder };
                        } else {
                            console.error('❌ Fallback Activity request failed:', response);
                            this.showError('Failed to load recent activity: ' + (response.data?.message || 'Unknown error'));
                        }
                        
                        this.isLoadingActivity = false;
                    },
                    error: (xhr, status, error) => {
                        console.error('❌ Fallback Recent activity AJAX error:', {xhr, status, error});
                        console.error('❌ Fallback Response text:', xhr.responseText);
                        this.showError('Network error loading recent activity');
                        this.isLoadingActivity = false;
                    }
                });
                
                return;
            } else {
                this.isLoadingActivity = false;
                this.showError('Analytics configuration not loaded and no WordPress AJAX available');
                return;
            }
        }
        
        // Main AJAX request for normal operation
        const filters = this.getFilters();
        
        // Check if filters are valid (getFilters returns null for invalid custom range)
        if (filters === null) {
            console.log('⚠️ Invalid filters for recent activity - aborting request');
            this.isLoadingActivity = false;
            return;
        }
        
        const activityData = {
            action: 'affiliate_hub_recent_activity',
            nonce: affiliateHubAnalytics.nonce,
            page: page,
            per_page: parseInt(currentPerPage),
            sort_by: sortBy,
            sort_order: sortOrder,
            ...filters
        };
        
        console.log('🔄 Loading recent activity page:', page, 'per page:', currentPerPage);
        console.log('🔍 Activity AJAX data:', activityData);
        
        jQuery.ajax({
            url: affiliateHubAnalytics.ajax_url,
            type: 'POST',
            data: activityData,
            success: (response) => {
                console.log('✅ Recent activity AJAX response:', response);
                
                if (response.success) {
                    console.log('📊 Activity data received:', response.data);
                    console.log('🔍 Data structure check:');
                    console.log('  - Items count:', response.data.items?.length || 0);
                    console.log('  - Pagination:', response.data.pagination);
                    console.log('  - First item:', response.data.items?.[0]);
                    
                    this.renderPaginatedActivity(response.data);
                    this.currentActivityPage = page;
                    this.currentActivitySort = { column: sortBy, order: sortOrder };
                } else {
                    console.error('❌ Activity request failed:', response);
                    this.showError('Failed to load recent activity: ' + (response.data?.message || 'Unknown error'));
                }
                
                this.isLoadingActivity = false;
            },
            error: (xhr, status, error) => {
                console.error('❌ Recent activity AJAX error:', {xhr, status, error});
                console.error('❌ Response text:', xhr.responseText);
                this.showError('Network error loading recent activity');
                this.isLoadingActivity = false;
            }
        });
};

AnalyticsDashboard.prototype.renderPaginatedActivity = function(data) {
    console.log('🎨 Rendering paginated activity:', data);
    console.log('🔍 Looking for table element...');
    
    const tableBody = document.querySelector('#recent-activity-table tbody');
    console.log('📋 Table body found:', !!tableBody);
    
    if (!tableBody) {
        console.error('❌ Table body not found! Looking for alternative selectors...');
        console.log('🔍 Available tables:', document.querySelectorAll('table'));
        console.log('🔍 Elements with recent-activity:', document.querySelectorAll('[id*="recent-activity"]'));
        return;
    }
    
    // Clear existing content
    console.log('🧹 Clearing table content...');
    tableBody.innerHTML = '';
    
    if (!data.items || data.items.length === 0) {
        console.log('❌ No items to display');
        tableBody.innerHTML = '<tr><td colspan="8" style="text-align: center; padding: 20px;">No recent activity found</td></tr>';
        this.updateActivityPagination(data.pagination || {});
        return;
    }
    
    console.log(`📝 Rendering ${data.items.length} activity rows...`);
    
    // Render rows
    data.items.forEach((activity, index) => {
        console.log(`🔍 Processing item ${index + 1}:`, activity);
        
        const row = document.createElement('tr');
        
        // Format referrer
        let referrerHtml = '';
        if (activity.referrer_url && activity.referrer_url !== 'Direct') {
            referrerHtml = `<a href="${activity.referrer_url}" target="_blank" class="referrer-link" title="${activity.referrer_url}">${activity.referrer_display}</a>`;
        } else {
            referrerHtml = '<span class="referrer-direct">Direct</span>';
        }
        
        row.innerHTML = `
            <td class="date-cell">${activity.formatted_date}</td>
            <td class="link-title">
                <a href="${activity.edit_url}" target="_blank" title="Edit link">
                    ${activity.link_title || 'Unknown'}
                </a>
            </td>
            <td class="referrer-cell">${referrerHtml}</td>
            <td class="ip-cell">${activity.ip_address || 'Unknown'}</td>
            <td class="browser-cell">${activity.browser || 'Unknown'}</td>
            <td class="os-device-cell">${activity.os || 'Unknown'}</td>
            <td class="os-device-cell">
                ${activity.device_icon} ${activity.device_type || 'Unknown'}
            </td>
            <td class="country-cell">${activity.country || 'Unknown'}</td>
        `;
        
        tableBody.appendChild(row);
    });
    
    // Update pagination
    this.updateActivityPagination(data.pagination);
    
    // Update sort indicators
    this.updateSortIndicators(data.sort);
    
    console.log('✅ Paginated activity rendered with', data.items.length, 'rows');
};

AnalyticsDashboard.prototype.updateActivityPagination = function(pagination) {
    console.log('📊 Updating pagination:', pagination);
    
    // Update info text
    const infoElement = document.getElementById('activity-showing');
    if (infoElement && pagination.total_items) {
        infoElement.textContent = `Showing ${pagination.showing_from}-${pagination.showing_to} of ${pagination.total_items} entries`;
    }
    
    // Update page buttons
    const pagesContainer = document.getElementById('activity-pages');
    if (pagesContainer) {
        pagesContainer.innerHTML = '';
        
        if (pagination.total_pages > 1) {
            this.generatePageButtons(pagesContainer, pagination.current_page, pagination.total_pages);
        }
    }
    
    // Update prev/next buttons
    const prevBtn = document.getElementById('activity-prev');
    const nextBtn = document.getElementById('activity-next');
    
    if (prevBtn) {
        prevBtn.disabled = pagination.current_page <= 1;
    }
    
    if (nextBtn) {
        nextBtn.disabled = pagination.current_page >= pagination.total_pages;
    }
};

AnalyticsDashboard.prototype.generatePageButtons = function(container, currentPage, totalPages) {
    const maxVisible = 7;
    let startPage = Math.max(1, currentPage - 3);
    let endPage = Math.min(totalPages, startPage + maxVisible - 1);
    
    if (endPage - startPage < maxVisible - 1) {
        startPage = Math.max(1, endPage - maxVisible + 1);
    }
    
    // First page
    if (startPage > 1) {
        this.createPageButton(container, 1, currentPage);
        if (startPage > 2) {
            const ellipsis = document.createElement('span');
            ellipsis.className = 'pagination-ellipsis';
            ellipsis.textContent = '...';
            container.appendChild(ellipsis);
        }
    }
    
    // Page numbers
    for (let i = startPage; i <= endPage; i++) {
        this.createPageButton(container, i, currentPage);
    }
    
    // Last page
    if (endPage < totalPages) {
        if (endPage < totalPages - 1) {
            const ellipsis = document.createElement('span');
            ellipsis.className = 'pagination-ellipsis';
            ellipsis.textContent = '...';
            container.appendChild(ellipsis);
        }
        this.createPageButton(container, totalPages, currentPage);
    }
};

AnalyticsDashboard.prototype.createPageButton = function(container, pageNum, currentPage) {
    const button = document.createElement('button');
    button.className = `pagination-page ${pageNum === currentPage ? 'active' : ''}`;
    button.textContent = pageNum;
    button.addEventListener('click', () => {
        if (pageNum !== currentPage) {
            this.loadRecentActivity(pageNum, null, this.currentActivitySort?.column, this.currentActivitySort?.order);
        }
    });
    container.appendChild(button);
};

AnalyticsDashboard.prototype.updateSortIndicators = function(sortData) {
    // Remove all existing sort classes
    document.querySelectorAll('#recent-activity-table th.sortable').forEach(th => {
        th.classList.remove('sorted-asc', 'sorted-desc');
    });
    
    // Add current sort class
    if (sortData && sortData.column) {
        const sortHeader = document.querySelector(`#recent-activity-table th[data-sort="${sortData.column}"]`);
        if (sortHeader) {
            sortHeader.classList.add(sortData.order === 'ASC' ? 'sorted-asc' : 'sorted-desc');
        }
    }
};

AnalyticsDashboard.prototype.initActivityControls = function() {
    // Per page selector
    const perPageSelect = document.getElementById('activity-per-page');
    if (perPageSelect) {
        perPageSelect.addEventListener('change', () => {
            this.loadRecentActivity(1, perPageSelect.value, this.currentActivitySort?.column, this.currentActivitySort?.order);
        });
    }
    
    // Previous/Next buttons
    const prevBtn = document.getElementById('activity-prev');
    if (prevBtn) {
        prevBtn.addEventListener('click', () => {
            if (this.currentActivityPage > 1) {
                this.loadRecentActivity(this.currentActivityPage - 1, null, this.currentActivitySort?.column, this.currentActivitySort?.order);
            }
        });
    }
    
    const nextBtn = document.getElementById('activity-next');
    if (nextBtn) {
        nextBtn.addEventListener('click', () => {
            this.loadRecentActivity(this.currentActivityPage + 1, null, this.currentActivitySort?.column, this.currentActivitySort?.order);
        });
    }
    
    // Sort headers
    document.querySelectorAll('#recent-activity-table th.sortable').forEach(th => {
        th.addEventListener('click', () => {
            const column = th.dataset.sort;
            let order = 'DESC';
            
            // Toggle order if clicking same column
            if (this.currentActivitySort?.column === column) {
                order = this.currentActivitySort.order === 'DESC' ? 'ASC' : 'DESC';
            }
            
            this.loadRecentActivity(1, null, column, order);
        });
    });
    
    console.log('🎯 Activity controls initialized');
    
    // Add debug button for testing endpoints - FIXED jQuery & analytics reference
    if (jQuery('#recent-activity-table').length) {
        if (!jQuery('#debug-endpoints-btn').length) {
            jQuery('<button id="debug-endpoints-btn" class="button" style="margin: 10px;">🔧 Test All Endpoints</button>')
                .insertBefore('#recent-activity-table')
                .on('click', function() {
                    console.log('🔧 Debug button clicked');
                    
                    if (window.analyticsDashboard && typeof window.analyticsDashboard.testAllEndpoints === 'function') {
                        console.log('✅ Found analytics dashboard, calling testAllEndpoints');
                        window.analyticsDashboard.testAllEndpoints();
                    } else {
                        console.error('❌ Analytics dashboard not found, trying direct test');
                        
                        // Direct test if analytics object is not available
                        console.log('🔧 Testing debug endpoint directly...');
                        jQuery.post(affiliateHubAnalytics.ajax_url, {
                            action: 'affiliate_hub_debug_endpoint',
                            test_param: 'direct_test'
                        })
                        .done(response => {
                            console.log('✅ Direct debug endpoint success:', response);
                        })
                        .fail((xhr, status, error) => {
                            console.log('❌ Direct debug endpoint failed:', error);
                            console.log('Response:', xhr.responseText);
                        });
                    }
                });
        }
    }
};

/**
 * Test all AJAX endpoints for debugging - FIXED jQuery
 */
AnalyticsDashboard.prototype.testAllEndpoints = function() {
    console.log('🔧 Testing all endpoints...');
    
    const testData = {
        action: 'affiliate_hub_debug_endpoint',
        test_param: 'debug_test'
    };
    
    // Test debug endpoint (no nonce required) - FIXED jQuery
    console.log('🔧 Testing debug endpoint...');
    jQuery.post(affiliateHubAnalytics.ajax_url, testData)
        .done(response => {
            console.log('✅ Debug endpoint success:', response);
            
            // Now test recent activity with nonce
            this.testRecentActivityEndpoint();
        })
        .fail((xhr, status, error) => {
            console.log('❌ Debug endpoint failed:', error);
            console.log('Response:', xhr.responseText);
        });
};

/**
 * Test recent activity endpoint specifically - FIXED jQuery
 */
AnalyticsDashboard.prototype.testRecentActivityEndpoint = function() {
    console.log('🔧 Testing recent activity endpoint...');
    
    const activityData = {
        action: 'affiliate_hub_recent_activity',
        nonce: affiliateHubAnalytics.nonce,
        page: 1,
        per_page: 5,
        sort_by: 'click_time',
        sort_order: 'DESC'
    };
    
    jQuery.post(affiliateHubAnalytics.ajax_url, activityData)
        .done(response => {
            console.log('✅ Recent activity endpoint success:', response);
        })
        .fail((xhr, status, error) => {
            console.log('❌ Recent activity endpoint failed:', error);
            console.log('Full response:', xhr.responseText);
            console.log('Status:', status);
        });
};

console.log('🎯 Analytics Dashboard script ready');

// Simple test function accessible from console
window.testAjaxSimple = function() {
    console.log('🧪 Testing simple AJAX endpoint...');
    
    jQuery.ajax({
        url: affiliateHubAnalytics.ajax_url,
        type: 'POST',
        data: {
            action: 'affiliate_hub_test_simple',
            test_data: 'console_test'
        },
        success: function(response) {
            console.log('✅ Simple AJAX test SUCCESS:', response);
            alert('AJAX Test Success: ' + JSON.stringify(response));
        },
        error: function(xhr, status, error) {
            console.error('❌ Simple AJAX test FAILED:', {xhr, status, error});
            console.error('Response text:', xhr.responseText);
            alert('AJAX Test Failed: ' + error + '\nResponse: ' + xhr.responseText);
        }
    });
};

// Test filtering with detailed logs
window.testFiltering = function() {
    console.log('🧪 Testing filtering functionality...');
    
    if (!window.analyticsDashboard) {
        console.error('❌ Analytics dashboard not available');
        alert('Analytics dashboard not available');
        return;
    }
    
    try {
        const filters = window.analyticsDashboard.getFilters();
        console.log('🔍 Current filters:', filters);
        
        if (filters === null) {
            console.log('⚠️ Filters returned null - checking custom date range');
            const dateRange = document.getElementById('date-range');
            const dateFrom = document.getElementById('date-from');
            const dateTo = document.getElementById('date-to');
            
            console.log('Date range:', dateRange ? dateRange.value : 'NOT FOUND');
            console.log('Date from:', dateFrom ? dateFrom.value : 'NOT FOUND');
            console.log('Date to:', dateTo ? dateTo.value : 'NOT FOUND');
        }
        
        alert('Filters test completed - check console for details');
        
    } catch (error) {
        console.error('❌ Filter test error:', error);
        alert('Filter test error: ' + error.message);
    }
};
