/**
 * Autolinker Admin JavaScript - Cleaned Version
 * Only local duplicate validation, no AJAX conflicts checking
 */
jQuery(document).ready(function($) {
    'use strict';

    // Prevent multiple initializations
    if (window.AffiliateHubAutolinkerInitialized) {
        console.log('🔧 Already initialized, skipping');
        return;
    }
    window.AffiliateHubAutolinkerInitialized = true;

    var AffiliateHubAutolinker = {
        conflictCheckTimeout: null,
        
        init: function() {
            console.log('🔧 Initializing AffiliateHubAutolinker');
            this.bindEvents();
            this.setupExistingKeywords();
            this.initSortable();
        },

        bindEvents: function() {
            var self = this;
            
            // Unbind existing events to prevent duplicates
            $(document).off('click', '#add-keyword');
            $(document).off('click', '.remove-keyword');
            $(document).off('input', 'input[name="autolink_keywords[]"]');
            $(document).off('focus', 'input[name="autolink_keywords[]"]');
            $(document).off('blur', 'input[name="autolink_keywords[]"]');
            
            // Add keyword button
            $(document).on('click', '#add-keyword', function(e) {
                e.preventDefault();
                self.addKeyword();
            });
            
            // Remove keyword button
            $(document).on('click', '.remove-keyword', function(e) {
                e.preventDefault();
                self.removeKeyword($(this));
            });
            
            // Input validation
            $(document).on('input', 'input[name="autolink_keywords[]"]', function() {
                var $input = $(this);
                // Use setTimeout to ensure the input value is fully updated
                setTimeout(function() {
                    self.validateKeyword($input);
                }, 10); // 10ms delay to allow input to complete
            });
            
            // Auto-focus new keywords
            $(document).on('focus', 'input[name="autolink_keywords[]"]', function() {
                $(this).closest('.keyword-tag').addClass('active');
            });
            
            $(document).on('blur', 'input[name="autolink_keywords[]"]', function() {
                $(this).closest('.keyword-tag').removeClass('active');
            });
        },

        setupExistingKeywords: function() {
            // Convert any old-style keyword inputs to new structure
            $('.keyword-tag').each(function() {
                var $tag = $(this);
                if ($tag.find('.keyword-status-message').length === 0) {
                    $tag.append('<div class="keyword-status-message" style="display: none;"></div>');
                }
            });
        },

        addKeyword: function() {
            var container = $('#autolink-keywords-container');
            var newKeyword = $(
                '<div class="keyword-tag">' +
                '<div class="keyword-input-wrapper">' +
                '<input type="text" name="autolink_keywords[]" value="" placeholder="Enter keyword" />' +
                '<button type="button" class="button remove-keyword">×</button>' +
                '</div>' +
                '<div class="keyword-status-message" style="display: none;"></div>' +
                '</div>'
            );
            
            container.append(newKeyword);
            newKeyword.find('input').focus();
            
            // Re-initialize sortable if it exists
            if (container.hasClass('ui-sortable')) {
                container.sortable('refresh');
            }
        },

        removeKeyword: function($button) {
            var $tag = $button.closest('.keyword-tag');
            var isLastKeyword = $tag.siblings('.keyword-tag').length === 0;
            
            if (isLastKeyword) {
                // Clear the input and status message instead
                $tag.find('input').val('').focus();
                $tag.find('.keyword-status-message').hide();
            } else {
                // Remove the entire keyword tag
                $tag.fadeOut(200, function() {
                    $(this).remove();
                });
            }
        },

        validateKeyword: function($input) {
            var value = $input.val().trim();
            var $tag = $input.closest('.keyword-tag');
            var $statusMessage = $tag.find('.keyword-status-message');
            
            // Hide status message for empty values
            if (value === '') {
                $statusMessage.hide();
                $tag.removeClass('invalid duplicate conflict');
                return;
            }
            
            // Check for duplicates within current form first
            var duplicate = false;
            var allInputs = $('input[name="autolink_keywords[]"]').not($input);
            var currentValue = value.toLowerCase();
            
            console.log('🔧 LOCAL DUPLICATE CHECK:');
            console.log('🔧 Current value: "' + currentValue + '"');
            console.log('🔧 Checking against ' + allInputs.length + ' other inputs');
            
            allInputs.each(function(index) {
                var otherValue = $(this).val().trim().toLowerCase();
                console.log('🔧 [' + index + '] Other value: "' + otherValue + '"');
                console.log('🔧 [' + index + '] Match: ' + (otherValue === currentValue && otherValue !== ''));
                
                if (otherValue === currentValue && otherValue !== '') {
                    console.log('🔧 DUPLICATE FOUND!');
                    duplicate = true;
                    return false;
                }
            });
            
            if (duplicate) {
                $statusMessage.removeClass('available conflict checking')
                              .addClass('conflict')
                              .html('❌ Duplicate keyword in this form')
                              .show();
                // Add a flag to prevent AJAX check
                $statusMessage.data('local-duplicate', true);
                return;
            }
            
            // Clear the local duplicate flag
            $statusMessage.removeData('local-duplicate');
            
            // Basic validation (no special characters that might break regex)
            if (!/^[a-zA-Z0-9\s\-_]+$/.test(value)) {
                $statusMessage.removeClass('available conflict checking')
                              .addClass('conflict')
                              .html('❌ Only letters, numbers, spaces, hyphens and underscores allowed')
                              .show();
                return;
            }
            
            // Check for conflicts with other affiliate links (real-time)
            this.checkKeywordConflict($input, value);
        },
        
        checkKeywordConflict: function($input, keyword) {
            var self = this;
            var $tag = $input.closest('.keyword-tag');
            var $statusMessage = $tag.find('.keyword-status-message');
            
            // Don't check AJAX conflicts if we have a local duplicate
            if ($statusMessage.data('local-duplicate')) {
                console.log('🚫 Skipping AJAX check - local duplicate detected');
                return;
            }
            
            // Get post ID - handle both new and existing posts
            var postId = $('#post_ID').val() || $('#ID').val() || 0;
            
            // Show checking status
            $statusMessage.removeClass('available conflict')
                          .addClass('checking')
                          .html('⏳ Sprawdzanie...')
                          .show();
            
            // Check if we have AJAX configuration
            if (!window.affiliateHubAutolinker || !window.affiliateHubAutolinker.ajaxUrl) {
                $statusMessage.removeClass('checking available')
                              .addClass('conflict')
                              .html('❌ Błąd konfiguracji')
                              .show();
                return;
            }
            
            // Clear previous timeout
            if (this.conflictCheckTimeout) {
                clearTimeout(this.conflictCheckTimeout);
            }
            
            // Debounce the check
            this.conflictCheckTimeout = setTimeout(function() {
                $.ajax({
                    url: window.affiliateHubAutolinker.ajaxUrl,
                    type: 'POST',
                    data: {
                        action: 'affiliate_hub_check_keyword_conflict',
                        keyword: keyword,
                        post_id: postId,
                        nonce: window.affiliateHubAutolinker.nonce
                    },
                    success: function(response) {
                        console.log('🔧 AJAX response:', response);
                        
                        // DEBUG: Show cache contents
                        if (response.data && response.data.debug_cache) {
                            console.log('🔧 CACHE CONTENTS:');
                            response.data.debug_cache.forEach(function(item, index) {
                                console.log('  [' + index + '] "' + item.keyword + '" → post ' + item.post_id);
                            });
                            console.log('🔧 Searched for: "' + response.data.debug_searched_keyword + '"');
                            console.log('🔧 Current post: ' + response.data.debug_current_post);
                        }
                        
                        if (response.success) {
                            if (response.data.conflict) {
                                // Show conflict with link details
                                var conflictHtml = '🚨 RISK! Keyword already in use<br>' +
                                                 '<strong>' + response.data.existing_link_name + '</strong> (ID: ' + response.data.existing_link_id + ')';
                                
                                if (response.data.existing_link_edit_url) {
                                    conflictHtml += '<br><a href="' + response.data.existing_link_edit_url + '" target="_blank" style="color: #0073aa;">➜ Edit link</a>';
                                }
                                
                                $statusMessage.removeClass('checking available')
                                              .addClass('conflict')
                                              .html(conflictHtml)
                                              .show();
                            } else {
                                // No conflict - hide status message
                                $statusMessage.hide();
                            }
                        } else {
                            $statusMessage.removeClass('checking available')
                                          .addClass('conflict')
                                          .html('❌ Błąd sprawdzania')
                                          .show();
                        }
                    },
                    error: function() {
                        $statusMessage.removeClass('checking available')
                                      .addClass('conflict')
                                      .html('❌ Błąd sieci')
                                      .show();
                    }
                });
            }, 300); // 300ms debounce
        },

        initSortable: function() {
            var container = $('#autolink-keywords-container');
            
            if (container.length && typeof $.fn.sortable !== 'undefined') {
                container.sortable({
                    items: '.keyword-tag',
                    cursor: 'move',
                    placeholder: 'ui-sortable-placeholder',
                    tolerance: 'pointer',
                    start: function(e, ui) {
                        ui.placeholder.height(ui.item.outerHeight());
                    },
                    stop: function(e, ui) {
                        // Optional: trigger change event to mark form as dirty
                        container.trigger('sortable-update');
                    }
                });
            }
        }
    };

    // Initialize when document is ready
    AffiliateHubAutolinker.init();
});
