/**
 * Debug version of Keywords Autolinker Admin JavaScript
 */
(function($) {
    'use strict';

    console.log('Autolinker admin script loaded');
    console.log('affiliateHubAutolinker object:', window.affiliateHubAutolinker);

    // Prevent multiple initializations
    if (window.AffiliateHubAutolinkerInitialized) {
        console.log('Already initialized, skipping');
        return;
    }
    window.AffiliateHubAutolinkerInitialized = true;

    var AffiliateHubAutolinker = {
        init: function() {
            console.log('Initializing AffiliateHubAutolinker');
            this.bindEvents();
            this.initSortable();
        },

        bindEvents: function() {
            var self = this;
            console.log('Binding events');
            
            // Debug: Check if input fields exist
            var keywordInputs = $('input[name="autolink_keywords[]"]');
            console.log('Found keyword inputs:', keywordInputs.length);
            
            // Unbind existing events to prevent duplicates
            $(document).off('click', '#add-keyword');
            $(document).off('click', '.remove-keyword');
            $(document).off('click', '.rebuild-cache-btn');
            $(document).off('input', 'input[name="autolink_keywords[]"]');
            $(document).off('focus', 'input[name="autolink_keywords[]"]');
            $(document).off('blur', 'input[name="autolink_keywords[]"]');
            
            // Add keyword button
            $(document).on('click', '#add-keyword', function(e) {
                e.preventDefault();
                console.log('Add keyword clicked');
                self.addKeyword();
            });
            
            // Remove keyword button
            $(document).on('click', '.remove-keyword', function(e) {
                e.preventDefault();
                console.log('Remove keyword clicked');
                self.removeKeyword($(this));
            });
            
            // Input validation with debugging
            $(document).on('input', 'input[name="autolink_keywords[]"]', function() {
                var value = $(this).val();
                console.log('Keyword input changed:', value);
                self.validateKeyword($(this));
            });
            
            // Auto-focus new keywords
            $(document).on('focus', 'input[name="autolink_keywords[]"]', function() {
                $(this).closest('.keyword-tag').addClass('active');
            });
            
            $(document).on('blur', 'input[name="autolink_keywords[]"]', function() {
                $(this).closest('.keyword-tag').removeClass('active');
            });
        },

        validateKeyword: function($input) {
            var value = $input.val().trim();
            var $tag = $input.closest('.keyword-tag');
            
            console.log('Validating keyword:', value);
            
            // Remove any existing validation classes
            $tag.removeClass('invalid duplicate conflict');
            
            if (value === '') {
                console.log('Empty keyword, skipping validation');
                return;
            }
            
            // Check for duplicates within current form
            var duplicate = false;
            $('input[name="autolink_keywords[]"]').not($input).each(function() {
                if ($(this).val().trim().toLowerCase() === value.toLowerCase()) {
                    duplicate = true;
                    return false;
                }
            });
            
            if (duplicate) {
                console.log('Duplicate keyword detected');
                $tag.addClass('duplicate');
                return;
            }
            
            // Check for conflicts with other affiliate links (real-time)
            console.log('Checking for conflicts...');
            this.checkKeywordConflict($input, value);
            
            // Basic validation (no special characters that might break regex)
            if (!/^[a-zA-Z0-9\s\-_]+$/.test(value)) {
                console.log('Invalid characters detected');
                $tag.addClass('invalid');
            }
        },
        
        checkKeywordConflict: function($input, keyword) {
            var self = this;
            var $tag = $input.closest('.keyword-tag');
            var postId = $('#post_ID').val() || 0;
            
            console.log('Checking conflict for keyword:', keyword, 'Post ID:', postId);
            console.log('AJAX URL:', window.affiliateHubAutolinker ? window.affiliateHubAutolinker.ajaxUrl : 'NOT FOUND');
            console.log('Nonce:', window.affiliateHubAutolinker ? window.affiliateHubAutolinker.nonce : 'NOT FOUND');
            
            // Clear any existing timeout
            if (this.conflictCheckTimeout) {
                clearTimeout(this.conflictCheckTimeout);
            }
            
            // Check if we have required data
            if (!window.affiliateHubAutolinker || !window.affiliateHubAutolinker.ajaxUrl) {
                console.error('affiliateHubAutolinker object not found or missing ajaxUrl');
                return;
            }
            
            // Debounce the check to avoid too many requests
            this.conflictCheckTimeout = setTimeout(function() {
                console.log('Sending AJAX request...');
                $.ajax({
                    url: window.affiliateHubAutolinker.ajaxUrl,
                    type: 'POST',
                    data: {
                        action: 'affiliate_hub_check_keyword_conflict',
                        keyword: keyword,
                        post_id: postId,
                        nonce: window.affiliateHubAutolinker.nonce
                    },
                    success: function(response) {
                        console.log('AJAX response:', response);
                        if (response.success) {
                            if (response.data.conflict) {
                                console.log('Conflict found:', response.data);
                                $tag.addClass('conflict');
                                $tag.attr('title', response.data.message);
                            } else {
                                console.log('No conflict found');
                                $tag.removeClass('conflict');
                                $tag.removeAttr('title');
                            }
                        } else {
                            console.error('AJAX error:', response);
                        }
                    },
                    error: function(xhr, status, error) {
                        console.error('AJAX request failed:', status, error);
                        console.error('Response:', xhr.responseText);
                        // Silently fail for UX
                        $tag.removeClass('conflict');
                    }
                });
            }, 500); // 500ms debounce
        },

        addKeyword: function() {
            console.log('Adding new keyword');
            var container = $('#autolink-keywords-container');
            var newKeyword = $(
                '<div class="keyword-tag">' +
                '<input type="text" name="autolink_keywords[]" value="" placeholder="Enter keyword" />' +
                '<button type="button" class="button remove-keyword">×</button>' +
                '</div>'
            );
            
            container.append(newKeyword);
            newKeyword.find('input').focus();
            
            // Re-initialize sortable if it exists
            if (container.hasClass('ui-sortable')) {
                container.sortable('refresh');
            }
        },

        removeKeyword: function($button) {
            console.log('Removing keyword');
            var container = $('#autolink-keywords-container');
            var keywordTags = container.find('.keyword-tag');
            
            // Don't remove if it's the last one
            if (keywordTags.length > 1) {
                $button.closest('.keyword-tag').fadeOut(300, function() {
                    $(this).remove();
                });
            } else {
                // Clear the input instead
                $button.siblings('input').val('').focus();
            }
        }
    };

    // Initialize when DOM is ready
    $(document).ready(function() {
        console.log('DOM ready, initializing...');
        AffiliateHubAutolinker.init();
    });

})(jQuery);
