/**
 * Keywords Autolinker Admin JavaScript
 */
(function($) {
    'use strict';

    console.log('🔧 Autolinker admin script loading...');
    console.log('🔧 affiliateHubAutolinker:', window.affiliateHubAutolinker);

    // Prevent multiple initializations
    if (window.AffiliateHubAutolinkerInitialized) {
        console.log('🔧 Already initialized, skipping');
        return;
    }
    window.AffiliateHubAutolinkerInitialized = true;

    var AffiliateHubAutolinker = {
        init: function() {
            console.log('🔧 Initializing AffiliateHubAutolinker');
            this.setupExistingKeywords();
            this.bindEvents();
            this.initSortable();
        },
        
        setupExistingKeywords: function() {
            console.log('🔧 Setting up existing keywords');
            var $existingTags = $('.keyword-tag');
            console.log('🔧 Found existing tags:', $existingTags.length);
            
            $existingTags.each(function() {
                var $tag = $(this);
                // Check if already has new structure
                if ($tag.find('.keyword-input-wrapper').length === 0) {
                    console.log('🔧 Converting old structure to new');
                    var $input = $tag.find('input[name="autolink_keywords[]"]');
                    var $removeBtn = $tag.find('.remove-keyword');
                    
                    // Wrap input and button
                    $input.add($removeBtn).wrapAll('<div class="keyword-input-wrapper"></div>');
                    
                    // Add status message div
                    $tag.append('<div class="keyword-status-message" style="display: none;"></div>');
                }
                
                // Validate existing keywords
                var $input = $tag.find('input[name="autolink_keywords[]"]');
                if ($input.val().trim() !== '') {
                    console.log('🔧 Validating existing keyword:', $input.val());
                    AffiliateHubAutolinker.validateKeyword($input);
                }
            });
        },

        bindEvents: function() {
            var self = this;
            console.log('🔧 Binding autolinker events...');
            console.log('🔧 Found keyword inputs:', $('input[name="autolink_keywords[]"]').length);
            console.log('🔧 Found add-keyword button:', $('#add-keyword').length);
            
            // Unbind existing events to prevent duplicates
            $(document).off('click', '#add-keyword');
            $(document).off('click', '.remove-keyword');
            $(document).off('click', '.rebuild-cache-btn');
            $(document).off('input', 'input[name="autolink_keywords[]"]');
            $(document).off('focus', 'input[name="autolink_keywords[]"]');
            $(document).off('blur', 'input[name="autolink_keywords[]"]');
            
            // Add keyword button
            $(document).on('click', '#add-keyword', function(e) {
                e.preventDefault();
                console.log('🔧 Add keyword button clicked - handler triggered');
                self.addKeyword();
            });
            
            // Remove keyword button
            $(document).on('click', '.remove-keyword', function(e) {
                e.preventDefault();
                self.removeKeyword($(this));
            });
            
            // Rebuild cache button
            $(document).on('click', '.rebuild-cache-btn', function(e) {
                e.preventDefault();
                self.rebuildCache($(this));
            });
            
            // Input validation
            $(document).on('input', 'input[name="autolink_keywords[]"]', function() {
                console.log('🔧 Input event triggered on keyword field:', $(this).val());
                self.validateKeyword($(this));
            });
            
            // Auto-focus new keywords
            $(document).on('focus', 'input[name="autolink_keywords[]"]', function() {
                $(this).closest('.keyword-tag').addClass('active');
            });
            
            $(document).on('blur', 'input[name="autolink_keywords[]"]', function() {
                $(this).closest('.keyword-tag').removeClass('active');
            });
        },

        addKeyword: function() {
            console.log('🔧 addKeyword() function called');
            var container = $('#autolink-keywords-container');
            console.log('🔧 Container found:', container.length);
            console.log('🔧 Existing keywords:', container.find('.keyword-tag').length);
            
            var newKeyword = $(
                '<div class="keyword-tag">' +
                '<div class="keyword-input-wrapper">' +
                '<input type="text" name="autolink_keywords[]" value="" placeholder="Enter keyword" />' +
                '<button type="button" class="button remove-keyword">×</button>' +
                '</div>' +
                '<div class="keyword-status-message" style="display: none;"></div>' +
                '</div>'
            );
            
            container.append(newKeyword);
            console.log('🔧 Keyword added, new total:', container.find('.keyword-tag').length);
            newKeyword.find('input').focus();
            
            // Re-initialize sortable if it exists
            if (container.hasClass('ui-sortable')) {
                container.sortable('refresh');
            }
        },

        removeKeyword: function($button) {
            var container = $('#autolink-keywords-container');
            var keywordTags = container.find('.keyword-tag');
            
            // Don't remove if it's the last one
            if (keywordTags.length > 1) {
                $button.closest('.keyword-tag').fadeOut(300, function() {
                    $(this).remove();
                });
            } else {
                // Clear the input and status message instead
                var $tag = $button.closest('.keyword-tag');
                $tag.find('input').val('').focus();
                $tag.find('.keyword-status-message').hide();
            }
        },

        validateKeyword: function($input) {
            var value = $input.val().trim();
            var $tag = $input.closest('.keyword-tag');
            var $statusMessage = $tag.find('.keyword-status-message');
            
            console.log('🔧 Validating keyword:', value);
            console.log('🔧 Status message element found:', $statusMessage.length);
            console.log('🔧 Current tag:', $tag.length);
            
            // Hide status message for empty values
            if (value === '') {
                console.log('🔧 Empty value, hiding status');
                $statusMessage.hide();
                $tag.removeClass('invalid duplicate conflict');
                return;
            }
            
            // Check for duplicates within current form first
            var duplicate = false;
            var allInputs = $('input[name="autolink_keywords[]"]').not($input);
            console.log('🔧 Checking against', allInputs.length, 'other inputs');
            console.log('🔧 Current input value:', value.toLowerCase());
            
            allInputs.each(function(index) {
                var otherValue = $(this).val().trim().toLowerCase();
                var currentValue = value.toLowerCase();
                console.log('🔧 [' + index + '] Comparing:', '"' + currentValue + '"', 'vs', '"' + otherValue + '"');
                console.log('🔧 [' + index + '] Are equal?:', otherValue === currentValue);
                if (otherValue === currentValue && otherValue !== '') {
                    console.log('🔧 DUPLICATE FOUND at index', index);
                    duplicate = true;
                    return false;
                }
            });
            
            if (duplicate) {
                console.log('🔧 Duplicate keyword detected - showing conflict message');
                $statusMessage.removeClass('available conflict checking')
                              .addClass('conflict')
                              .html('❌ Duplicate keyword in this form')
                              .show();
                console.log('🔧 Status message updated:', $statusMessage.attr('class'), $statusMessage.html());
                console.log('🔧 Status message visible:', $statusMessage.is(':visible'));
                // Don't check conflicts with other links if we already have a local duplicate
                return;
            }
            
            // Basic validation (no special characters that might break regex)
            if (!/^[a-zA-Z0-9\s\-_]+$/.test(value)) {
                console.log('🔧 Invalid characters detected');
                $statusMessage.removeClass('available conflict checking')
                              .addClass('conflict')
                              .html('❌ Only letters, numbers, spaces, hyphens and underscores allowed')
                              .show();
                return;
            }
            
            // Don't check conflicts with other links - only local duplicates matter
            console.log('🔧 Keyword validation passed - hiding status message');
            $statusMessage.hide();
            $tag.removeClass('invalid duplicate conflict');
        },
        
        checkKeywordConflict: function($input, keyword) {
            var self = this;
            var $tag = $input.closest('.keyword-tag');
            var $statusMessage = $tag.find('.keyword-status-message');
            
            // Get post ID - handle both new and existing posts
            var postId = $('#post_ID').val() || $('#ID').val() || 0;
            console.log('🔧 Post ID found:', postId, '(from', $('#post_ID').length ? '#post_ID' : $('#ID').length ? '#ID' : 'fallback', ')');
            
            console.log('🔧 Checking keyword conflict:', keyword);
            console.log('🔧 AJAX URL:', window.affiliateHubAutolinker ? window.affiliateHubAutolinker.ajaxUrl : 'MISSING');
            console.log('🔧 Nonce:', window.affiliateHubAutolinker ? window.affiliateHubAutolinker.nonce : 'MISSING');
            
            // Show checking status
            $statusMessage.removeClass('available conflict')
                          .addClass('checking')
                          .html('⏳ Checking availability...')
                          .show();
            
            // Clear any existing timeout
            if (this.conflictCheckTimeout) {
                clearTimeout(this.conflictCheckTimeout);
            }
            
            // Check if we have required data
            if (!window.affiliateHubAutolinker || !window.affiliateHubAutolinker.ajaxUrl) {
                console.error('affiliateHubAutolinker object not found or missing ajaxUrl');
                $statusMessage.removeClass('checking available')
                              .addClass('conflict')
                              .html('❌ Configuration error')
                              .show();
                return;
            }
            
            // Debounce the check to avoid too many requests
            this.conflictCheckTimeout = setTimeout(function() {
                console.log('🔧 Sending AJAX request...');
                $.ajax({
                    url: window.affiliateHubAutolinker.ajaxUrl,
                    type: 'POST',
                    data: {
                        action: 'affiliate_hub_check_keyword_conflict',
                        keyword: keyword,
                        post_id: postId,
                        nonce: window.affiliateHubAutolinker.nonce
                    },
                    success: function(response) {
                        console.log('🔧 AJAX response:', response);
                        console.log('🔧 Response data:', response.data);
                        
                        if (response.success) {
                            if (response.data.conflict) {
                                console.log('🔧 Conflict found:', response.data);
                                var conflictHtml = '❌ ' + response.data.message;
                                
                                // Add action links if available
                                if (response.data.existing_link_edit_url || response.data.existing_link_view_url) {
                                    conflictHtml += '<div class="keyword-status-actions">';
                                    if (response.data.existing_link_edit_url) {
                                        conflictHtml += '<a href="' + response.data.existing_link_edit_url + '" target="_blank">Edit Link</a>';
                                    }
                                    if (response.data.existing_link_view_url) {
                                        conflictHtml += '<a href="' + response.data.existing_link_view_url + '" target="_blank">View Link</a>';
                                    }
                                    conflictHtml += '</div>';
                                }
                                
                                $statusMessage.removeClass('checking available')
                                              .addClass('conflict')
                                              .html(conflictHtml)
                                              .show();
                            } else {
                                console.log('🔧 No conflict found');
                                $statusMessage.removeClass('checking conflict')
                                              .addClass('available')
                                              .html('✅ ' + response.data.message)
                                              .show();
                            }
                        } else {
                            console.error('🔧 AJAX error:', response);
                            $statusMessage.removeClass('checking available')
                                          .addClass('conflict')
                                          .html('❌ Error checking keyword')
                                          .show();
                        }
                    },
                    error: function(xhr, status, error) {
                        console.error('🔧 AJAX request failed:', status, error);
                        console.error('🔧 Response:', xhr.responseText);
                        $statusMessage.removeClass('checking available')
                                      .addClass('conflict')
        },

        initSortable: function() {
            var container = $('#autolink-keywords-container');
            
            if (container.length && typeof $.fn.sortable !== 'undefined') {
                container.sortable({
                    items: '.keyword-tag',
                    cursor: 'move',
                    placeholder: 'ui-sortable-placeholder',
                    tolerance: 'pointer',
                    start: function(e, ui) {
                        ui.placeholder.height(ui.item.outerHeight());
                    },
                    stop: function(e, ui) {
                        // Optional: trigger change event to mark form as dirty
                        $(document).trigger('affiliate-hub-keywords-reordered');
                    }
                });
            }
        },

        rebuildCache: function($button) {
            if (!confirm(affiliateHubAutolinker.strings.confirmRebuildCache)) {
                return;
            }
            
            var $container = $button.closest('.affiliate-hub-info-box');
            var originalText = $button.text();
            
            // Show loading state
            $button.text(affiliateHubAutolinker.strings.rebuildingCache).prop('disabled', true);
            $container.addClass('affiliate-hub-loading');
            
            $.ajax({
                url: affiliateHubAutolinker.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'affiliate_hub_rebuild_autolinker_cache',
                    nonce: affiliateHubAutolinker.nonce
                },
                success: function(response) {
                    if (response.success) {
                        $button.text(affiliateHubAutolinker.strings.cacheRebuilt);
                        
                        // Show success message
                        $container.prepend(
                            '<div class="notice notice-success is-dismissible inline">' +
                            '<p>' + response.data.message + '</p>' +
                            '</div>'
                        );
                        
                        // Reload page after 2 seconds to show updated stats
                        setTimeout(function() {
                            location.reload();
                        }, 2000);
                        
                    } else {
                        $button.text(affiliateHubAutolinker.strings.errorRebuildingCache);
                        
                        // Show error message
                        $container.prepend(
                            '<div class="notice notice-error is-dismissible inline">' +
                            '<p>' + (response.data ? response.data.message : 'Unknown error') + '</p>' +
                            '</div>'
                        );
                    }
                },
                error: function() {
                    $button.text(affiliateHubAutolinker.strings.errorRebuildingCache);
                    
                    $container.prepend(
                        '<div class="notice notice-error is-dismissible inline">' +
                        '<p>' + affiliateHubAutolinker.strings.errorRebuildingCache + '</p>' +
                        '</div>'
                    );
                },
                complete: function() {
                    // Restore button state
                    setTimeout(function() {
                        $button.text(originalText).prop('disabled', false);
                        $container.removeClass('affiliate-hub-loading');
                    }, 3000);
                }
            });
        },

        // Utility method to get keyword statistics
        getKeywordStats: function() {
            var keywords = [];
            $('input[name="autolink_keywords[]"]').each(function() {
                var value = $(this).val().trim();
                if (value !== '') {
                    keywords.push(value);
                }
            });
            
            return {
                total: keywords.length,
                unique: [...new Set(keywords.map(k => k.toLowerCase()))].length,
                duplicates: keywords.length - [...new Set(keywords.map(k => k.toLowerCase()))].length
            };
        },

        // Method to export keywords (for potential future use)
        exportKeywords: function() {
            var keywords = [];
            $('input[name="autolink_keywords[]"]').each(function() {
                var value = $(this).val().trim();
                if (value !== '') {
                    keywords.push(value);
                }
            });
            
            return JSON.stringify(keywords);
        },

        // Method to import keywords (for potential future use)
        importKeywords: function(keywordData) {
            try {
                var keywords = JSON.parse(keywordData);
                var container = $('#autolink-keywords-container');
                
                // Clear existing keywords
                container.empty();
                
                // Add imported keywords
                keywords.forEach(function(keyword) {
                    container.append(
                        '<div class="keyword-tag">' +
                        '<input type="text" name="autolink_keywords[]" value="' + $('<div>').text(keyword).html() + '" />' +
                        '<button type="button" class="button remove-keyword">×</button>' +
                        '</div>'
                    );
                });
                
                // Ensure at least one empty field
                if (keywords.length === 0) {
                    this.addKeyword();
                }
                
                // Re-initialize sortable
                this.initSortable();
                
                return true;
            } catch (e) {
                console.error('Error importing keywords:', e);
                return false;
            }
        }
    };

    // Initialize when document is ready
    $(document).ready(function() {
        AffiliateHubAutolinker.init();
    });

    // Expose to global scope for potential external use
    window.AffiliateHubAutolinker = AffiliateHubAutolinker;

})(jQuery);
