/**
 * Click Tracker for Affiliate Hub
 */
(function($) {
    'use strict';

    var AffiliateHubTracker = {
        init: function() {
            this.bindEvents();
        },

        bindEvents: function() {
            var self = this;
            
            // Track clicks on affiliate links - DON'T prevent default navigation
            $(document).on('click', 'a', function(e) {
                var href = this.href;
                var linkPrefix = affiliateHubTracker.linkPrefix || 'go';
                
                // Check if this is an affiliate link
                if (href && href.indexOf('/' + linkPrefix + '/') !== -1) {
                    // Track click asynchronously without blocking navigation
                    self.trackClick(href);
                    // DON'T prevent default - let browser navigate normally
                }
            });
        },

        trackClick: function(url) {
            // Input validation
            if (typeof url !== 'string' || !url || !affiliateHubTracker.ajax_url || !affiliateHubTracker.nonce) {
                return;
            }

            // Extract slug from URL
            var slug = this.extractSlugFromUrl(url);
            
            if (!slug || typeof slug !== 'string') {
                return;
            }

            $.ajax({
                url: affiliateHubTracker.ajax_url,
                type: 'POST',
                data: {
                    action: 'affiliate_hub_track_click',
                    url: url,
                    slug: slug,
                    nonce: affiliateHubTracker.nonce
                },
                async: true
            });
        },

        extractSlugFromUrl: function(url) {
            try {
                // Input validation
                if (typeof url !== 'string' || !url) {
                    return null;
                }
                
                var linkPrefix = affiliateHubTracker.linkPrefix || 'go';
                
                // Simple string-based approach
                var searchPattern = '/' + linkPrefix + '/';
                var startIndex = url.indexOf(searchPattern);
                
                if (startIndex === -1) {
                    return null;
                }
                
                var slugStart = startIndex + searchPattern.length;
                var slugPart = url.substring(slugStart);
                
                // Extract slug until first slash, question mark, or hash
                var endChars = ['/', '?', '#'];
                var endIndex = slugPart.length;
                
                for (var i = 0; i < endChars.length; i++) {
                    var charIndex = slugPart.indexOf(endChars[i]);
                    if (charIndex !== -1 && charIndex < endIndex) {
                        endIndex = charIndex;
                    }
                }
                
                return slugPart.substring(0, endIndex) || null;
                
            } catch (e) {
                return null;
            }
        }
    };

    // Initialize when document is ready
    $(document).ready(function() {
        AffiliateHubTracker.init();
    });

})(jQuery);