/**
 * GeoLocation Admin Dashboard JavaScript
 * 
 * Handles charts, tools, and AJAX requests for geolocation admin page
 */
(function($) {
    'use strict';

    var GeoLocationAdmin = {
        chart: null,
        
        init: function() {
            console.log('GeoLocationAdmin.init() called');
            this.initChart();
            this.bindEvents();
            this.loadGeoStats();
            console.log('GeoLocationAdmin.init() completed');
        },
        
        initChart: function() {
            var ctx = document.getElementById('geo-distribution-chart');
            if (!ctx) return;
            
            this.chart = new Chart(ctx, {
                type: 'doughnut',
                data: {
                    labels: [],
                    datasets: [{
                        data: [],
                        backgroundColor: [
                            '#2271b1', '#72aee6', '#00a32a', '#00ba37', 
                            '#ffb900', '#ff6900', '#d63638', '#e65054',
                            '#8c8f94', '#a7aaad'
                        ],
                        borderWidth: 2,
                        borderColor: '#fff'
                    }]
                },
                options: {
                    responsive: true,
                    maintainAspectRatio: false,
                    plugins: {
                        legend: {
                            position: 'right'
                        },
                        tooltip: {
                            callbacks: {
                                label: function(context) {
                                    var label = context.label || '';
                                    var value = context.parsed || 0;
                                    var total = context.dataset.data.reduce((a, b) => a + b, 0);
                                    var percentage = ((value / total) * 100).toFixed(1);
                                    return label + ': ' + value + ' clicks (' + percentage + '%)';
                                }
                            }
                        }
                    }
                }
            });
        },
        
        bindEvents: function() {
            console.log('bindEvents() called');
            console.log('Install button exists:', $('#install-native-btn').length > 0);
            console.log('Benchmark button exists:', $('#run-benchmark-btn').length > 0);
            
            $('#bulk-enrich-btn').on('click', this.bulkEnrich.bind(this));
            $('#cache-cleanup-btn').on('click', this.cleanupCache.bind(this));
            $('#maxmind-update-btn').on('click', this.updateMaxMind.bind(this));
            $('#install-native-btn').on('click', this.installNativeGeoIP2.bind(this));
            $('#uninstall-native-btn').on('click', this.uninstallNativeGeoIP2.bind(this));
            $('#run-benchmark-btn').on('click', this.runBenchmark.bind(this));
            $('#reset-performance-metrics').on('click', this.resetPerformanceMetrics.bind(this));
            $('#advanced-mode-toggle').on('change', this.toggleAdvancedMode.bind(this));
            $('#export-geo-btn').on('click', this.exportData.bind(this));
            
            console.log('bindEvents() completed');
        },
        
        loadGeoStats: function() {
            $.ajax({
                url: affiliateHubGeo.ajax_url,
                type: 'POST',
                data: {
                    action: 'affiliate_hub_geo_stats',
                    nonce: affiliateHubGeo.nonce
                },
                success: function(response) {
                    if (response.success && this.chart) {
                        this.chart.data.labels = response.data.labels;
                        this.chart.data.datasets[0].data = response.data.data;
                        this.chart.update();
                    }
                }.bind(this),
                error: function() {
                    console.error('Failed to load geo statistics');
                }
            });
        },
        
        bulkEnrich: function() {
            var $button = $('#bulk-enrich-btn');
            var $status = $('#bulk-enrich-status');
            
            $button.prop('disabled', true).text('Processing...');
            $status.html('<div class="notice notice-info inline"><p>Starting bulk enrichment process...</p></div>');
            
            $.ajax({
                url: affiliateHubGeo.ajax_url,
                type: 'POST',
                data: {
                    action: 'affiliate_hub_bulk_enrich_geo',
                    nonce: affiliateHubGeo.nonce
                },
                success: function(response) {
                    if (response.success) {
                        $status.html('<div class="notice notice-success inline"><p>' + response.data.message + '</p></div>');
                        
                        // Refresh page after delay to show updated stats
                        setTimeout(function() {
                            location.reload();
                        }, 3000);
                    } else {
                        $status.html('<div class="notice notice-error inline"><p>Error: ' + (response.data || 'Unknown error') + '</p></div>');
                    }
                },
                error: function() {
                    $status.html('<div class="notice notice-error inline"><p>AJAX request failed.</p></div>');
                },
                complete: function() {
                    $button.prop('disabled', false).text('Enrich Old Data');
                }
            });
        },
        
        updateMaxMind: function() {
            var $button = $('#maxmind-update-btn');
            var $status = $('#maxmind-update-status');
            
            if (!$status.length) {
                $status = $('<div id="maxmind-update-status"></div>').insertAfter($button);
            }
            
            $button.prop('disabled', true).text('Updating...');
            $status.html('<div class="notice notice-info inline"><p>Downloading MaxMind database (~70MB)...</p></div>');
            
            $.ajax({
                url: affiliateHubGeo.ajax_url,
                type: 'POST',
                data: {
                    action: 'affiliate_hub_maxmind_update',
                    nonce: affiliateHubGeo.nonce
                },
                timeout: 180000, // 3 minutes timeout
                success: function(response) {
                    if (response.success) {
                        $status.html('<div class="notice notice-success inline"><p>' + response.data.message + '</p></div>');
                        // Refresh page after 2 seconds to show updated status
                        setTimeout(function() {
                            location.reload();
                        }, 2000);
                    } else {
                        $status.html('<div class="notice notice-error inline"><p>Error: ' + (response.data || 'Unknown error') + '</p></div>');
                    }
                },
                error: function(xhr, status, error) {
                    if (status === 'timeout') {
                        $status.html('<div class="notice notice-error inline"><p>Request timeout. MaxMind database download takes 1-3 minutes.</p></div>');
                    } else {
                        $status.html('<div class="notice notice-error inline"><p>AJAX request failed: ' + error + '</p></div>');
                    }
                },
                complete: function() {
                    $button.prop('disabled', false).text('Update Database');
                }
            });
        },
        
        cleanupCache: function() {
            var $button = $('#cache-cleanup-btn');
            var $status = $('#cache-cleanup-status');
            
            $button.prop('disabled', true).text('Cleaning...');
            $status.html('<div class="notice notice-info inline"><p>Cleaning expired cache entries...</p></div>');
            
            $.ajax({
                url: affiliateHubGeo.ajax_url,
                type: 'POST',
                data: {
                    action: 'affiliate_hub_cleanup_cache',
                    nonce: affiliateHubGeo.nonce
                },
                success: function(response) {
                    if (response.success) {
                        $status.html('<div class="notice notice-success inline"><p>Cache cleanup completed. ' + (response.data.deleted || 0) + ' entries removed.</p></div>');
                        
                        // Refresh page after delay
                        setTimeout(function() {
                            location.reload();
                        }, 2000);
                    } else {
                        $status.html('<div class="notice notice-error inline"><p>Error: ' + (response.data || 'Unknown error') + '</p></div>');
                    }
                },
                error: function() {
                    $status.html('<div class="notice notice-error inline"><p>AJAX request failed.</p></div>');
                },
                complete: function() {
                    $button.prop('disabled', false).text('Clean Cache');
                }
            });
        },
        
        installNativeGeoIP2: function() {
            console.log('Install Native GeoIP2 button clicked');
            var $button = $('#install-native-btn');
            var $status = $('#installer-status-display');
            
            if (!$button.length) {
                console.error('Install button not found');
                return;
            }
            
            if (!$status.length) {
                console.error('Status display element not found');
                return;
            }
            
            console.log('AJAX URL:', affiliateHubGeo.ajax_url);
            console.log('Nonce:', affiliateHubGeo.nonce);
            
            $button.prop('disabled', true).text('Installing...');
            $status.html('<div class="notice notice-info inline"><p>Downloading and installing GeoIP2 library from GitHub...</p></div>');
            
            $.ajax({
                url: affiliateHubGeo.ajax_url,
                type: 'POST',
                data: {
                    action: 'affiliate_hub_install_native_geoip2',
                    nonce: affiliateHubGeo.nonce
                },
                timeout: 180000, // 3 minutes timeout
                success: function(response) {
                    console.log('Install response:', response);
                    if (response.success) {
                        $status.html('<div class="notice notice-success inline"><p>' + response.data.message + '</p></div>');
                        setTimeout(function() {
                            location.reload();
                        }, 2000);
                    } else {
                        $status.html('<div class="notice notice-error inline"><p>Error: ' + (response.data || 'Installation failed') + '</p></div>');
                    }
                },
                error: function(xhr, status, error) {
                    console.error('Install error:', xhr, status, error);
                    if (status === 'timeout') {
                        $status.html('<div class="notice notice-error inline"><p>Installation timeout. This may take several minutes.</p></div>');
                    } else {
                        $status.html('<div class="notice notice-error inline"><p>Installation failed: ' + error + '</p></div>');
                    }
                },
                complete: function() {
                    $button.prop('disabled', false).text('Install Native GeoIP2');
                }
            });
        },
        
        uninstallNativeGeoIP2: function() {
            if (!confirm('Are you sure you want to uninstall the native GeoIP2 library? This will revert to fallback mode.')) {
                return;
            }
            
            var $button = $('#uninstall-native-btn');
            var $status = $('#installer-status-display');
            
            $button.prop('disabled', true).text('Uninstalling...');
            $status.html('<div class="notice notice-info inline"><p>Removing native GeoIP2 library...</p></div>');
            
            $.ajax({
                url: affiliateHubGeo.ajax_url,
                type: 'POST',
                data: {
                    action: 'affiliate_hub_uninstall_native_geoip2',
                    nonce: affiliateHubGeo.nonce
                },
                success: function(response) {
                    if (response.success) {
                        $status.html('<div class="notice notice-success inline"><p>' + response.data.message + '</p></div>');
                        setTimeout(function() {
                            location.reload();
                        }, 2000);
                    } else {
                        $status.html('<div class="notice notice-error inline"><p>Error: ' + (response.data || 'Uninstallation failed') + '</p></div>');
                    }
                },
                error: function(xhr, status, error) {
                    $status.html('<div class="notice notice-error inline"><p>Uninstallation failed: ' + error + '</p></div>');
                },
                complete: function() {
                    $button.prop('disabled', false).text('Uninstall Native');
                }
            });
        },
        
        runBenchmark: function() {
            var $button = $('#run-benchmark-btn');
            var $results = $('#benchmark-results');
            
            $button.prop('disabled', true).text('Running Benchmark...');
            $results.show().html('<div class="notice notice-info inline"><p>Running performance benchmark...</p></div>');
            
            $.ajax({
                url: affiliateHubGeo.ajax_url,
                type: 'POST',
                data: {
                    action: 'affiliate_hub_run_benchmark',
                    nonce: affiliateHubGeo.nonce
                },
                timeout: 60000, // 1 minute timeout
                success: function(response) {
                    if (response.success && response.data.results) {
                        var results = response.data.results;
                        var html = '<h4>Benchmark Results</h4>';
                        html += '<div class="benchmark-summary">';
                        html += '<div class="benchmark-item">';
                        html += '<strong>Fallback Mode:</strong> ' + (results.fallback_avg || 'N/A') + 'ms average';
                        html += '</div>';
                        html += '<div class="benchmark-item">';
                        html += '<strong>Native Mode:</strong> ' + (results.native_avg || 'N/A') + 'ms average';
                        html += '</div>';
                        if (results.improvement > 0) {
                            html += '<div class="benchmark-item improvement">';
                            html += '<strong>Performance Improvement:</strong> ' + results.improvement + '%';
                            html += '</div>';
                        }
                        html += '</div>';
                        $results.html(html);
                    } else {
                        $results.html('<div class="notice notice-error inline"><p>Benchmark failed: ' + (response.data || 'Unknown error') + '</p></div>');
                    }
                },
                error: function(xhr, status, error) {
                    if (status === 'timeout') {
                        $results.html('<div class="notice notice-error inline"><p>Benchmark timeout. Network may be slow.</p></div>');
                    } else {
                        $results.html('<div class="notice notice-error inline"><p>Benchmark failed: ' + error + '</p></div>');
                    }
                },
                complete: function() {
                    $button.prop('disabled', false).text('Run Benchmark');
                }
            });
        },
        
        resetPerformanceMetrics: function() {
            if (!confirm('Are you sure you want to reset all performance metrics? This cannot be undone.')) {
                return;
            }
            
            $.ajax({
                url: affiliateHubGeo.ajax_url,
                type: 'POST',
                data: {
                    action: 'affiliate_hub_reset_performance_metrics',
                    nonce: affiliateHubGeo.nonce
                },
                success: function(response) {
                    if (response.success) {
                        alert(response.data.message);
                        location.reload();
                    } else {
                        alert('Error: ' + (response.data || 'Reset failed'));
                    }
                },
                error: function() {
                    alert('Reset request failed');
                }
            });
        },
        
        toggleAdvancedMode: function() {
            var $toggle = $('#advanced-mode-toggle');
            var $options = $('#advanced-options');
            
            if ($toggle.is(':checked')) {
                $options.slideDown();
            } else {
                $options.slideUp();
            }
        },
        
        exportData: function() {
            // Create form and submit for file download
            var form = document.createElement('form');
            form.method = 'POST';
            form.action = affiliateHubGeo.ajax_url;
            form.style.display = 'none';
            
            // Add form fields
            var fields = {
                action: 'affiliate_hub_export_geo',
                nonce: affiliateHubGeo.nonce
            };
            
            for (var key in fields) {
                var input = document.createElement('input');
                input.type = 'hidden';
                input.name = key;
                input.value = fields[key];
                form.appendChild(input);
            }
            
            document.body.appendChild(form);
            form.submit();
            document.body.removeChild(form);
        }
    };
    
    // Initialize when DOM is ready
    $(document).ready(function() {
        console.log('GeoLocationAdmin JavaScript loaded');
        console.log('affiliateHubGeo object:', window.affiliateHubGeo);
        GeoLocationAdmin.init();
    });
    
})(jQuery);
