(function($){
    var chart = null;
    var scanId = null;

    // Defensive fallback: if the localized config object wasn't printed for any reason,
    // create a minimal default so the UI doesn't crash with a ReferenceError.
    if (typeof window.AffiliateHubLinkScanner === 'undefined') {
        window.AffiliateHubLinkScanner = {
            ajax_url: (typeof ajaxurl !== 'undefined') ? ajaxurl : '/wp-admin/admin-ajax.php',
            nonce: '',
            chart: { disable_animations: false, debounce_ms: 200 },
            strings: {}
        };
    }

    var lastChartData = null;
        var chartCanvas = null;
        var chartCtx = null;

        // Read settings from localized object
        var chartDisableAnimations = false;
        var pollDebounceMs = 2000;
        try {
            chartDisableAnimations = !!window.AffiliateHubLinkScanner.chart.disable_animations;
            pollDebounceMs = parseInt(window.AffiliateHubLinkScanner.chart.debounce_ms, 10) || 2000;
        } catch (e) {
            // ignore and use defaults
        }

    // Debug helpers (temporary) - set to true to enable banner and detailed logs
    var AH_DEBUG = true;
    function initDebugBanner(){
        if (!AH_DEBUG) return;
        if ($('#ah-debug-banner').length === 0) {
            var $b = $('<div id="ah-debug-banner" style="font-size:12px;padding:8px;border:1px solid #d1d1d1;background:#fffbea;color:#333;margin-bottom:10px;border-radius:3px;">Debug: idle</div>');
            // insert near the controls so it's visible in the admin area
            $b.insertBefore('#ah-scan-status');
        }
    }

    function debugLog(entry){
        try {
            if (AH_DEBUG) {
                console.log('[AffiliateHub LinkScanner DEBUG]', entry);
                var $b = $('#ah-debug-banner');
                if ($b.length) {
                    var text = (new Date()).toLocaleTimeString() + ' — ' + (entry.action || 'unknown') + ' — ' + (entry.status || '') + ' ' + (entry.note || '');
                    $b.text(text);
                }
            }
        } catch(e) { /* ignore */ }
    }

    // Ensure the ApexCharts container is sized appropriately 
    function ensureSquareChart() {
        try {
            var $wrapper = $('.chart-wrapper');
            if ($wrapper.length === 0) return;
            var $container = $wrapper.find('#ah-scan-chart');
            if ($container.length === 0) return;
            var cw = Math.floor($wrapper.innerWidth());
            // limit to max width from CSS (420) to match layout
            cw = Math.min(cw, 420);
            // ApexCharts handles responsive sizing automatically
            if (chart && typeof chart.updateOptions === 'function') {
                try { 
                    chart.updateOptions({
                        chart: {
                            width: cw,
                            height: cw
                        }
                    });
                } catch(e) { /* ignore */ }
            }
        } catch(e) { /* ignore */ }
    }

    // Debounced resize handler
    var _resizeTimer = null;
    $(window).on('resize', function(){ clearTimeout(_resizeTimer); _resizeTimer = setTimeout(ensureSquareChart, 150); });

    // Centralized AJAX wrapper with debug logging
    function ajaxCall(method, url, data, cb) {
        var t0 = Date.now();
        debugLog({action: data.action, note: 'request', params: data});
        var jq;
        if (method === 'GET') {
            jq = $.get(url, data)
        } else {
            jq = $.post(url, data);
        }
        jq.done(function(resp, textStatus, jqxhr){
            debugLog({action: data.action, status: jqxhr.status || 200, note: 'done'});
            if (typeof cb === 'function') cb(resp, textStatus, jqxhr);
        }).fail(function(jqxhr, textStatus, errorThrown){
            var status = jqxhr.status || 0;
            debugLog({action: data.action, status: status, note: 'fail: ' + (errorThrown || textStatus)});
            // try to parse json body
            var body = jqxhr.responseText || '';
            try { console.error('AJAX fail body:', JSON.parse(body)); } catch(e){ console.error('AJAX fail body (raw):', body); }
            if (typeof cb === 'function') cb(null, textStatus, jqxhr);
        });
    }

    function renderChart(data, labels, colors){
        // Check if ApexCharts is available
        if (typeof ApexCharts === 'undefined') {
            console.warn('ApexCharts not loaded yet, skipping chart render');
            return;
        }
        
        // Check if chart element exists
        if (!chartCtx) {
            chartCanvas = document.getElementById('ah-scan-chart');
            if (!chartCanvas) {
                console.warn('Chart container element not found, skipping chart render');
                return;
            }
            chartCtx = chartCanvas; // For ApexCharts, we use the div directly
        }

        // OPTIMIZED: Skip redraw if data unchanged
        try {
            var dataString = JSON.stringify({data: data, labels: labels});
            if (lastChartData === dataString) return;
            lastChartData = dataString;
        } catch(e) { /* ignore and proceed to update */ }
        
        var container = chartCtx;
        if (!chart) {
            try {
                var options = {
                    series: data || [0, 0, 0, 0],
                    labels: labels || ['Active','Issues','Errors','Pending'],
                    colors: colors || ['#28a745','#ffc107','#dc3545','#6c757d'],
                    chart: {
                        type: 'donut',
                        width: 420,
                        height: 420,
                        animations: {
                            enabled: !chartDisableAnimations,
                            easing: 'easeinout',
                            speed: chartDisableAnimations ? 0 : 200,
                            animateGradually: {
                                enabled: !chartDisableAnimations,
                                delay: 150
                            },
                            dynamicAnimation: {
                                enabled: !chartDisableAnimations,
                                speed: 150
                            }
                        }
                    },
                    plotOptions: {
                        pie: {
                            donut: {
                                size: '45%'
                            }
                        }
                    },
                    legend: {
                        position: 'bottom',
                        fontSize: window.innerWidth < 480 ? '10px' : '12px',
                        markers: {
                            width: 12,
                            height: 12
                        },
                        itemMargin: {
                            horizontal: 15,
                            vertical: 5
                        }
                    },
                    tooltip: {
                        theme: 'light',
                        style: {
                            fontSize: window.innerWidth < 480 ? '11px' : '12px'
                        }
                    },
                    responsive: [{
                        breakpoint: 480,
                        options: {
                            chart: {
                                width: 300,
                                height: 300
                            },
                            legend: {
                                fontSize: '10px'
                            }
                        }
                    }]
                };
                
                chart = new ApexCharts(container, options);
                chart.render();
            } catch(e) {
                console.error('Error creating ApexCharts chart:', e);
                return;
            }
        } else {
            // OPTIMIZED: Update existing chart efficiently
            try {
                chart.updateSeries(data || [0, 0, 0, 0]);
                if (labels) {
                    chart.updateOptions({
                        labels: labels,
                        colors: colors || ['#28a745','#ffc107','#dc3545','#6c757d']
                    });
                }
            } catch (e) {
                console.error('Error updating ApexCharts chart:', e);
            }
        }
    }

    function pollStatus(){
        if(!scanId) return;
        ajaxCall('GET', AffiliateHubLinkScanner.ajax_url, {action:'affiliate_hub_get_scan_status', scan_id:scanId, nonce:AffiliateHubLinkScanner.nonce}, function(resp, status, jqxhr){
            if(resp && resp.success){
                var s = resp.data;
                // OPTIMIZED: Simplified status calculation
                var active = s.counts ? (s.counts.active || 0) : 0;
                var broken = s.counts ? (s.counts.broken || 0) : 0;
                var forbidden = s.counts ? (s.counts.forbidden || 0) : 0;
                var not_found = s.counts ? (s.counts.not_found || 0) : 0;
                var server_error = s.counts ? (s.counts.server_error || 0) : 0;
                var network_error = s.counts ? (s.counts.network_error || 0) : 0;
                var skipped = s.counts ? (s.counts.skipped || 0) : 0;
                var failed = s.counts ? (s.counts.failed || 0) : 0;
                
                var total_processed = active + broken + forbidden + not_found + server_error + network_error + skipped + failed;
                var pending = Math.max(0, s.total_urls - total_processed);
                // If a scan is very large, use a lightweight progress bar instead of ApexCharts to avoid rAF
                var largeThreshold = 1000;
                if (s.total_urls >= largeThreshold) {
                    // Enter progress mode
                    $('.chart-wrapper').addClass('progress-mode');
                    var processed = s.processed || 0;
                    var pct = s.total_urls > 0 ? Math.min(100, Math.round((processed / s.total_urls) * 100)) : 0;
                    $('.ah-progress-fill').css('width', pct + '%');
                    $('.ah-progress-text').text(processed + ' / ' + s.total_urls);
                } else {
                    $('.chart-wrapper').removeClass('progress-mode');
                }
                // OPTIMIZED: Group similar errors for cleaner chart
                var errors = broken + server_error + failed; // Group generic errors including failed
                var issues = forbidden + not_found + network_error; // Group access issues
                
                // Only show categories with non-zero values for cleaner visualization
                var chartData = [];
                var chartLabels = [];
                var chartColors = [];
                
                if (active > 0) {
                    chartData.push(active);
                    chartLabels.push('Active');
                    chartColors.push('#28a745');
                }
                if (issues > 0) {
                    chartData.push(issues);
                    chartLabels.push('Access Issues');
                    chartColors.push('#ffc107');
                }
                if (errors > 0) {
                    chartData.push(errors);
                    chartLabels.push('Errors');
                    chartColors.push('#dc3545');
                }
                if (pending > 0) {
                    chartData.push(pending);
                    chartLabels.push('Pending');
                    chartColors.push('#6c757d');
                }
                
                renderChart(chartData, chartLabels, chartColors);
                // Update status text only when changed
                var statusText = 'Processed: '+s.processed+' / '+s.total_urls+' Status: '+s.status;
                if ($('#ah-scan-status-text').data('status-text') !== statusText) {
                    $('#ah-scan-status-text').data('status-text', statusText).text(statusText);
                }
                // reflect paused/canceled state
                if (s.paused && parseInt(s.paused) === 1) {
                    $('#ah-pause-scan').hide();
                    $('#ah-resume-scan').show();
                } else {
                    $('#ah-pause-scan').show();
                    $('#ah-resume-scan').hide();
                }

                // BUGFIX: Show force finish button for stuck scans (all processed but still running)
                var isStuck = (s.status === 'running' && s.processed >= s.total_urls && s.total_urls > 0);
                if (isStuck) {
                    $('#ah-force-finish').show();
                } else {
                    $('#ah-force-finish').hide();
                }

                if (s.canceled && parseInt(s.canceled) === 1) {
                    // Scan canceled: reset controls
                    $('#ah-pause-scan,#ah-resume-scan,#ah-cancel-scan,#ah-force-finish').hide();
                    $('#ah-start-scan').prop('disabled', false);
                    scanId = null;
                }

                if(s.status !== 'finished' && s.status !== 'completed'){
                    // Poll again after configured debounce interval.
                    // Only fetch incremental updates for visible rows so status/code badges update live
                    // Ask for recent rows since lastMaxId to fetch any updated statuses
                    console.log('[DEBUG] Polling continues, status:', s.status, 'processed:', s.processed, 'total:', s.total_urls);
                    fetchAndApplyRowUpdates();
                    // loadDetailsPage() removed to prevent duplicate AJAX calls during polling
                    setTimeout(pollStatus, pollDebounceMs);
                } else {
                    // Scan finished - reset button states and keep chart visible with final results
                    console.log('[DEBUG] Scan finished, status:', s.status, 'resetting button states but keeping chart visible');
                    
                    // Reset button states to initial state
                    $('#ah-start-scan').prop('disabled', false).show();
                    $('#ah-pause-scan, #ah-resume-scan, #ah-cancel-scan, #ah-force-finish').hide();
                    
                    // Remove progress mode if it was enabled
                    $('.chart-wrapper').removeClass('progress-mode');
                    
                    // IMPORTANT: Keep chart visible with final results - don't reload page
                    // Update status text to show completion
                    $('#ah-scan-status-text').text('Scan completed: '+s.processed+' / '+s.total_urls+' links processed');
                    
                    // Clear scan ID to stop polling
                    scanId = null;
                    
                    // Load final detailed results in table without page reload
                    loadDetailsPage();
                    
                    // Make sure persistent controls are visible (we render them server-side)
                    $('.ah-legend-controls').addClass('show');
                }
            }
        });
    }

    // Fetch recent details (using since_id) and update any visible rows in the table
    function fetchAndApplyRowUpdates(){
        if (!scanId) return;
        console.log('[DEBUG] fetchAndApplyRowUpdates called, scanId:', scanId);
        
        // Get filter values from our form elements 
        var includeIgnored = $('#ah-show-ignored').is(':checked') ? 1 : 0;
        var search = $('#ah-search').val() || '';
        var status = $('#ah-filter-status').val() || '';
        var since_id = lastMaxIdByScan[scanId] || 0;
        
        ajaxCall('GET', AffiliateHubLinkScanner.ajax_url, {action: AffiliateHubLinkScanner.endpoints.details || 'affiliate_hub_scan_details', scan_id:scanId, page:1, per_page:20, include_ignored:includeIgnored, search: search, status: status, since_id: since_id, nonce:AffiliateHubLinkScanner.nonce}, function(resp){
            if (resp && resp.success) {
                var links = resp.data.links || [];
                links.forEach(function(l){
                    // Update existing row if visible
                    var $row = $('tr[data-link-id="'+l.id+'"]');
                    if ($row.length) {
                        // Update status badge
                        var status = l.status || '';
                        var statusColor = '#6c757d';
                        if (status === 'active') statusColor = '#28a745';
                        if (status === 'broken') statusColor = '#dc3545';
                        if (status === 'forbidden') statusColor = '#ffc107';
                        
                        // Try both badge classes for compatibility
                        var $statusBadge = $row.find('.column-status .ah-status-badge, .column-status .affiliate-hub-redirect-type');
                        if ($statusBadge.length) {
                            $statusBadge.css('background-color', statusColor).text(status);
                        } else {
                            $row.find('.column-status').html('<span class="affiliate-hub-redirect-type" style="background-color:'+statusColor+';color:#fff;">'+(status || '')+'</span>');
                        }

                        // Update code badge
                        var code = l.status_code ? l.status_code : '';
                        var codeColor = '#6c757d';
                        if (code >= 200 && code < 300) codeColor = '#28a745';
                        else if (code >= 400 && code < 500) codeColor = '#dc3545';
                        else if (code >= 500) codeColor = '#b02a37';
                        
                        var $codeBadge = $row.find('.column-code .ah-code-badge, .column-code .affiliate-hub-redirect-type');
                        if ($codeBadge.length) {
                            $codeBadge.css('background-color', codeColor).text(code);
                        } else {
                            $row.find('.column-code').html('<span class="affiliate-hub-redirect-type" style="background-color:'+codeColor+';color:#fff;">'+(code || '')+'</span>');
                        }

                        // Update note
                        var note = l.note ? $('<div>').text(l.note).html() : '';
                        $row.find('.column-note').html(note);
                        
                        // Update post-type if it exists
                        var postType = l.post_type || '';
                        var postTypeLabel = '';
                        if (postType === 'post') postTypeLabel = 'Wpis';
                        else if (postType === 'page') postTypeLabel = 'Strona';  
                        else if (postType) postTypeLabel = postType;
                        var $postTypeCell = $row.find('.column-post-type');
                        if ($postTypeCell.length) {
                            $postTypeCell.html(postTypeLabel);
                        }

                        // Quick highlight animation to indicate update
                        $row.addClass('ah-row-updated');
                        setTimeout(function(){ $row.removeClass('ah-row-updated'); }, 900);
                    }
                });
                // update lastMaxId for incremental polling
                if (resp.data.max_id && scanId) {
                    lastMaxIdByScan[scanId] = Math.max(lastMaxIdByScan[scanId] || 0, resp.data.max_id);
                }
            }
        });
    }

    // Initialize on page load
    $(function(){
        initDebugBanner();
        // ApexCharts handles canvas sizing automatically
    });

    // Ensure the WP list-table has visible column headers; if thead is missing
    // or headers are empty (some admin themes may strip them), inject a
    // defensive header row that matches our columns so layout is clear.
    function ensureTableHeaders() {
        try {
            var $table = $('#linkscanner-list-form table.wp-list-table');
            if ($table.length === 0) {
                $table = $('.ah-details-table-wrap table.wp-list-table');
            }
            if ($table.length === 0) return;

            var $thead = $table.children('thead');
            var hasVisible = false;
            if ($thead.length) {
                // check if any th contains non-empty text
                $thead.find('th').each(function(){ if ($(this).text().trim().length > 0) hasVisible = true; });
            }

            if (!hasVisible) {
                // build header markup
                var ths = '';
                ths += '<th scope="col" id="cb" class="manage-column column-cb check-column"><input type="checkbox" /></th>';
                ths += '<th scope="col" class="manage-column column-link">Link</th>';
                ths += '<th scope="col" class="manage-column column-label">Link Label</th>';
                ths += '<th scope="col" class="manage-column column-post">Post ID</th>';
                ths += '<th scope="col" class="manage-column column-status">Status</th>';
                ths += '<th scope="col" class="manage-column column-code">Code</th>';
                ths += '<th scope="col" class="manage-column column-note">Note</th>';
                ths += '<th scope="col" class="manage-column column-actions">Actions</th>';

                var $newThead = $('<thead class="ah-generated-thead"><tr>' + ths + '</tr></thead>');

                // remove existing empty thead if present
                if ($thead.length) { $thead.remove(); }
                // insert before tbody
                $table.prepend($newThead);
                // ensure CSS overrides apply
                $newThead.find('th').css({ 'visibility': 'visible' });
                // After injecting headers, sync widths to avoid misalignment
                try { syncTableHeaderWidths($table); } catch(e) { /* ignore */ }
            }
        } catch(e) { /* ignore */ }
    }

    // Run once on DOM ready to guard against missing headers
    $(function(){ ensureTableHeaders(); });

    // Sync column layout by inserting/updating a <colgroup> based on the
    // computed widths of the first visible tbody row. Using <colgroup>
    // ensures the browser uses identical column sizing for both thead
    // and tbody even when admin themes apply strange styles.
    var _syncTimer = null;
    // Temporary debug logging for colgroup syncing
    function debugSyncTableHeaderWidths($table, widths) {
        try {
            console.log('Syncing table header widths:', widths);
            var $colgroup = $table.children('colgroup');
            if ($colgroup.length === 0) {
                console.warn('No colgroup found, creating a new one.');
            }
            widths.forEach(function(w, i) {
                console.log(`Column ${i + 1}: ${w}px`);
            });
        } catch (e) {
            console.error('Error during debugSyncTableHeaderWidths:', e);
        }
    }

    // Ensure colgroup syncing happens after table rendering
    function syncTableHeaderWidths($table) {
        try {
            $table = $table || $('#linkscanner-list-form table.wp-list-table');
            if ($table.length === 0) {
                $table = $('.ah-details-table-wrap table.wp-list-table');
            }
            if ($table.length === 0) return;

            var $tbody = $table.find('tbody');
            if ($tbody.length === 0) return;
            var $firstRow = $tbody.find('tr').filter(function(){ return $(this).find('td').length > 0; }).first();
            if ($firstRow.length === 0) return;

            // Measure widths of each TD in the first row
            var widths = [];
            $firstRow.children('td').each(function(i, td){
                try {
                    var rect = td.getBoundingClientRect();
                    widths.push(Math.max(30, Math.ceil(rect.width))); // enforce small min
                } catch(e) { widths.push(80); }
            });

            if (widths.length === 0) return;

            // Build or update a <colgroup> with matching widths
            var $colgroup = $table.children('colgroup');
            if ($colgroup.length === 0) {
                $colgroup = $('<colgroup></colgroup>');
                $table.prepend($colgroup);
            } else {
                $colgroup.empty();
            }
            widths.forEach(function(w){
                var $c = $('<col>').css('width', w + 'px');
                $colgroup.append($c);
            });

            // Force table to use border-box so col widths match measured pixels
            $table.css('box-sizing', 'border-box');

            // Apply defensive CSS rules to ensure alignment
            $table.find('thead th, tbody td').css({
                'box-sizing': 'border-box',
                'padding-left': '12px',
                'padding-right': '12px'
            });
        } catch(e) {
            console.error('Error during syncTableHeaderWidths:', e);
        }
    }

    // Ensure syncing happens after table rendering
    $(document).ready(function() {
        var $table = $('#linkscanner-list-form table.wp-list-table');
        if ($table.length === 0) {
            $table = $('.ah-details-table-wrap table.wp-list-table');
        }
        syncTableHeaderWidths($table);
    });

    // Debounced window resize to re-sync column group
    $(window).on('resize', function(){ clearTimeout(_syncTimer); _syncTimer = setTimeout(function(){ syncTableHeaderWidths(); }, 150); });

    // Intercept WP_List_Table bulk action form submit
    $(document).on('submit', '#linkscanner-list-form', function(e){
        // If form is simple GET search, let it pass through
        var hasBulk = $('#bulk-action-selector-top, #bulk-action-selector-bottom').length > 0;
        if (!hasBulk) return;
        // prevent full page submit when our ajax handler available
        e.preventDefault();
        var action = $('#bulk-action-selector-top').val() || $('#bulk-action-selector-bottom').val();
        if (!action || action === '-1') {
            // fallback to normal behavior (search)
            this.submit();
            return;
        }
        // collect checked items
        var items = [];
        $('#linkscanner-list-form input[name="scanned[]"]:checked').each(function(){ items.push($(this).val()); });
        if (items.length === 0) { alert('No items selected'); return; }
        $.post(AffiliateHubLinkScanner.ajax_url, {action:'affiliate_hub_linkscanner_bulk_action', bulk_action: action, items: items, nonce: AffiliateHubLinkScanner.nonce}, function(resp){
            if (resp && resp.success) {
                // reload to show updates
                location.reload();
            } else {
                alert('Bulk action failed');
            }
        });
    });

    // Global AJAX error handler to surface server errors in admin UI and console
    $(document).ajaxError(function(event, jqxhr, settings, thrownError){
        console.error('AffiliateHub LinkScanner AJAX error:', settings, jqxhr.status, thrownError);
        // Show a transient notice near the controls
        if ($('#ah-ajax-error').length === 0) {
            $('<div id="ah-ajax-error" style="background:#fff3cd;border:1px solid #ffeeba;padding:8px;margin-top:10px;border-radius:3px;color:#856404;">Server error communicating with Link Scanner — check logs.</div>').insertBefore('#ah-details-table');
            setTimeout(function(){ $('#ah-ajax-error').fadeOut(300, function(){ $(this).remove(); }); }, 8000);
        }
    });

    // Populate post-types multiselect
    $(function(){
        try {
            var pts = window.AffiliateHubLinkScanner.post_types || [];
            var $sel = $('#ah-post-types');
            pts.forEach(function(p){
                var $opt = $('<option>').attr('value', p.name).text(p.label || p.name);
                $sel.append($opt);
            });
        } catch(e) {
            // ignore
        }
    });

    $(document).on('click', '#ah-start-scan', function(e){
        e.preventDefault();
        
        // Get scan type
        var scanType = $('input[name="ah-scan-type"]:checked').val() || 'content';
        
        // Gather selected post types (only for content scan)
        var pts = [];
        if (scanType === 'content') {
            pts = $('#ah-post-types').val() || [];
        }
        
        $.post(AffiliateHubLinkScanner.ajax_url, {
            action: AffiliateHubLinkScanner.endpoints.start || 'affiliate_hub_start_scan', 
            nonce: AffiliateHubLinkScanner.nonce, 
            post_types: pts,
            scan_type: scanType
        }, function(resp){
            if(resp.success){
                scanId = resp.data.scan_id;
                // reset incremental tracking
                lastMaxIdByScan[scanId] = 0;
                pollStatus();
                // show pause/cancel controls
                $('#ah-pause-scan').show();
                $('#ah-cancel-scan').show();
                $('#ah-start-scan').prop('disabled', true);
            }
        });
    });

    $(document).on('click', '#ah-pause-scan', function(e){
        e.preventDefault();
        if(!scanId) return;
        $.post(AffiliateHubLinkScanner.ajax_url, {action: AffiliateHubLinkScanner.endpoints.pause, scan_id: scanId, nonce: AffiliateHubLinkScanner.nonce}, function(resp){
            if(resp.success){
                $('#ah-pause-scan').hide();
                $('#ah-resume-scan').show();
            }
        });
    });

    $(document).on('click', '#ah-resume-scan', function(e){
        e.preventDefault();
        if(!scanId) return;
        $.post(AffiliateHubLinkScanner.ajax_url, {action: AffiliateHubLinkScanner.endpoints.resume, scan_id: scanId, nonce: AffiliateHubLinkScanner.nonce}, function(resp){
            if(resp.success){
                $('#ah-resume-scan').hide();
                $('#ah-pause-scan').show();
            }
        });
    });

    $(document).on('click', '#ah-cancel-scan', function(e){
        e.preventDefault();
        if(!scanId) return;
        if(!confirm('Cancel this scan?')) return;
        $.post(AffiliateHubLinkScanner.ajax_url, {action: AffiliateHubLinkScanner.endpoints.cancel, scan_id: scanId, nonce: AffiliateHubLinkScanner.nonce}, function(resp){
            if(resp.success){
                // reset UI
                scanId = null;
                $('#ah-pause-scan,#ah-resume-scan,#ah-cancel-scan,#ah-force-finish').hide();
                $('#ah-start-scan').prop('disabled', false);
                renderChart([0,0,0,0]);
                // Reset progress bar but keep list table visible
                $('.ah-progress-fill').css('width', '0%');
                $('.ah-progress-text').text('0 / 0');
            }
        });
    });

    // BUGFIX: Force finish stuck scan
    $(document).on('click', '#ah-force-finish', function(e){
        e.preventDefault();
        if(!scanId) return;
        if(!confirm('Force finish this scan? This will mark it as completed even if some links are still pending.')) return;
        $.post(AffiliateHubLinkScanner.ajax_url, {action: AffiliateHubLinkScanner.endpoints.force_finish, scan_id: scanId, nonce: AffiliateHubLinkScanner.nonce}, function(resp){
            if(resp.success){
                // reset UI
                scanId = null;
                $('#ah-pause-scan,#ah-resume-scan,#ah-cancel-scan,#ah-force-finish').hide();
                $('#ah-start-scan').prop('disabled', false);
                $('#ah-scan-status-text').text('Scan force-finished');
            }
        });
    });

    // Temporary on-page toggle: disable/enable chart animations and persist via AJAX
    $(document).on('change', '#ah-disable-animations', function(){
        var disabled = $(this).is(':checked') ? 1 : 0;
        // Optimistically apply client-side
        chartDisableAnimations = !!disabled;
        if (chart) {
            try { chart.options.animation = chartDisableAnimations ? false : { duration: 200 }; } catch(e) {}
        }
        // Persist to server
        $.post(AffiliateHubLinkScanner.ajax_url, {action:'affiliate_hub_toggle_scanner_animation', nonce:AffiliateHubLinkScanner.nonce, disable: disabled}, function(resp){
            // ignore response for now
        });
    });

    $(document).on('click', '#ah-export-scan', function(e){
        if(!scanId) return;
        window.location = AffiliateHubLinkScanner.ajax_url + '?action=affiliate_hub_export_scan&scan_id=' + scanId + '&nonce=' + AffiliateHubLinkScanner.nonce;
    });

    $(document).on('click', '#ah-show-details', function(e){
        if(!scanId) return;
        loadDetailsPage(1);
    });

    var lastMaxIdByScan = {};
    // Initialize scanId from server-rendered form if available
    $(function(){
        var $form = $('#linkscanner-list-form');
        if ($form.length && $form.data('scan-id')) {
            scanId = parseInt($form.data('scan-id'), 10) || null;
            if (scanId) lastMaxIdByScan[scanId] = 0;
        }
    });
    function loadDetailsPage(page, incremental){
        incremental = !!incremental;
        console.log('[DEBUG] loadDetailsPage called with page:', page, 'incremental:', incremental);
        if (!scanId) return; // avoid firing requests without a scan id
        var includeIgnored = $('#ah-show-ignored').is(':checked') ? 1 : 0;
        var search = $('#ah-search').val() || '';
        var status = $('#ah-filter-status').val() || '';
        var post_type = ($('#ah-post-types').val() || []).join(',');
        var since_id = 0;
        if (incremental && scanId && lastMaxIdByScan[scanId]) since_id = lastMaxIdByScan[scanId];
        ajaxCall('GET', AffiliateHubLinkScanner.ajax_url, {action: AffiliateHubLinkScanner.endpoints.details || 'affiliate_hub_scan_details', scan_id:scanId, page:page, per_page:20, include_ignored:includeIgnored, search: search, status: status, post_type: post_type, since_id: since_id, nonce:AffiliateHubLinkScanner.nonce}, function(resp){
            if(resp && resp.success){
                var links = resp.data.links;
                var total = resp.data.total;
                var per_page = resp.data.per_page;
                var cur_page = resp.data.page;
                // Prefer WP_List_Table tbody if present, otherwise fallback to legacy #ah-details-table
                var $tbody = $('#linkscanner-list-form table.wp-list-table tbody');
                if ($tbody.length === 0) {
                    $tbody = $('#ah-details-table tbody');
                }
                // If not incremental, clear and render
                if (!incremental) { $tbody.empty(); }
                links.forEach(function(l){
                    // small SVG icons
                    var svgOpen = '<svg width="14" height="14" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg" style="vertical-align:middle;margin-right:6px"><path d="M14 3H21V10" stroke="#444" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"/><path d="M10 14L21 3" stroke="#444" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"/><path d="M21 21H3V3H11" stroke="#444" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"/></svg>';
                    var svgLabel = '<svg width="12" height="12" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg" style="vertical-align:middle;margin-right:6px"><path d="M20 12v7a1 1 0 0 1-1 1H5a1 1 0 0 1-1-1V7a1 1 0 0 1 1-1h7" stroke="#666" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"/><path d="M17 3v4" stroke="#666" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"/></svg>';
                    var svgPost = '<svg width="12" height="12" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg" style="vertical-align:middle;margin-right:6px"><path d="M3 7h18" stroke="#444" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"/><path d="M6 7v11a2 2 0 0 0 2 2h8" stroke="#444" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"/></svg>';

                    var label = l.anchor_text ? $('<div>').text(l.anchor_text).html() : '';
                    var postLink = l.post_id ? '<a href="'+(typeof ajaxurl !== 'undefined' ? ('/wp-admin/post.php?post='+l.post_id+'&action=edit') : '#')+'" target="_blank" title="Open post">'+ svgPost + '#' + l.post_id +'</a>' : '';
                    var status = l.status || '';
                    var statusColor = '#6c757d';
                    if (status === 'active') statusColor = '#28a745';
                    if (status === 'broken') statusColor = '#dc3545';
                    if (status === 'forbidden') statusColor = '#ffc107';

                    var code = l.status_code ? l.status_code : '';
                    var codeColor = '#6c757d';
                    if (code >= 200 && code < 300) codeColor = '#28a745';
                    else if (code >= 400 && code < 500) codeColor = '#dc3545';
                    else if (code >= 500) codeColor = '#b02a37';

                    var $tr = $('<tr>').attr('data-link-id', l.id);
                    // If appending to the WP_List_Table, include the checkbox column first to match server table
                    var $tableElem = $tbody.closest('table');
                    var needsCheckbox = $tableElem.find('th.check-column, th.manage-column.column-cb').length > 0;
                    if (needsCheckbox) {
                        $tr.append($('<td class="check-column">').html('<input type="checkbox" name="scanned[]" value="'+l.id+'" />'));
                    }
                    // Link column: icon + URL
                    var openLinkHtml = '<a href="'+l.url+'" target="_blank" rel="noopener noreferrer" title="Open link">'+ svgOpen + $('<div>').text(l.url).html() +'</a>';
                    $tr.append($('<td class="column-link">').html(openLinkHtml));
                    // Link Label column: small icon + label text
                    $tr.append($('<td class="column-label">').html(svgLabel + '<span>'+ label +'</span>'));
                    // Post ID column
                    $tr.append($('<td class="column-post">').html(postLink));
                    // Post Type column
                    var postType = l.post_type || '';
                    var postTypeLabel = '';
                    if (postType === 'post') postTypeLabel = 'Wpis';
                    else if (postType === 'page') postTypeLabel = 'Strona';  
                    else if (postType) postTypeLabel = postType;
                    $tr.append($('<td class="column-post-type">').html(postTypeLabel));
                    // Status column (colored badge)
                    $tr.append($('<td class="column-status">').html('<span class="ah-status-badge" style="display:inline-block;padding:4px 8px;border-radius:12px;background:'+statusColor+';color:#fff;font-weight:600;">'+(status || '')+'</span>'));
                    // Code column (colored small badge)
                    $tr.append($('<td class="column-code">').html('<span class="ah-code-badge" style="display:inline-block;padding:3px 7px;border-radius:8px;background:'+codeColor+';color:#fff;">'+(code || '')+'</span>'));
                    // Note
                    var note = l.note ? $('<div>').text(l.note).html() : '';
                    $tr.append($('<td class="column-note">').html(note));
                    // Actions: same as before
                    if (l.ignored && parseInt(l.ignored) === 1) {
                        $tr.append($('<td class="column-actions">').html('<button class="button ah-unignore-link">Un-ignore</button>'));
                    } else {
                        $tr.append($('<td class="column-actions">').html('<button class="button ah-retry-link">Retry</button> <button class="button ah-ignore-link">Ignore</button>'));
                    }
                    $tbody.append($tr);
                });
                // update lastMaxId for incremental polling
                if (resp.data.max_id && scanId) {
                    lastMaxIdByScan[scanId] = Math.max(lastMaxIdByScan[scanId] || 0, resp.data.max_id);
                }
                // pagination
                var pages = Math.ceil(total / per_page);
                var html = '<div class="ah-pagination" style="margin-top:12px;">';
                for(var p=1;p<=pages;p++){
                    if (p === cur_page) {
                        html += '<button class="button ah-page active" data-page="'+p+'">'+p+'</button> ';
                    } else {
                        html += '<button class="button ah-page" data-page="'+p+'">'+p+'</button> ';
                    }
                }
                html += '</div>';
                // Place pagination after the table we are using
                var $table = $('#linkscanner-list-form table.wp-list-table');
                if ($table.length === 0) {
                    $table = $('#ah-details-table');
                }
                if ($('#ah-details-pagination').length === 0) {
                    $('<div id="ah-details-pagination"></div>').insertAfter($table);
                }
                $('#ah-details-pagination').html(html);
                // Ensure headers exist after rendering/adding rows
                ensureTableHeaders();
            }
        });
    }

    $(document).on('click', '.ah-page', function(){
        var p = $(this).data('page');
        loadDetailsPage(p);
    });

    // Re-load details when filters change
    // Wire both inline filters and top filters to refresh table
    $(document).on('change', '#ah-filter-status, #ah-post-types, #ah-show-ignored, #ah-filter-status-top, #ah-show-ignored-top', function(){
        // Sync top/bottom controls
        var topVal = $('#ah-filter-status-top').val();
        if (typeof topVal !== 'undefined') { $('#ah-filter-status').val(topVal); }
        var topShow = $('#ah-show-ignored-top').is(':checked');
        $('#ah-show-ignored').prop('checked', topShow);
        loadDetailsPage(1);
    });

    // Debounced search for top search box
    var searchTimer = null;
    $(document).on('input', '#ah-search-top', function(){
        clearTimeout(searchTimer);
        searchTimer = setTimeout(function(){ loadDetailsPage(1); }, 450);
    });
    // Prevent full form submit and use AJAX
    $(document).on('submit', '#ah-details-search-form', function(e){
        e.preventDefault();
        loadDetailsPage(1);
    });

    $(document).on('click', '.ah-retry-link', function(){
        var tr = $(this).closest('tr');
        var id = tr.data('link-id');
        $.post(AffiliateHubLinkScanner.ajax_url, {action:'affiliate_hub_retry_link', link_id:id, nonce:AffiliateHubLinkScanner.nonce}, function(resp){
            if(resp.success){
                tr.find('.status').text('pending');
            }
        });
    });

    $(document).on('click', '.ah-ignore-link', function(){
        var tr = $(this).closest('tr');
        var id = tr.data('link-id');
        if (!confirm('Are you sure you want to ignore this link?')) {
            return;
        }
        $.post(AffiliateHubLinkScanner.ajax_url, {action:'affiliate_hub_ignore_link', link_id:id, ignore:1, nonce:AffiliateHubLinkScanner.nonce}, function(resp){
            if(resp.success){
                tr.remove();
            }
        });
    });

    $(document).on('click', '.ah-unignore-link', function(){
        var tr = $(this).closest('tr');
        var id = tr.data('link-id');
        $.post(AffiliateHubLinkScanner.ajax_url, {action:'affiliate_hub_unignore_link', link_id:id, nonce:AffiliateHubLinkScanner.nonce}, function(resp){
            if(resp.success){
                // refresh current page
                var cur = $('.ah-page.active').data('page') || 1;
                loadDetailsPage(cur);
            }
        });
    });

    // Toggle include ignored
    $(document).on('change', '#ah-show-ignored', function(){
        loadDetailsPage(1);
    });

    // Initialize chart when ApexCharts is ready
    function initializeChart() {
        try {
            // Check if ApexCharts and element are available
            if (typeof ApexCharts === 'undefined') {
                console.log('ApexCharts not available yet');
                return false; // Not ready yet
            }
            
            var chartElement = document.getElementById('ah-scan-chart');
            if (!chartElement) {
                console.log('Chart canvas element not found');
                return false;
            }
            
            console.log('Attempting to create ApexCharts chart...');
            // Create an empty chart immediately so UI doesn't look blank before a scan starts
            renderChart([0,0,0,1], ['No Data'], ['#6c757d']);
            console.log('ApexCharts chart created successfully');
            return true; // Successfully initialized
        } catch (e) { 
            console.error('Error initializing empty chart on load', e);
            return false;
        }
    }

    // Ensure chart is visible immediately on page load (show empty doughnut)
    $(function(){
        // Try multiple initialization strategies
        var chartInitialized = false;
        
        // Strategy 1: DOM Ready
        $(document).ready(function() {
            if (!chartInitialized && initializeChart()) {
                chartInitialized = true;
                console.log('Chart initialized on DOM ready');
                return;
            }
        });
        
        // Strategy 2: Window Load (after all resources including scripts)
        $(window).on('load', function() {
            if (!chartInitialized && initializeChart()) {
                chartInitialized = true;
                console.log('Chart initialized on window load');
                return;
            }
        });
        
        // Strategy 3: Retry mechanism with timeout
        var maxAttempts = 15;
        var attempt = 0;
        
        function tryInitChart() {
            if (chartInitialized) return;
            
            if (initializeChart()) {
                chartInitialized = true;
                console.log('Chart initialized successfully on retry attempt', attempt + 1);
                return; // Success!
            }
            
            attempt++;
            if (attempt < maxAttempts) {
                setTimeout(tryInitChart, 300);
            } else {
                console.warn('Failed to initialize ApexCharts chart after', maxAttempts, 'attempts');
                console.log('ApexCharts available:', typeof ApexCharts !== 'undefined');
                console.log('Canvas element:', !!document.getElementById('ah-scan-chart'));
                
                // Simply log the failure - the chart container will remain empty
                console.log('Chart initialization failed - container will remain empty');
            }
        }
        
        // Start retry mechanism after a short delay
        setTimeout(tryInitChart, 100);
    });

    // Initialize table functionality when DOM is ready
    $(document).ready(function() {
        initTableFunctionality();
        
        // Initialize chart with empty data to make it visible from start
        renderChart([0, 0, 0, 1], ['No Data'], ['#6c757d']);
        
        // Initialize real-time updates if we have a scan ID
        var scanIdFromForm = $('#linkscanner-list-form').data('scan-id');
        if (scanIdFromForm) {
            scanId = scanIdFromForm;
            // Don't start polling automatically - let the user start the scan
        }
    });

    // Table functionality
    function initTableFunctionality() {
        // Select all checkbox functionality
        $('#cb-select-all-1').on('change', function() {
            var isChecked = $(this).is(':checked');
            $('input[name="scanned[]"]').prop('checked', isChecked);
        });

        // Individual checkbox change
        $(document).on('change', 'input[name="scanned[]"]', function() {
            var totalCheckboxes = $('input[name="scanned[]"]').length;
            var checkedCheckboxes = $('input[name="scanned[]"]:checked').length;
            
            $('#cb-select-all-1').prop('checked', totalCheckboxes === checkedCheckboxes);
        });
        // Search functionality
        $('#ah-search-input').on('keypress', function(e) {
            if (e.which === 13) {
                filterTable();
            }
        });

        $('#search-submit').on('click', filterTable);
        $('#doaction').on('click', filterTable);
        $('#ah-include-ignored').on('change', filterTable);
        $('#ah-filter-post-type').on('change', filterTable);
        $('#ah-filter-status-table').on('change', filterTable);

        // Copy URL functionality
        $(document).on('click', '.copy-url-btn', function(e) {
            e.preventDefault();
            var url = $(this).data('url');
            if (url) {
                copyToClipboard(url);
                $(this).find('.dashicons').removeClass('dashicons-clipboard').addClass('dashicons-yes');
                setTimeout(() => {
                    $(this).find('.dashicons').removeClass('dashicons-yes').addClass('dashicons-clipboard');
                }, 2000);
            }
        });

        // Action buttons
        $(document).on('click', '.ah-retry-link', function(e) {
            e.preventDefault();
            var linkId = $(this).data('id');
            retryLink(linkId);
        });

        $(document).on('click', '.ah-ignore-link', function(e) {
            e.preventDefault();
            var linkId = $(this).data('id');
            ignoreLink(linkId);
        });

        $(document).on('click', '.ah-unignore-link', function(e) {
            e.preventDefault();
            var linkId = $(this).data('id');
            unignoreLink(linkId);
        });
    }

    function filterTable() {
        var search = $('#link-search-input').val() || '';
        var status = $('#ah-filter-status-table').val() || '';
        var postTypeFilter = $('#ah-filter-post-type').val() || '';
        var includeIgnored = $('#ah-include-ignored').is(':checked');

        var params = {
            s: search,
            status: status,
            post_type_filter: postTypeFilter,
            include_ignored: includeIgnored ? 1 : 0,
            post_type: 'affiliate_link',
            page: 'affiliate-hub-link-scanner'
        };

        window.location.href = window.location.pathname + '?' + $.param(params);
    }

    function copyToClipboard(text) {
        if (navigator.clipboard) {
            navigator.clipboard.writeText(text).then(function() {
                showCopySuccess();
            }).catch(function(err) {
                console.error('Clipboard API failed:', err);
                fallbackCopyToClipboard(text);
            });
        } else {
            fallbackCopyToClipboard(text);
        }
    }

    function fallbackCopyToClipboard(text) {
        var temp = $('<input>');
        $('body').append(temp);
        temp.val(text).select();
        document.execCommand('copy');
        temp.remove();
        showCopySuccess();
    }

    function showCopySuccess() {
        // We could add a temporary success message or just use the icon change
        // The icon change is already handled in the click handler
    }

    function retryLink(linkId) {
        $.post(window.AffiliateHubLinkScanner.ajax_url, {
            action: 'affiliate_hub_retry_link',
            nonce: window.AffiliateHubLinkScanner.nonce,
            link_id: linkId
        }, function(response) {
            if (response.success) {
                showNotice('Link queued for retry.', 'success');
                // Update the row status
                updateRowStatus(linkId, 'pending', 0);
            } else {
                showNotice(response.data || 'Failed to retry link.', 'error');
            }
        });
    }

    function ignoreLink(linkId) {
        $.post(window.AffiliateHubLinkScanner.ajax_url, {
            action: 'affiliate_hub_ignore_link',
            nonce: window.AffiliateHubLinkScanner.nonce,
            link_id: linkId
        }, function(response) {
            if (response.success) {
                showNotice('Link ignored.', 'success');
                // Update the row
                $('tr[data-id="' + linkId + '"]').fadeOut();
            } else {
                showNotice(response.data || 'Failed to ignore link.', 'error');
            }
        });
    }

    function unignoreLink(linkId) {
        $.post(window.AffiliateHubLinkScanner.ajax_url, {
            action: 'affiliate_hub_unignore_link',
            nonce: window.AffiliateHubLinkScanner.nonce,
            link_id: linkId
        }, function(response) {
            if (response.success) {
                showNotice('Link unignored.', 'success');
                // Update the row
                updateRowStatus(linkId, 'pending', 0);
            } else {
                showNotice(response.data || 'Failed to unignore link.', 'error');
            }
        });
    }

    function updateRowStatus(linkId, status, code) {
        var $row = $('tr[data-id="' + linkId + '"]');
        if ($row.length) {
            // Update status
            var statusColor = getStatusColor(status);
            $row.find('.column-status .affiliate-hub-redirect-type')
                .css('background-color', statusColor)
                .text(status);

            // Update code if provided
            if (code) {
                var codeColor = getCodeColor(code);
                $row.find('.column-code .affiliate-hub-redirect-type')
                    .css('background-color', codeColor)
                    .text(code);
            }

            // Update actions
            if (status === 'pending') {
                $row.find('.column-actions').html(
                    '<button class="button ah-retry-link" data-id="' + linkId + '">Retry</button> ' +
                    '<button class="button ah-ignore-link" data-id="' + linkId + '">Ignore</button>'
                );
            }
        }
    }

    function getStatusColor(status) {
        switch (status) {
            case 'active': return '#28a745';
            case 'broken': return '#dc3545';
            case 'forbidden': return '#ffc107';
            default: return '#6c757d';
        }
    }

    function getCodeColor(code) {
        code = parseInt(code);
        if (code >= 200 && code < 300) return '#28a745';
        if (code >= 400 && code < 500) return '#dc3545';
        if (code >= 500) return '#b02a37';
        return '#6c757d';
    }

    function showNotice(message, type) {
        var className = type === 'error' ? 'notice-error' : 'notice-success';
        var $notice = $('<div class="notice ' + className + ' is-dismissible"><p>' + message + '</p></div>');
        $('.wrap h1').after($notice);
        
        setTimeout(function() {
            $notice.fadeOut();
        }, 3000);
    }

    // Responsive chart handling - ApexCharts handles this automatically
    function handleResponsiveChart() {
        // ApexCharts has built-in responsive handling
        // No manual intervention needed
    }

    // Window resize handler for responsive design
    var resizeTimeout;
    $(window).on('resize', function() {
        clearTimeout(resizeTimeout);
        resizeTimeout = setTimeout(function() {
            handleResponsiveChart();
            
            // Update table horizontal scroll on mobile
            if (window.innerWidth <= 768) {
                $('.ah-details-table-wrap').addClass('mobile-scroll');
            } else {
                $('.ah-details-table-wrap').removeClass('mobile-scroll');
            }
        }, 250);
    });
    
    // Handle scan type radio button changes
    $(document).on('change', 'input[name="ah-scan-type"]', function() {
        var scanType = $(this).val();
        
        if (scanType === 'content') {
            // Show post types selection for content scan
            $('#ah-post-types-container').show();
        } else if (scanType === 'affiliates') {
            // Hide post types selection for affiliate scan
            $('#ah-post-types-container').hide();
        }
    });
    
    // Initialize scan type visibility on page load
    $(document).ready(function() {
        $('input[name="ah-scan-type"]:checked').trigger('change');
    });

    // Initialize responsive state
    $(document).ready(function() {
        if (window.innerWidth <= 768) {
            $('.ah-details-table-wrap').addClass('mobile-scroll');
        }
    });

})(jQuery);
