/**
 * Simplified Affiliate Hub Statistics Dashboard JavaScript
 * Uses ApexCharts for better performance and visual appearance
 */
(function($) {
    'use strict';
    
    console.log('stats-simplified.js: File loaded');
    
    var AffiliateStats = {
        clicksChart: null,
        initialized: false,
        chartLibraryLoaded: false,
        pendingChartData: null,
        
        init: function() {
            if (this.initialized) {
                return;
            }
            
            // Check for ApexCharts availability with retry mechanism
            this.checkChartLibrary();
            this.bindEvents();
            this.loadInitialData();
            this.initialized = true;
        },
        
        checkChartLibrary: function() {
            var self = this;
            var maxRetries = 10;
            var retryCount = 0;
            
            function checkChart() {
                if (typeof ApexCharts !== 'undefined') {
                    self.chartLibraryLoaded = true;
                    console.log('ApexCharts loaded successfully');
                    
                    // If we have pending chart data, render it now
                    if (self.pendingChartData) {
                        self.updateClicksChart(self.pendingChartData.total, self.pendingChartData.unique);
                        self.pendingChartData = null;
                    }
                    return;
                }
                
                retryCount++;
                if (retryCount < maxRetries) {
                    setTimeout(checkChart, 200); // Check every 200ms
                } else {
                    console.warn('ApexCharts not available after retries. Using table fallback.');
                    self.chartLibraryLoaded = false;
                }
            }
            
            checkChart();
        },
        
        bindEvents: function() {
            var self = this;
            
            console.log('AffiliateStats: Binding events...');
            console.log('Custom date button exists:', $('#apply-custom-date').length);
            console.log('Start date input exists:', $('#start-date').length);
            console.log('End date input exists:', $('#end-date').length);
            
            // Date filter tabs
            $('.date-filter-tab').on('click', function(e) {
                e.preventDefault();
                var $tab = $(this);
                var period = $tab.data('period');
                
                console.log('Date filter tab clicked:', period);
                
                $('.date-filter-tab').removeClass('active');
                $tab.addClass('active');
                
                if (period === 'custom') {
                    $('#custom-date-range').show();
                    console.log('Showing custom date range');
                } else {
                    $('#custom-date-range').hide();
                    self.loadData({ period: period });
                }
            });
            
            // Custom date range
            $('#apply-custom-date').on('click', function(e) {
                e.preventDefault();
                var startDate = $('#start-date').val();
                var endDate = $('#end-date').val();
                
                console.log('Apply custom date clicked:', startDate, 'to', endDate);
                
                if (startDate && endDate) {
                    // Make sure custom tab is active
                    $('.date-filter-tab').removeClass('active');
                    $('.date-filter-tab[data-period="custom"]').addClass('active');
                    
                    self.loadData({ 
                        period: 'custom',
                        start_date: startDate,
                        end_date: endDate
                    });
                } else {
                    console.warn('Start date or end date is missing');
                    alert('Please select both start and end dates.');
                }
            });
            
            // Debug: Test if button click is registered at all
            $('#apply-custom-date').on('click', function() {
                console.log('DIRECT CLICK HANDLER - Custom date button clicked!');
            });
            
            // General filter
            $('#general-filter').on('change', function() {
                self.loadData({ link_id: $(this).val() });
            });
            
            // Chart granularity
            $('#chart-granularity').on('change', function() {
                self.loadData({ granularity: $(this).val() });
            });
            
            // Fetch report functionality
            $('#search-affiliate-link').on('keypress', function(e) {
                if (e.keyCode === 13) { // Enter key
                    $('#fetch-report').click();
                }
            });
            
            $('#fetch-report').on('click', function() {
                var searchTerm = $('#search-affiliate-link').val();
                self.fetchSpecificLinkReport(searchTerm);
            });
        },
        
        loadInitialData: function() {
            this.loadData({ period: 'this_month' });
        },
        
        loadData: function(params) {
            var self = this;
            params = params || {};
            
            // Set defaults tylko jeśli period nie został już przekazany
            if (!params.period) {
                var $activeTab = $('.date-filter-tab.active');
                params.period = $activeTab.data('period') || 'this_month';
            }
            
            // Add AJAX parameters
            params.action = 'affiliate_hub_get_stats';
            params.nonce = affiliateHubStats.nonce;
            
            // Debug log for custom date ranges
            if (params.period === 'custom') {
                console.log('Loading custom date range:', params.start_date, 'to', params.end_date);
            }
            
            // Show loading
            this.showLoading();
            
            $.ajax({
                url: affiliateHubStats.ajax_url,
                method: 'POST',
                data: params,
                success: function(response) {
                    if (response.success) {
                        console.log('Stats AJAX success for period:', params.period);
                        if (params.period === 'custom') {
                            console.log('Custom dates sent:', params.start_date, 'to', params.end_date);
                            console.log('Data returned for custom range:', response.data);
                        }
                        console.log('Stats data received:', response.data);
                        self.updateStats(response.data);
                    } else {
                        console.error('Stats AJAX error:', response);
                    }
                },
                error: function(xhr, status, error) {
                    console.error('Stats AJAX request failed:', status, error);
                },
                complete: function() {
                    self.hideLoading();
                }
            });
        },
        
        updateStats: function(data) {
            // Check if data exists and has required properties
            if (!data) {
                return;
            }
            
            // Update metric cards with safe defaults
            $('#total-clicks').text(data.total_clicks || '0');
            $('#unique-clicks').text(data.unique_clicks || '0');
            $('#ctr-rate').text((data.ctr_rate || '0') + '%');
            $('#active-links').text(data.active_links || '0');
            
            // Update chart with both total and unique clicks (with safe defaults)
            var totalClicks = data.clicks_over_time || [];
            var uniqueClicks = data.unique_clicks_over_time || [];
            this.updateClicksChart(totalClicks, uniqueClicks);
        },
        
        updateClicksChart: function(totalClicksData, uniqueClicksData) {
            var ctx = document.getElementById('clicks-chart');
            if (!ctx) {
                return;
            }

            // Ensure data is arrays
            totalClicksData = totalClicksData || [];
            uniqueClicksData = uniqueClicksData || [];

            // If ApexCharts is not yet loaded, store data for later
            if (!this.chartLibraryLoaded && typeof ApexCharts === 'undefined') {
                this.pendingChartData = {
                    total: totalClicksData,
                    unique: uniqueClicksData
                };
                // Show fallback table immediately
                this.createFallbackTable(ctx, totalClicksData, uniqueClicksData);
                return;
            }

            // ApexCharts is available, proceed with chart creation
            // Destroy existing chart
            if (this.clicksChart) {
                this.clicksChart.destroy();
            }

            // Handle empty data case
            if (totalClicksData.length === 0) {
                if (typeof ApexCharts === 'undefined') {
                    // Use fallback table for empty data
                    this.createFallbackTable(ctx, [], []);
                    return;
                }
                
                // Create empty chart with ApexCharts
                var emptyOptions = {
                    series: [{
                        name: 'Total Clicks',
                        data: [0]
                    }, {
                        name: 'Unique Clicks', 
                        data: [0]
                    }],
                    chart: {
                        type: 'line',
                        height: 350,
                        animations: {
                            enabled: true,
                            easing: 'easeinout',
                            speed: 600
                        },
                        toolbar: {
                            show: true,
                            tools: {
                                download: true,
                                pan: false,
                                reset: true,
                                zoom: false
                            }
                        }
                    },
                    xaxis: {
                        categories: ['No Data']
                    },
                    colors: ['#2271b1', '#00a32a'],
                    stroke: {
                        width: 2,
                        curve: 'smooth'
                    },
                    legend: {
                        position: 'top'
                    },
                    responsive: [{
                        breakpoint: 768,
                        options: {
                            chart: {
                                height: 250
                            },
                            legend: {
                                position: 'bottom'
                            }
                        }
                    }]
                };
                
                this.clicksChart = new ApexCharts(ctx, emptyOptions);
                this.clicksChart.render();
                return;
            }
            
            // Prepare chart data for ApexCharts format
            var categories = [];
            var totalData = [];
            var uniqueData = [];
            
            // Process total clicks data
            totalClicksData.forEach(function(item) {
                if (item && item.date && typeof item.clicks !== 'undefined') {
                    categories.push(item.date);
                    totalData.push(item.clicks);
                }
            });
            
            // Process unique clicks data - match by date
            categories.forEach(function(date) {
                var uniqueItem = uniqueClicksData.find(function(u) { 
                    return u && u.date === date; 
                });
                uniqueData.push(uniqueItem ? uniqueItem.clicks : 0);
            });
            
            // Debug logging to check data
            console.log('Chart data - Total:', totalData);
            console.log('Chart data - Unique:', uniqueData);
            console.log('Chart categories:', categories);
            
            // Ensure we have valid numeric data
            totalData = totalData.map(function(val) { return Number(val) || 0; });
            uniqueData = uniqueData.map(function(val) { return Number(val) || 0; });
            
            // Debug logging to check data - but always use single Y axis for comparison
            console.log('Chart data - Total:', totalData);
            console.log('Chart data - Unique:', uniqueData);
            console.log('Chart categories:', categories);
            
            // Create ApexCharts options - podstawowa konfiguracja
            var options = {
                series: [{
                    name: 'Total Clicks',
                    data: totalData
                }, {
                    name: 'Unique Clicks',
                    data: uniqueData
                }],
                chart: {
                    type: 'line',
                    height: 400
                },
                xaxis: {
                    categories: categories
                },
                stroke: {
                    curve: 'smooth',
                    width: 2
                },
                markers: {
                    size: 4
                },
                legend: {
                    position: 'top'
                }
            };
            
            // Destroy existing chart if it exists
            if (this.clicksChart) {
                this.clicksChart.destroy();
            }
            
            // Create and render the chart
            var ctx = document.getElementById('clicks-chart');
            if (ctx) {
                this.clicksChart = new ApexCharts(ctx, options);
                this.clicksChart.render();
            }
        },
        
        createFallbackTable: function(ctx, totalClicksData, uniqueClicksData) {
            // Create a simple HTML table as fallback when ApexCharts is not available
            var container = ctx.parentElement;
            
            // Clear the canvas and create table
            var loadingMessage = this.chartLibraryLoaded ? 
                '<p class="description">ApexCharts is loading... This table will be replaced with a chart shortly.</p>' :
                '<p class="description">ApexCharts could not be loaded. Showing data in table format.</p>';
                
            container.innerHTML = '<div class="chart-fallback">' +
                '<h4>Click Statistics Over Time</h4>' +
                loadingMessage +
                '<table class="widefat striped">' +
                '<thead><tr><th>Date</th><th>Total Clicks</th><th>Unique Clicks</th></tr></thead>' +
                '<tbody id="stats-table-body"></tbody>' +
                '</table>' +
                '</div>';
            
            var tbody = document.getElementById('stats-table-body');
            
            if (totalClicksData.length === 0) {
                tbody.innerHTML = '<tr><td colspan="3" style="text-align: center; color: #666; font-style: italic;">No click data available for the selected period</td></tr>';
                return;
            }
            
            // Populate table with data
            var maxRows = 10;
            for (var i = 0; i < Math.min(totalClicksData.length, maxRows); i++) {
                var totalPoint = totalClicksData[i] || {};
                var uniquePoint = uniqueClicksData[i] || {};
                
                var date = totalPoint.x || totalPoint.date || 'N/A';
                var totalClicks = totalPoint.y || totalPoint.clicks || 0;
                var uniqueClicks = uniquePoint.y || uniquePoint.clicks || 0;
                
                tbody.innerHTML += '<tr>' +
                    '<td><strong>' + date + '</strong></td>' +
                    '<td><span class="total-clicks">' + totalClicks + '</span></td>' +
                    '<td><span class="unique-clicks">' + uniqueClicks + '</span></td>' +
                    '</tr>';
            }
            
            if (totalClicksData.length > maxRows) {
                tbody.innerHTML += '<tr style="background-color: #f9f9f9;"><td colspan="3" style="text-align: center; font-style: italic; color: #666;">... and ' + (totalClicksData.length - maxRows) + ' more entries (showing most recent ' + maxRows + ')</td></tr>';
            }
            
            // Add summary row
            if (totalClicksData.length > 0) {
                var totalSum = totalClicksData.reduce(function(sum, item) { return sum + (item.y || item.clicks || 0); }, 0);
                var uniqueSum = uniqueClicksData.reduce(function(sum, item) { return sum + (item.y || item.clicks || 0); }, 0);
                
                tbody.innerHTML += '<tr style="border-top: 2px solid #ddd; background-color: #f0f0f1; font-weight: bold;">' +
                    '<td><strong>Total for Period</strong></td>' +
                    '<td><strong>' + totalSum + '</strong></td>' +
                    '<td><strong>' + uniqueSum + '</strong></td>' +
                    '</tr>';
            }
        },
        
        showLoading: function() {
            // Update metric cards
            $('#total-clicks, #unique-clicks, #ctr-rate, #active-links').text('—');
        },
        
        hideLoading: function() {
            // Loading is automatically hidden when data is updated
        },
        
        // Handle window resize for responsive chart
        handleResize: function() {
            var self = this;
            if (this.clicksChart && typeof this.clicksChart.updateOptions === 'function') {
                var newHeight;
                var screenWidth = window.innerWidth;
                
                // Adjust height based on screen size
                if (screenWidth <= 480) {
                    newHeight = 250;
                } else if (screenWidth <= 768) {
                    newHeight = 300;
                } else if (screenWidth <= 1200) {
                    newHeight = 350;
                } else {
                    newHeight = 400;
                }
                
                try {
                    this.clicksChart.updateOptions({
                        chart: {
                            height: newHeight,
                            width: '100%'
                        }
                    }, false, true);
                } catch (e) {
                    // Silently handle any update errors
                    console.log('Chart resize update skipped:', e.message);
                }
            }
        },
        
        fetchSpecificLinkReport: function(searchTerm) {
            if (!searchTerm) {
                alert('Please enter a search term');
                return;
            }
            
            console.log('🔍 Fetching report for:', searchTerm);
            
            // Search in the general filter dropdown for matching link
            var generalFilter = $('#general-filter');
            var found = false;
            
            generalFilter.find('option').each(function() {
                var optionText = $(this).text().toLowerCase();
                var searchLower = searchTerm.toLowerCase();
                
                // Check if option text contains the search term
                if (optionText.includes(searchLower)) {
                    generalFilter.val($(this).val());
                    found = true;
                    console.log('✅ Found matching link:', $(this).text());
                    return false; // break the loop
                }
            });
            
            if (found) {
                // Refresh data with the selected link
                this.loadData({ link_id: generalFilter.val() });
                
                // Update active tab to reflect current state
                $('.date-filter-tab.active').click();
                
                // Show success message
                this.showNotice('Report loaded for: ' + searchTerm, 'success');
            } else {
                console.log('❌ No matching link found for:', searchTerm);
                alert('No matching affiliate link found for: "' + searchTerm + '"');
            }
        },
        
        showNotice: function(message, type) {
            // Simple notice display (you can enhance this)
            var noticeClass = type === 'success' ? 'notice-success' : 'notice-error';
            var notice = '<div class="notice ' + noticeClass + ' is-dismissible"><p>' + message + '</p></div>';
            
            // Remove existing notices
            $('.wrap .notice').remove();
            
            // Add new notice at the top
            $('.wrap').prepend(notice);
            
            // Auto-dismiss after 3 seconds
            setTimeout(function() {
                $('.wrap .notice').fadeOut();
            }, 3000);
        },

        handleResize: function() {
            if (this.clicksChart) {
                var chartContainer = $('#clicks-chart').closest('.chart-wrapper');
                var containerWidth = chartContainer.width();
                var newHeight = Math.min(400, Math.max(250, containerWidth * 0.4));
                
                try {
                    this.clicksChart.updateOptions({
                        chart: {
                            height: newHeight,
                            width: '100%'
                        }
                    }, false, true);
                } catch (e) {
                    // Silently handle any update errors
                    console.log('Chart resize update skipped:', e.message);
                }
            }
        }
    };
    
    // Window resize handler with debounce
    var resizeTimeout;
    $(window).on('resize', function() {
        clearTimeout(resizeTimeout);
        resizeTimeout = setTimeout(function() {
            if (AffiliateStats.initialized) {
                AffiliateStats.handleResize();
            }
        }, 300);
    });
    
    // Initialize when document is ready
    $(document).ready(function() {
        console.log('Document ready - stats-simplified.js loaded');
        console.log('Stats page elements found:', $('.affiliate-hub-stats-page').length);
        console.log('All .wrap elements:', $('.wrap').length);
        
        // Check if we're on the stats page
        if ($('.affiliate-hub-stats-page').length > 0) {
            console.log('Stats page detected - initializing AffiliateStats');
            AffiliateStats.init();
        } else {
            console.log('Stats page NOT detected - checking all page classes');
            $('.wrap').each(function(i) {
                console.log('Wrap element', i, 'classes:', $(this).attr('class'));
            });
        }
    });
    
})(jQuery);
