/**
 * Affiliate Hub Statistics Dashboard JavaScript - ApexCharts Version
 */
(function($) {
    'use strict';
    
    // Prevent multiple initializations with improved detection
    if (window.AffiliateStatsInitialized) {
        return;
    }
    
    // Ensure jQuery is ready and available
    if (typeof $ === 'undefined' || typeof jQuery === 'undefined') {
        console.error('AffiliateStats: jQuery not available');
        return;
    }
    
    var AffiliateStats = {
        charts: {},
        initialized: false,
        loadingPromise: null, // Cache loading state
        
        init: function() {
            if (this.initialized) {
                return;
            }
            
            // Production mode - minimal logging
            if (typeof affiliateHubStats !== 'undefined' && affiliateHubStats.debug_mode) {
                console.log('🚀 AffiliateStats: Initializing...');
            }
            
            this.bindEvents();
            this.loadInitialData();
            
            this.initialized = true;
            window.AffiliateStatsInitialized = true;
        },
        
        bindEvents: function() {
            var self = this;
            
            // Optimize event binding - use delegation and namespaces
            $(document).off('.affiliate-stats'); // Remove old events
            $('#apply-custom-date').off('click.affiliate-stats');
            $('#general-filter').off('change.affiliate-stats');
            
            // Date filter tabs (new interface) - with debugging
            $('.date-filter-tab').on('click.affiliate-stats', function(e) {
                e.preventDefault();
                e.stopPropagation();
                
                var $clickedTab = $(this);
                var period = $clickedTab.data('period');
                
                console.log('📅 Date filter clicked!');
                console.log('📅 Clicked element:', $clickedTab);
                console.log('📅 Period:', period);
                console.log('📅 All tabs:', $('.date-filter-tab'));
                
                // Remove active class from all tabs
                $('.date-filter-tab').removeClass('active');
                // Add active class to clicked tab
                $clickedTab.addClass('active');
                
                console.log('📅 Active class set. New active tab:', $('.date-filter-tab.active'));
                
                if (period === 'custom') {
                    $('#custom-date-range').show();
                    console.log('📅 Showing custom date inputs');
                } else {
                    $('#custom-date-range').hide();
                    console.log('📅 Loading data for period:', period);
                    
                    // Add delay to ensure DOM is updated
                    setTimeout(function() {
                        self.refreshAllData();
                    }, 50);
                }
            });
            
            // Debug: Check if date tabs exist and log them
            var $dateTabs = $('.date-filter-tab');
            console.log('📅 Date filter tabs found:', $dateTabs.length);
            $dateTabs.each(function(index) {
                console.log('📅 Tab ' + index + ':', {
                    element: this,
                    text: $(this).text(),
                    period: $(this).data('period'),
                    hasClass: $(this).hasClass('date-filter-tab'),
                    isActive: $(this).hasClass('active')
                });
            });
            
            // Custom date apply button
            $('#apply-custom-date').on('click.affiliate-stats', function(e) {
                e.preventDefault();
                console.log('📅 Custom date apply clicked');
                self.refreshAllData();
            });
            
            // General filter (new interface)
            $('#general-filter').on('change.affiliate-stats', function() {
                console.log('🔍 General filter changed to:', $(this).val());
                self.refreshAllData();
            });
            
            // Search affiliate link
            $('#search-affiliate-link').on('keyup', function(e) {
                if (e.keyCode === 13) { // Enter key
                    $('#fetch-report').click();
                }
            });
            
            // Fetch report button
            $('#fetch-report').on('click', function() {
                var searchTerm = $('#search-affiliate-link').val();
                self.fetchSpecificLinkReport(searchTerm);
            });
            
            // Table filters
            $('#table-categories-filter, #table-tags-filter, #table-redirect-filter').on('change', function() {
                self.filterTable();
            });
            
            // Table search
            $('#search-table-links').on('keyup', function() {
                self.searchTable();
            });
            
            // Filter table button
            $('#filter-table').on('click', function() {
                self.filterTable();
            });
            
            // Legacy date range (keep for compatibility)
            $('#date-range').on('change', function() {
                if ($(this).val() === 'custom') {
                    $('#custom-date-range').show();
                } else {
                    $('#custom-date-range').hide();
                    self.refreshAllData();
                }
            });
            
            // Custom date range (legacy)
            $('#start-date, #end-date').on('change', function() {
                if ($('#date-range').val() === 'custom') {
                    self.refreshAllData();
                }
            });
            
            // Refresh button (legacy)
            $('#refresh-stats').on('click', function() {
                self.refreshAllData();
            });
            
            // Export button (legacy)
            $('#export-stats').on('click', function() {
                self.exportData();
            });
            
            // Chart granularity
            $('#chart-granularity').on('change', function() {
                self.refreshClicksChart();
            });
            
            // Generate sample data button (debug only)
            $('#generate-sample-clicks').on('click', function() {
                self.generateSampleClicks();
            });
        },
        
        loadInitialData: function() {
            this.refreshAllData();
        },
        
        // New unified refresh function
        refreshAllData: function() {
            console.log('🔄 Refreshing all data...');
            this.showLoading();
            this.loadDashboardStats();
            
            // Only load top links if we're not viewing specific link stats
            var generalFilter = $('#general-filter').val();
            var hasSpecificLink = (affiliateHubStats.link_id && affiliateHubStats.link_id > 0) || generalFilter;
            
            if (!hasSpecificLink) {
                this.loadTopLinks();
                $('#top-links-widget').show();
            } else {
                console.log('📊 Hiding top links - viewing specific link stats');
                $('#top-links-widget').hide();
            }
            
            this.refreshClicksChart();
            this.loadTableData();
        },
        
        // Get current date range from new tab interface
        getCurrentDateRange: function() {
            var activePeriod = $('.date-filter-tab.active').data('period');
            console.log('📅 getCurrentDateRange - Active tab period:', activePeriod);
            console.log('📅 Active tab element:', $('.date-filter-tab.active'));
            
            if (activePeriod === 'custom') {
                var startDate = $('#start-date').val();
                var endDate = $('#end-date').val();
                console.log('📅 Custom date range:', startDate, 'to', endDate);
                if (startDate && endDate) {
                    return {
                        type: 'custom',
                        start: startDate,
                        end: endDate
                    };
                }
            }
            
            var result = activePeriod || 'this_month';
            console.log('📅 Final date range result:', result);
            return result;
        },
        
        // Load table data for detailed statistics
        loadTableData: function() {
            console.log('🔄 Loading table data...');
            var self = this;
            var dateRange = this.getCurrentDateRange();
            var generalFilter = $('#general-filter').val();
            
            // Convert date range to backend format
            var backendDateRange = dateRange;
            if (typeof dateRange === 'object' && dateRange.type === 'custom') {
                backendDateRange = dateRange.start + '|' + dateRange.end;
            }
            
            var ajaxData = {
                action: 'affiliate_hub_get_table_data',
                date_range: backendDateRange,
                general_filter: generalFilter,
                nonce: affiliateHubStats.nonce
            };
            
            console.log('📤 loadTableData AJAX request:', ajaxData);
            
            $.ajax({
                url: affiliateHubStats.ajax_url,
                type: 'POST',
                data: ajaxData,
                success: function(response) {
                    console.log('📥 loadTableData AJAX response:', response);
                    
                    if (response.success) {
                        console.log('✅ Success - table data:', response.data);
                        self.updateTable(response.data);
                    } else {
                        console.error('❌ Failed to load table data:', response);
                        self.showTableError();
                    }
                },
                error: function(xhr, status, error) {
                    console.error('🔥 AJAX error loading table data:', status, error);
                    console.error('🔥 Response text:', xhr.responseText);
                    self.showTableError();
                }
            });
        },
        
        // Update the detailed statistics table with click data
        updateTable: function(data) {
            console.log('🔍 updateTable received data:', data);
            
            var tbody = $('#stats-table-body');
            tbody.empty();
            
            if (data && data.length > 0) {
                console.log('📊 Processing', data.length, 'rows');
                console.log('📊 Sample row:', data[0]);
                
                data.forEach(function(click, index) {
                    console.log('📊 Row', index, ':', click);
                    
                    var row = '<tr>';
                    row += '<td class="column-title">' + (click.title || 'NO TITLE') + '</td>';
                    row += '<td class="column-ip">' + (click.ip_address || 'NO IP') + '</td>';
                    row += '<td class="column-timestamp">' + (click.timestamp || '—') + '</td>';
                    row += '<td class="column-referer" title="' + (click.referer || '') + '">' + (click.referer || '—') + '</td>';
                    row += '<td class="column-traffic-source">' + (click.traffic_source || '—') + '</td>';
                    row += '<td class="column-os">' + (click.os || '—') + '</td>';
                    row += '<td class="column-browser">' + (click.browser || '—') + '</td>';
                    row += '<td class="column-device-type">' + (click.device_type || '—') + '</td>';
                    row += '<td class="column-country">' + (click.country || '—') + '</td>';
                    row += '<td class="column-clicks"><span class="affiliate-hub-clicks-cell">' + (click.clicks || 0) + '</span></td>';
                    row += '</tr>';
                    tbody.append(row);
                });
            } else {
                console.warn('⚠️ No data received or empty array');
                tbody.append('<tr><td colspan="10" class="loading-row"><div class="loading">No click data available for the selected period.</div></td></tr>');
            }
        },
        
        // Show table error
        showTableError: function() {
            var tbody = $('#stats-table-body');
            tbody.empty();
            tbody.append('<tr><td colspan="10" class="loading-row"><div class="loading" style="color: #dc3232;">Error loading data. Please try again.</div></td></tr>');
        },
        
        // Filter table by categories, tags, redirect type
        filterTable: function() {
            var categoryFilter = $('#table-categories-filter').val();
            var tagFilter = $('#table-tags-filter').val();
            var redirectFilter = $('#table-redirect-filter').val();
            
            $('#stats-table-body tr').each(function() {
                var row = $(this);
                var show = true;
                
                if (categoryFilter && !row.find('td:eq(4)').text().toLowerCase().includes(categoryFilter.toLowerCase())) {
                    show = false;
                }
                
                if (tagFilter && !row.find('td:eq(5)').text().toLowerCase().includes(tagFilter.toLowerCase())) {
                    show = false;
                }
                
                if (redirectFilter && !row.find('.affiliate-hub-redirect-type').hasClass('redirect-' + redirectFilter)) {
                    show = false;
                }
                
                if (show) {
                    row.show();
                } else {
                    row.hide();
                }
            });
        },
        
        // Search table
        searchTable: function() {
            var searchTerm = $('#search-table-links').val().toLowerCase();
            
            $('#stats-table-body tr').each(function() {
                var row = $(this);
                var text = row.text().toLowerCase();
                
                if (searchTerm === '' || text.includes(searchTerm)) {
                    row.show();
                } else {
                    row.hide();
                }
            });
        },
        
        // Fetch specific link report
        fetchSpecificLinkReport: function(searchTerm) {
            if (!searchTerm) {
                alert('Please enter a search term');
                return;
            }
            
            console.log('🔍 Fetching report for:', searchTerm);
            
            // Here you could implement search for specific link
            // For now, just filter the general dropdown
            var generalFilter = $('#general-filter');
            var found = false;
            
            generalFilter.find('option').each(function() {
                if ($(this).text().toLowerCase().includes(searchTerm.toLowerCase())) {
                    generalFilter.val($(this).val());
                    found = true;
                    return false;
                }
            });
            
            if (found) {
                this.refreshAllData();
            } else {
                alert('Link not found. Please check the spelling or try a different search term.');
            }
        },
        
        refreshData: function() {
            // Legacy function - redirect to new unified function
            this.refreshAllData();
        },
        
        loadDashboardStats: function() {
            console.log('🔄 AffiliateStats: Loading dashboard stats');
            var self = this;
            var dateRange = this.getCurrentDateRange();
            console.log('📅 AffiliateStats: Date range:', dateRange);
            
            // Convert date range to backend format
            var backendDateRange = dateRange;
            if (typeof dateRange === 'object' && dateRange.type === 'custom') {
                backendDateRange = dateRange.start + '|' + dateRange.end;
                console.log('📅 Converted custom range to:', backendDateRange);
            }
            
            var ajaxData = {
                action: 'affiliate_hub_get_stats_data',
                date_range: backendDateRange,
                nonce: affiliateHubStats.nonce
            };
            
            // Add general filter
            var generalFilter = $('#general-filter').val();
            if (generalFilter) {
                ajaxData.link_ids = [generalFilter];
                console.log('🔍 AffiliateStats: Filtering by general filter:', generalFilter);
            }
            
            // Add link_id if we're viewing specific link stats (legacy)
            if (affiliateHubStats.link_id && affiliateHubStats.link_id > 0) {
                ajaxData.link_ids = [affiliateHubStats.link_id];
                console.log('🔍 AffiliateStats: Filtering by link_id:', affiliateHubStats.link_id);
                console.log('📤 AffiliateStats: Sending link_ids array:', ajaxData.link_ids);
            } else if (!generalFilter) {
                console.log('📊 AffiliateStats: Loading all stats (no filter)');
            }
            
            console.log('📤 AffiliateStats: Full AJAX data:', ajaxData);
            
            $.ajax({
                url: affiliateHubStats.ajax_url,
                type: 'POST',
                data: ajaxData,
                success: function(response) {
                    console.log('📥 AffiliateStats: AJAX response received:', response);
                    if (response.success) {
                        console.log('✅ AffiliateStats: Processing successful response');
                        console.log('📊 AffiliateStats: Stats data:', response.data);
                        self.updateDashboardStats(response.data);
                        self.updateCharts(response.data);
                    } else {
                        console.error('❌ AffiliateStats: Server returned error:', response);
                        self.showError('Server error: ' + (response.data ? response.data.message : 'Unknown error'));
                    }
                },
                error: function(xhr, status, error) {
                    console.error('❌ AffiliateStats: AJAX request failed');
                    console.error('Status:', status);
                    console.error('Error:', error);
                    console.error('Response Text:', xhr.responseText);
                    console.error('Status Code:', xhr.status);
                    
                    var errorMsg = 'Failed to load dashboard statistics.';
                    if (xhr.status === 403) {
                        errorMsg = 'Access denied. Please check your permissions.';
                    } else if (xhr.status === 500) {
                        errorMsg = 'Server error. Please check error logs.';
                        // Try to parse error details from response
                        try {
                            var errorData = JSON.parse(xhr.responseText);
                            if (errorData && errorData.data && errorData.data.message) {
                                errorMsg += ' Details: ' + errorData.data.message;
                            }
                        } catch (e) {
                            // Response is not JSON, use raw text if helpful
                            if (xhr.responseText && xhr.responseText.length < 200) {
                                errorMsg += ' Details: ' + xhr.responseText;
                            }
                        }
                    }
                    
                    self.showError(errorMsg);
                }
            });
        },
        
        updateDashboardStats: function(data) {
            console.log('📊 Updating dashboard stats with data:', data);
            
            var totalClicks = data.total_clicks || 0;
            var uniqueClicks = data.unique_clicks || 0;
            var ctr = data.ctr || 0;
            var activeLinks = 0;
            
            // Calculate active links from various data sources
            if (data.active_links) {
                activeLinks = data.active_links;
            } else if (data.traffic_sources && data.traffic_sources.length > 0) {
                activeLinks = data.traffic_sources.length;
            } else if (data.link_count) {
                activeLinks = data.link_count;
            }
            
            $('#total-clicks').text(this.formatNumber(totalClicks));
            $('#unique-clicks').text(this.formatNumber(uniqueClicks));
            $('#ctr-rate').text(ctr.toFixed(1) + '%');
            $('#active-links').text(this.formatNumber(activeLinks));
            
            console.log('✅ Dashboard updated - Total:', totalClicks, 'Unique:', uniqueClicks, 'CTR:', ctr + '%', 'Active:', activeLinks);
        },
        
        updateCharts: function(data) {
            console.log('📈 Updating charts with data:', data);
            
            // Update charts with fallback for empty data
            this.updateTrafficSourcesChart(data.traffic_sources || []);
            this.updateBrowsersChart(data.browsers || []);
            this.updateOSChart(data.operating_systems || []);
            this.updateDevicesChart(data.devices || []);
            this.updateCountriesList(data.countries || []);
            
            // If no data available, show placeholder message
            if (!data.total_clicks || data.total_clicks == 0) {
                this.showNoDataMessage();
            }
        },
        
        showNoDataMessage: function() {
            // Show friendly "no data" message in chart areas
            var noDataMsg = '<div class="no-chart-data" style="padding: 60px; text-align: center; color: #666; font-style: italic;">📊 No data available for the selected period.<br><small>Try a different date range or add some affiliate links.</small></div>';
            
            $('#top-links').html(noDataMsg);
            $('.chart-wrapper').each(function() {
                if ($(this).find('canvas').length > 0) {
                    $(this).html(noDataMsg);
                }
            });
        },
        
        refreshClicksChart: function() {
            console.log('📊 Refreshing clicks chart...');
            var self = this;
            var dateRange = this.getCurrentDateRange();
            var granularity = $('#chart-granularity').val();
            
            // Convert date range to backend format
            var backendDateRange = dateRange;
            if (typeof dateRange === 'object' && dateRange.type === 'custom') {
                backendDateRange = dateRange.start + '|' + dateRange.end;
            }
            
            var ajaxData = {
                action: 'affiliate_hub_get_chart_data',
                date_range: backendDateRange,
                granularity: granularity,
                nonce: affiliateHubStats.nonce
            };
            
            // Add general filter for specific link
            var generalFilter = $('#general-filter').val();
            if (generalFilter) {
                ajaxData.link_ids = [generalFilter];
                console.log('📊 Chart: Filtering by general filter:', generalFilter);
            }
            
            // Add link_id if we're viewing specific link stats (legacy)
            if (affiliateHubStats.link_id && affiliateHubStats.link_id > 0) {
                ajaxData.link_ids = [affiliateHubStats.link_id];
                console.log('📊 Chart: Filtering by link_id:', affiliateHubStats.link_id);
            }
            
            console.log('📤 Chart AJAX data:', ajaxData);
            
            $.ajax({
                url: affiliateHubStats.ajax_url,
                type: 'POST',
                data: ajaxData,
                success: function(response) {
                    console.log('📊 Chart AJAX response:', response);
                    if (response.success) {
                        console.log('📊 Chart data received:', response.data);
                        self.updateClicksChart(response.data);
                    } else {
                        console.error('❌ Chart: Server returned error:', response);
                        self.showChartError('Failed to load chart data: ' + (response.data ? response.data.message : 'Unknown error'));
                    }
                },
                error: function(xhr, status, error) {
                    console.error('❌ Chart: AJAX request failed', {
                        status: status,
                        error: error,
                        responseText: xhr.responseText
                    });
                    self.showChartError('Failed to load chart data');
                }
            });
        },
        
        updateClicksChart: function(data) {
            console.log('📊 Updating clicks chart with data:', data);
            
            // Handle empty data
            if (!data || data.length === 0) {
                console.warn('⚠️ No chart data available');
                this.showChartError('No data available for the selected period');
                return;
            }
            
            var ctx = document.getElementById('clicks-chart');
            if (!ctx) {
                console.error('❌ Chart container element not found');
                return;
            }
            
            var labels = data.map(function(item) {
                var date = new Date(item.date_group);
                console.log('📅 Processing date:', item.date_group, '→', date);
                
                // Validate date to prevent 1970 issues
                if (isNaN(date.getTime()) || date.getFullYear() < 2020) {
                    console.warn('⚠️ Invalid date detected:', item.date_group, date);
                    // Try to parse manually
                    var parts = item.date_group.split('-');
                    if (parts.length >= 3) {
                        date = new Date(parseInt(parts[0]), parseInt(parts[1]) - 1, parseInt(parts[2]));
                    }
                }
                
                return date.toLocaleDateString();
            });
            
            console.log('📅 Chart labels generated:', labels);
            
            var totalClicks = data.map(function(item) {
                return parseInt(item.total_clicks);
            });
            
            var uniqueClicks = data.map(function(item) {
                return parseInt(item.unique_clicks);
            });
            
            if (this.charts.clicks) {
                this.charts.clicks.destroy();
            }
            
            var options = {
                series: [{
                    name: affiliateHubStats.labels.total_clicks || 'Total Clicks',
                    data: totalClicks
                }, {
                    name: affiliateHubStats.labels.unique_clicks || 'Unique Clicks', 
                    data: uniqueClicks
                }],
                chart: {
                    type: 'line',
                    height: 350,
                    animations: {
                        enabled: true,
                        easing: 'easeinout',
                        speed: 600
                    },
                    toolbar: {
                        show: true,
                        tools: {
                            download: true,
                            pan: false,
                            reset: true,
                            zoom: true,
                            zoomin: true,
                            zoomout: true
                        }
                    }
                },
                xaxis: {
                    categories: labels,
                    title: {
                        text: 'Date'
                    }
                },
                yaxis: {
                    title: {
                        text: 'Clicks'
                    },
                    min: 0,
                    labels: {
                        formatter: function(value) {
                            return Math.floor(value).toLocaleString();
                        }
                    }
                },
                colors: ['#0073aa', '#00a32a'],
                stroke: {
                    width: 3,
                    curve: 'smooth'
                },
                fill: {
                    type: 'gradient',
                    gradient: {
                        shade: 'light',
                        type: 'vertical',
                        shadeIntensity: 0.25,
                        gradientToColors: undefined,
                        inverseColors: false,
                        opacityFrom: 0.85,
                        opacityTo: 0.1,
                        stops: [50, 0, 100]
                    }
                },
                legend: {
                    position: 'top',
                    horizontalAlign: 'left'
                },
                grid: {
                    borderColor: '#f1f1f1',
                    strokeDashArray: 3
                },
                tooltip: {
                    theme: 'light',
                    x: {
                        format: 'dd MMM yyyy'
                    },
                    y: {
                        formatter: function(value) {
                            return value.toLocaleString() + ' clicks';
                        }
                    }
                },
                responsive: [{
                    breakpoint: 768,
                    options: {
                        chart: {
                            height: 250,
                            toolbar: {
                                show: false
                            }
                        },
                        legend: {
                            position: 'bottom'
                        }
                    }
                }]
            };
            
            this.charts.clicks = new ApexCharts(ctx, options);
            this.charts.clicks.render();
        },
        
        updateTrafficSourcesChart: function(data) {
            // Render traffic sources as a line chart
            var el = document.getElementById('traffic-sources-chart');
            if (!el) return;

            var labels = data.map(function(item) { return item.traffic_source || 'Unknown'; });
            var values = data.map(function(item) { return parseInt(item.clicks) || 0; });

            // Destroy previous instance if present
            if (this.charts.trafficSources) {
                try { this.charts.trafficSources.destroy(); } catch (e) { /* ignore */ }
            }

            // Fallback label
            var datasetLabel = (affiliateHubStats && affiliateHubStats.labels && affiliateHubStats.labels.traffic_sources) ? affiliateHubStats.labels.traffic_sources : 'Traffic Sources';

            var options = {
                series: [{
                    name: datasetLabel,
                    data: values
                }],
                chart: {
                    type: 'line',
                    height: 300,
                    animations: {
                        enabled: true,
                        easing: 'easeinout',
                        speed: 500
                    },
                    toolbar: {
                        show: true,
                        tools: {
                            download: true,
                            pan: false,
                            reset: true,
                            zoom: false
                        }
                    }
                },
                xaxis: {
                    categories: labels,
                    title: {
                        text: 'Traffic Source'
                    }
                },
                yaxis: {
                    title: {
                        text: 'Clicks'
                    },
                    min: 0
                },
                colors: ['#0073aa'],
                stroke: {
                    width: 3,
                    curve: 'smooth'
                },
                markers: {
                    size: 6,
                    strokeWidth: 2,
                    strokeColors: '#fff',
                    fillOpacity: 1
                },
                fill: {
                    type: 'gradient',
                    gradient: {
                        shade: 'light',
                        type: 'vertical',
                        opacityFrom: 0.4,
                        opacityTo: 0.1
                    }
                },
                legend: {
                    position: 'top'
                },
                grid: {
                    borderColor: '#f1f1f1',
                    strokeDashArray: 3
                },
                tooltip: {
                    theme: 'light',
                    y: {
                        formatter: function(value) {
                            return value + ' clicks';
                        }
                    }
                },
                responsive: [{
                    breakpoint: 768,
                    options: {
                        chart: {
                            height: 250,
                            toolbar: {
                                show: false
                            }
                        }
                    }
                }]
            };

            this.charts.trafficSources = new ApexCharts(el, options);
            this.charts.trafficSources.render();
        },
        
        updateBrowsersChart: function(data) {
            var ctx = document.getElementById('browsers-chart');
            if (!ctx) return;
            
            var labels = data.map(function(item) {
                return item.browser || 'Unknown';
            });
            
            var values = data.map(function(item) {
                return parseInt(item.clicks);
            });
            
            var colors = ['#0073aa', '#00a32a', '#d63638', '#dba617', '#9b51e0'];
            
            if (this.charts.browsers) {
                this.charts.browsers.destroy();
            }
            
            var options = {
                series: [{
                    name: 'Clicks',
                    data: values
                }],
                chart: {
                    type: 'bar',
                    height: 300,
                    animations: {
                        enabled: true,
                        easing: 'easeinout',
                        speed: 600
                    },
                    toolbar: {
                        show: true,
                        tools: {
                            download: true,
                            pan: false,
                            reset: false,
                            zoom: false
                        }
                    }
                },
                xaxis: {
                    categories: labels,
                    title: {
                        text: 'Browser'
                    }
                },
                yaxis: {
                    title: {
                        text: 'Clicks'
                    },
                    min: 0
                },
                colors: colors.slice(0, values.length),
                plotOptions: {
                    bar: {
                        borderRadius: 4,
                        horizontal: false,
                        columnWidth: '60%',
                        distributed: true
                    }
                },
                dataLabels: {
                    enabled: true,
                    style: {
                        colors: ['#fff'],
                        fontWeight: 'bold'
                    }
                },
                legend: {
                    show: false
                },
                grid: {
                    borderColor: '#f1f1f1',
                    strokeDashArray: 3
                },
                tooltip: {
                    theme: 'light',
                    y: {
                        formatter: function(value) {
                            return value + ' clicks';
                        }
                    }
                },
                responsive: [{
                    breakpoint: 768,
                    options: {
                        chart: {
                            height: 250,
                            toolbar: {
                                show: false
                            }
                        },
                        plotOptions: {
                            bar: {
                                columnWidth: '80%'
                            }
                        }
                    }
                }]
            };

            this.charts.browsers = new ApexCharts(ctx, options);
            this.charts.browsers.render();
        },
        
        updateOSChart: function(data) {
            var ctx = document.getElementById('os-chart');
            if (!ctx) return;
            
            var labels = data.map(function(item) {
                return item.os || 'Unknown';
            });
            
            var values = data.map(function(item) {
                return parseInt(item.clicks);
            });
            
            var colors = ['#0073aa', '#00a32a', '#d63638', '#dba617', '#9b51e0'];
            
            if (this.charts.os) {
                this.charts.os.destroy();
            }
            
            var options = {
                series: values,
                labels: labels,
                colors: colors.slice(0, values.length),
                chart: {
                    type: 'pie',
                    height: 350,
                    animations: {
                        enabled: true,
                        easing: 'easeinout',
                        speed: 600
                    },
                    toolbar: {
                        show: true,
                        tools: {
                            download: true,
                            pan: false,
                            reset: false,
                            zoom: false
                        }
                    }
                },
                plotOptions: {
                    pie: {
                        expandOnClick: true,
                        donut: {
                            size: '0%'
                        }
                    }
                },
                dataLabels: {
                    enabled: true,
                    formatter: function(val, opts) {
                        return opts.w.config.labels[opts.seriesIndex] + ": " + Math.round(val) + "%";
                    },
                    style: {
                        fontSize: '12px',
                        fontWeight: 'bold',
                        colors: ['#fff']
                    }
                },
                legend: {
                    position: 'bottom',
                    fontSize: '12px',
                    markers: {
                        width: 12,
                        height: 12
                    }
                },
                tooltip: {
                    theme: 'light',
                    y: {
                        formatter: function(value, opts) {
                            return opts.w.globals.series[opts.seriesIndex] + ' clicks (' + Math.round(value) + '%)';
                        }
                    }
                },
                responsive: [{
                    breakpoint: 768,
                    options: {
                        chart: {
                            height: 300,
                            toolbar: {
                                show: false
                            }
                        },
                        legend: {
                            position: 'bottom',
                            fontSize: '10px'
                        }
                    }
                }]
            };

            this.charts.os = new ApexCharts(ctx, options);
            this.charts.os.render();
        },
        
        updateDevicesChart: function(data) {
            var ctx = document.getElementById('devices-chart');
            if (!ctx) return;
            
            var labels = data.map(function(item) {
                return item.device_type || 'Unknown';
            });
            
            var values = data.map(function(item) {
                return parseInt(item.clicks);
            });
            
            var colors = ['#0073aa', '#00a32a', '#d63638'];
            
            if (this.charts.devices) {
                this.charts.devices.destroy();
            }
            
            var options = {
                series: values,
                labels: labels,
                colors: colors.slice(0, values.length),
                chart: {
                    type: 'donut',
                    height: 350,
                    animations: {
                        enabled: true,
                        easing: 'easeinout',
                        speed: 600
                    },
                    toolbar: {
                        show: true,
                        tools: {
                            download: true,
                            pan: false,
                            reset: false,
                            zoom: false
                        }
                    }
                },
                plotOptions: {
                    pie: {
                        donut: {
                            size: '45%',
                            labels: {
                                show: true,
                                total: {
                                    show: true,
                                    label: 'Total',
                                    formatter: function(w) {
                                        return w.globals.seriesTotals.reduce((a, b) => a + b, 0);
                                    }
                                }
                            }
                        }
                    }
                },
                dataLabels: {
                    enabled: true,
                    formatter: function(val, opts) {
                        return opts.w.config.labels[opts.seriesIndex] + ": " + Math.round(val) + "%";
                    },
                    style: {
                        fontSize: '12px',
                        fontWeight: 'bold',
                        colors: ['#fff']
                    }
                },
                legend: {
                    position: 'bottom',
                    fontSize: '12px',
                    markers: {
                        width: 12,
                        height: 12
                    }
                },
                tooltip: {
                    theme: 'light',
                    y: {
                        formatter: function(value, opts) {
                            return opts.w.globals.series[opts.seriesIndex] + ' clicks (' + Math.round(value) + '%)';
                        }
                    }
                },
                responsive: [{
                    breakpoint: 768,
                    options: {
                        chart: {
                            height: 300,
                            toolbar: {
                                show: false
                            }
                        },
                        legend: {
                            position: 'bottom',
                            fontSize: '10px'
                        }
                    }
                }]
            };

            this.charts.devices = new ApexCharts(ctx, options);
            this.charts.devices.render();
        },
        
        loadTopLinks: function() {
            var self = this;
            var dateRange = this.getCurrentDateRange();
            
            // Convert date range to backend format
            var backendDateRange = dateRange;
            if (typeof dateRange === 'object' && dateRange.type === 'custom') {
                backendDateRange = dateRange.start + '|' + dateRange.end;
            }
            
            $.ajax({
                url: affiliateHubStats.ajax_url,
                type: 'POST',
                data: {
                    action: 'affiliate_hub_get_top_links',
                    date_range: backendDateRange,
                    limit: 10,
                    nonce: affiliateHubStats.nonce
                },
                success: function(response) {
                    if (response.success) {
                        self.updateTopLinks(response.data);
                    }
                },
                error: function() {
                    console.error('Failed to load top links');
                }
            });
        },
        
        updateTopLinks: function(data) {
            var container = $('#top-links');
            container.empty();
            
            if (data.length === 0) {
                container.append('<div class="no-data">' + affiliateHubStats.labels.no_data + '</div>');
                return;
            }
            
            data.forEach(function(link) {
                var item = $('<div class="top-link-item"></div>');
                var title = $('<div class="top-link-title"></div>').text(link.post_title || 'Untitled');
                var stats = $('<div class="top-link-stats"></div>').text(
                    link.total_clicks + ' clicks (' + link.unique_clicks + ' unique) - ' + link.ctr + '% CTR'
                );
                
                item.append(title).append(stats);
                container.append(item);
            });
        },
        
        updateCountriesList: function(data) {
            var container = $('#countries-list');
            container.empty();
            
            if (data.length === 0) {
                container.append('<div class="no-data">' + affiliateHubStats.labels.no_data + '</div>');
                return;
            }
            
            data.forEach(function(country) {
                var item = $('<div class="country-item"></div>');
                var name = $('<span></span>').text(country.country || 'Unknown');
                var clicks = $('<span></span>').text(country.clicks + ' clicks');
                
                item.append(name).append(clicks);
                container.append(item);
            });
        },
        
        exportData: function() {
            var dateRange = this.getCurrentDateRange();
            
            // Convert date range to backend format
            var backendDateRange = dateRange;
            if (typeof dateRange === 'object' && dateRange.type === 'custom') {
                backendDateRange = dateRange.start + '|' + dateRange.end;
            }
            
            var form = $('<form>', {
                method: 'POST',
                action: affiliateHubStats.ajax_url
            });
            
            form.append($('<input>', {
                type: 'hidden',
                name: 'action',
                value: 'affiliate_hub_export_stats'
            }));
            
            form.append($('<input>', {
                type: 'hidden',
                name: 'date_range',
                value: backendDateRange
            }));
            
            form.append($('<input>', {
                type: 'hidden',
                name: 'type',
                value: 'summary'
            }));
            
            form.append($('<input>', {
                type: 'hidden',
                name: 'nonce',
                value: affiliateHubStats.nonce
            }));
            
            $('body').append(form);
            form.submit();
            form.remove();
        },
        
        getDateRange: function() {
            // First try legacy date range selector
            var range = $('#date-range').val();
            
            if (range === 'custom') {
                var start = $('#start-date').val();
                var end = $('#end-date').val();
                
                if (start && end) {
                    return start + '|' + end;
                }
                return '30'; // fallback
            }
            
            if (range) {
                return range;
            }
            
            // Fallback to new interface
            return this.getCurrentDateRange();
        },
        
        showLoading: function() {
            console.log('⏳ Showing loading state...');
            $('.stat-number').text('⏳');
            $('.loading').show();
        },
        
        showError: function(message) {
            console.error('💥 Showing error:', message);
            
            // Update stats cards with error message
            $('.stat-number').text('❌');
            
            // Show error in dashboard
            var errorHtml = '<div class="notice notice-error inline" style="margin: 10px 0;"><p><strong>Error:</strong> ' + message + '</p></div>';
            
            // Remove existing error messages
            $('.affiliate-stats-dashboard .notice-error').remove();
            
            // Add new error message at top of dashboard
            $('.affiliate-stats-dashboard').prepend(errorHtml);
            
            // Hide loading indicators
            $('.loading').hide();
        },
        
        showChartError: function(message) {
            console.error('📊 Chart Error:', message);
            
            // Show error message in chart area
            var chartContainer = $('#clicks-chart').parent();
            var errorHtml = '<div class="chart-error" style="padding: 40px; text-align: center; color: #dc3232; font-style: italic;">📊 ' + message + '</div>';
            
            // Replace chart with error message
            chartContainer.html(errorHtml);
        },
        
        formatNumber: function(num) {
            return parseInt(num).toLocaleString();
        },
        
        // Generate sample click data for testing
        generateSampleClicks: function() {
            console.log('🔧 Generating sample click data...');
            var self = this;
            
            if (!confirm('Generate sample click data? This will add test data to your database.')) {
                return;
            }
            
            $.ajax({
                url: affiliateHubStats.ajax_url,
                type: 'POST',
                data: {
                    action: 'affiliate_hub_generate_sample_clicks',
                    nonce: affiliateHubStats.nonce
                },
                success: function(response) {
                    if (response.success) {
                        alert('Generated ' + response.data.generated + ' sample clicks!');
                        self.refreshAllData(); // Refresh the dashboard
                    } else {
                        alert('Error: ' + response.data);
                    }
                },
                error: function(xhr, status, error) {
                    console.error('AJAX error:', error);
                    alert('Error generating sample data. Check console for details.');
                }
            });
        }
    };
    
    // Make AffiliateStats globally available
    window.AffiliateStats = AffiliateStats;
    
})(jQuery);
