/**
 * TinyMCE Plugin for Affiliate Hub
 */
(function() {
    'use strict';

    tinymce.PluginManager.add('affiliate_hub', function(editor, url) {
        
        // Add button to toolbar
        editor.addButton('affiliate_hub', {
            title: 'Insert Affiliate Link',
            icon: 'link',
            onclick: function() {
                openAffiliateModal();
            }
        });

        // Add menu item
        editor.addMenuItem('affiliate_hub', {
            text: 'Affiliate Link',
            icon: 'link',
            context: 'insert',
            onclick: function() {
                openAffiliateModal();
            }
        });

        function openAffiliateModal() {
            // Get selected text
            var selectedText = editor.selection.getContent({format: 'text'});
            
            editor.windowManager.open({
                title: 'Insert Affiliate Link',
                width: 600,
                height: 400,
                body: [
                    {
                        type: 'container',
                        html: '<div id="affiliate-hub-modal-content">Loading...</div>'
                    }
                ],
                onOpen: function() {
                    loadAffiliateLinks(selectedText);
                },
                buttons: [
                    {
                        text: 'Cancel',
                        onclick: function() {
                            editor.windowManager.close();
                        }
                    },
                    {
                        text: 'Insert',
                        subtype: 'primary',
                        onclick: function() {
                            insertSelectedLink();
                            editor.windowManager.close();
                        }
                    }
                ]
            });
        }

        function loadAffiliateLinks(selectedText) {
            jQuery.ajax({
                url: ajaxurl,
                type: 'POST',
                data: {
                    action: 'affiliate_hub_get_links_for_editor',
                    nonce: affiliateHubTinyMCE.nonce
                },
                success: function(response) {
                    if (response.success) {
                        renderAffiliateModal(response.data.links, selectedText);
                    } else {
                        jQuery('#affiliate-hub-modal-content').html('<p>Error loading affiliate links.</p>');
                    }
                },
                error: function() {
                    jQuery('#affiliate-hub-modal-content').html('<p>Error loading affiliate links.</p>');
                }
            });
        }

        function renderAffiliateModal(links, selectedText) {
            var html = '<div class="affiliate-hub-modal">';
            
            // Search box
            html += '<div class="search-box">';
            html += '<input type="text" id="affiliate-search" placeholder="Search affiliate links..." />';
            html += '</div>';
            
            // Link text input
            html += '<div class="link-text-box">';
            html += '<label for="link-text">Link Text:</label>';
            html += '<input type="text" id="link-text" value="' + (selectedText || '') + '" placeholder="Enter link text..." />';
            html += '</div>';
            
            // Links list
            html += '<div class="links-container" style="max-height: 200px; overflow-y: auto;">';
            
            if (links.length > 0) {
                html += '<table class="wp-list-table widefat fixed striped">';
                html += '<thead><tr><th>Title</th><th>URL</th><th>Clicks</th></tr></thead>';
                html += '<tbody>';
                
                links.forEach(function(link) {
                    html += '<tr class="affiliate-link-row" data-id="' + link.id + '" data-short-url="' + link.short_url + '">';
                    html += '<td><strong>' + link.title + '</strong></td>';
                    html += '<td><span class="url-preview">' + link.destination_url.substring(0, 50) + '...</span></td>';
                    html += '<td>' + (link.click_count || 0) + '</td>';
                    html += '</tr>';
                });
                
                html += '</tbody></table>';
            } else {
                html += '<p>No affiliate links found. <a href="' + affiliateHubTinyMCE.addNewUrl + '" target="_blank">Add your first link</a>.</p>';
            }
            
            html += '</div>';
            html += '</div>';
            
            jQuery('#affiliate-hub-modal-content').html(html);
            
            // Bind events
            bindModalEvents();
        }

        function bindModalEvents() {
            // Search functionality
            jQuery('#affiliate-search').on('keyup', function() {
                var query = jQuery(this).val().toLowerCase();
                jQuery('.affiliate-link-row').each(function() {
                    var $row = jQuery(this);
                    var title = $row.find('td:first').text().toLowerCase();
                    var url = $row.find('.url-preview').text().toLowerCase();
                    
                    if (title.indexOf(query) !== -1 || url.indexOf(query) !== -1) {
                        $row.show();
                    } else {
                        $row.hide();
                    }
                });
            });
            
            // Row selection
            jQuery('.affiliate-link-row').on('click', function() {
                jQuery('.affiliate-link-row').removeClass('selected');
                jQuery(this).addClass('selected');
            });
            
            // Double click to insert
            jQuery('.affiliate-link-row').on('dblclick', function() {
                insertSelectedLink();
                editor.windowManager.close();
            });
        }

        function insertSelectedLink() {
            var $selectedRow = jQuery('.affiliate-link-row.selected');
            
            if ($selectedRow.length === 0) {
                alert('Please select an affiliate link.');
                return;
            }
            
            var linkId = $selectedRow.data('id');
            var shortUrl = $selectedRow.data('short-url');
            var linkText = jQuery('#link-text').val() || $selectedRow.find('td:first').text();
            
            // Insert the link
            var linkHtml = '<a href="' + shortUrl + '" class="affiliate-link" data-link-id="' + linkId + '">' + linkText + '</a>';
            editor.insertContent(linkHtml);
        }

        // Add CSS for modal styling
        editor.on('init', function() {
            editor.dom.addStyle(`
                .affiliate-hub-modal .search-box {
                    margin-bottom: 15px;
                }
                .affiliate-hub-modal .search-box input {
                    width: 100%;
                    padding: 8px;
                    border: 1px solid #ddd;
                    border-radius: 4px;
                }
                .affiliate-hub-modal .link-text-box {
                    margin-bottom: 15px;
                }
                .affiliate-hub-modal .link-text-box label {
                    display: block;
                    margin-bottom: 5px;
                    font-weight: bold;
                }
                .affiliate-hub-modal .link-text-box input {
                    width: 100%;
                    padding: 8px;
                    border: 1px solid #ddd;
                    border-radius: 4px;
                }
                .affiliate-hub-modal .affiliate-link-row {
                    cursor: pointer;
                }
                .affiliate-hub-modal .affiliate-link-row:hover {
                    background-color: #f5f5f5;
                }
                .affiliate-hub-modal .affiliate-link-row.selected {
                    background-color: #e3f2fd;
                }
                .affiliate-hub-modal .url-preview {
                    color: #666;
                    font-style: italic;
                }
            `);
        });
    });

})();
