<?php
namespace AffiliateHub\Admin;

use AffiliateHub\Modules\EnhancedClickTracker;
use AffiliateHub\Modules\UserAgentParser;
use AffiliateHub\Core\Constants;
use Exception;

// Import all required WordPress functions to avoid namespace conflicts
use function add_action;
use function add_submenu_page;
use function wp_enqueue_script;
use function wp_enqueue_style;
use function wp_localize_script;
use function wp_add_inline_script;
use function admin_url;
use function wp_create_nonce;
use function wp_verify_nonce;
use function wp_send_json_success;
use function wp_send_json_error;
use function wp_doing_ajax;
use function current_user_can;
use function check_admin_referer;
use function has_action;
use function get_posts;
use function get_terms;
use function current_time;
use function wp_die;
use function error_log;
use function esc_html;
use function esc_attr;
use function __;
use function _e;
use function is_wp_error;
use function sprintf;
use function intval;
use function sanitize_text_field;
use function max;
use function min;
use function date;
use function count;
use function array_merge;
use function implode;
use function array_fill;
use function in_array;
use function strtoupper;
use function ceil;
use function strtotime;
use function parse_url;
use function json_encode;
use function headers_sent;
use function header;
use function ob_clean;
use function get_current_user_id;

/**
 * Enhanced Click Analytics Dashboard
 * 
 * Provides comprehensive analytics interface with:
 * - Click trends and performance metrics
 * - Geographic distribution
 * - Browser/OS analytics
 * - Session and unique visitor tracking
 * - Filtering and export capabilities
 */
class AnalyticsDashboard {
    
    private $tracker;
    private $user_agent_parser;
    
    public function __construct() {
        error_log('🚀 AnalyticsDashboard constructor START');
        
        $this->tracker = EnhancedClickTracker::get_instance();
        $this->user_agent_parser = new UserAgentParser();
        
        error_log('AnalyticsDashboard constructor - registering AJAX handlers');
        
        add_action('admin_menu', [$this, 'add_admin_menu'], 20);
        add_action('admin_enqueue_scripts', [$this, 'enqueue_admin_scripts']);
        add_action('wp_ajax_affiliate_hub_analytics_data', [$this, 'ajax_get_analytics_data']);
        add_action('wp_ajax_affiliate_hub_export_analytics', [$this, 'ajax_export_analytics']);
        add_action('wp_ajax_affiliate_hub_recent_activity', [$this, 'ajax_get_recent_activity']);
        add_action('wp_ajax_affiliate_hub_test_table', [$this, 'ajax_test_table']);
        add_action('wp_ajax_affiliate_hub_test_simple', [$this, 'ajax_test_simple']);
        
        // Add debug endpoint that ALWAYS works
        add_action('wp_ajax_affiliate_hub_debug_endpoint', [$this, 'ajax_debug_endpoint']);
        
        // Add WordPress init hook to verify registration
        add_action('init', function() {
            error_log('🔧 WordPress init - checking AJAX hooks:');
            error_log('  - test_table hook exists: ' . (has_action('wp_ajax_affiliate_hub_test_table') ? 'YES' : 'NO'));
            error_log('  - debug_endpoint hook exists: ' . (has_action('wp_ajax_affiliate_hub_debug_endpoint') ? 'YES' : 'NO'));
            error_log('  - recent_activity hook exists: ' . (has_action('wp_ajax_affiliate_hub_recent_activity') ? 'YES' : 'NO'));
        });
        
        error_log('AnalyticsDashboard AJAX handlers registered');
        error_log('🎯 AnalyticsDashboard constructor COMPLETE');
    }
    
    /**
     * Add analytics menu to WordPress admin
     */
    public function add_admin_menu() {
        add_submenu_page(
            'edit.php?post_type=' . Constants::POST_TYPE_AFFILIATE_LINK,
            __('Enhanced Analytics', 'affiliate-hub'),
            __('Analytics', 'affiliate-hub'),
            'manage_options',
            'affiliate-hub-analytics',
            [$this, 'render_analytics_page']
        );
    }
    
    /**
     * Enqueue admin scripts and styles
     */
    public function enqueue_admin_scripts($hook) {
        error_log('AnalyticsDashboard enqueue_admin_scripts called with hook: ' . $hook);
        
        if (strpos($hook, 'affiliate-hub-analytics') === false) {
            error_log('Skipping analytics scripts - not analytics page');
            return;
        }
        
        error_log('Loading analytics scripts for hook: ' . $hook);
        
        // Load ApexCharts with CDN + local fallback
        $apexcharts_local = AFFILIATE_HUB_URL . 'assets/js/apexcharts.min.js';
        $apexcharts_cdn = 'https://cdn.jsdelivr.net/npm/apexcharts@3.45.2/dist/apexcharts.min.js';
        
        // Try CDN first
        wp_enqueue_script(
            'apexcharts',
            $apexcharts_cdn,
            [],
            '3.45.2',
            true
        );
        
        // Enhanced fallback script with better error handling
        wp_add_inline_script('apexcharts', '
            window.addEventListener("load", function() {
                if (typeof ApexCharts === "undefined") {
                    console.log("ApexCharts CDN failed, loading local fallback...");
                    var script = document.createElement("script");
                    script.src = "' . $apexcharts_local . '";
                    script.onload = function() {
                        console.log("ApexCharts local fallback loaded");
                        if (typeof window.initAnalyticsCharts === "function") {
                            window.initAnalyticsCharts();
                        }
                    };
                    script.onerror = function() {
                        console.error("Failed to load ApexCharts library");
                        document.getElementById("analytics-loading").innerHTML = 
                            "<div class=\"error\"><p>Error: ApexCharts library could not be loaded.</p></div>";
                    };
                    document.head.appendChild(script);
                } else {
                    console.log("ApexCharts CDN loaded successfully");
                }
            });
        ');
        
        wp_enqueue_script(
            'affiliate-hub-analytics',
            AFFILIATE_HUB_URL . 'assets/js/analytics-dashboard-simple.js',
            ['jquery', 'wp-util'],
            AFFILIATE_HUB_VERSION . '-' . time(), // Force cache refresh
            true
        );
        
        wp_enqueue_script(
            'affiliate-hub-advanced-multiselect',
            AFFILIATE_HUB_URL . 'assets/js/advanced-multi-select.js',
            ['jquery', 'affiliate-hub-analytics'],
            AFFILIATE_HUB_VERSION . '-' . time(),
            true
        );
        
        wp_enqueue_style(
            'affiliate-hub-analytics',
            AFFILIATE_HUB_URL . 'assets/css/analytics-dashboard.css',
            [],
            AFFILIATE_HUB_VERSION
        );
        
        // Localize script for the analytics dashboard
        wp_localize_script('affiliate-hub-analytics', 'affiliateHubAnalytics', [
            'ajax_url' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('affiliate_hub_analytics_nonce'),
            'labels' => [
                'total_clicks' => __('Total Clicks', 'affiliate-hub'),
                'unique_clicks' => __('Unique Clicks (24h)', 'affiliate-hub'),
                'unique_sessions' => __('Unique Sessions', 'affiliate-hub'),
                'avg_processing_time' => __('Avg Processing Time (ms)', 'affiliate-hub'),
                'geo_cache_rate' => __('Geo Cache Hit Rate (%)', 'affiliate-hub'),
                'loading' => __('Loading...', 'affiliate-hub'),
                'error' => __('Error loading data', 'affiliate-hub')
            ]
        ]);
        
        error_log('Analytics scripts and localization enqueued for hook: ' . $hook);
    }
    
    /**
     * Render main analytics dashboard page
     */
    public function render_analytics_page() {
        // Handle sync clicks request
        if (isset($_GET['sync_clicks']) && $_GET['sync_clicks'] === '1') {
            if (current_user_can('manage_options') && check_admin_referer('sync_clicks_nonce', '_wpnonce', false)) {
                $result = $this->sync_clicks_data();
                
                if ($result['success']) {
                    echo '<div class="notice notice-success is-dismissible"><p>';
                    printf(
                        __('Data synchronized successfully! Updated %d links with %d discrepancies found.', 'affiliate-hub'),
                        $result['updated_links'],
                        count($result['discrepancies'])
                    );
                    echo '</p></div>';
                } else {
                    echo '<div class="notice notice-error is-dismissible"><p>';
                    echo esc_html($result['message']);
                    echo '</p></div>';
                }
            } else {
                echo '<div class="notice notice-error is-dismissible"><p>';
                _e('Security check failed.', 'affiliate-hub');
                echo '</p></div>';
            }
        }
        
        ?>
        <div class="wrap">
            <h1><?php _e('Enhanced Click Analytics', 'affiliate-hub'); ?></h1>
            
            <div id="analytics-loading" class="analytics-loading">
                <p><?php _e('Loading analytics dashboard...', 'affiliate-hub'); ?></p>
            </div>
            
            <div id="analytics-content" style="display:none;">
            <!-- Filters -->
            <div class="analytics-filters">
                <form id="analytics-filters-form">
                    <div class="filter-group">
                        <label for="date-range"><?php _e('Date Range:', 'affiliate-hub'); ?></label>
                        <select id="date-range" name="date_range">
                            <option value="7"><?php _e('Last 7 days', 'affiliate-hub'); ?></option>
                            <option value="30" selected><?php _e('Last 30 days', 'affiliate-hub'); ?></option>
                            <option value="90"><?php _e('Last 90 days', 'affiliate-hub'); ?></option>
                            <option value="custom"><?php _e('Custom range', 'affiliate-hub'); ?></option>
                        </select>
                    </div>
                    
                    <div class="filter-group custom-date-range" style="display:none;">
                        <label for="date-from"><?php _e('From:', 'affiliate-hub'); ?></label>
                        <input type="date" id="date-from" name="date_from">
                        
                        <label for="date-to"><?php _e('To:', 'affiliate-hub'); ?></label>
                        <input type="date" id="date-to" name="date_to">
                    </div>
                    
                    <div class="filter-group">
                        <label for="link-filter"><?php _e('Affiliate Link:', 'affiliate-hub'); ?></label>
                        <select id="link-filter" name="link_id">
                            <option value=""><?php _e('All Links', 'affiliate-hub'); ?></option>
                            <?php $this->render_link_options(); ?>
                        </select>
                    </div>
                    
                    <div class="filter-group filter-group-wide">
                        <label for="category-filter"><?php _e('Categories:', 'affiliate-hub'); ?></label>
                        <div class="multi-select-container" id="category-multi-select">
                            <div class="multi-select-input">
                                <input type="text" 
                                       id="category-search" 
                                       placeholder="<?php _e('Search categories...', 'affiliate-hub'); ?>"
                                       autocomplete="off">
                                <div class="selected-tags" id="selected-categories"></div>
                                <div class="dropdown-arrow">▼</div>
                            </div>
                            <div class="multi-select-dropdown" id="category-dropdown" style="display:none;">
                                <div class="quick-filters">
                                    <span class="quick-filter-label"><?php _e('Popular:', 'affiliate-hub'); ?></span>
                                    <?php $this->render_popular_categories(); ?>
                                </div>
                                <div class="dropdown-options" id="category-options">
                                    <?php $this->render_category_options_advanced(); ?>
                                </div>
                            </div>
                        </div>
                    </div>
                    
                    <div class="filter-group filter-group-wide">
                        <label for="tag-filter"><?php _e('Tags:', 'affiliate-hub'); ?></label>
                        <div class="multi-select-container" id="tag-multi-select">
                            <div class="multi-select-input">
                                <input type="text" 
                                       id="tag-search" 
                                       placeholder="<?php _e('Search tags...', 'affiliate-hub'); ?>"
                                       autocomplete="off">
                                <div class="selected-tags" id="selected-tags"></div>
                                <div class="dropdown-arrow">▼</div>
                            </div>
                            <div class="multi-select-dropdown" id="tag-dropdown" style="display:none;">
                                <div class="quick-filters">
                                    <span class="quick-filter-label"><?php _e('Popular:', 'affiliate-hub'); ?></span>
                                    <?php $this->render_popular_tags(); ?>
                                </div>
                                <div class="dropdown-options" id="tag-options">
                                    <?php $this->render_tag_options_advanced(); ?>
                                </div>
                            </div>
                        </div>
                    </div>
                    
                    <div class="filter-group">
                        <label for="unique-only">
                            <input type="checkbox" id="unique-only" name="unique_only" value="1">
                            <?php _e('Unique clicks only (24h)', 'affiliate-hub'); ?>
                        </label>
                    </div>
                    
                    <button type="button" id="apply-filters" class="button button-primary">
                        <?php _e('Apply Filters', 'affiliate-hub'); ?>
                    </button>
                    
                    <button type="button" id="export-data" class="button">
                        <?php _e('Export CSV', 'affiliate-hub'); ?>
                    </button>
                    
                    <a href="<?php echo esc_url(wp_nonce_url(add_query_arg('sync_clicks', '1'), 'sync_clicks_nonce')); ?>" 
                       class="button button-secondary" 
                       onclick="return confirm('<?php _e('This will synchronize total clicks data between meta fields and database table. Continue?', 'affiliate-hub'); ?>');"
                       title="<?php _e('Synchronize total clicks count with database table', 'affiliate-hub'); ?>">
                        <?php _e('Sync Clicks Data', 'affiliate-hub'); ?>
                    </a>
                </form>
            </div>
            
            <!-- Summary Cards -->
            <div class="analytics-summary">
                <div class="summary-card">
                    <h3><?php _e('Total Clicks', 'affiliate-hub'); ?></h3>
                    <div class="summary-value" id="total-clicks">-</div>
                </div>
                
                <div class="summary-card">
                    <h3><?php _e('Unique Clicks (24h)', 'affiliate-hub'); ?></h3>
                    <div class="summary-value" id="unique-clicks">-</div>
                </div>
                
                <div class="summary-card">
                    <h3><?php _e('Unique Sessions', 'affiliate-hub'); ?></h3>
                    <div class="summary-value" id="unique-sessions">-</div>
                </div>
                
                <div class="summary-card">
                    <h3><?php _e('Avg Processing Time', 'affiliate-hub'); ?></h3>
                    <div class="summary-value" id="avg-processing-time">-</div>
                    <div class="summary-unit">ms</div>
                </div>
                
                <div class="summary-card">
                    <h3><?php _e('Geo Cache Hit Rate', 'affiliate-hub'); ?></h3>
                    <div class="summary-value" id="geo-cache-rate">-</div>
                    <div class="summary-unit">%</div>
                </div>
            </div>
            
            <!-- Click Trends Section - Full Width -->
            <div class="analytics-full-width">
                <div class="chart-container full-width">
                    <h3><?php _e('Click Trends', 'affiliate-hub'); ?></h3>
                    <div id="clicks-chart" class="chart-placeholder"></div>
                </div>
            </div>
            
            <!-- Charts and Tables Section - Side by Side -->
            <div class="analytics-side-by-side">
                <div class="left-column">
                    <div class="table-container">
                        <h3><?php _e('Top Performing Links', 'affiliate-hub'); ?></h3>
                        <table id="top-links-table" class="analytics-table">
                            <thead>
                                <tr>
                                    <th><?php _e('Link Name', 'affiliate-hub'); ?></th>
                                    <th><?php _e('URL', 'affiliate-hub'); ?></th>
                                    <th><?php _e('Total Clicks', 'affiliate-hub'); ?></th>
                                    <th><?php _e('Unique Clicks', 'affiliate-hub'); ?></th>
                                    <th><?php _e('CTR', 'affiliate-hub'); ?></th>
                                    <th><?php _e('Last Click', 'affiliate-hub'); ?></th>
                                </tr>
                            </thead>
                            <tbody>
                                <!-- Data will be populated by JavaScript -->
                            </tbody>
                        </table>
                    </div>
                </div>
                
                <div class="right-column">
                    <div class="chart-container">
                        <h3><?php _e('Top Links Performance', 'affiliate-hub'); ?></h3>
                        <div id="top-links-chart" class="chart-placeholder"></div>
                    </div>
                </div>
            </div>

            <!-- Recent Activity Section - Full Width -->
            <div class="recent-activity-section">
                <div class="recent-activity-header">
                    <h3><?php _e('Recent Activity', 'affiliate-hub'); ?></h3>
                    <div class="activity-controls">
                        <div class="pagination-info">
                            <span id="activity-showing">Showing 1-25 of 0 entries</span>
                        </div>
                        <div class="pagination-controls">
                            <label for="activity-per-page"><?php _e('Show:', 'affiliate-hub'); ?></label>
                            <select id="activity-per-page">
                                <option value="25">25</option>
                                <option value="50">50</option>
                                <option value="100">100</option>
                                <option value="200">200</option>
                            </select>
                            <span><?php _e('entries', 'affiliate-hub'); ?></span>
                        </div>
                    </div>
                </div>
                
                <div class="table-container activity-table-container">
                    <table id="recent-activity-table" class="analytics-table activity-table">
                        <thead>
                            <tr>
                                <th class="sortable" data-sort="click_time"><?php _e('Date/Time', 'affiliate-hub'); ?> <span class="sort-icon">↕</span></th>
                                <th class="sortable" data-sort="link_title"><?php _e('Link', 'affiliate-hub'); ?> <span class="sort-icon">↕</span></th>
                                <th class="sortable" data-sort="referrer"><?php _e('Referrer', 'affiliate-hub'); ?> <span class="sort-icon">↕</span></th>
                                <th class="sortable" data-sort="ip_address"><?php _e('IP Address', 'affiliate-hub'); ?> <span class="sort-icon">↕</span></th>
                                <th class="sortable" data-sort="browser"><?php _e('Browser', 'affiliate-hub'); ?> <span class="sort-icon">↕</span></th>
                                <th class="sortable" data-sort="os"><?php _e('OS', 'affiliate-hub'); ?> <span class="sort-icon">↕</span></th>
                                <th class="sortable" data-sort="device"><?php _e('Device', 'affiliate-hub'); ?> <span class="sort-icon">↕</span></th>
                                <th class="sortable" data-sort="country"><?php _e('Country', 'affiliate-hub'); ?> <span class="sort-icon">↕</span></th>
                            </tr>
                        </thead>
                        <tbody>
                            <!-- Data will be populated by JavaScript -->
                        </tbody>
                    </table>
                </div>
                
                <div class="activity-pagination">
                    <div class="pagination-nav">
                        <button id="activity-prev" class="pagination-btn" disabled>
                            <span class="dashicons dashicons-arrow-left-alt2"></span>
                            <?php _e('Previous', 'affiliate-hub'); ?>
                        </button>
                        
                        <div class="pagination-pages" id="activity-pages">
                            <!-- Page numbers will be generated by JavaScript -->
                        </div>
                        
                        <button id="activity-next" class="pagination-btn" disabled>
                            <?php _e('Next', 'affiliate-hub'); ?>
                            <span class="dashicons dashicons-arrow-right-alt2"></span>
                        </button>
                    </div>
                </div>
            </div>
            
            <div id="loading-indicator" style="display:none;">
                <?php _e('Loading analytics data...', 'affiliate-hub'); ?>
            </div>
            </div> <!-- End analytics-content -->
        </div>
        <?php
    }
    
    /**
     * Render affiliate link options for filter dropdown
     */
    private function render_link_options() {
        $links = get_posts([
            'post_type' => Constants::POST_TYPE_AFFILIATE_LINK,
            'post_status' => 'publish',
            'numberposts' => -1,
            'orderby' => 'title',
            'order' => 'ASC'
        ]);
        
        foreach ($links as $link) {
            echo '<option value="' . esc_attr($link->ID) . '">' . esc_html($link->post_title) . '</option>';
        }
    }
    
    /**
     * Render category options for filter dropdown
     */
    private function render_category_options() {
        $categories = get_terms([
            'taxonomy' => Constants::TAXONOMY_AFFILIATE_CATEGORY,
            'hide_empty' => false,
            'orderby' => 'name',
            'order' => 'ASC'
        ]);
        
        if (!is_wp_error($categories) && ($categories && count($categories) > 0)) {
            foreach ($categories as $category) {
                $count = $this->get_category_links_count($category->term_id);
                echo '<option value="' . esc_attr($category->term_id) . '">' . 
                     esc_html($category->name) . ' (' . $count . ')</option>';
            }
        }
    }
    
    /**
     * Render advanced category options for multi-select
     */
    private function render_category_options_advanced() {
        $categories = get_terms([
            'taxonomy' => Constants::TAXONOMY_AFFILIATE_CATEGORY,
            'hide_empty' => false,
            'orderby' => 'count',
            'order' => 'DESC'
        ]);
        
        if (!is_wp_error($categories) && ($categories && count($categories) > 0)) {
            foreach ($categories as $category) {
                $count = $this->get_category_links_count($category->term_id);
                echo '<div class="dropdown-option" data-value="' . esc_attr($category->term_id) . '" data-text="' . esc_attr($category->name) . '">';
                echo '<span class="option-name">' . esc_html($category->name) . '</span>';
                echo '<span class="option-count">(' . $count . ')</span>';
                echo '</div>';
            }
        }
    }
    
    /**
     * Render popular categories as quick filter buttons
     */
    private function render_popular_categories() {
        $categories = get_terms([
            'taxonomy' => Constants::TAXONOMY_AFFILIATE_CATEGORY,
            'hide_empty' => false,
            'orderby' => 'count',
            'order' => 'DESC',
            'number' => 5 // Top 5 most used categories
        ]);
        
        if (!is_wp_error($categories) && $categories && count($categories) > 0) {
            foreach ($categories as $category) {
                echo '<button type="button" class="quick-filter-btn" data-type="category" data-value="' . esc_attr($category->term_id) . '">';
                echo esc_html($category->name);
                echo '</button>';
            }
        }
    }
    
    /**
     * Render tag options for filter dropdown
     */
    private function render_tag_options() {
        $tags = get_terms([
            'taxonomy' => Constants::TAXONOMY_AFFILIATE_TAG,
            'hide_empty' => false,
            'orderby' => 'name',
            'order' => 'ASC'
        ]);
        
        if (!is_wp_error($tags) && ($tags && count($tags) > 0)) {
            foreach ($tags as $tag) {
                $count = $this->get_tag_links_count($tag->term_id);
                echo '<option value="' . esc_attr($tag->term_id) . '">' . 
                     esc_html($tag->name) . ' (' . $count . ')</option>';
            }
        }
    }
    
    /**
     * Render advanced tag options for multi-select
     */
    private function render_tag_options_advanced() {
        $tags = get_terms([
            'taxonomy' => Constants::TAXONOMY_AFFILIATE_TAG,
            'hide_empty' => false,
            'orderby' => 'count',
            'order' => 'DESC'
        ]);
        
        if (!is_wp_error($tags) && ($tags && count($tags) > 0)) {
            foreach ($tags as $tag) {
                $count = $this->get_tag_links_count($tag->term_id);
                echo '<div class="dropdown-option" data-value="' . esc_attr($tag->term_id) . '" data-text="' . esc_attr($tag->name) . '">';
                echo '<span class="option-name">' . esc_html($tag->name) . '</span>';
                echo '<span class="option-count">(' . $count . ')</span>';
                echo '</div>';
            }
        }
    }
    
    /**
     * Render popular tags as quick filter buttons
     */
    private function render_popular_tags() {
        $tags = get_terms([
            'taxonomy' => Constants::TAXONOMY_AFFILIATE_TAG,
            'hide_empty' => false,
            'orderby' => 'count',
            'order' => 'DESC',
            'number' => 5 // Top 5 most used tags
        ]);
        
        if (!is_wp_error($tags) && ($tags && count($tags) > 0)) {
            foreach ($tags as $tag) {
                echo '<button type="button" class="quick-filter-btn" data-type="tag" data-value="' . esc_attr($tag->term_id) . '">';
                echo esc_html($tag->name);
                echo '</button>';
            }
        }
    }
    
    /**
     * Get count of links in specific category
     */
    private function get_category_links_count($category_id) {
        $posts = get_posts([
            'post_type' => Constants::POST_TYPE_AFFILIATE_LINK,
            'post_status' => 'publish',
            'numberposts' => -1,
            'tax_query' => [
                [
                    'taxonomy' => Constants::TAXONOMY_AFFILIATE_CATEGORY,
                    'field' => 'term_id',
                    'terms' => $category_id
                ]
            ]
        ]);
        
        return count($posts);
    }
    
    /**
     * Get count of links with specific tag
     */
    private function get_tag_links_count($tag_id) {
        $posts = get_posts([
            'post_type' => Constants::POST_TYPE_AFFILIATE_LINK,
            'post_status' => 'publish',
            'numberposts' => -1,
            'tax_query' => [
                [
                    'taxonomy' => Constants::TAXONOMY_AFFILIATE_TAG,
                    'field' => 'term_id',
                    'terms' => $tag_id
                ]
            ]
        ]);
        
        return count($posts);
    }
    
    /**
     * AJAX handler for analytics data
     */
    public function ajax_get_analytics_data() {
        error_log('=== ANALYTICS AJAX CALLED ===');
        error_log('POST data: ' . print_r($_POST, true));
        error_log('Nonce from POST: ' . ($_POST['nonce'] ?? 'NOT SET'));
        error_log('Expected nonce: ' . wp_create_nonce('affiliate_hub_analytics_nonce'));
        
        try {
            // Verify nonce
            if (!check_ajax_referer('affiliate_hub_analytics_nonce', 'nonce', false)) {
                error_log('AJAX nonce verification failed');
                wp_send_json_error('Security check failed');
                return;
            }
            
            error_log('Nonce verification passed');
            
            $filters = $this->parse_filters($_POST);
            error_log('Parsed filters: ' . print_r($filters, true));
            
            $data = [
                'summary' => $this->get_summary_data($filters),
                'trends' => $this->get_trends_data($filters),
                'browser_stats' => $this->get_browser_stats($filters),
                'os_stats' => $this->get_os_stats($filters),
                'device_stats' => $this->get_device_stats($filters),
                'geo_stats' => $this->get_geo_stats($filters),
                'top_links' => $this->get_top_links($filters),
                'recent_activity' => $this->get_recent_activity($filters)
            ];
            
            error_log('Analytics data prepared successfully');
            error_log('Data summary count: ' . count($data['summary'] ?? []));
            error_log('Top links count: ' . count($data['top_links'] ?? []));
            error_log('Recent activity count: ' . count($data['recent_activity'] ?? []));
            
            wp_send_json_success($data);
            
        } catch (\Exception $e) {
            error_log('Analytics AJAX Exception: ' . $e->getMessage());
            error_log('Stack trace: ' . $e->getTraceAsString());
            wp_send_json_error('Failed to load analytics data: ' . $e->getMessage());
        } catch (\Error $e) {
            error_log('Analytics AJAX Fatal Error: ' . $e->getMessage());
            error_log('Stack trace: ' . $e->getTraceAsString());
            wp_send_json_error('Fatal error in analytics: ' . $e->getMessage());
        }
    }
    
    /**
     * Parse filters from request
     */
    private function parse_filters($request) {
        $filters = [];
        
        error_log('Raw request data: ' . print_r($request, true));
        
        // Date range
        if ($request['date_range'] && $request['date_range'] !== '') {
            $date_range = sanitize_text_field($request['date_range']);
            error_log('Date range value: ' . $date_range);
            
            // Only process numeric date ranges, not 'custom'
            if (is_numeric($date_range)) {
                $range = intval($date_range);
                if ($range > 0) {
                    // Start from beginning of day X days ago
                    $filters['date_from'] = date('Y-m-d 00:00:00', strtotime("-{$range} days"));
                    // End at end of today
                    $filters['date_to'] = date('Y-m-d 23:59:59');
                    error_log("Set date range filter for $range days: {$filters['date_from']} to {$filters['date_to']}");
                }
            } elseif ($date_range === 'custom') {
                // For custom range, we need both date_from and date_to
                if ((!$request['date_from'] || $request['date_from'] === '') || (!$request['date_to'] || $request['date_to'] === '')) {
                    error_log('Custom range selected but missing dates - using default 30 days');
                    // Fallback to 30 days if custom dates are not provided
                    $filters['date_from'] = date('Y-m-d 00:00:00', strtotime('-30 days'));
                    $filters['date_to'] = date('Y-m-d 23:59:59');
                }
            }
        }
        
        // Custom date range - these should override any date_range settings
        if ($request['date_from'] && $request['date_from'] !== '') {
            $filters['date_from'] = sanitize_text_field($request['date_from']) . ' 00:00:00';
            error_log('Custom date_from: ' . $filters['date_from']);
        }
        
        if ($request['date_to'] && $request['date_to'] !== '') {
            $filters['date_to'] = sanitize_text_field($request['date_to']) . ' 23:59:59';
            error_log('Custom date_to: ' . $filters['date_to']);
        }
        
        // Link filter
        if ($request['link_id'] && $request['link_id'] !== '') {
            $filters['link_id'] = intval($request['link_id']);
        }
        
        // Category filter - support multiple categories
        if ($request['category_ids'] && (is_array($request['category_ids']) ? count($request['category_ids']) > 0 : $request['category_ids'] !== '')) {
            if (is_array($request['category_ids'])) {
                $filters['category_ids'] = array_map('intval', $request['category_ids']);
            } else {
                // Single category as string (backward compatibility)
                $filters['category_ids'] = [intval($request['category_ids'])];
            }
        } elseif ($request['category_id'] && $request['category_id'] !== '') {
            // Backward compatibility with old single category filter
            $filters['category_ids'] = [intval($request['category_id'])];
        }
        
        // Tag filter - support multiple tags
        if ($request['tag_ids'] && (is_array($request['tag_ids']) ? count($request['tag_ids']) > 0 : $request['tag_ids'] !== '')) {
            if (is_array($request['tag_ids'])) {
                $filters['tag_ids'] = array_map('intval', $request['tag_ids']);
            } else {
                // Single tag as string (backward compatibility)
                $filters['tag_ids'] = [intval($request['tag_ids'])];
            }
        } elseif ($request['tag_id'] && $request['tag_id'] !== '') {
            // Backward compatibility with old single tag filter
            $filters['tag_ids'] = [intval($request['tag_id'])];
        }
        
        // Unique only
        if ($request['unique_only'] && $request['unique_only'] !== '') {
            $filters['unique_only'] = true;
        }
        
        error_log('Final parsed filters: ' . print_r($filters, true));
        
        return $filters;
    }
    
    /**
     * Get summary analytics data
     */
    private function get_summary_data($filters) {
        error_log('Getting summary data with filters: ' . print_r($filters, true));
        
        try {
            global $wpdb;
            $table_clicks = $wpdb->prefix . Constants::TABLE_LINK_CLICKS;
            $where_clause = $this->build_where_clause($filters);
            
            error_log('Summary WHERE clause: ' . $where_clause);
            
            $sql = "SELECT 
                        COUNT(*) as total_clicks,
                        COUNT(CASE WHEN is_unique_24h = 1 THEN 1 END) as unique_clicks,
                        COUNT(DISTINCT ip_address) as unique_sessions,
                        AVG(processing_time_ms) as avg_processing_time,
                        AVG(CASE WHEN country IS NOT NULL AND country != '' THEN 100 ELSE 0 END) as geo_cache_hit_rate
                    FROM $table_clicks 
                    WHERE $where_clause";
            
            error_log('Summary SQL: ' . $sql);
            $result = $wpdb->get_row($sql, ARRAY_A);
            error_log('Summary data result: ' . print_r($result, true));
            
            // Ensure numeric values
            if ($result) {
                $result['total_clicks'] = intval($result['total_clicks']);
                $result['unique_clicks'] = intval($result['unique_clicks']);
                $result['unique_sessions'] = intval($result['unique_sessions']);
                $result['avg_processing_time'] = round(floatval($result['avg_processing_time']), 2);
                $result['geo_cache_hit_rate'] = round(floatval($result['geo_cache_hit_rate']), 1);
            }
            
            return $result ?: [
                'total_clicks' => 0,
                'unique_clicks' => 0,
                'unique_sessions' => 0,
                'avg_processing_time' => 0,
                'geo_cache_hit_rate' => 0
            ];
            
        } catch (\Exception $e) {
            error_log('Error in get_summary_data: ' . $e->getMessage());
            return [
                'total_clicks' => 0,
                'unique_clicks' => 0,
                'unique_sessions' => 0,
                'avg_processing_time' => 0,
                'geo_cache_hit_rate' => 0
            ];
        }
    }
    
    /**
     * Get click trends data
     */
    private function get_trends_data($filters) {
        try {
            error_log('Getting trends data with filters: ' . print_r($filters, true));
            
            global $wpdb;
            $table_clicks = $wpdb->prefix . Constants::TABLE_LINK_CLICKS;
            
            // Use the centralized build_where_clause method that handles all filters including categories and tags
            $where_clause = $this->build_where_clause($filters);
            
            // Fallback to days filter if no date range specified
            if ((!isset($filters['date_from']) || $filters['date_from'] === "") && 
                (!isset($filters['date_to']) || $filters['date_to'] === "")) {
                $days = 30; // default
                $where_clause .= " AND clicked_at >= DATE_SUB(NOW(), INTERVAL $days DAY)";
                error_log('Using fallback days filter: ' . $days);
            }
            
            error_log('Final WHERE clause for trends: ' . $where_clause);
            
            $sql = "
                SELECT 
                    DATE(clicked_at) as date,
                    COUNT(*) as total_clicks,
                    COUNT(CASE WHEN is_unique_24h = 1 THEN 1 END) as unique_clicks
                FROM $table_clicks 
                WHERE $where_clause
                GROUP BY DATE(clicked_at)
                ORDER BY date ASC
            ";
            
            error_log('Final SQL for trends: ' . $sql);
            $result = $wpdb->get_results($sql, ARRAY_A) ?: [];
            error_log('Trends data result count: ' . count($result));
            
            if (!empty($result)) {
                error_log('First trend result: ' . print_r($result[0], true));
                error_log('Last trend result: ' . print_r(end($result), true));
            }
            
            return $result;
            
        } catch (\Exception $e) {
            error_log('Error in get_trends_data: ' . $e->getMessage());
            return [];
        }
    }
    
    /**
     * Get browser statistics
     */
    private function get_browser_stats($filters) {
        global $wpdb;
        
        $table_clicks = $wpdb->prefix . Constants::TABLE_LINK_CLICKS;
        $where_clause = $this->build_where_clause($filters);
        
        // Check if table exists
        $table_exists = $wpdb->get_var("SHOW TABLES LIKE '$table_clicks'") == $table_clicks;
        error_log("Table $table_clicks exists: " . ($table_exists ? 'YES' : 'NO'));
        
        if (!$table_exists) {
            error_log("Analytics table missing: $table_clicks");
            return [];
        }
        
        // Check total row count
        $total_rows = $wpdb->get_var("SELECT COUNT(*) FROM $table_clicks");
        error_log("Total rows in $table_clicks: $total_rows");
        
        $sql = "SELECT browser, COUNT(*) as count
                FROM $table_clicks 
                WHERE $where_clause AND browser != ''
                GROUP BY browser
                ORDER BY count DESC
                LIMIT 10";
        
        error_log('Browser stats SQL: ' . $sql);
        $result = $wpdb->get_results($sql, ARRAY_A) ?: [];
        error_log('Browser stats result: ' . print_r($result, true));
        
        return $result;
    }
    
    /**
     * Get operating system statistics
     */
    private function get_os_stats($filters) {
        global $wpdb;
        
        $table_clicks = $wpdb->prefix . Constants::TABLE_LINK_CLICKS;
        $where_clause = $this->build_where_clause($filters);
        
        $sql = "SELECT os, COUNT(*) as count
                FROM $table_clicks 
                WHERE $where_clause AND os != ''
                GROUP BY os
                ORDER BY count DESC
                LIMIT 10";
        
        return $wpdb->get_results($sql, ARRAY_A) ?: [];
    }
    
    /**
     * Get device type statistics
     */
    private function get_device_stats($filters) {
        global $wpdb;
        
        $table_clicks = $wpdb->prefix . Constants::TABLE_LINK_CLICKS;
        $where_clause = $this->build_where_clause($filters);
        
        $sql = "SELECT device_type, COUNT(*) as count
                FROM $table_clicks 
                WHERE $where_clause AND device_type != ''
                GROUP BY device_type
                ORDER BY count DESC";
        
        return $wpdb->get_results($sql, ARRAY_A) ?: [];
    }
    
    /**
     * Get geographic statistics
     */
    private function get_geo_stats($filters) {
        global $wpdb;
        
        $table_clicks = $wpdb->prefix . Constants::TABLE_LINK_CLICKS;
        $where_clause = $this->build_where_clause($filters);
        
        $sql = "SELECT country, COUNT(*) as count
                FROM $table_clicks 
                WHERE $where_clause AND country != ''
                GROUP BY country
                ORDER BY count DESC
                LIMIT 15";
        
        return $wpdb->get_results($sql, ARRAY_A) ?: [];
    }
    
    /**
     * Get top performing links
     */
    private function get_top_links($filters) {
        global $wpdb;
        
        error_log('=== GET_TOP_LINKS DEBUG ===');
        
        $table_clicks = $wpdb->prefix . Constants::TABLE_LINK_CLICKS;
        $where_clause = $this->build_where_clause($filters);
        
        error_log('Table clicks: ' . $table_clicks);
        error_log('Where clause: ' . $where_clause);
        
        // First, let's see what link_ids we have in clicks table
        $link_ids_sql = "SELECT DISTINCT link_id, COUNT(*) as count FROM $table_clicks WHERE $where_clause GROUP BY link_id ORDER BY count DESC LIMIT 20";
        $link_ids_result = $wpdb->get_results($link_ids_sql, ARRAY_A);
        error_log('Link IDs in clicks table: ' . print_r($link_ids_result, true));
        
        // Now let's check if these posts exist and what post type they are
        if (($link_ids_result && count($link_ids_result) > 0)) {
            $ids = array_map(function($row) { return intval($row['link_id']); }, $link_ids_result);
            $ids_str = implode(',', $ids);
            $posts_sql = "SELECT ID, post_title, post_type, post_status FROM {$wpdb->posts} WHERE ID IN ($ids_str)";
            $posts_result = $wpdb->get_results($posts_sql, ARRAY_A);
            error_log('Posts for these IDs: ' . print_r($posts_result, true));
            
            // Check unique clicks values for debugging
            $unique_debug_sql = "SELECT link_id, is_unique_24h, COUNT(*) as count FROM $table_clicks WHERE link_id IN ($ids_str) GROUP BY link_id, is_unique_24h ORDER BY link_id, is_unique_24h";
            $unique_debug = $wpdb->get_results($unique_debug_sql, ARRAY_A);
            error_log('Unique clicks debug: ' . print_r($unique_debug, true));
        }
        
        $sql = "SELECT 
                    c.link_id,
                    p.post_title as link_title,
                    p.post_type,
                    p.post_status,
                    m.meta_value as destination_url,
                    COUNT(*) as total_clicks,
                    COUNT(DISTINCT c.ip_address) as unique_clicks,
                    AVG(c.processing_time_ms) as avg_processing_time,
                    ROUND(
                        CASE 
                            WHEN COUNT(*) > 0 THEN (COUNT(DISTINCT c.ip_address) / COUNT(*)) * 100 
                            ELSE 0 
                        END, 2
                    ) as ctr_rate,
                    MAX(c.clicked_at) as last_click
                FROM $table_clicks c
                LEFT JOIN {$wpdb->posts} p ON c.link_id = p.ID
                LEFT JOIN {$wpdb->postmeta} m ON p.ID = m.post_id AND m.meta_key = '" . Constants::META_DESTINATION_URL . "'
                WHERE $where_clause
                GROUP BY c.link_id, p.post_title, p.post_type, p.post_status, m.meta_value
                ORDER BY total_clicks DESC
                LIMIT 10";
        
        error_log('Final SQL: ' . $sql);
        $result = $wpdb->get_results($sql, ARRAY_A);
        error_log('Top links result: ' . print_r($result, true));
        
        return $result ?: [];
    }
    
    /**
     * Get recent activity
     */
    private function get_recent_activity($filters) {
        global $wpdb;
        
        $table_clicks = $wpdb->prefix . Constants::TABLE_LINK_CLICKS;
        $where_clause = $this->build_where_clause($filters);
        
        $sql = "SELECT 
                    c.clicked_at,
                    c.link_id,
                    p.post_title as link_title,
                    c.ip_address,
                    c.user_agent,
                    c.country,
                    c.browser,
                    c.device_type,
                    c.is_unique_24h,
                    c.processing_time_ms
                FROM $table_clicks c
                LEFT JOIN {$wpdb->posts} p ON c.link_id = p.ID
                WHERE $where_clause
                ORDER BY c.clicked_at DESC
                LIMIT 50";
        
        return $wpdb->get_results($sql, ARRAY_A) ?: [];
    }
    
    /**
     * Build WHERE clause from filters
     */
    private function build_where_clause($filters) {
        global $wpdb;
        $conditions = ['1=1'];
        
        if (!empty($filters['link_id'])) {
            $conditions[] = 'link_id = ' . intval($filters['link_id']);
        }
        
        // Category filter - get all link IDs that belong to selected categories
        if (!empty($filters['category_ids']) && is_array($filters['category_ids'])) {
            $category_links = get_posts([
                'post_type' => Constants::POST_TYPE_AFFILIATE_LINK,
                'post_status' => 'publish',
                'numberposts' => -1,
                'fields' => 'ids',
                'tax_query' => [
                    [
                        'taxonomy' => Constants::TAXONOMY_AFFILIATE_CATEGORY,
                        'field' => 'term_id',
                        'terms' => $filters['category_ids'],
                        'operator' => 'IN'
                    ]
                ]
            ]);
            
            if (!empty($category_links)) {
                $link_ids_str = implode(',', array_map('intval', $category_links));
                $conditions[] = "link_id IN ($link_ids_str)";
                error_log('Category filter: link_ids = ' . $link_ids_str . ' for categories: ' . implode(',', $filters['category_ids']));
            } else {
                // No links in selected categories, return no results
                $conditions[] = '1=0';
                error_log('Category filter: no links found for categories ' . implode(',', $filters['category_ids']));
            }
        }
        
        // Tag filter - get all link IDs that have selected tags
        if (!empty($filters['tag_ids']) && is_array($filters['tag_ids'])) {
            $tag_links = get_posts([
                'post_type' => Constants::POST_TYPE_AFFILIATE_LINK,
                'post_status' => 'publish',
                'numberposts' => -1,
                'fields' => 'ids',
                'tax_query' => [
                    [
                        'taxonomy' => Constants::TAXONOMY_AFFILIATE_TAG,
                        'field' => 'term_id',
                        'terms' => $filters['tag_ids'],
                        'operator' => 'IN'
                    ]
                ]
            ]);
            
            if (!empty($tag_links)) {
                $link_ids_str = implode(',', array_map('intval', $tag_links));
                $conditions[] = "link_id IN ($link_ids_str)";
                error_log('Tag filter: link_ids = ' . $link_ids_str . ' for tags: ' . implode(',', $filters['tag_ids']));
            } else {
                // No links with selected tags, return no results
                $conditions[] = '1=0';
                error_log('Tag filter: no links found for tags ' . implode(',', $filters['tag_ids']));
            }
        }
        
        if (!empty($filters['date_from'])) {
            $conditions[] = "clicked_at >= '" . esc_sql($filters['date_from']) . "'";
        }
        
        if (!empty($filters['date_to'])) {
            $conditions[] = "clicked_at <= '" . esc_sql($filters['date_to']) . "'";
        }
        
        if (!empty($filters['unique_only'])) {
            $conditions[] = 'is_unique_24h = 1';
        }
        
        $where_clause = implode(' AND ', $conditions);
        error_log('Built WHERE clause: ' . $where_clause);
        
        return $where_clause;
    }
    
    /**
     * AJAX handler for data export
     */
    public function ajax_export_analytics() {
        check_ajax_referer('affiliate_hub_analytics_nonce', 'nonce');
        
        $filters = $this->parse_filters($_POST);
        
        try {
            $data = $this->get_export_data($filters);
            $filename = 'affiliate-hub-analytics-' . DATE('Y-m-d-H-i-s') . '.csv';
            
            header('Content-Type: text/csv');
            header('Content-Disposition: attachment; filename="' . $filename . '"');
            
            $output = fopen('php://output', 'w');
            
            // CSV Headers
            fputcsv($output, [
                'Date',
                'Link ID',
                'Link Title',
                'IP Address',
                'Country',
                'Browser',
                'OS',
                'Device Type',
                'Is Unique 24h',
                'Processing Time (ms)',
                'Geo Provider',
                'Geo Cached'
            ]);
            
            // CSV Data
            foreach ($data as $row) {
                fputcsv($output, [
                    $row['clicked_at'],
                    $row['link_id'],
                    $row['link_title'],
                    $row['ip_address'],
                    $row['country'],
                    $row['browser'],
                    $row['os'],
                    $row['device_type'],
                    $row['is_unique_24h'] ? 'Yes' : 'No',
                    $row['processing_time_ms'],
                    $row['geo_provider'],
                    $row['geo_cached'] ? 'Yes' : 'No'
                ]);
            }
            
            fclose($output);
            exit;
            
        } catch (\Exception $e) {
            wp_send_json_error('Export failed: ' . $e->getMessage());
        }
    }
    
    /**
     * Get data for export
     */
    private function get_export_data($filters) {
        global $wpdb;
        
        $table_clicks = $wpdb->prefix . Constants::TABLE_LINK_CLICKS;
        $where_clause = $this->build_where_clause($filters);
        
        $sql = "SELECT 
                    c.clicked_at,
                    c.link_id,
                    p.post_title as link_title,
                    c.ip_address,
                    c.country,
                    c.browser,
                    c.os,
                    c.device_type,
                    c.is_unique_24h,
                    c.processing_time_ms,
                    c.geo_provider,
                    c.geo_cached
                FROM $table_clicks c
                LEFT JOIN {$wpdb->posts} p ON c.link_id = p.ID
                WHERE $where_clause
                ORDER BY c.clicked_at DESC
                LIMIT 10000";
        
        return $wpdb->get_results($sql, ARRAY_A) ?: [];
    }
    
    /**
     * Synchronize meta fields with actual database table data
     * This fixes discrepancies between total clicks meta field and table count
     */
    public function sync_clicks_data() {
        global $wpdb;
        
        // Check if user has permission
        if (!current_user_can('manage_options')) {
            return [
                'success' => false,
                'message' => 'Insufficient permissions'
            ];
        }
        
        $table_clicks = $wpdb->prefix . Constants::TABLE_LINK_CLICKS;
        
        // Check if table exists
        if ($wpdb->get_var("SHOW TABLES LIKE '$table_clicks'") !== $table_clicks) {
            return [
                'success' => false,
                'message' => 'Clicks table not found'
            ];
        }
        
        // Get actual counts from database
        $results = $wpdb->get_results(
            "SELECT 
                link_id,
                COUNT(*) as actual_total_clicks,
                COUNT(DISTINCT ip_address) as actual_unique_clicks,
                MAX(clicked_at) as last_clicked
             FROM $table_clicks 
             GROUP BY link_id",
            ARRAY_A
        );
        
        $updated_links = 0;
        $discrepancies = [];
        
        foreach ($results as $row) {
            $link_id = intval($row['link_id']);
            $actual_total = intval($row['actual_total_clicks']);
            $actual_unique = intval($row['actual_unique_clicks']);
            $last_clicked = $row['last_clicked'];
            
            // Get current meta values
            $current_total = intval(get_post_meta($link_id, Constants::META_CLICK_COUNT, true));
            $current_last = get_post_meta($link_id, Constants::META_LAST_CLICKED, true);
            
            // Check for discrepancies
            if ($current_total != $actual_total) {
                $discrepancies[] = [
                    'link_id' => $link_id,
                    'current_total' => $current_total,
                    'actual_total' => $actual_total,
                    'difference' => $actual_total - $current_total
                ];
                
                // Update meta field with actual count
                update_post_meta($link_id, Constants::META_CLICK_COUNT, $actual_total);
                
                // Update last clicked if we have newer data
                if ((!$current_last || $current_last === "") || ($last_clicked && strtotime($last_clicked) > strtotime($current_last))) {
                    update_post_meta($link_id, Constants::META_LAST_CLICKED, $last_clicked);
                }
                
                $updated_links++;
            }
        }
        
        return [
            'success' => true,
            'updated_links' => $updated_links,
            'discrepancies' => $discrepancies,
            'message' => "Synchronized $updated_links links. Found " . count($discrepancies) . " discrepancies."
        ];
    }

    /**
     * AJAX handler for paginated recent activity
     */
    public function ajax_get_recent_activity() {
        error_log('🔄 AJAX RECENT ACTIVITY CALLED - START');
        error_log('📋 POST data: ' . json_encode($_POST));
        
        header('Content-Type: application/json');
        
        // Check if nonce exists
        if (!isset($_POST['nonce'])) {
            error_log('❌ No nonce provided');
            echo \json_encode(['success' => false, 'data' => ['message' => 'No security token provided']]);
            \wp_die();
        }
        
        // Verify nonce
        if (!\wp_verify_nonce($_POST['nonce'], 'affiliate_hub_analytics_nonce')) {
            error_log('❌ Nonce verification failed. Provided: ' . $_POST['nonce']);
            echo \json_encode(['success' => false, 'data' => ['message' => 'Security check failed']]);
            \wp_die();
        }
        
        error_log('✅ Nonce verified successfully');
        
        try {
            // Get parameters safely
            $page = \max(1, \intval($_POST['page'] ?? 1));
            $per_page = \max(1, \min(200, \intval($_POST['per_page'] ?? 25)));
            $sort_by = \sanitize_text_field($_POST['sort_by'] ?? 'click_time');
            $sort_order = \sanitize_text_field($_POST['sort_order'] ?? 'DESC');
            
            error_log("📊 Parameters: page=$page, per_page=$per_page, sort=$sort_by $sort_order");
            
            // Get filters
            $filters = [];
            if (!empty($_POST['date_from'])) {
                $filters['date_from'] = \sanitize_text_field($_POST['date_from']);
            }
            if (!empty($_POST['date_to'])) {
                $filters['date_to'] = \sanitize_text_field($_POST['date_to']);
            }
            if (!empty($_POST['link_id'])) {
                $filters['link_id'] = \intval($_POST['link_id']);
            }
            if (!empty($_POST['category_ids']) && is_array($_POST['category_ids'])) {
                $filters['category_ids'] = array_map('intval', $_POST['category_ids']);
            }
            if (!empty($_POST['tag_ids']) && is_array($_POST['tag_ids'])) {
                $filters['tag_ids'] = array_map('intval', $_POST['tag_ids']);
            }
            
            error_log('🔄 Calling get_recent_activity_paginated...');
            
            // Get data using existing method
            $data = $this->get_recent_activity_paginated($filters, $page, $per_page, $sort_by, $sort_order);
            
            error_log('✅ Data retrieved: ' . count($data['items']) . ' items, total: ' . $data['pagination']['total_items']);
            
            echo \json_encode(['success' => true, 'data' => $data]);
            \wp_die();
            
        } catch (\Exception $e) {
            error_log('❌ Recent activity error: ' . $e->getMessage());
            error_log('❌ Stack trace: ' . $e->getTraceAsString());
            
            echo \json_encode([
                'success' => false,
                'data' => [
                    'message' => 'Error: ' . $e->getMessage(),
                    'debug' => [
                        'file' => $e->getFile(),
                        'line' => $e->getLine()
                    ]
                ]
            ]);
            \wp_die();
        }
    }

    /**
     * Get paginated recent activity data
     */
    private function get_recent_activity_paginated($filters, $page = 1, $per_page = 25, $sort_by = 'click_time', $sort_order = 'DESC') {
        global $wpdb;
        
        error_log('🔍 get_recent_activity_paginated called with filters: ' . \json_encode($filters));
        
        $table_name = $wpdb->prefix . 'affiliate_hub_link_clicks';
        
        // Check if table exists
        $table_exists = $wpdb->get_var("SHOW TABLES LIKE '$table_name'");
        if (!$table_exists) {
            error_log("❌ Table $table_name does not exist!");
            throw new Exception("Clicks table does not exist");
        }
        
        error_log("✅ Table $table_name exists");
        
        // Check table structure
        $columns = $wpdb->get_results("DESCRIBE $table_name", ARRAY_A);
        error_log("📋 Table structure: " . \json_encode($columns));
        
        // Simple query first - just count all records
        $total_count = $wpdb->get_var("SELECT COUNT(*) FROM $table_name");
        error_log("📊 Total records in table: $total_count");
        
        if ($total_count == 0) {
            error_log("ℹ️ No records found - returning empty data");
            return [
                'items' => [],
                'pagination' => [
                    'current_page' => 1,
                    'per_page' => $per_page,
                    'total_items' => 0,
                    'total_pages' => 0,
                    'showing_from' => 0,
                    'showing_to' => 0
                ],
                'sort' => [
                    'column' => $sort_by,
                    'order' => $sort_order
                ]
            ];
        }
        
        // If we have records, continue with the full query
        error_log("🔄 Proceeding with full query for $total_count records");
        
        // Build WHERE clause
        $where_conditions = ['1=1'];
        $where_values = [];
        
        // Date filters
        if (!empty($filters['date_from'])) {
            $where_conditions[] = 'clicked_at >= %s';
            $where_values[] = $filters['date_from'];
        }
        
        if (!empty($filters['date_to'])) {
            $where_conditions[] = 'clicked_at <= %s';
            $where_values[] = $filters['date_to'];
        }
        
        // Link filter
        if (!empty($filters['link_id'])) {
            $where_conditions[] = 'link_id = %d';
            $where_values[] = $filters['link_id'];
        }
        
        // Category filter
        if (!empty($filters['category_ids']) && is_array($filters['category_ids'])) {
            $category_placeholders = implode(',', array_fill(0, count($filters['category_ids']), '%d'));
            $where_conditions[] = "link_id IN (
                SELECT DISTINCT object_id 
                FROM {$wpdb->term_relationships} tr 
                JOIN {$wpdb->term_taxonomy} tt ON tr.term_taxonomy_id = tt.term_taxonomy_id 
                WHERE tt.taxonomy = 'affiliate_category' 
                AND tt.term_id IN ($category_placeholders)
            )";
            $where_values = \array_merge($where_values, $filters['category_ids']);
        }
        
        // Tag filter
        if (!empty($filters['tag_ids']) && is_array($filters['tag_ids'])) {
            $tag_placeholders = implode(',', array_fill(0, count($filters['tag_ids']), '%d'));
            $where_conditions[] = "link_id IN (
                SELECT DISTINCT object_id 
                FROM {$wpdb->term_relationships} tr 
                JOIN {$wpdb->term_taxonomy} tt ON tr.term_taxonomy_id = tt.term_taxonomy_id 
                WHERE tt.taxonomy = 'affiliate_tag' 
                AND tt.term_id IN ($tag_placeholders)
            )";
            $where_values = \array_merge($where_values, $filters['tag_ids']);
        }
        
        $where_clause = implode(' AND ', $where_conditions);
        
        error_log("🔍 Where conditions: " . \json_encode($where_conditions));
        error_log("🔍 Where clause: " . $where_clause);
        error_log("🔍 Where values: " . \json_encode($where_values));
        
        // Validate sort column
        $allowed_sort_columns = [
            'click_time' => 'clicked_at',
            'link_title' => 'link_title',
            'referrer' => 'referer',
            'ip_address' => 'ip_address',
            'browser' => 'browser',
            'os' => 'os',
            'device' => 'device_type',
            'country' => 'country'
        ];
        
        $sort_column = $allowed_sort_columns[$sort_by] ?? 'clicked_at';
        $sort_order = \in_array(\strtoupper($sort_order), ['ASC', 'DESC']) ? \strtoupper($sort_order) : 'DESC';
        
        // Get total count
        $count_query = "
            SELECT COUNT(*) 
            FROM $table_name c
            LEFT JOIN {$wpdb->posts} p ON c.link_id = p.ID
            WHERE $where_clause
        ";
        
        if (!empty($where_values)) {
            $count_query = $wpdb->prepare($count_query, $where_values);
        }
        
        error_log("🔍 Count query: " . $count_query);
        
        $total_items = $wpdb->get_var($count_query);
        
        // Check for database errors
        if ($wpdb->last_error) {
            error_log("❌ Count query error: " . $wpdb->last_error);
            throw new Exception("Count query failed: " . $wpdb->last_error);
        }
        
        error_log("📊 Total items found: " . $total_items);
        
        // Calculate pagination
        $offset = ($page - 1) * $per_page;
        $total_pages = \ceil($total_items / $per_page);
        
        // Get data
        $data_query = "
            SELECT 
                c.clicked_at,
                c.link_id,
                COALESCE(p.post_title, 'Unknown') as link_title,
                c.referer as referrer,
                c.ip_address,
                c.browser,
                c.os,
                c.device_type,
                c.country,
                c.is_unique_24h,
                c.processing_time_ms
            FROM $table_name c
            LEFT JOIN {$wpdb->posts} p ON c.link_id = p.ID
            WHERE $where_clause
            ORDER BY $sort_column $sort_order
            LIMIT %d OFFSET %d
        ";
        
        $query_values = \array_merge($where_values, [$per_page, $offset]);
        $prepared_query = $wpdb->prepare($data_query, $query_values);
        
        error_log("🔍 SQL Query: " . $prepared_query);
        error_log("🔍 Query values: " . \json_encode($query_values));
        
        $items = $wpdb->get_results($prepared_query, ARRAY_A);
        
        // Check for database errors
        if ($wpdb->last_error) {
            error_log("❌ Database error: " . $wpdb->last_error);
            throw new Exception("Database query failed: " . $wpdb->last_error);
        }
        
        error_log("📊 Retrieved " . count($items) . " items from database");
        
        // Process items for display
        foreach ($items as &$item) {
            // Format date
            $item['formatted_date'] = date('Y-m-d H:i:s', strtotime($item['clicked_at']));
            
            // Process referrer
            if ((!$item['referrer'] || $item['referrer'] === "") || $item['referrer'] === 'Direct') {
                $item['referrer_display'] = 'Direct';
                $item['referrer_url'] = '';
            } else {
                $parsed_url = parse_url($item['referrer']);
                $item['referrer_display'] = $parsed_url['host'] ?? $item['referrer'];
                $item['referrer_url'] = $item['referrer'];
            }
            
            // Device icon
            $device_icons = [
                'Desktop' => '🖥️',
                'Mobile' => '📱',
                'Tablet' => '📱',
                'Bot' => '🤖'
            ];
            $item['device_icon'] = $device_icons[$item['device_type']] ?? '❓';
            
            // Link edit URL
            $item['edit_url'] = \admin_url("post.php?post={$item['link_id']}&action=edit");
        }
        
        $result = [
            'items' => $items,
            'pagination' => [
                'current_page' => $page,
                'per_page' => $per_page,
                'total_items' => (int)$total_items,
                'total_pages' => $total_pages,
                'showing_from' => \min($offset + 1, $total_items),
                'showing_to' => \min($offset + $per_page, $total_items)
            ],
            'sort' => [
                'column' => $sort_by,
                'order' => $sort_order
            ]
        ];
        
        error_log("✅ Returning " . count($items) . " items out of $total_items total");
        error_log("📄 Pagination: page $page of $total_pages, showing " . $result['pagination']['showing_from'] . "-" . $result['pagination']['showing_to']);
        
        return $result;
    }

    /**
     * Test endpoint to check table status
     */
    public function ajax_test_table() {
        // Ensure we don't output anything before JSON
        ob_clean();
        
        // Set proper headers
        header('Content-Type: application/json');
        
        try {
            global $wpdb;
            
            error_log('🧪 TEST TABLE ENDPOINT CALLED');
            
            $table_name = $wpdb->prefix . 'affiliate_hub_link_clicks';
            $table_exists = $wpdb->get_var("SHOW TABLES LIKE '$table_name'");
            
            $response = [
                'table_name' => $table_name,
                'table_exists' => ($table_exists && count($table_exists) > 0),
                'record_count' => 0,
                'sample_data' => [],
                'debug_info' => [
                    'ajax_url' => admin_url('admin-ajax.php'),
                    'doing_ajax' => wp_doing_ajax(),
                    'current_action' => $_POST['action'] ?? 'none'
                ]
            ];
            
            if ($table_exists) {
                $response['record_count'] = $wpdb->get_var("SELECT COUNT(*) FROM $table_name");
                
                if ($response['record_count'] == 0) {
                    // Add sample data if none exists
                    $wpdb->insert($table_name, [
                        'link_id' => 1,
                        'ip_address' => '127.0.0.1',
                        'user_agent' => 'Mozilla/5.0 Test Browser',
                        'referrer' => 'https://example.com',
                        'country' => 'Poland',
                        'country_code' => 'PL',
                        'browser' => 'Chrome',
                        'os' => 'Windows',
                        'device_type' => 'Desktop',
                        'is_unique' => 1,
                        'is_unique_24h' => 1,
                        'session_fingerprint' => 'test_session_' . time(),
                        'clicked_at' => current_time('mysql')
                    ]);
                    
                    $wpdb->insert($table_name, [
                        'link_id' => 1,
                        'ip_address' => '192.168.1.100',
                        'user_agent' => 'Mozilla/5.0 (iPhone; CPU iPhone OS 15_0 like Mac OS X)',
                        'referrer' => 'Direct',
                        'country' => 'United States',
                        'country_code' => 'US',
                        'browser' => 'Safari',
                        'os' => 'iOS',
                        'device_type' => 'Mobile',
                        'is_unique' => 1,
                        'is_unique_24h' => 1,
                        'session_fingerprint' => 'test_session_mobile_' . time(),
                        'clicked_at' => current_time('mysql')
                    ]);
                    
                    $response['record_count'] = 2;
                    $response['message'] = 'Added sample data';
                }
                
                $response['sample_data'] = $wpdb->get_results("SELECT * FROM $table_name LIMIT 3", ARRAY_A);
            }
            
            error_log('🧪 Test table response: ' . \json_encode($response));
            wp_send_json_success($response);
            
        } catch (Exception $e) {
            error_log('🧪 Test table error: ' . $e->getMessage());
            wp_send_json_error([
                'message' => 'Test failed: ' . $e->getMessage(),
                'debug' => [
                    'file' => $e->getFile(),
                    'line' => $e->getLine()
                ]
            ]);
        }
        
        // Ensure we exit properly
        wp_die();
    }

    /**
     * Debug endpoint that ALWAYS works
     */
    public function ajax_debug_endpoint() {
        // Ensure clean output
        ob_clean();
        header('Content-Type: application/json');
        
        try {
            error_log('🔧 DEBUG ENDPOINT CALLED');
            
            // NO NONCE CHECK - for debugging only
            $response = [
                'message' => 'Debug endpoint working!',
                'post_data' => $_POST,
                'endpoint_registered' => true,
                'timestamp' => current_time('mysql'),
                'is_ajax' => wp_doing_ajax(),
                'action_hook_exists' => has_action('wp_ajax_affiliate_hub_recent_activity'),
                'test_table_hook_exists' => has_action('wp_ajax_affiliate_hub_test_table'),
                'ajax_url' => admin_url('admin-ajax.php'),
                'current_user' => get_current_user_id(),
                'debug_info' => [
                    'REQUEST_METHOD' => $_SERVER['REQUEST_METHOD'] ?? 'none',
                    'HTTP_X_REQUESTED_WITH' => $_SERVER['HTTP_X_REQUESTED_WITH'] ?? 'none',
                    'CONTENT_TYPE' => $_SERVER['CONTENT_TYPE'] ?? 'none'
                ]
            ];
            
            error_log('🔧 Debug response: ' . \json_encode($response));
            wp_send_json_success($response);
            
        } catch (Exception $e) {
            error_log('🔧 Debug endpoint error: ' . $e->getMessage());
            wp_send_json_error([
                'message' => 'Debug endpoint failed: ' . $e->getMessage(),
                'trace' => $e->getTrace()
            ]);
        }
        
        wp_die();
    }
    
    /**
     * Simple test endpoint for debugging AJAX issues
     */
    public function ajax_test_simple() {
        error_log('🧪 SIMPLE TEST ENDPOINT CALLED');
        
        header('Content-Type: application/json');
        
        try {
            // Very basic response without nonce check
            $response = [
                'success' => true,
                'data' => [
                    'message' => 'Simple test endpoint working!',
                    'timestamp' => current_time('mysql'),
                    'post_data' => $_POST
                ]
            ];
            
            error_log('🧪 Simple test response: ' . json_encode($response));
            echo json_encode($response);
            
        } catch (Exception $e) {
            error_log('🧪 Simple test error: ' . $e->getMessage());
            echo json_encode([
                'success' => false,
                'data' => ['message' => 'Simple test error: ' . $e->getMessage()]
            ]);
        }
        
        wp_die();
    }
}











