<?php
/**
 * Dashboard Widget for Affiliate Hub
 *
 * @package AffiliateHub
 * @subpackage Admin
 */

namespace AffiliateHub\Admin;

use AffiliateHub\Core\Constants;
use AffiliateHub\Models\AffiliateLink;

/**
 * Dashboard Widget class
 */
class DashboardWidget {
    
    /**
     * Initialize the dashboard widget
     */
    public function __construct() {
        add_action('wp_dashboard_setup', array($this, 'add_dashboard_widget'));
    }
    
    /**
     * Add dashboard widget
     */
    public function add_dashboard_widget() {
        // Check if user can view affiliate stats
        if (!current_user_can(Constants::CAP_VIEW_AFFILIATE_STATS)) {
            return;
        }
        
        wp_add_dashboard_widget(
            'affiliate_hub_stats_widget',
            __('Affiliate Hub - Quick Stats', 'affiliate-hub'),
            array($this, 'dashboard_widget_content'),
            array($this, 'dashboard_widget_control')
        );
    }
    
    /**
     * Dashboard widget content
     */
    public function dashboard_widget_content() {
        // Get basic stats
        $stats = $this->get_quick_stats();
        
        ?>
        <div class="affiliate-hub-dashboard-widget">
            <div class="affiliate-hub-widget-stats">
                <div class="stat-box">
                    <div class="stat-number"><?php echo esc_html($stats['total_links']); ?></div>
                    <div class="stat-label"><?php esc_html_e('Total Links', 'affiliate-hub'); ?></div>
                </div>
                <div class="stat-box">
                    <div class="stat-number"><?php echo esc_html($stats['total_clicks']); ?></div>
                    <div class="stat-label"><?php esc_html_e('Total Clicks', 'affiliate-hub'); ?></div>
                </div>
                <div class="stat-box">
                    <div class="stat-number"><?php echo esc_html($stats['clicks_today']); ?></div>
                    <div class="stat-label"><?php esc_html_e('Clicks Today', 'affiliate-hub'); ?></div>
                </div>
                <div class="stat-box">
                    <div class="stat-number"><?php echo esc_html($stats['clicks_this_month']); ?></div>
                    <div class="stat-label"><?php esc_html_e('Clicks This Month', 'affiliate-hub'); ?></div>
                </div>
            </div>
            
            <?php if (!empty($stats['top_links'])): ?>
            <div class="affiliate-hub-widget-top-links">
                <h4><?php esc_html_e('Top Performing Links', 'affiliate-hub'); ?></h4>
                <ul>
                    <?php foreach ($stats['top_links'] as $link): ?>
                    <li>
                        <a href="<?php echo esc_url(admin_url('post.php?post=' . $link['id'] . '&action=edit')); ?>">
                            <?php echo esc_html($link['title']); ?>
                        </a>
                        <span class="clicks-count">(<?php echo esc_html($link['clicks']); ?> <?php esc_html_e('clicks', 'affiliate-hub'); ?>)</span>
                    </li>
                    <?php endforeach; ?>
                </ul>
            </div>
            <?php endif; ?>
            
            <div class="affiliate-hub-widget-actions">
                <a href="<?php echo esc_url(admin_url('post-new.php?post_type=' . Constants::POST_TYPE_AFFILIATE_LINK)); ?>" class="button button-primary">
                    <?php esc_html_e('Add New Link', 'affiliate-hub'); ?>
                </a>
                <a href="<?php echo esc_url(admin_url('edit.php?post_type=' . Constants::POST_TYPE_AFFILIATE_LINK)); ?>" class="button">
                    <?php esc_html_e('Manage Links', 'affiliate-hub'); ?>
                </a>
                <?php if (get_option(Constants::OPTION_ENABLE_STATS, true)): ?>
                <a href="<?php echo esc_url(admin_url('admin.php?page=affiliate-hub-stats')); ?>" class="button">
                    <?php esc_html_e('View Stats', 'affiliate-hub'); ?>
                </a>
                <?php endif; ?>
            </div>
        </div>
        
        <style>
        .affiliate-hub-dashboard-widget {
            padding: 0;
        }
        
        .affiliate-hub-widget-stats {
            display: flex;
            flex-wrap: wrap;
            gap: 10px;
            margin-bottom: 15px;
        }
        
        .stat-box {
            flex: 1;
            min-width: 100px;
            text-align: center;
            padding: 10px;
            background: #f9f9f9;
            border-radius: 4px;
        }
        
        .stat-number {
            font-size: 24px;
            font-weight: bold;
            color: #0073aa;
            line-height: 1;
        }
        
        .stat-label {
            font-size: 11px;
            color: #666;
            text-transform: uppercase;
            margin-top: 5px;
        }
        
        .affiliate-hub-widget-top-links h4 {
            margin: 0 0 10px 0;
            font-size: 13px;
        }
        
        .affiliate-hub-widget-top-links ul {
            margin: 0;
            list-style: none;
        }
        
        .affiliate-hub-widget-top-links li {
            padding: 5px 0;
            border-bottom: 1px solid #eee;
            display: flex;
            justify-content: space-between;
            align-items: center;
        }
        
        .affiliate-hub-widget-top-links li:last-child {
            border-bottom: none;
        }
        
        .affiliate-hub-widget-top-links a {
            text-decoration: none;
            color: #0073aa;
        }
        
        .affiliate-hub-widget-top-links a:hover {
            text-decoration: underline;
        }
        
        .clicks-count {
            color: #666;
            font-size: 12px;
        }
        
        .affiliate-hub-widget-actions {
            margin-top: 15px;
            padding-top: 10px;
            border-top: 1px solid #eee;
            text-align: center;
        }
        
        .affiliate-hub-widget-actions .button {
            margin: 0 3px;
        }
        </style>
        <?php
    }
    
    /**
     * Dashboard widget control (configuration)
     */
    public function dashboard_widget_control() {
        // Widget configuration options could go here
        // For now, we'll keep it simple
        echo '<p>' . esc_html__('Configure widget settings here if needed.', 'affiliate-hub') . '</p>';
    }
    
    /**
     * Get quick stats for dashboard
     *
     * @return array
     */
    private function get_quick_stats() {
        global $wpdb;

        // Short cache to avoid repeated queries on dashboard loads
        $cache_key = 'affiliate_hub_quick_stats';
        $cached = \get_transient($cache_key);
        if ($cached !== false && is_array($cached)) {
            return $cached;
        }

        // Total links
        $total_links = \wp_count_posts(Constants::POST_TYPE_AFFILIATE_LINK);
        $total_links_count = isset($total_links->publish) ? (int) $total_links->publish : 0;

    // Total clicks
        $table_clicks = $wpdb->prefix . Constants::TABLE_LINK_CLICKS;
        $total_clicks = 0;
        $clicks_today = 0;
        $clicks_this_month = 0;

        // Check if clicks table exists (prepared, safe)
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching -- Safe table existence check; overall method is transient-cached
    // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Table name cannot be parameterized; value is built from $wpdb->prefix + constant
    $table_exists = ($wpdb->get_var($wpdb->prepare("SHOW TABLES LIKE %s", $table_clicks)) === $table_clicks);
        if ($table_exists) {
            // Count ALL clicks
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching -- Aggregates from custom table; function-level transient caching applied
            // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Table name interpolation; query has no user input
            $total_clicks = (int) $wpdb->get_var("SELECT COUNT(*) FROM $table_clicks"); // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Table name interpolation only; no user input

            // Clicks today (sargable range, uses WP-local time)
            $today = \current_time('Y-m-d');
            $today_start = $today . ' 00:00:00';
            $today_end = $today . ' 23:59:59';
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching -- Prepared aggregate; function-level transient caching applied
            // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Table name interpolation only; user inputs are parameterized below
            $clicks_today = (int) $wpdb->get_var($wpdb->prepare(
                "SELECT COUNT(*) FROM $table_clicks WHERE clicked_at >= %s AND clicked_at <= %s",
                $today_start,
                $today_end
            )); // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Table name interpolation only; parameters prepared

            // Clicks this month (start of month to start of next month)
            $month_start = \current_time('Y-m-01 00:00:00');
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching -- Prepared aggregate; function-level transient caching applied
            // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Table name interpolation only; user inputs are parameterized below
            $clicks_this_month = (int) $wpdb->get_var($wpdb->prepare(
                "SELECT COUNT(*) FROM $table_clicks WHERE clicked_at >= %s AND clicked_at < DATE_ADD(%s, INTERVAL 1 MONTH)",
                $month_start,
                $month_start
            )); // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Table name interpolation only; parameters prepared

            // Optional debug logs guarded by WP_DEBUG
            if (defined('WP_DEBUG') && constant('WP_DEBUG')) {
                // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- Debug-only logs under WP_DEBUG
                error_log('AffiliateHub Dashboard - Total clicks: ' . $total_clicks);
                // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- Debug-only logs under WP_DEBUG
                error_log('AffiliateHub Dashboard - Clicks today: ' . $clicks_today);
                // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- Debug-only logs under WP_DEBUG
                error_log('AffiliateHub Dashboard - Clicks this month: ' . $clicks_this_month);
            }
        }

        // Top performing links
        $top_links = $this->get_top_performing_links(5);

    $result = array(
            'total_links'      => $total_links_count,
            'total_clicks'     => $total_clicks,
            'clicks_today'     => $clicks_today,
            'clicks_this_month'=> $clicks_this_month,
            'top_links'        => $top_links,
        );
        
    $ttl = (\defined('MINUTE_IN_SECONDS') ? \constant('MINUTE_IN_SECONDS') : 60);
    \set_transient($cache_key, $result, 1 * $ttl);

        return $result;
    }
    
    /**
     * Get top performing links
     *
     * @param int $limit Number of links to return
     * @return array
     */
    private function get_top_performing_links($limit = 5) {
        global $wpdb;

        $limit = max(1, intval($limit));
        $cache_key = 'affiliate_hub_top_links_' . $limit;
    $cached = \get_transient($cache_key);
        if ($cached !== false && is_array($cached)) {
            return $cached;
        }

    $top_links = array();
    $table_clicks = $wpdb->prefix . Constants::TABLE_LINK_CLICKS;

    // Prefer the indexed clicks table for performance, if it exists (cache existence for a short period)
    $exists_cache_key = 'affiliate_hub_tbl_exists_' . md5($table_clicks);
    $exists_cached = \wp_cache_get($exists_cache_key, 'affiliate_hub');
    if (false === $exists_cached) {
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching -- Single schema existence check; result cached via wp_cache for 5 minutes
        $exists_cached = ($wpdb->get_var($wpdb->prepare("SHOW TABLES LIKE %s", $table_clicks)) === $table_clicks) ? 'yes' : 'no';
        \wp_cache_set($exists_cache_key, $exists_cached, 'affiliate_hub', 300);
    }
    $table_exists = ($exists_cached === 'yes');

        if ($table_exists) {
            // Aggregate by link_id and join posts to ensure published affiliate links
            $sql = "
                SELECT p.ID AS id, p.post_title AS title, COUNT(c.id) AS clicks
                FROM {$table_clicks} c
                INNER JOIN {$wpdb->posts} p ON p.ID = c.link_id
                WHERE p.post_type = %s AND p.post_status = 'publish'
                GROUP BY p.ID
                ORDER BY clicks DESC
                LIMIT %d
            ";

            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching -- Prepared aggregate; method-level transient caching applied
            // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared -- The $sql uses placeholders and is prepared below
            $rows = $wpdb->get_results($wpdb->prepare($sql, Constants::POST_TYPE_AFFILIATE_LINK, $limit), 'ARRAY_A');
            if (!empty($rows)) {
                foreach ($rows as $row) {
                    $top_links[] = array(
                        'id' => intval($row['id']),
                        'title' => (string) $row['title'],
                        'clicks' => intval($row['clicks'])
                    );
                }
            }
        }

        // Fallback to meta counts (slower) if clicks table missing or empty
        if (empty($top_links)) {
            // Fallback uses postmeta ordering; acceptable only when analytics table is missing.
            // phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_meta_key,WordPress.DB.SlowDBQuery.slow_db_query_meta_query
            // phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_meta_key
            // phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_meta_query
            $args = array(
                'post_type'      => Constants::POST_TYPE_AFFILIATE_LINK,
                'post_status'    => 'publish',
                'posts_per_page' => $limit,
                'meta_key'       => Constants::META_CLICK_COUNT,
                'orderby'        => 'meta_value_num',
                'order'          => 'DESC',
                'meta_query'     => array(
                    array(
                        'key'     => Constants::META_CLICK_COUNT,
                        'value'   => '0',
                        'compare' => '>'
                    )
                )
            );

            $posts = \get_posts($args);
            foreach ($posts as $post) {
                $clicks = \get_post_meta($post->ID, Constants::META_CLICK_COUNT, true);
                $top_links[] = array(
                    'id' => $post->ID,
                    'title' => $post->post_title,
                    'clicks' => $clicks ? intval($clicks) : 0
                );
            }
        }

        // Cache for a short period to reduce load on the dashboard
    $ttl = (\defined('MINUTE_IN_SECONDS') ? \constant('MINUTE_IN_SECONDS') : 60);
    \set_transient($cache_key, $top_links, 5 * $ttl);

        return $top_links;
    }
}
