<?php
/**
 * GeoLocation Admin Page
 * 
 * Dashboard and management for geolocation module
 *
 * @package AffiliateHub
 * @subpackage Admin
 */

namespace AffiliateHub\Admin;

use AffiliateHub\Core\Constants;
use AffiliateHub\Modules\GeoLocation;

class GeoLocationAdmin {
    
    private $geolocation_module;
    
    public function __construct() {
        add_action('admin_enqueue_scripts', array($this, 'enqueue_admin_scripts'));
        add_action('wp_ajax_affiliate_hub_geo_stats', array($this, 'ajax_get_geo_stats'));
        add_action('wp_ajax_affiliate_hub_bulk_enrich_geo', array($this, 'ajax_bulk_enrich_geo'));
        add_action('wp_ajax_affiliate_hub_maxmind_update', array($this, 'ajax_maxmind_update'));
        add_action('wp_ajax_affiliate_hub_install_native_geoip2', array($this, 'ajax_install_native_geoip2'));
        add_action('wp_ajax_affiliate_hub_uninstall_native_geoip2', array($this, 'ajax_uninstall_native_geoip2'));
        add_action('wp_ajax_affiliate_hub_run_benchmark', array($this, 'ajax_run_benchmark'));
        add_action('wp_ajax_affiliate_hub_reset_performance_metrics', array($this, 'ajax_reset_performance_metrics'));
        
        // Fix submenu highlighting - use earlier hook
        add_action('current_screen', array($this, 'fix_submenu_highlighting'));
        add_filter('parent_file', array($this, 'fix_parent_file'));
        add_filter('submenu_file', array($this, 'fix_submenu_file'));
        
        // Get GeoLocation module instance from ModuleManager
        $plugin = \AffiliateHub\Core\Plugin::get_instance();
        if ($plugin->modules) {
            $this->geolocation_module = $plugin->modules->get_module('GeoLocation');
        }
        
        // Fallback if module manager is not available
        if (!$this->geolocation_module) {
            $this->geolocation_module = new \AffiliateHub\Modules\GeoLocation();
        }
    }
    
    /**
     * Fix submenu highlighting for GeoLocation page
     */
    public function fix_submenu_highlighting() {
        global $submenu_file, $parent_file;
        
        // Check if we're on the geolocation page
        if (isset($_GET['post_type']) && $_GET['post_type'] === 'affiliate_link' && 
            isset($_GET['page']) && $_GET['page'] === 'affiliate-hub-geolocation') {
            
            $parent_file = 'edit.php?post_type=affiliate_link';
            $submenu_file = 'affiliate-hub-geolocation';
        }
    }
    
    /**
     * Fix parent file for menu highlighting
     */
    public function fix_parent_file($parent_file) {
        if (isset($_GET['post_type']) && $_GET['post_type'] === 'affiliate_link' && 
            isset($_GET['page']) && $_GET['page'] === 'affiliate-hub-geolocation') {
            return 'edit.php?post_type=affiliate_link';
        }
        return $parent_file;
    }
    
    /**
     * Fix submenu file for menu highlighting
     */
    public function fix_submenu_file($submenu_file) {
        if (isset($_GET['post_type']) && $_GET['post_type'] === 'affiliate_link' && 
            isset($_GET['page']) && $_GET['page'] === 'affiliate-hub-geolocation') {
            return 'affiliate-hub-geolocation';
        }
        return $submenu_file;
    }
    
    public function enqueue_admin_scripts($hook) {
        error_log('enqueue_admin_scripts called with hook: ' . $hook);
        error_log('Current GET parameters: ' . print_r($_GET, true));
        
        // Check if we're on the geolocation page
        $is_geolocation_page = false;
        
        // Method 1: Check hook
        if ($hook === 'affiliate_link_page_affiliate-hub-geolocation') {
            $is_geolocation_page = true;
        }
        
        // Method 2: Check GET parameters
        if (isset($_GET['post_type']) && $_GET['post_type'] === 'affiliate_link' && 
            isset($_GET['page']) && $_GET['page'] === 'affiliate-hub-geolocation') {
            $is_geolocation_page = true;
        }
        
        // Method 3: Check current screen
        if (function_exists('get_current_screen')) {
            $screen = get_current_screen();
            if ($screen && $screen->id === 'affiliate_link_page_affiliate-hub-geolocation') {
                $is_geolocation_page = true;
            }
        }
        
        error_log('Is geolocation page: ' . ($is_geolocation_page ? 'YES' : 'NO'));
        
        if (!$is_geolocation_page) {
            error_log('Skipping script enqueue - not geolocation page');
            return;
        }
        
        error_log('Loading GeoLocation admin scripts');
        
        wp_enqueue_script('chart-js', 'https://cdn.jsdelivr.net/npm/chart.js', array(), '3.9.1', true);
        wp_enqueue_script(
            'affiliate-hub-geo-admin',
            AFFILIATE_HUB_URL . 'assets/js/geo-admin.js',
            array('jquery', 'chart-js'),
            AFFILIATE_HUB_VERSION,
            true
        );
        
        $localize_data = array(
            'ajax_url' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('affiliate_hub_admin_nonce')
        );
        
        error_log('Localizing script with data: ' . print_r($localize_data, true));
        
        wp_localize_script('affiliate-hub-geo-admin', 'affiliateHubGeo', $localize_data);
    }
    
    public function render_admin_page() {
        $stats = $this->geolocation_module->get_statistics();
        ?>
        <div class="wrap">
            <h1><span class="dashicons dashicons-location-alt"></span> <?php _e('GeoLocation Analytics', 'affiliate-hub'); ?></h1>
            
            <div class="affiliate-hub-geo-dashboard">
                <!-- Overview Cards -->
                <div class="geo-stats-cards">
                    <div class="geo-card">
                        <div class="geo-card-header">
                            <h3><?php _e('Cache Performance', 'affiliate-hub'); ?></h3>
                        </div>
                        <div class="geo-card-body">
                            <div class="geo-stat-number"><?php echo number_format($stats['total_cached_ips'] ?? 0); ?></div>
                            <div class="geo-stat-label"><?php _e('Cached IP Addresses', 'affiliate-hub'); ?></div>
                            <div class="geo-stat-meta">
                                <?php printf(__('%d expired entries', 'affiliate-hub'), $stats['expired_cache_entries'] ?? 0); ?>
                            </div>
                        </div>
                    </div>
                    
                    <div class="geo-card">
                        <div class="geo-card-header">
                            <h3><?php _e('Data Coverage', 'affiliate-hub'); ?></h3>
                        </div>
                        <div class="geo-card-body">
                            <div class="geo-stat-number"><?php echo number_format($stats['clicks_with_geo'] ?? 0); ?></div>
                            <div class="geo-stat-label"><?php _e('Clicks with Geo Data', 'affiliate-hub'); ?></div>
                            <div class="geo-stat-meta">
                                <?php 
                                $without_geo = $stats['clicks_without_geo'] ?? 0;
                                if ($without_geo > 0) {
                                    printf(__('%d clicks need enrichment', 'affiliate-hub'), $without_geo);
                                } else {
                                    _e('All clicks have geo data', 'affiliate-hub');
                                }
                                ?>
                            </div>
                        </div>
                    </div>
                    
                    <div class="geo-card">
                        <div class="geo-card-header">
                            <h3><?php _e('API Providers', 'affiliate-hub'); ?></h3>
                        </div>
                        <div class="geo-card-body">
                            <div class="provider-stats">
                                <?php foreach ($stats['providers'] ?? array() as $provider_key => $provider_data): ?>
                                    <div class="provider-stat">
                                        <strong><?php echo esc_html($provider_data['name'] ?? $provider_key); ?></strong><br>
                                        <small>
                                            <?php 
                                            echo number_format($provider_data['successful_requests'] ?? 0) . ' / ' . 
                                                 number_format($provider_data['total_requests'] ?? 0) . ' requests';
                                            ?>
                                        </small>
                                    </div>
                                <?php endforeach; ?>
                            </div>
                        </div>
                    </div>
                </div>
                
                <!-- Geographic Distribution Chart -->
                <div class="geo-charts-section">
                    <div class="geo-card full-width">
                        <div class="geo-card-header">
                            <h3><?php _e('Geographic Distribution', 'affiliate-hub'); ?></h3>
                        </div>
                        <div class="geo-card-body">
                            <canvas id="geo-distribution-chart" width="400" height="200"></canvas>
                        </div>
                    </div>
                </div>
                
                <!-- MaxMind Database Status -->
                <div class="geo-maxmind-section">
                    <div class="geo-card full-width">
                        <div class="geo-card-header">
                            <h3><?php _e('MaxMind GeoLite2 Database (Primary)', 'affiliate-hub'); ?></h3>
                        </div>
                        <div class="geo-card-body">
                            <?php $maxmind_status = $this->geolocation_module->get_maxmind_status(); ?>
                            
                            <div class="maxmind-status">
                                <div class="status-indicator <?php echo $maxmind_status['available'] ? 'status-good' : 'status-error'; ?>">
                                    <span class="status-dot"></span>
                                    <?php echo esc_html($maxmind_status['message']); ?>
                                </div>
                                
                                <?php if ($maxmind_status['available']): ?>
                                    <div class="maxmind-info">
                                        <p><strong><?php _e('Database Size:', 'affiliate-hub'); ?></strong> <?php echo esc_html($maxmind_status['file_size']); ?></p>
                                        <p><strong><?php _e('Last Updated:', 'affiliate-hub'); ?></strong> <?php echo date_i18n(get_option('date_format') . ' ' . get_option('time_format'), $maxmind_status['last_update']); ?></p>
                                        <?php if ($maxmind_status['needs_update']): ?>
                                            <p class="maxmind-warning"><?php _e('⚠️ Database is older than 2 weeks. Consider updating.', 'affiliate-hub'); ?></p>
                                        <?php endif; ?>
                                    </div>
                                    
                                    <button id="maxmind-update-btn" class="button">
                                        <?php _e('Update Database', 'affiliate-hub'); ?>
                                    </button>
                                <?php else: ?>
                                    <div class="maxmind-setup">
                                        <p><?php _e('MaxMind GeoLite2 provides the most accurate and fastest geolocation data.', 'affiliate-hub'); ?></p>
                                        <p><?php _e('To get started:', 'affiliate-hub'); ?></p>
                                        <ol>
                                            <li><?php _e('Register for a free MaxMind account:', 'affiliate-hub'); ?> <a href="https://www.maxmind.com/en/geolite2/signup" target="_blank">maxmind.com/signup</a></li>
                                            <li><?php _e('Generate a license key in your account', 'affiliate-hub'); ?></li>
                                            <li><?php _e('Enter the license key in GeoLocation Settings', 'affiliate-hub'); ?></li>
                                            <li><?php _e('Database will be automatically downloaded (~70MB)', 'affiliate-hub'); ?></li>
                                        </ol>
                                        
                                        <div class="notice notice-info inline" style="margin: 15px 0;">
                                            <p><strong><?php _e('Performance Note:', 'affiliate-hub'); ?></strong></p>
                                            <p><?php _e('Currently using API fallback mode (works but slower). For best performance, install composer and geoip2 package:', 'affiliate-hub'); ?></p>
                                            <code style="display: block; background: #f8f9fa; padding: 10px; margin: 8px 0; border-left: 4px solid #0073aa;">
                                                cd <?php echo esc_html(AFFILIATE_HUB_PATH); ?><br>
                                                composer require geoip2/geoip2:~2.0
                                            </code>
                                        </div>
                                        
                                        <?php if (!$maxmind_status['geoip2_available']): ?>
                                            <div class="notice notice-warning">
                                                <p><?php _e('⚠️ GeoIP2 PHP library not detected. MaxMind functionality may be limited.', 'affiliate-hub'); ?></p>
                                            </div>
                                        <?php endif; ?>
                                        
                                        <a href="<?php echo admin_url('admin.php?page=affiliate-hub-settings#geolocation'); ?>" class="button button-primary">
                                            <?php _e('Configure MaxMind Settings', 'affiliate-hub'); ?>
                                        </a>
                                    </div>
                                <?php endif; ?>
                            </div>
                        </div>
                    </div>
                </div>
                
                <!-- Auto-installer & Performance Dashboard -->
                <div class="geo-installer-section">
                    <div class="geo-card full-width">
                        <div class="geo-card-header">
                            <h3><?php _e('Performance & Auto-installer', 'affiliate-hub'); ?></h3>
                        </div>
                        <div class="geo-card-body">
                            <?php $auto_installer = $this->geolocation_module->get_auto_installer(); ?>
                            <?php $installer_status = $auto_installer->get_status(); ?>
                            <?php $performance_monitor = $this->geolocation_module->get_performance_monitor(); ?>
                            <?php $performance = $performance_monitor->get_performance_summary(); ?>
                            
                            <div class="installer-status">
                                <div class="status-row">
                                    <div class="status-column">
                                        <h4><?php _e('Current Mode', 'affiliate-hub'); ?></h4>
                                        <div class="mode-indicator <?php echo $installer_status['current_mode']; ?>-mode">
                                            <?php if ($installer_status['current_mode'] === 'native'): ?>
                                                <span class="dashicons dashicons-yes-alt"></span>
                                                <?php _e('Native GeoIP2 (Fast)', 'affiliate-hub'); ?>
                                            <?php else: ?>
                                                <span class="dashicons dashicons-clock"></span>
                                                <?php _e('Fallback Mode (Slower)', 'affiliate-hub'); ?>
                                            <?php endif; ?>
                                        </div>
                                    </div>
                                    
                                    <div class="status-column">
                                        <h4><?php _e('Performance', 'affiliate-hub'); ?></h4>
                                        <?php if ($performance['performance_improvement'] > 0): ?>
                                            <div class="performance-improvement">
                                                <strong><?php echo esc_html($performance['performance_improvement']); ?>%</strong>
                                                <?php _e('faster with native', 'affiliate-hub'); ?>
                                            </div>
                                        <?php endif; ?>
                                        
                                        <div class="response-times">
                                            <?php if ($performance['fallback_avg_time'] > 0): ?>
                                                <div><?php _e('Fallback:', 'affiliate-hub'); ?> <strong><?php echo number_format($performance['fallback_avg_time'], 1); ?>ms</strong></div>
                                            <?php endif; ?>
                                            <?php if ($performance['native_avg_time'] > 0): ?>
                                                <div><?php _e('Native:', 'affiliate-hub'); ?> <strong><?php echo number_format($performance['native_avg_time'], 1); ?>ms</strong></div>
                                            <?php endif; ?>
                                        </div>
                                    </div>
                                    
                                    <div class="status-column">
                                        <h4><?php _e('Actions', 'affiliate-hub'); ?></h4>
                                        <?php if (!$installer_status['native_installed']): ?>
                                            <button id="install-native-btn" class="button button-primary">
                                                <span class="dashicons dashicons-download"></span>
                                                <?php _e('Install Native GeoIP2', 'affiliate-hub'); ?>
                                            </button>
                                        <?php else: ?>
                                            <button id="uninstall-native-btn" class="button">
                                                <?php _e('Uninstall Native', 'affiliate-hub'); ?>
                                            </button>
                                        <?php endif; ?>
                                        
                                        <button id="run-benchmark-btn" class="button">
                                            <span class="dashicons dashicons-chart-line"></span>
                                            <?php _e('Run Benchmark', 'affiliate-hub'); ?>
                                        </button>
                                    </div>
                                </div>
                                
                                <!-- Performance Recommendation -->
                                <?php if ($performance['recommendation']): ?>
                                    <div class="performance-recommendation <?php echo esc_attr($performance['recommendation']['type']); ?>">
                                        <div class="recommendation-content">
                                            <span class="dashicons dashicons-info"></span>
                                            <?php echo esc_html($performance['recommendation']['message']); ?>
                                            
                                            <?php if ($performance['recommendation']['type'] === 'install_native'): ?>
                                                <div class="expected-improvement">
                                                    <strong><?php _e('Expected improvement: 80-95% faster lookups', 'affiliate-hub'); ?></strong>
                                                </div>
                                            <?php endif; ?>
                                        </div>
                                    </div>
                                <?php endif; ?>
                                
                                <!-- Installation Status -->
                                <div id="installer-status-display"></div>
                                
                                <!-- Benchmark Results -->
                                <div id="benchmark-results" style="display: none;">
                                    <h4><?php _e('Benchmark Results', 'affiliate-hub'); ?></h4>
                                    <canvas id="benchmark-chart" width="400" height="200"></canvas>
                                </div>
                            </div>
                            
                            <!-- Advanced Mode Toggle -->
                            <div class="advanced-mode-toggle">
                                <h4>
                                    <label>
                                        <input type="checkbox" id="advanced-mode-toggle"> 
                                        <?php _e('Advanced Mode', 'affiliate-hub'); ?>
                                    </label>
                                </h4>
                                
                                <div id="advanced-options" style="display: none;">
                                    <div class="advanced-option">
                                        <label for="custom-vendor-dir"><?php _e('Custom Vendor Directory:', 'affiliate-hub'); ?></label>
                                        <input type="text" id="custom-vendor-dir" value="<?php echo esc_attr(AFFILIATE_HUB_PATH . 'vendor/'); ?>" class="regular-text">
                                        <p class="description"><?php _e('Custom path for GeoIP2 library installation', 'affiliate-hub'); ?></p>
                                    </div>
                                    
                                    <div class="advanced-option">
                                        <label>
                                            <input type="checkbox" id="verbose-logging"> 
                                            <?php _e('Verbose Installation Logging', 'affiliate-hub'); ?>
                                        </label>
                                        <p class="description"><?php _e('Enable detailed logging during installation process', 'affiliate-hub'); ?></p>
                                    </div>
                                    
                                    <div class="advanced-option">
                                        <button id="reset-performance-metrics" class="button">
                                            <?php _e('Reset Performance Metrics', 'affiliate-hub'); ?>
                                        </button>
                                        <p class="description"><?php _e('Clear all collected performance data', 'affiliate-hub'); ?></p>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
                
                <!-- Management Tools -->
                <div class="geo-tools-section">
                    <div class="geo-card">
                        <div class="geo-card-header">
                            <h3><?php _e('Management Tools', 'affiliate-hub'); ?></h3>
                        </div>
                        <div class="geo-card-body">
                            <div class="geo-tool">
                                <h4><?php _e('Bulk Enrichment', 'affiliate-hub'); ?></h4>
                                <p><?php _e('Process clicks without geographic data using API providers.', 'affiliate-hub'); ?></p>
                                <button id="bulk-enrich-btn" class="button button-primary">
                                    <?php _e('Enrich Old Data', 'affiliate-hub'); ?>
                                </button>
                                <div id="bulk-enrich-status"></div>
                            </div>
                            
                            <div class="geo-tool">
                                <h4><?php _e('Cache Cleanup', 'affiliate-hub'); ?></h4>
                                <p><?php _e('Remove expired geolocation cache entries.', 'affiliate-hub'); ?></p>
                                <button id="cache-cleanup-btn" class="button">
                                    <?php _e('Clean Cache', 'affiliate-hub'); ?>
                                </button>
                                <div id="cache-cleanup-status"></div>
                            </div>
                            
                            <div class="geo-tool">
                                <h4><?php _e('Export Data', 'affiliate-hub'); ?></h4>
                                <p><?php _e('Export geolocation data for analysis.', 'affiliate-hub'); ?></p>
                                <button id="export-geo-btn" class="button">
                                    <?php _e('Export CSV', 'affiliate-hub'); ?>
                                </button>
                            </div>
                        </div>
                    </div>
                    
                    <div class="geo-card">
                        <div class="geo-card-header">
                            <h3><?php _e('System Status', 'affiliate-hub'); ?></h3>
                        </div>
                        <div class="geo-card-body">
                            <div class="system-status">
                                <div class="status-item">
                                    <span class="status-indicator <?php echo get_option(Constants::OPTION_ENABLE_GEOLOCATION, true) ? 'active' : 'inactive'; ?>"></span>
                                    <strong><?php _e('GeoLocation Module:', 'affiliate-hub'); ?></strong>
                                    <?php echo get_option(Constants::OPTION_ENABLE_GEOLOCATION, true) ? __('Enabled', 'affiliate-hub') : __('Disabled', 'affiliate-hub'); ?>
                                </div>
                                
                                <div class="status-item">
                                    <span class="status-indicator <?php echo get_option(Constants::OPTION_ANONYMIZE_IP, false) ? 'active' : 'inactive'; ?>"></span>
                                    <strong><?php _e('IP Anonymization:', 'affiliate-hub'); ?></strong>
                                    <?php echo get_option(Constants::OPTION_ANONYMIZE_IP, false) ? __('Enabled (GDPR)', 'affiliate-hub') : __('Disabled', 'affiliate-hub'); ?>
                                </div>
                                
                                <div class="status-item">
                                    <span class="status-indicator active"></span>
                                    <strong><?php _e('Primary Provider:', 'affiliate-hub'); ?></strong>
                                    <?php echo esc_html(get_option('affiliate_hub_geo_primary_provider', 'ipapi_co')); ?>
                                </div>
                                
                                <div class="status-item">
                                    <span class="status-indicator active"></span>
                                    <strong><?php _e('Cache Tables:', 'affiliate-hub'); ?></strong>
                                    <?php 
                                    global $wpdb;
                                    $table_exists = $wpdb->get_var("SHOW TABLES LIKE '{$wpdb->prefix}affiliate_hub_geo_cache'");
                                    echo $table_exists ? __('Created', 'affiliate-hub') : __('Missing', 'affiliate-hub');
                                    ?>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
                
                <!-- Recent Activity -->
                <div class="geo-activity-section">
                    <div class="geo-card full-width">
                        <div class="geo-card-header">
                            <h3><?php _e('Recent Geolocation Activity', 'affiliate-hub'); ?></h3>
                        </div>
                        <div class="geo-card-body">
                            <table class="wp-list-table widefat fixed striped">
                                <thead>
                                    <tr>
                                        <th><?php _e('IP Address', 'affiliate-hub'); ?></th>
                                        <th><?php _e('Location', 'affiliate-hub'); ?></th>
                                        <th><?php _e('Provider', 'affiliate-hub'); ?></th>
                                        <th><?php _e('Cached At', 'affiliate-hub'); ?></th>
                                        <th><?php _e('Expires', 'affiliate-hub'); ?></th>
                                    </tr>
                                </thead>
                                <tbody id="recent-geo-activity">
                                    <?php $this->render_recent_activity(); ?>
                                </tbody>
                            </table>
                        </div>
                    </div>
                </div>
            </div>
        </div>
        
        <style>
        .affiliate-hub-geo-dashboard {
            margin-top: 20px;
        }
        
        .geo-stats-cards {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(300px, 1fr));
            gap: 20px;
            margin-bottom: 30px;
        }
        
        .geo-card {
            background: #fff;
            border: 1px solid #ccd0d4;
            border-radius: 4px;
            overflow: hidden;
        }
        
        .geo-card.full-width {
            grid-column: 1 / -1;
        }
        
        .geo-card-header {
            background: #f1f1f1;
            padding: 15px 20px;
            border-bottom: 1px solid #ccd0d4;
        }
        
        .geo-card-header h3 {
            margin: 0;
            font-size: 16px;
        }
        
        .geo-card-body {
            padding: 20px;
        }
        
        .geo-stat-number {
            font-size: 32px;
            font-weight: bold;
            color: #2271b1;
            margin-bottom: 5px;
        }
        
        .geo-stat-label {
            font-size: 14px;
            color: #666;
            margin-bottom: 10px;
        }
        
        .geo-stat-meta {
            font-size: 12px;
            color: #999;
        }
        
        .provider-stats .provider-stat {
            margin-bottom: 10px;
            padding-bottom: 10px;
            border-bottom: 1px solid #eee;
        }
        
        .provider-stats .provider-stat:last-child {
            border-bottom: none;
            margin-bottom: 0;
        }
        
        .geo-charts-section,
        .geo-tools-section,
        .geo-activity-section {
            margin-bottom: 30px;
        }
        
        .geo-tools-section {
            display: grid;
            grid-template-columns: 2fr 1fr;
            gap: 20px;
        }
        
        .geo-tool {
            margin-bottom: 25px;
            padding-bottom: 25px;
            border-bottom: 1px solid #eee;
        }
        
        .geo-tool:last-child {
            border-bottom: none;
            margin-bottom: 0;
        }
        
        .geo-tool h4 {
            margin: 0 0 10px 0;
        }
        
        .geo-tool p {
            color: #666;
            margin-bottom: 15px;
        }
        
        .system-status .status-item {
            display: flex;
            align-items: center;
            margin-bottom: 10px;
        }
        
        .status-indicator {
            display: inline-block;
            width: 12px;
            height: 12px;
            border-radius: 50%;
            margin-right: 10px;
        }
        
        .status-indicator.active {
            background-color: #46b450;
        }
        
        .status-indicator.inactive {
            background-color: #dc3232;
        }
        
        #bulk-enrich-status,
        #cache-cleanup-status {
            margin-top: 15px;
        }
        
        .notice.inline {
            display: block;
            margin: 10px 0;
            padding: 10px;
        }
        </style>
        <?php
    }
    
    private function render_recent_activity() {
        global $wpdb;
        
        $table_cache = $wpdb->prefix . 'affiliate_hub_geo_cache';
        $results = $wpdb->get_results("
            SELECT ip_address, country_name, city, provider_used, cached_at, expires_at 
            FROM $table_cache 
            ORDER BY cached_at DESC 
            LIMIT 10
        ");
        
        if (empty($results)) {
            echo '<tr><td colspan="5">' . __('No geolocation activity yet.', 'affiliate-hub') . '</td></tr>';
            return;
        }
        
        foreach ($results as $row) {
            echo '<tr>';
            echo '<td>' . esc_html($row->ip_address) . '</td>';
            echo '<td>' . esc_html($row->country_name . ($row->city ? ', ' . $row->city : '')) . '</td>';
            echo '<td>' . esc_html($row->provider_used) . '</td>';
            echo '<td>' . esc_html($row->cached_at) . '</td>';
            echo '<td>' . esc_html($row->expires_at) . '</td>';
            echo '</tr>';
        }
    }
    
    public function ajax_get_geo_stats() {
        check_ajax_referer('affiliate_hub_admin_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Insufficient permissions');
        }
        
        global $wpdb;
        $table_clicks = $wpdb->prefix . Constants::TABLE_LINK_CLICKS;
        
        // Get country distribution
        $countries = $wpdb->get_results("
            SELECT country, COUNT(*) as clicks 
            FROM $table_clicks 
            WHERE country != '' AND country IS NOT NULL
            GROUP BY country 
            ORDER BY clicks DESC 
            LIMIT 10
        ");
        
        $chart_data = array(
            'labels' => array(),
            'data' => array()
        );
        
        foreach ($countries as $country) {
            $chart_data['labels'][] = $country->country;
            $chart_data['data'][] = $country->clicks;
        }
        
        wp_send_json_success($chart_data);
    }
    
    public function ajax_bulk_enrich_geo() {
        check_ajax_referer('affiliate_hub_admin_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Insufficient permissions');
        }
        
        // Delegate to GeoLocation module's bulk enrich method
        $this->geolocation_module->ajax_bulk_enrich_old_data();
    }
    
    /**
     * AJAX handler for MaxMind database update
     */
    public function ajax_maxmind_update() {
        check_ajax_referer('affiliate_hub_admin_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Insufficient permissions');
        }
        
        require_once AFFILIATE_HUB_PATH . 'includes/Modules/MaxMindManager.php';
        $maxmind_manager = new \AffiliateHub\Modules\MaxMindManager();
        
        $result = $maxmind_manager->update_database();
        
        if (is_wp_error($result)) {
            wp_send_json_error($result->get_error_message());
        } else {
            wp_send_json_success(array(
                'message' => __('MaxMind database updated successfully', 'affiliate-hub'),
                'status' => $this->geolocation_module->get_maxmind_status()
            ));
        }
    }
    
    /**
     * AJAX handler for native GeoIP2 installation
     */
    public function ajax_install_native_geoip2() {
        error_log('=== AJAX INSTALL START ===');
        
        try {
            error_log('Step 1: Checking nonce');
            check_ajax_referer('affiliate_hub_admin_nonce', 'nonce');
            error_log('Step 1: Nonce OK');
            
            error_log('Step 2: Checking permissions');
            if (!current_user_can('manage_options')) {
                error_log('Ajax install: Insufficient permissions');
                wp_send_json_error('Insufficient permissions');
                return;
            }
            error_log('Step 2: Permissions OK');
            
            error_log('Step 3: Getting geolocation module');
            if (!$this->geolocation_module) {
                error_log('Ajax install: GeoLocation module not found');
                wp_send_json_error('GeoLocation module not available');
                return;
            }
            error_log('Step 3: GeoLocation module OK');
            
            error_log('Step 4: Getting auto installer');
            $auto_installer = $this->geolocation_module->get_auto_installer();
            if (!$auto_installer) {
                error_log('Ajax install: Auto installer not found');
                wp_send_json_error('Auto installer not available');
                return;
            }
            error_log('Step 4: Auto installer OK - class: ' . get_class($auto_installer));
            
            error_log('Step 5: Starting installation');
            $result = $auto_installer->install();
            error_log('Step 5: Installation completed with result: ' . print_r($result, true));
            
            if ($result['success']) {
                wp_send_json_success($result);
            } else {
                wp_send_json_error($result['message'] ?? 'Installation failed');
            }
        } catch (\Exception $e) {
            error_log('Ajax install exception: ' . $e->getMessage());
            error_log('Ajax install stack trace: ' . $e->getTraceAsString());
            wp_send_json_error('Installation failed: ' . $e->getMessage());
        } catch (\Error $e) {
            error_log('Ajax install PHP error: ' . $e->getMessage());
            error_log('Ajax install error trace: ' . $e->getTraceAsString());
            wp_send_json_error('Installation failed (PHP Error): ' . $e->getMessage());
        } catch (\Throwable $e) {
            error_log('Ajax install throwable: ' . $e->getMessage());
            error_log('Ajax install throwable trace: ' . $e->getTraceAsString());
            wp_send_json_error('Installation failed (Throwable): ' . $e->getMessage());
        }
    }
    
    /**
     * AJAX handler for native GeoIP2 uninstallation
     */
    public function ajax_uninstall_native_geoip2() {
        check_ajax_referer('affiliate_hub_admin_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Insufficient permissions');
        }
        
        $auto_installer = $this->geolocation_module->get_auto_installer();
        $result = $auto_installer->uninstall();
        
        if ($result['success']) {
            wp_send_json_success($result);
        } else {
            wp_send_json_error($result['message']);
        }
    }
    
    /**
     * AJAX handler for benchmark test
     */
    public function ajax_run_benchmark() {
        error_log('Ajax run benchmark called');
        
        check_ajax_referer('affiliate_hub_admin_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            error_log('Ajax benchmark: Insufficient permissions');
            wp_send_json_error('Insufficient permissions');
        }
        
        $performance_monitor = $this->geolocation_module->get_performance_monitor();
        if (!$performance_monitor) {
            error_log('Ajax benchmark: Performance monitor not found');
            wp_send_json_error('Performance monitor not available');
            return;
        }
        
        error_log('Ajax benchmark: Starting benchmark');
        $results = $performance_monitor->run_benchmark();
        error_log('Ajax benchmark result: ' . print_r($results, true));
        
        wp_send_json_success(array(
            'message' => __('Benchmark completed successfully', 'affiliate-hub'),
            'results' => $results
        ));
    }
    
    /**
     * AJAX handler for resetting performance metrics
     */
    public function ajax_reset_performance_metrics() {
        check_ajax_referer('affiliate_hub_admin_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Insufficient permissions');
        }
        
        $performance_monitor = $this->geolocation_module->get_performance_monitor();
        $result = $performance_monitor->reset_metrics();
        
        wp_send_json_success($result);
    }
}
