<?php
/**
 * Import/Export functionality for Affiliate Hub
 *
 * @package AffiliateHub
 * @subpackage Admin
 */

namespace AffiliateHub\Admin;

use AffiliateHub\Core\Constants;
use AffiliateHub\Models\AffiliateLink;

/**
 * ImportExport class
 */
class ImportExport {
    
    /**
     * Initialize import/export functionality
     */
    public function __construct() {
    \add_action('admin_init', array($this, 'handle_export'));
    \add_action('admin_init', array($this, 'handle_import'));
    \add_action('wp_ajax_affiliate_hub_export_links', array($this, 'ajax_export_links'));
    \add_action('wp_ajax_affiliate_hub_import_preview', array($this, 'ajax_import_preview'));
    }
    
    /**
     * Add import/export section to settings page
     */
    public function add_import_export_section() {
        ?>
        <div class="affiliate-hub-import-export">
            <h3><?php \esc_html_e('Import / Export', 'affiliate-hub'); ?></h3>
            
            <!-- Export Section -->
            <div class="export-section">
                <h4><?php \esc_html_e('Export Affiliate Links', 'affiliate-hub'); ?></h4>
                <p><?php \esc_html_e('Export your affiliate links to CSV or JSON format for backup or migration.', 'affiliate-hub'); ?></p>
                
                <form method="post" action="<?php echo \esc_url(\admin_url('admin.php?page=affiliate-hub-settings')); ?>">
                    <?php \wp_nonce_field('affiliate_hub_export', 'export_nonce'); ?>
                    <input type="hidden" name="action" value="export_links">
                    
                    <table class="form-table">
                        <tr>
                            <th scope="row"><?php \esc_html_e('Export Format', 'affiliate-hub'); ?></th>
                            <td>
                                <label><input type="radio" name="export_format" value="csv" checked> <?php \esc_html_e('CSV', 'affiliate-hub'); ?></label><br>
                                <label><input type="radio" name="export_format" value="json"> <?php \esc_html_e('JSON', 'affiliate-hub'); ?></label>
                            </td>
                        </tr>
                        <tr>
                            <th scope="row"><?php \esc_html_e('Include Statistics', 'affiliate-hub'); ?></th>
                            <td>
                                <label><input type="checkbox" name="include_stats" value="1" checked> <?php \esc_html_e('Include click statistics in export', 'affiliate-hub'); ?></label>
                            </td>
                        </tr>
                        <tr>
                            <th scope="row"><?php \esc_html_e('Date Range', 'affiliate-hub'); ?></th>
                            <td>
                                <select name="date_range">
                                    <option value="all"><?php \esc_html_e('All time', 'affiliate-hub'); ?></option>
                                    <option value="last_30_days"><?php \esc_html_e('Last 30 days', 'affiliate-hub'); ?></option>
                                    <option value="last_3_months"><?php \esc_html_e('Last 3 months', 'affiliate-hub'); ?></option>
                                    <option value="last_year"><?php \esc_html_e('Last year', 'affiliate-hub'); ?></option>
                                </select>
                            </td>
                        </tr>
                    </table>
                    
                    <p class="submit">
                        <input type="submit" class="button-primary" value="<?php \esc_attr_e('Export Links', 'affiliate-hub'); ?>">
                    </p>
                </form>
            </div>
            
            <!-- Import Section -->
            <div class="import-section">
                <h4><?php \esc_html_e('Import Affiliate Links', 'affiliate-hub'); ?></h4>
                <p><?php \esc_html_e('Import affiliate links from CSV or JSON format.', 'affiliate-hub'); ?></p>
                
                <form method="post" action="<?php echo \esc_url(\admin_url('admin.php?page=affiliate-hub-settings')); ?>" enctype="multipart/form-data">
                    <?php \wp_nonce_field('affiliate_hub_import', 'import_nonce'); ?>
                    <input type="hidden" name="action" value="import_links">
                    
                    <table class="form-table">
                        <tr>
                            <th scope="row"><?php \esc_html_e('Import File', 'affiliate-hub'); ?></th>
                            <td>
                                <input type="file" name="import_file" accept=".csv,.json" required>
                                <p class="description"><?php \esc_html_e('Select a CSV or JSON file to import. Maximum file size: 2MB.', 'affiliate-hub'); ?></p>
                            </td>
                        </tr>
                        <tr>
                            <th scope="row"><?php \esc_html_e('Import Options', 'affiliate-hub'); ?></th>
                            <td>
                                <label><input type="checkbox" name="skip_existing" value="1" checked> <?php \esc_html_e('Skip existing links (based on slug)', 'affiliate-hub'); ?></label><br>
                                <label><input type="checkbox" name="update_existing" value="1"> <?php \esc_html_e('Update existing links', 'affiliate-hub'); ?></label>
                            </td>
                        </tr>
                    </table>
                    
                    <p class="submit">
                        <input type="submit" class="button-primary" value="<?php \esc_attr_e('Import Links', 'affiliate-hub'); ?>">
                    </p>
                </form>
            </div>
            
            <!-- Import Template -->
            <div class="import-template-section">
                <h4><?php \esc_html_e('Import Template', 'affiliate-hub'); ?></h4>
                <p><?php \esc_html_e('Download a template file to see the required format for imports.', 'affiliate-hub'); ?></p>
                <a href="<?php echo \esc_url(\admin_url('admin.php?page=affiliate-hub-settings&action=download_template&format=csv&_wpnonce=' . \wp_create_nonce('affiliate_hub_template'))); ?>" class="button">
                    <?php \esc_html_e('Download CSV Template', 'affiliate-hub'); ?>
                </a>
                <a href="<?php echo \esc_url(\admin_url('admin.php?page=affiliate-hub-settings&action=download_template&format=json&_wpnonce=' . \wp_create_nonce('affiliate_hub_template'))); ?>" class="button">
                    <?php \esc_html_e('Download JSON Template', 'affiliate-hub'); ?>
                </a>
            </div>
        </div>
        
        <style>
        .affiliate-hub-import-export {
            margin: 20px 0;
        }
        
        .export-section, .import-section, .import-template-section {
            margin: 30px 0;
            padding: 20px;
            border: 1px solid #ddd;
            background: #f9f9f9;
        }
        
        .export-section h4, .import-section h4, .import-template-section h4 {
            margin-top: 0;
        }
        </style>
        <?php
    }
    
    /**
     * Handle export request
     */
    public function handle_export() {
        if (!isset($_POST['action']) || $_POST['action'] !== 'export_links') {
            return;
        }
        
    $nonce = isset($_POST['export_nonce']) ? \sanitize_text_field(\wp_unslash($_POST['export_nonce'])) : '';
    if (!\wp_verify_nonce($nonce, 'affiliate_hub_export')) {
            \wp_die(\esc_html__('Security check failed.', 'affiliate-hub'));
        }
        
        if (!\current_user_can(Constants::CAP_MANAGE_AFFILIATE_LINKS)) {
            \wp_die(\esc_html__('You do not have permission to export affiliate links.', 'affiliate-hub'));
        }
        
    $format = isset($_POST['export_format']) ? \sanitize_text_field(\wp_unslash($_POST['export_format'])) : 'csv';
    $format = in_array($format, array('csv', 'json'), true) ? $format : 'csv';
    $include_stats = !empty($_POST['include_stats']);
    $date_range = isset($_POST['date_range']) ? \sanitize_text_field(\wp_unslash($_POST['date_range'])) : 'all';
        
        $links = $this->get_links_for_export($date_range);
        
        if ($format === 'json') {
            $this->export_json($links, $include_stats);
        } else {
            $this->export_csv($links, $include_stats);
        }
    }
    
    /**
     * Handle import request
     */
    public function handle_import() {
        if (!isset($_POST['action']) || $_POST['action'] !== 'import_links') {
            return;
        }
        
        $nonce = isset($_POST['import_nonce']) ? \sanitize_text_field(\wp_unslash($_POST['import_nonce'])) : '';
        if (!\wp_verify_nonce($nonce, 'affiliate_hub_import')) {
            \wp_die(\esc_html__('Security check failed.', 'affiliate-hub'));
        }
        
        if (!\current_user_can(Constants::CAP_MANAGE_AFFILIATE_LINKS)) {
            \wp_die(\esc_html__('You do not have permission to import affiliate links.', 'affiliate-hub'));
        }
        
        if (
            !isset($_FILES['import_file']) ||
            !is_array($_FILES['import_file']) ||
            !isset($_FILES['import_file']['error']) ||
            (int) $_FILES['import_file']['error'] !== UPLOAD_ERR_OK
        ) {
            \wp_die(\esc_html__('No file uploaded or upload error.', 'affiliate-hub'));
        }
        
        $file = $_FILES['import_file'];
        $original_name = isset($file['name']) ? \sanitize_file_name(\wp_unslash($file['name'])) : '';
        $tmp_name = isset($file['tmp_name']) ? $file['tmp_name'] : '';
        $size = isset($file['size']) ? (int) $file['size'] : 0;

        if (empty($original_name) || empty($tmp_name) || !file_exists($tmp_name)) {
            \wp_die(\esc_html__('Invalid upload data.', 'affiliate-hub'));
        }

        // Enforce a 2MB size limit as documented in the UI.
        if ($size > 2 * 1024 * 1024) {
            \wp_die(\esc_html__('File too large. Maximum size is 2MB.', 'affiliate-hub'));
        }

        $file_extension = strtolower((string) pathinfo($original_name, PATHINFO_EXTENSION));

        if (!in_array($file_extension, array('csv', 'json'), true)) {
            \wp_die(\esc_html__('Invalid file format. Only CSV and JSON files are allowed.', 'affiliate-hub'));
        }
        
        $skip_existing = !empty($_POST['skip_existing']);
        $update_existing = !empty($_POST['update_existing']);
        
        if ($file_extension === 'json') {
            $result = $this->import_json($tmp_name, $skip_existing, $update_existing);
        } else {
            $result = $this->import_csv($tmp_name, $skip_existing, $update_existing);
        }
        
        // Redirect with result message
        $redirect_url = \add_query_arg(
            array('page' => 'affiliate-hub-settings', 'import_result' => rawurlencode($result)),
            \admin_url('admin.php')
        );
        \wp_safe_redirect($redirect_url);
        exit;
    }
    
    /**
     * Export links as CSV
     */
    private function export_csv($links, $include_stats = false) {
        $filename = 'affiliate-links-' . gmdate('Y-m-d-H-i-s') . '.csv';
        
        header('Content-Type: text/csv');
        header('Content-Disposition: attachment; filename="' . $filename . '"');
        
    // phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_read_fopen -- Using php://output for streaming a CSV download is acceptable here.
    // phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_fopen -- Using php://output for streaming a CSV download is acceptable here.
        $output = fopen('php://output', 'w');
        
        // CSV headers
        $headers = array('Title', 'Slug', 'Destination URL', 'Redirect Type', 'Nofollow', 'New Window', 'Categories');
        if ($include_stats) {
            $headers = array_merge($headers, array('Total Clicks', 'Unique Clicks', 'Last Clicked'));
        }
        
        fputcsv($output, $headers);
        
        // Export data
        foreach ($links as $link_data) {
            $row = array(
                $link_data['title'],
                $link_data['slug'],
                $link_data['destination_url'],
                $link_data['redirect_type'],
                $link_data['nofollow'] ? 'Yes' : 'No',
                $link_data['new_window'] ? 'Yes' : 'No',
                implode(', ', $link_data['categories'])
            );
            if ($include_stats) {
                $row = array_merge($row, array(
                    $link_data['total_clicks'],
                    $link_data['unique_clicks'],
                    $link_data['last_clicked']
                ));
            }
            
            fputcsv($output, $row);
        }
        
    // phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_read_fclose -- Closing php://output stream after writing response.
    // phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_fclose -- Closing php://output stream after writing response.
        fclose($output);
        exit;
    }
    
    /**
     * Export links as JSON
     */
    private function export_json($links, $include_stats = false) {
        $filename = 'affiliate-links-' . gmdate('Y-m-d-H-i-s') . '.json';
        
        header('Content-Type: application/json');
        header('Content-Disposition: attachment; filename="' . $filename . '"');
        
        $export_data = array(
            'export_date' => gmdate('Y-m-d H:i:s'),
            'plugin_version' => AFFILIATE_HUB_VERSION,
            'include_stats' => $include_stats,
            'total_links' => count($links),
            'links' => $links
        );
        
        echo json_encode($export_data, JSON_PRETTY_PRINT);
        exit;
    }
    
    /**
     * Get links for export based on date range
     */
    private function get_links_for_export($date_range) {
        $date_query = array();
        
        switch ($date_range) {
            case 'last_30_days':
                $date_query = array(
                    'after' => '30 days ago'
                );
                break;
            case 'last_3_months':
                $date_query = array(
                    'after' => '3 months ago'
                );
                break;
            case 'last_year':
                $date_query = array(
                    'after' => '1 year ago'
                );
                break;
        }
        
        $args = array(
            'post_type' => Constants::POST_TYPE_AFFILIATE_LINK,
            'post_status' => 'publish',
            'posts_per_page' => -1
        );
        
        if (!empty($date_query)) {
            $args['date_query'] = array($date_query);
        }
        
    $posts = \get_posts($args);
        $links = array();
        
        foreach ($posts as $post) {
            try {
                $affiliate_link = new \AffiliateHub\Models\AffiliateLink($post->ID);
                
                $links[] = array(
                    'id' => $post->ID,
                    'title' => $post->post_title,
                    'slug' => $post->post_name,
                    'destination_url' => $affiliate_link->get_destination_url(),
                    'redirect_type' => $affiliate_link->get_redirect_type(),
                    'nofollow' => $affiliate_link->is_nofollow(),
                    'new_window' => $affiliate_link->is_new_window(),
                    'categories' => \wp_list_pluck($affiliate_link->get_categories(), 'name'),
                    'total_clicks' => $affiliate_link->get_click_count(),
                    'unique_clicks' => $affiliate_link->get_unique_clicks(),
                    'last_clicked' => $affiliate_link->get_last_clicked(),
                    'created' => $post->post_date,
                    'modified' => $post->post_modified
                );
            } catch (\Exception $e) {
                // Skip invalid links
                continue;
            }
        }
        
        return $links;
    }
    
    /**
     * Import CSV file
     */
    private function import_csv($file_path, $skip_existing, $update_existing) {
        $imported = 0;
        $skipped = 0;
        $errors = 0;
        
    // phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_read_fopen -- Reading uploaded tmp file for CSV import.
    // phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_fopen -- Reading uploaded tmp file for CSV import.
        if (($handle = fopen($file_path, 'r')) !== false) {
            // Skip header row
            fgetcsv($handle);
            
            while (($data = fgetcsv($handle)) !== false) {
                $result = $this->import_single_link($data, $skip_existing, $update_existing, 'csv');
                
                switch ($result) {
                    case 'imported':
                        $imported++;
                        break;
                    case 'skipped':
                        $skipped++;
                        break;
                    case 'error':
                        $errors++;
                        break;
                }
            }
            
            // phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_read_fclose -- Closing handle for uploaded tmp file.
            // phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_fclose -- Closing handle for uploaded tmp file.
            fclose($handle);
        }
        
        // translators: %1$d: number of imported links, %2$d: number of skipped links, %3$d: number of error links
    return sprintf(\__('Import completed. Imported: %1$d, Skipped: %2$d, Errors: %3$d', 'affiliate-hub'), $imported, $skipped, $errors);
    }
    
    /**
     * Import JSON file
     */
    private function import_json($file_path, $skip_existing, $update_existing) {
        // phpcs:ignore WordPress.WP.AlternativeFunctions.file_get_contents_file_get_contents -- Reading uploaded tmp file for JSON import.
        $json_data = file_get_contents($file_path);
        $data = json_decode($json_data, true);
        
        if (json_last_error() !== JSON_ERROR_NONE) {
            return \__('Invalid JSON file.', 'affiliate-hub');
        }
        
        if (!isset($data['links']) || !is_array($data['links'])) {
            return \__('Invalid JSON structure. Missing links data.', 'affiliate-hub');
        }
        
        $imported = 0;
        $skipped = 0;
        $errors = 0;
        
        foreach ($data['links'] as $link_data) {
            $result = $this->import_single_link($link_data, $skip_existing, $update_existing, 'json');
            
            switch ($result) {
                case 'imported':
                    $imported++;
                    break;
                case 'skipped':
                    $skipped++;
                    break;
                case 'error':
                    $errors++;
                    break;
            }
        }
        
        // translators: %1$d: number of imported links, %2$d: number of skipped links, %3$d: number of error links
    return sprintf(\__('Import completed. Imported: %1$d, Skipped: %2$d, Errors: %3$d', 'affiliate-hub'), $imported, $skipped, $errors);
    }
    
    /**
     * Import a single link
     */
    private function import_single_link($data, $skip_existing, $update_existing, $format) {
        try {
            // Parse data based on format
            if ($format === 'csv') {
                $link_data = array(
                    'title' => $data[0],
                    'slug' => $data[1],
                    'destination_url' => $data[2],
                    'redirect_type' => $data[3],
                    'nofollow' => $data[4] === 'Yes',
                    'new_window' => $data[5] === 'Yes',
                    'categories' => explode(', ', $data[6])
                );
            } else {
                $link_data = $data;
            }
            
            // Validate required fields
            if (empty($link_data['title']) || empty($link_data['destination_url'])) {
                return 'error';
            }
            
            // Check if link already exists
            $existing_post = \get_page_by_path($link_data['slug'], \OBJECT, Constants::POST_TYPE_AFFILIATE_LINK);
            
            if ($existing_post) {
                if ($skip_existing) {
                    return 'skipped';
                } elseif (!$update_existing) {
                    return 'error';
                }
                
                // Update existing link
                $post_id = $existing_post->ID;
                \wp_update_post(array(
                    'ID' => $post_id,
                    'post_title' => \sanitize_text_field($link_data['title'])
                ));
            } else {
                // Create new link
                $post_data = array(
                    'post_title' => \sanitize_text_field($link_data['title']),
                    'post_name' => \sanitize_title($link_data['slug']),
                    'post_type' => Constants::POST_TYPE_AFFILIATE_LINK,
                    'post_status' => 'publish',
                    'post_author' => \get_current_user_id()
                );
                
                $post_id = \wp_insert_post($post_data);
                if (\is_wp_error($post_id)) {
                    return 'error';
                }
            }
            
            // Set affiliate link metadata
            $affiliate_link = new \AffiliateHub\Models\AffiliateLink($post_id);
            $affiliate_link->set_destination_url($link_data['destination_url']);
            
            if (!empty($link_data['redirect_type'])) {
                $affiliate_link->set_redirect_type($link_data['redirect_type']);
            }
            
            if (isset($link_data['nofollow'])) {
                $affiliate_link->set_nofollow($link_data['nofollow']);
            }
            
            if (isset($link_data['new_window'])) {
                $affiliate_link->set_new_window($link_data['new_window']);
            }
            
            // Set categories
            if (!empty($link_data['categories']) && is_array($link_data['categories'])) {
                $category_ids = array();
                foreach ($link_data['categories'] as $category_name) {
                    $category_name = trim($category_name);
                    if (!empty($category_name)) {
                        $term = \get_term_by('name', $category_name, Constants::TAXONOMY_AFFILIATE_CATEGORY);
                        if (!$term) {
                            $term = \wp_insert_term($category_name, Constants::TAXONOMY_AFFILIATE_CATEGORY);
                            if (!\is_wp_error($term)) {
                                $category_ids[] = $term['term_id'];
                            }
                        } else {
                            $category_ids[] = $term->term_id;
                        }
                    }
                }
                
                if (!empty($category_ids)) {
                    \wp_set_object_terms($post_id, $category_ids, Constants::TAXONOMY_AFFILIATE_CATEGORY);
                }
            }
            
            return 'imported';
            
        } catch (\Exception $e) {
            return 'error';
        }
    }
}
