<?php
/**
 * Meta Boxes for Affiliate Links
 *
 * @package AffiliateHub
 * @subpackage Admin
 */

namespace AffiliateHub\Admin;

use AffiliateHub\Core\Constants;
use AffiliateHub\Models\AffiliateLink;

/**
 * MetaBoxes class
 */
class MetaBoxes {
    
    /**
     * Constructor
     */
    public function __construct() {
        // Skip initialization if safe mode is active
        if (get_option('affiliate_hub_safe_mode', false)) {
            return;
        }
        
        add_action('add_meta_boxes', array($this, 'add_meta_boxes'));
        add_action('save_post', array($this, 'save_meta_boxes'), 5, 1); // Higher priority
    }
    
    /**
     * Add meta boxes
     */
    public function add_meta_boxes() {
        add_meta_box(
            'affiliate-hub-link-details',
            __('Link Details', 'affiliate-hub'),
            array($this, 'link_details_meta_box'),
            Constants::POST_TYPE_AFFILIATE_LINK,
            'normal',
            'high'
        );
        
        add_meta_box(
            'affiliate-hub-link-options',
            __('Link Options', 'affiliate-hub'),
            array($this, 'link_options_meta_box'),
            Constants::POST_TYPE_AFFILIATE_LINK,
            'side',
            'default'
        );
        
        add_meta_box(
            'affiliate-hub-link-stats',
            __('Analytics', 'affiliate-hub'),
            array($this, 'link_stats_meta_box'),
            Constants::POST_TYPE_AFFILIATE_LINK,
            'side',
            'default'
        );
        
        // Add Keywords Autolinker meta box if module is enabled
        if (get_option(Constants::OPTION_AUTOLINKER_ENABLED, false)) {
            add_meta_box(
                'affiliate-hub-autolinker-keywords',
                __('Autolink Keywords', 'affiliate-hub'),
                array($this, 'autolinker_keywords_meta_box'),
                Constants::POST_TYPE_AFFILIATE_LINK,
                'normal',
                'default'
            );
        }
    }
    
    /**
     * Link details meta box
     */
    public function link_details_meta_box($post) {
        wp_nonce_field('affiliate_hub_meta_box', 'affiliate_hub_meta_box_nonce');
        
        $affiliate_link = new \AffiliateHub\Models\AffiliateLink($post->ID);
        $destination_url = $affiliate_link->get_destination_url();
        $link_slug = $affiliate_link->get_slug();
        $global_prefix = get_option(Constants::OPTION_LINK_PREFIX, 'go');
        
        // Check if this link has a custom prefix saved
        $custom_prefix = get_post_meta($post->ID, '_affiliate_custom_prefix', true);
        
        // For new posts, check if meta exists. For existing posts that were saved with the new system,
        // the meta key will exist even if the value is empty string (no prefix)
        $meta_all = get_post_meta($post->ID, '_affiliate_custom_prefix');
        $has_custom_setting = !empty($meta_all); // Meta key exists
        
        // Determine the display value for the input field
        if ($has_custom_setting) {
            // This link was saved with the new system - respect saved prefix (even if empty)
            if ($custom_prefix === '') {
                $display_value = $link_slug; // No prefix
            } else {
                $display_value = $custom_prefix . '/' . $link_slug;
            }
        } else if ($link_slug) {
            // Old link or first edit - use global prefix for backward compatibility
            $display_value = $global_prefix . '/' . $link_slug;
        } else {
            // New link - start with global prefix (if exists)
            $display_value = $global_prefix ? $global_prefix . '/' : '';
        }
        
        $home_url = trailingslashit(home_url());
        ?>
        <table class="form-table">
            <tr>
                <th scope="row">
                    <label for="destination_url"><?php esc_html_e('Destination URL', 'affiliate-hub'); ?></label>
                </th>
                <td>
                    <input type="url" id="destination_url" name="destination_url" value="<?php echo esc_attr($destination_url); ?>" class="large-text" required />
                    <p class="description"><?php esc_html_e('The URL where visitors will be redirected when they click your affiliate link.', 'affiliate-hub'); ?></p>
                </td>
            </tr>
            <tr>
                <th scope="row">
                    <label for="link_slug"><?php esc_html_e('Link Slug', 'affiliate-hub'); ?></label>
                </th>
                <td>
                    <div class="affiliate-hub-slug-container">
                        <span class="slug-prefix"><?php echo esc_html($home_url); ?></span>
                        <input type="text" id="link_slug" name="link_slug" 
                               value="<?php echo esc_attr($display_value); ?>" 
                               class="regular-text" 
                               placeholder="<?php echo esc_attr($global_prefix ? $global_prefix . '/link-name' : 'link-name'); ?>" />
                        <span class="slug-suffix">/</span>
                        <button type="button" class="button button-small" id="generate-slug-btn" style="margin-left: 10px;">
                            <?php esc_html_e('Generate from title', 'affiliate-hub'); ?>
                        </button>
                    </div>
                    <div id="slug-validation" style="margin-top: 5px;"></div>
                    <p class="description">
                        <?php esc_html_e('Enter the complete path for your affiliate link. You can modify or remove the prefix as needed.', 'affiliate-hub'); ?>
                        <br>
                        <?php esc_html_e('Examples:', 'affiliate-hub'); ?>
                        <br>
                        • <code><?php echo esc_html($global_prefix); ?>/amazon-product</code> - <?php esc_html_e('Using default prefix', 'affiliate-hub'); ?>
                        <br>
                        • <code>shop/amazon-product</code> - <?php esc_html_e('Using custom prefix', 'affiliate-hub'); ?>
                        <br>
                        • <code>amazon-product</code> - <?php esc_html_e('No prefix', 'affiliate-hub'); ?>
                        <br>
                        <?php esc_html_e('Final URL will be:', 'affiliate-hub'); ?> <code><?php echo esc_html($home_url); ?>[your-input]/</code>
                    </p>
                </td>
            </tr>
            <tr>
                <th scope="row">
                    <label><?php esc_html_e('Generated Cloaked URL', 'affiliate-hub'); ?></label>
                </th>
                <td>
                    <input type="url" id="cloaked_url_preview" class="large-text" readonly style="background-color: #f9f9f9;" />
                    <p class="description"><?php esc_html_e('This is your final cloaked URL that will be generated based on the slug above.', 'affiliate-hub'); ?></p>
                    
                    <div id="cloaked-url-actions" style="margin-top: 10px;">
                        <button type="button" class="button button-secondary" id="test-link-btn" style="display: none;" 
                                title="<?php esc_attr_e('Test the link (for unpublished posts: tests destination URL directly)', 'affiliate-hub'); ?>">
                            <?php esc_html_e('Test Link', 'affiliate-hub'); ?>
                        </button>
                        <button type="button" class="button button-secondary" id="copy-url-btn" style="display: none;">
                            <?php esc_html_e('Copy URL', 'affiliate-hub'); ?>
                        </button>
                        <span id="copy-success" style="color: green; margin-left: 10px; display: none;">
                            <?php esc_html_e('Copied!', 'affiliate-hub'); ?>
                        </span>
                    </div>
                    
                    <p class="description" style="margin-top: 10px; color: #666;">
                        <strong><?php esc_html_e('Note:', 'affiliate-hub'); ?></strong> 
                        <?php esc_html_e('Test Link will open the destination URL directly until this post is published and the cloaked URL is active.', 'affiliate-hub'); ?>
                    </p>
                </td>
            </tr>
        </table>
        
        <style>
        .affiliate-hub-slug-container {
            display: flex;
            align-items: center;
            gap: 0;
        }
        .slug-prefix, .slug-suffix {
            background: #f1f1f1;
            border: 1px solid #ddd;
            padding: 3px 8px;
            color: #666;
            font-family: monospace;
        }
        .slug-prefix {
            border-right: none;
            border-radius: 4px 0 0 4px;
        }
        .slug-suffix {
            border-left: none;
            border-radius: 0 4px 4px 0;
        }
        #link_slug {
            border-radius: 0;
            border-left: none;
            border-right: none;
            font-family: monospace;
        }
        #slug-validation .validation-message {
            font-size: 12px;
            padding: 2px 0;
        }
        .validation-available { color: #46b450; }
        .validation-unavailable { color: #dc3232; }
        .validation-checking { color: #0073aa; }
        .validation-suggestion { color: #ffb900; }
        </style>
        
        <script type="text/javascript">
        jQuery(document).ready(function($) {
            var $slugField = $('#link_slug');
            var $previewField = $('#cloaked_url_preview');
            var $testBtn = $('#test-link-btn');
            var $copyBtn = $('#copy-url-btn');
            var $validation = $('#slug-validation');
            var $generateBtn = $('#generate-slug-btn');
            var $titleField = $('#title');
            var linkPrefix = <?php echo json_encode($global_prefix); ?>;
            var homeUrl = <?php echo json_encode($home_url); ?>;
            var currentPostId = <?php echo intval($post->ID); ?>;
            
            // Function to generate slug from title
            function generateSlugFromTitle(title) {
                var slug = title
                    .toLowerCase()
                    .trim()
                    // Replace Polish characters
                    .replace(/ą/g, 'a').replace(/ć/g, 'c').replace(/ę/g, 'e')
                    .replace(/ł/g, 'l').replace(/ń/g, 'n').replace(/ó/g, 'o')
                    .replace(/ś/g, 's').replace(/ź/g, 'z').replace(/ż/g, 'z')
                    // Replace German characters
                    .replace(/ä/g, 'ae').replace(/ö/g, 'oe').replace(/ü/g, 'ue')
                    .replace(/ß/g, 'ss')
                    // Replace Spanish characters
                    .replace(/ñ/g, 'n').replace(/á/g, 'a').replace(/é/g, 'e')
                    .replace(/í/g, 'i').replace(/ó/g, 'o').replace(/ú/g, 'u')
                    // Remove other special characters
                    .replace(/[^\w\s-]/g, '')
                    // Replace spaces with hyphens
                    .replace(/\s+/g, '-')
                    // Replace multiple hyphens with single hyphen
                    .replace(/-+/g, '-')
                    // Remove leading/trailing hyphens
                    .replace(/^-|-$/g, '');
                
                // Check if user has already customized the prefix
                var currentValue = $slugField.val().trim();
                var currentPrefix = '';
                
                if (currentValue && currentValue !== linkPrefix + '/') {
                    // User has custom content, try to preserve their prefix
                    var parsed = parseCombinedSlug(currentValue);
                    if (parsed.prefix) {
                        currentPrefix = parsed.prefix;
                    }
                }
                
                // Use custom prefix if exists, otherwise default
                var finalPrefix = currentPrefix || linkPrefix;
                
                if (finalPrefix) {
                    return finalPrefix + '/' + slug;
                } else {
                    return slug; // No prefix
                }
            }
            
            // Function to extract prefix and slug from combined input
            function parseCombinedSlug(combined) {
                if (!combined) return { prefix: '', slug: '' };
                
                // Find the last slash to split prefix and slug
                var lastSlashIndex = combined.lastIndexOf('/');
                
                if (lastSlashIndex === -1) {
                    // No slash found, treat entire string as slug
                    return { prefix: '', slug: combined };
                }
                
                var prefix = combined.substring(0, lastSlashIndex);
                var slug = combined.substring(lastSlashIndex + 1);
                
                return { prefix: prefix, slug: slug };
            }
            
            // Function to sanitize slug part only
            function sanitizeSlugPart(slug) {
                return slug
                    .toLowerCase()
                    .replace(/[^a-z0-9\-]/g, '-')
                    .replace(/-+/g, '-')
                    .replace(/^-|-$/g, '');
            }
            
            // Function to sanitize combined input while preserving custom prefix
            function sanitizeCombinedSlug(value) {
                if (!value) return '';
                
                var parsed = parseCombinedSlug(value);
                var sanitizedSlug = sanitizeSlugPart(parsed.slug);
                
                // Allow completely custom format - no restrictions
                if (parsed.prefix === '') {
                    return sanitizedSlug;
                }
                
                // Sanitize prefix part too (allow letters, numbers, hyphens)
                var sanitizedPrefix = parsed.prefix
                    .toLowerCase()
                    .replace(/[^a-z0-9\-]/g, '-')
                    .replace(/-+/g, '-')
                    .replace(/^-|-$/g, '');
                
                return sanitizedPrefix + '/' + sanitizedSlug;
            }
            
            // Function to update preview URL
            function updatePreview() {
                var combinedValue = $slugField.val().trim();
                console.log('updatePreview called with:', combinedValue);
                
                if (combinedValue) {
                    var fullUrl = homeUrl + combinedValue + '/';
                    console.log('Setting preview URL to:', fullUrl);
                    $previewField.val(fullUrl);
                    $testBtn.show().attr('data-url', fullUrl);
                    $copyBtn.show().attr('data-url', fullUrl);
                    console.log('Buttons should now be visible with data-url:', fullUrl);
                    
                    // Extract slug for validation
                    var parsed = parseCombinedSlug(combinedValue);
                    if (parsed.slug) {
                        validateSlug(parsed.slug);
                    }
                } else {
                    console.log('No combined value, hiding buttons');
                    $previewField.val('');
                    $testBtn.hide();
                    $copyBtn.hide();
                    $validation.html('');
                }
            }
            
            // Function to validate slug
            function validateSlug(slug) {
                console.log('AffiliateHub DEBUG - validateSlug called with:', slug);
                $validation.html('<span class="validation-message validation-checking">Checking availability...</span>');
                
                $.ajax({
                    url: ajaxurl,
                    type: 'POST',
                    data: {
                        action: 'check_affiliate_slug',
                        slug: slug,
                        post_id: currentPostId,
                        nonce: '<?php echo esc_attr(wp_create_nonce('affiliate_hub_admin_nonce')); ?>'
                    },
                    success: function(response) {
                        console.log('AffiliateHub DEBUG - AJAX response:', response);
                        if (response.success) {
                            console.log('AffiliateHub DEBUG - Available:', response.data.available);
                            console.log('AffiliateHub DEBUG - Conflicts:', response.data.conflicts);
                            if (response.data.available) {
                                $validation.html('<span class="validation-message validation-available">✓ Slug is available</span>');
                            } else {
                                var message = '<span class="validation-message validation-unavailable">✗ Slug conflicts detected:</span>';
                                if (response.data.conflicts && response.data.conflicts.length > 0) {
                                    message += '<ul style="margin: 5px 0; padding-left: 20px;">';
                                    response.data.conflicts.forEach(function(conflict) {
                                        message += '<li style="color: #d63638;">' + conflict + '</li>';
                                    });
                                    message += '</ul>';
                                }
                                message += '<span class="validation-message validation-suggestion">Suggestion: ' + response.data.suggested_slug + '</span>';
                                $validation.html(message);
                            }
                        } else {
                            console.log('AffiliateHub DEBUG - AJAX error:', response);
                            $validation.html('<span class="validation-message validation-unavailable">Error checking availability</span>');
                        }
                    },
                    error: function(xhr, status, error) {
                        console.log('AffiliateHub DEBUG - AJAX call failed:', status, error);
                        $validation.html('<span class="validation-message validation-unavailable">Connection error</span>');
                    }
                });
            }
            
            // Generate slug from title button
            $generateBtn.on('click', function() {
                var title = $titleField.val().trim();
                if (title) {
                    var generatedSlug = generateSlugFromTitle(title);
                    $slugField.val(generatedSlug);
                    updatePreview();
                } else {
                    alert('<?php echo \esc_js(\__('Please enter a title first.', 'affiliate-hub')); ?>');
                }
            });
            
            // Auto-sanitize input - allow full freedom while maintaining URL safety
            $slugField.on('input', function() {
                var currentValue = $(this).val();
                var sanitized = sanitizeCombinedSlug(currentValue);
                
                // Only apply sanitization if user pasted invalid content
                // Allow free typing without interference
                if (sanitized !== currentValue && 
                    (currentValue.indexOf(' ') !== -1 || 
                     currentValue.match(/[^a-zA-Z0-9\-\/]/))) {
                    var currentCursor = this.selectionStart;
                    $(this).val(sanitized);
                    this.setSelectionRange(currentCursor, currentCursor);
                }
                updatePreview();
            });
            
            // Initialize field with default PREFIX/ only if completely empty (new post)
            if (!$slugField.val() && window.location.href.indexOf('post-new.php') !== -1) {
                if (linkPrefix) {
                    $slugField.val(linkPrefix + '/');
                } else {
                    $slugField.val(''); // No prefix configured
                }
            }
            
            // Force update after a short delay to ensure all elements are loaded
            setTimeout(function() {
                console.log('Force updating preview after delay...');
                updatePreview();
            }, 100);
        });
        </script>
        <?php
    }
    
    /**
     * Link options meta box
     */
    public function link_options_meta_box($post) {
        $affiliate_link = new \AffiliateHub\Models\AffiliateLink($post->ID);
        $redirect_type = $affiliate_link->get_redirect_type();
        $is_nofollow = $affiliate_link->is_nofollow();
        $is_sponsored = $affiliate_link->is_sponsored();
        $is_tracking = $affiliate_link->is_tracking();
        $is_new_window = $affiliate_link->is_new_window();
        $redirect_types = Constants::get_redirect_types();
        ?>
        <p>
            <label for="redirect_type"><strong><?php esc_html_e('Redirect Type', 'affiliate-hub'); ?></strong></label><br>
            <select id="redirect_type" name="redirect_type" style="width: 100%;">
                <?php foreach ($redirect_types as $type => $label): ?>
                    <option value="<?php echo esc_attr($type); ?>" <?php selected($redirect_type, $type); ?>>
                        <?php echo esc_html($label); ?>
                    </option>
                <?php endforeach; ?>
            </select>
        </p>
        
        <p>
            <label>
                <input type="checkbox" name="nofollow" value="1" <?php checked($is_nofollow); ?> />
                <?php esc_html_e('Add rel="nofollow"', 'affiliate-hub'); ?>
            </label>
        </p>
        
        <p>
            <label>
                <input type="checkbox" name="sponsored" value="1" <?php checked($is_sponsored); ?> />
                <?php esc_html_e('Add rel="sponsored"', 'affiliate-hub'); ?>
            </label>
        </p>
        
        <p>
            <label>
                <input type="checkbox" name="tracking" value="1" <?php checked($is_tracking); ?> />
                <?php esc_html_e('Add rel="ugc" (User Generated Content/Tracking)', 'affiliate-hub'); ?>
            </label>
        </p>
        
        <p>
            <label>
                <input type="checkbox" name="new_window" value="1" <?php checked($is_new_window); ?> />
                <?php esc_html_e('Open in new window', 'affiliate-hub'); ?>
            </label>
        </p>
        <?php
    }
    
    /**
     * Link analytics meta box
     */
    public function link_stats_meta_box($post) {
        $affiliate_link = new \AffiliateHub\Models\AffiliateLink($post->ID);
        $click_count = $affiliate_link->get_click_count();
        $last_clicked = $affiliate_link->get_last_clicked();
        ?>
        <div class="affiliate-hub-stats">
            <p>
                <strong><?php esc_html_e('Total Clicks:', 'affiliate-hub'); ?></strong>
                <span class="click-count"><?php echo esc_html($click_count); ?></span>
            </p>
            
            <?php if ($last_clicked): ?>
                <p>
                    <strong><?php esc_html_e('Last Clicked:', 'affiliate-hub'); ?></strong><br>
                    <?php echo esc_html(date_i18n(get_option('date_format') . ' ' . get_option('time_format'), strtotime($last_clicked))); ?>
                </p>
            <?php else: ?>
                <p>
                    <em><?php esc_html_e('No clicks yet', 'affiliate-hub'); ?></em>
                </p>
            <?php endif; ?>
            
            <?php if ($post->post_status === 'publish'): ?>
                <p>
                    <?php 
                    $stats_url = add_query_arg(
                        array(
                            'page'    => 'affiliate-hub-analytics',
                            'link_id' => absint($post->ID),
                        ),
                        admin_url('admin.php')
                    );
                    ?>
                    <a href="<?php echo esc_url($stats_url); ?>" class="button button-secondary">
                        <?php esc_html_e('View Detailed Stats', 'affiliate-hub'); ?>
                    </a>
                </p>
            <?php endif; ?>
        </div>
        
        <style>
        .affiliate-hub-stats .click-count {
            font-size: 18px;
            font-weight: bold;
            color: #0073aa;
        }
        </style>
        <?php
    }
    
    /**
     * Save meta box data
     */
    public function save_meta_boxes($post_id) {
        // Check if this is an autosave
        if (defined('DOING_AUTOSAVE') && DOING_AUTOSAVE) {
            return;
        }
        
        // Check the post type
        if (get_post_type($post_id) !== Constants::POST_TYPE_AFFILIATE_LINK) {
            return;
        }
        
        // Check user permissions
        if (!current_user_can('edit_post', $post_id)) {
            return;
        }
        
        // Verify nonce
        if (!isset($_POST['affiliate_hub_meta_box_nonce']) || 
            !wp_verify_nonce($_POST['affiliate_hub_meta_box_nonce'], 'affiliate_hub_meta_box')) {
            return;
        }
        
        // Prevent infinite recursion
        static $saved_posts = array();
        if (in_array($post_id, $saved_posts)) {
            return;
        }
        $saved_posts[] = $post_id;
        
        $affiliate_link = new \AffiliateHub\Models\AffiliateLink($post_id);
        
        // Save custom slug if provided
        if (isset($_POST['link_slug'])) {
            $input_slug = sanitize_text_field($_POST['link_slug']);
            
            if (!empty($input_slug)) {
                // Parse the combined input to extract prefix and slug parts
                $last_slash_pos = strrpos($input_slug, '/');
                
                if ($last_slash_pos !== false) {
                    // Has prefix/slug format
                    $custom_prefix = substr($input_slug, 0, $last_slash_pos);
                    $actual_slug = substr($input_slug, $last_slash_pos + 1);
                } else {
                    // No slash, treat entire input as slug (no prefix)
                    $custom_prefix = '';
                    $actual_slug = $input_slug;
                }
                
                // Sanitize the actual slug part
                $actual_slug = sanitize_title($actual_slug);
                
                if (!empty($actual_slug)) {
                    // Save the full slug path for complex routing
                    update_post_meta($post_id, '_affiliate_full_slug', $input_slug);
                    
                    // Save the custom prefix as meta field (can be different from global setting)
                    update_post_meta($post_id, '_affiliate_custom_prefix', $custom_prefix);
                    
                    // Force rewrite rules regeneration to make new link work immediately
                    if (class_exists('AffiliateHub\\Core\\Plugin')) {
                        $plugin_instance = \AffiliateHub\Core\Plugin::get_instance();
                        $plugin_instance->regenerate_affiliate_rewrite_rules();
                    }
                    
                    // Remove this hook temporarily to prevent recursion
                    remove_action('save_post', array($this, 'save_meta_boxes'));
                    
                    // Update post slug with just the last segment for WordPress compatibility
                    wp_update_post(array(
                        'ID' => $post_id,
                        'post_name' => $actual_slug
                    ));
                    
                    // Re-add the hook
                    add_action('save_post', array($this, 'save_meta_boxes'));
                    
                    // CRITICAL: Flush rewrite rules to make new/updated link immediately accessible
                    flush_rewrite_rules();
                    
                } else {
                    // Remove custom prefix and full slug if slug is empty
                    delete_post_meta($post_id, '_affiliate_custom_prefix');
                    delete_post_meta($post_id, '_affiliate_full_slug');
                }
            } else {
                // Remove custom prefix and full slug if input is empty
                delete_post_meta($post_id, '_affiliate_custom_prefix');
                delete_post_meta($post_id, '_affiliate_full_slug');
            }
        }
        
        // Save destination URL
        if (isset($_POST['destination_url'])) {
            $destination_url = esc_url_raw($_POST['destination_url']);
            if (filter_var($destination_url, FILTER_VALIDATE_URL)) {
                $affiliate_link->set_destination_url($destination_url);
            }
        }
        
        // Save redirect type
        if (isset($_POST['redirect_type'])) {
            $redirect_type = sanitize_text_field($_POST['redirect_type']);
            $affiliate_link->set_redirect_type($redirect_type);
        }
        
        // Save nofollow setting
        $nofollow = isset($_POST['nofollow']) ? true : false;
        $affiliate_link->set_nofollow($nofollow);
        
        // Save sponsored setting
        $sponsored = isset($_POST['sponsored']) ? true : false;
        $affiliate_link->set_sponsored($sponsored);
        
        // Save tracking setting
        $tracking = isset($_POST['tracking']) ? true : false;
        $affiliate_link->set_tracking($tracking);
        
        // Save new window setting
        $new_window = isset($_POST['new_window']) ? true : false;
        $affiliate_link->set_new_window($new_window);
        
        // Save Keywords Autolinker settings if module is enabled
        if (get_option(Constants::OPTION_AUTOLINKER_ENABLED, false)) {
            $this->save_autolinker_keywords($post_id);
        }
        
        // Remove post from saved array to allow future saves
        if (($key = array_search($post_id, $saved_posts)) !== false) {
            unset($saved_posts[$key]);
        }
    }
    
    /**
     * Autolinker keywords meta box
     */
    public function autolinker_keywords_meta_box($post) {
        // Get existing keywords
        $keywords = get_post_meta($post->ID, Constants::META_AUTOLINK_KEYWORDS, true);
        $keyword_limit = get_post_meta($post->ID, Constants::META_AUTOLINK_KEYWORD_LIMIT, true);
        
        // Default values
        $keywords = !empty($keywords) ? $keywords : array();
        $keyword_limit = !empty($keyword_limit) ? intval($keyword_limit) : 0;
        
        ?>
        <div class="affiliate-hub-autolinker-keywords">
            <p class="description">
                <?php esc_html_e('Enter a list of keywords you wish to automatically link with this affiliate link (case-insensitive).', 'affiliate-hub'); ?>
            </p>
            
            <table class="form-table">
                <tr>
                    <th scope="row">
                        <label for="autolink_keywords"><?php esc_html_e('Keywords', 'affiliate-hub'); ?></label>
                    </th>
                    <td>
                        <div id="autolink-keywords-container">
                            <?php if (!empty($keywords)): ?>
                                <?php foreach ($keywords as $index => $keyword): ?>
                                    <div class="keyword-tag">
                                        <input type="text" name="autolink_keywords[]" value="<?php echo esc_attr($keyword); ?>" placeholder="<?php esc_attr_e('Enter keyword', 'affiliate-hub'); ?>" />
                                        <button type="button" class="button remove-keyword">×</button>
                                    </div>
                                <?php endforeach; ?>
                            <?php else: ?>
                                <div class="keyword-tag">
                                    <input type="text" name="autolink_keywords[]" value="" placeholder="<?php esc_attr_e('Enter keyword', 'affiliate-hub'); ?>" />
                                    <button type="button" class="button remove-keyword">×</button>
                                </div>
                            <?php endif; ?>
                        </div>
                        <p>
                            <button type="button" id="add-keyword" class="button"><?php esc_html_e('Add Keyword', 'affiliate-hub'); ?></button>
                        </p>
                        <p class="description">
                            <?php esc_html_e('Note: Place your keywords in order of precedence. Type your entries rather than copy/pasting to eliminate encoding issues.', 'affiliate-hub'); ?>
                        </p>
                    </td>
                </tr>
                <tr>
                    <th scope="row">
                        <label for="autolink_keyword_limit"><?php esc_html_e('Limit (per keyword)', 'affiliate-hub'); ?></label>
                    </th>
                    <td>
                        <input type="number" id="autolink_keyword_limit" name="autolink_keyword_limit" value="<?php echo esc_attr($keyword_limit); ?>" min="0" max="100" class="small-text" />
                        <p class="description">
                            <?php esc_html_e('Maximum number of links per keyword (0 = use global setting). If you have 2 keywords and set limit to 3, you\'ll have maximum 6 links total.', 'affiliate-hub'); ?>
                        </p>
                    </td>
                </tr>
            </table>
        </div>
        <?php
    }
    
    /**
     * Save autolinker keywords
     *
     * @param int $post_id Post ID
     */
    private function save_autolinker_keywords($post_id) {
        // Save keywords
        if (isset($_POST['autolink_keywords']) && is_array($_POST['autolink_keywords'])) {
            $keywords = array();
            foreach ($_POST['autolink_keywords'] as $keyword) {
                $keyword = trim(sanitize_text_field($keyword));
                if (!empty($keyword)) {
                    $keywords[] = $keyword;
                }
            }
            update_post_meta($post_id, Constants::META_AUTOLINK_KEYWORDS, $keywords);
        } else {
            delete_post_meta($post_id, Constants::META_AUTOLINK_KEYWORDS);
        }
        
        // Save keyword limit
        if (isset($_POST['autolink_keyword_limit'])) {
            $keyword_limit = intval($_POST['autolink_keyword_limit']);
            if ($keyword_limit > 0) {
                update_post_meta($post_id, Constants::META_AUTOLINK_KEYWORD_LIMIT, $keyword_limit);
            } else {
                delete_post_meta($post_id, Constants::META_AUTOLINK_KEYWORD_LIMIT);
            }
        }
        
        // Update cache for this link
        if (class_exists('\\AffiliateHub\\Modules\\KeywordsAutolinker\\Cache')) {
            $cache = new \AffiliateHub\Modules\KeywordsAutolinker\Cache();
            $cache->update_link_cache($post_id);
        }
    }
}
