<?php
/**
 * Settings Page
 *
 * @package AffiliateHub
 * @subpackage Admin
 */

namespace AffiliateHub\Admin;

use AffiliateHub\Core\Constants;

/**
 * Settings class
 */
class Settings {
    
    /**
     * Constructor
     */
    public function __construct() {
        add_action('admin_init', array($this, 'init_settings'));
        add_action('wp_ajax_affiliate_hub_maxmind_download', array($this, 'ajax_maxmind_download'));
    }

    /**
     * Link Scanner: disable animations field callback
     */
    public function link_scanner_disable_animations_callback() {
        $value = get_option(Constants::OPTION_LINK_SCANNER_DISABLE_ANIMATIONS, false);
        $field_id = Constants::OPTION_LINK_SCANNER_DISABLE_ANIMATIONS;
        echo '<input type="checkbox" id="' . esc_attr($field_id) . '" name="' . esc_attr($field_id) . '" value="1"' . checked($value, true, false) . ' />';
        echo '<label for="' . esc_attr($field_id) . '">' . esc_html__('Disable ApexCharts animations for the Link Scanner (reduces CPU during scans)', 'affiliate-hub') . '</label>';
    }

    /**
     * Link Scanner: debounce ms field callback
     */
    public function link_scanner_debounce_ms_callback() {
        $value = get_option(Constants::OPTION_LINK_SCANNER_DEBOUNCE_MS, 200);
        $field_id = Constants::OPTION_LINK_SCANNER_DEBOUNCE_MS;
        echo '<input type="number" id="' . esc_attr($field_id) . '" name="' . esc_attr($field_id) . '" min="100" max="10000" value="' . esc_attr($value) . '" class="small-text" />';
        echo '<p class="description">' . esc_html__('Debounce interval in milliseconds between UI update polls. Increase to reduce CPU/load during large scans.', 'affiliate-hub') . '</p>';
    }

    /**
     * Link Scanner: whitelist field callback
     */
    public function link_scanner_whitelist_callback() {
        $value = get_option(Constants::OPTION_LINK_SCANNER_WHITELIST, '');
        $field_id = Constants::OPTION_LINK_SCANNER_WHITELIST;
        echo '<input type="text" id="' . esc_attr($field_id) . '" name="' . esc_attr($field_id) . '" value="' . esc_attr($value) . '" class="regular-text" placeholder="amazon.com, ebay.com, allegro.pl" />';
        echo '<p class="description">' . esc_html__('Comma-separated list of allowed domains. If filled, ONLY links from these domains will be scanned.', 'affiliate-hub') . '</p>';
        echo '<p class="description"><strong>' . esc_html__('Example:', 'affiliate-hub') . '</strong> amazon.com, ebay.com, allegro.pl</p>';
        echo '<p class="description"><em>' . esc_html__('Leave empty to scan all domains (except those in blacklist).', 'affiliate-hub') . '</em></p>';
    }

    /**
     * Link Scanner: blacklist field callback
     */
    public function link_scanner_blacklist_callback() {
        $value = get_option(Constants::OPTION_LINK_SCANNER_BLACKLIST, '');
        $field_id = Constants::OPTION_LINK_SCANNER_BLACKLIST;
        echo '<input type="text" id="' . esc_attr($field_id) . '" name="' . esc_attr($field_id) . '" value="' . esc_attr($value) . '" class="regular-text" placeholder="facebook, google, localhost" />';
        echo '<p class="description">' . esc_html__('Comma-separated list of domain fragments to exclude from scanning. Links containing these fragments will be skipped.', 'affiliate-hub') . '</p>';
        echo '<p class="description"><strong>' . esc_html__('Example:', 'affiliate-hub') . '</strong> facebook, google, localhost, 127.0.0.1</p>';
        echo '<p class="description"><em>' . esc_html__('Use for excluding social media, internal links, or problematic domains.', 'affiliate-hub') . '</em></p>';
    }

    /**
     * Link Scanner: rate limit field callback
     */
    public function link_scanner_rate_callback() {
        $value = get_option(Constants::OPTION_LINK_SCANNER_RATE, 5);
        $field_id = Constants::OPTION_LINK_SCANNER_RATE;
        echo '<input type="number" id="' . esc_attr($field_id) . '" name="' . esc_attr($field_id) . '" min="1" max="20" value="' . esc_attr($value) . '" class="small-text" />';
        echo '<p class="description">' . esc_html__('Maximum number of HTTP requests per second during scanning.', 'affiliate-hub') . '</p>';
        echo '<p class="description"><strong>' . esc_html__('Recommended:', 'affiliate-hub') . '</strong> ' . esc_html__('1-5 for gentle scanning, 10-15 for faster scanning', 'affiliate-hub') . '</p>';
        echo '<p class="description"><em>' . esc_html__('Lower values prevent being blocked by target servers but scan slower.', 'affiliate-hub') . '</em></p>';
    }
    
    /**
     * Initialize settings
     */
    public function init_settings() {
        // Register settings group and individual settings
        register_setting(
            'affiliate_hub_settings',
            Constants::OPTION_LINK_PREFIX,
            array(
                'sanitize_callback' => 'sanitize_title',
                'default' => 'go'
            )
        );
        
        register_setting(
            'affiliate_hub_settings',
            Constants::OPTION_REDIRECT_TYPE,
            array(
                'sanitize_callback' => array($this, 'sanitize_redirect_type'),
                'default' => Constants::REDIRECT_301
            )
        );
        
        \register_setting(
            'affiliate_hub_settings',
            Constants::OPTION_ENABLE_STATS,
            array(
                'sanitize_callback' => array($this, 'sanitize_checkbox'),
                'default' => true
            )
        );
        
        \register_setting(
            'affiliate_hub_settings',
            Constants::OPTION_ENABLE_FAST_REDIRECT,
            array(
                'sanitize_callback' => array($this, 'sanitize_checkbox'),
                'default' => true
            )
        );
        
        \register_setting(
            'affiliate_hub_settings',
            Constants::OPTION_NOFOLLOW_LINKS,
            array(
                'sanitize_callback' => array($this, 'sanitize_checkbox'),
                'default' => true
            )
        );
        
        register_setting(
            'affiliate_hub_settings',
            Constants::OPTION_SPONSORED_LINKS,
            array(
                'sanitize_callback' => array($this, 'sanitize_checkbox'),
                'default' => false
            )
        );
        
        register_setting(
            'affiliate_hub_settings',
            Constants::OPTION_TRACKING_LINKS,
            array(
                'sanitize_callback' => array($this, 'sanitize_checkbox'),
                'default' => false
            )
        );
        
        register_setting(
            'affiliate_hub_settings',
            Constants::OPTION_OPEN_NEW_WINDOW,
            array(
                'sanitize_callback' => array($this, 'sanitize_checkbox'),
                'default' => false
            )
        );
        
        // GeoLocation settings
        register_setting(
            'affiliate_hub_settings',
            Constants::OPTION_ENABLE_GEOLOCATION,
            array(
                'sanitize_callback' => array($this, 'sanitize_checkbox'),
                'default' => true
            )
        );
        
        register_setting(
            'affiliate_hub_settings',
            Constants::OPTION_ANONYMIZE_IP,
            array(
                'sanitize_callback' => array($this, 'sanitize_checkbox'),
                'default' => false
            )
        );
        
        register_setting(
            'affiliate_hub_settings',
            'affiliate_hub_geo_primary_provider',
            array(
                'sanitize_callback' => 'sanitize_text_field',
                'default' => 'ipapi_co'
            )
        );
        
        register_setting(
            'affiliate_hub_settings',
            Constants::OPTION_FALLBACK_COUNTRY,
            array(
                'sanitize_callback' => 'sanitize_text_field',
                'default' => 'Poland'
            )
        );
        
        register_setting(
            'affiliate_hub_settings',
            Constants::OPTION_FALLBACK_CITY,
            array(
                'sanitize_callback' => 'sanitize_text_field',
                'default' => 'Warsaw'
            )
        );
        
        // MaxMind license key
        register_setting(
            'affiliate_hub_settings',
            'affiliate_hub_maxmind_license_key',
            array(
                'sanitize_callback' => 'sanitize_text_field',
                'default' => ''
            )
        );
        
        // Add settings sections
        add_settings_section(
            'affiliate_hub_general',
            __('General Settings', 'affiliate-hub'),
            array($this, 'general_section_callback'),
            'affiliate_hub_settings'
        );
        
        add_settings_section(
            'affiliate_hub_links',
            __('Link Settings', 'affiliate-hub'),
            array($this, 'links_section_callback'),
            'affiliate_hub_settings'
        );
        
        \add_settings_section(
            'affiliate_hub_stats',
            \__('Statistics Settings', 'affiliate-hub'),
            array($this, 'stats_section_callback'),
            'affiliate_hub_settings'
        );
        
        \add_settings_section(
            'affiliate_hub_performance',
            \__('Performance Settings', 'affiliate-hub'),
            array($this, 'performance_section_callback'),
            'affiliate_hub_settings'
        );
        
        \add_settings_section(
            'affiliate_hub_link_scanner',
            __('Link Scanner Settings', 'affiliate-hub'),
            array($this, 'link_scanner_section_callback'),
            'affiliate_hub_settings'
        );
        
        \add_settings_section(
            'affiliate_hub_geolocation',
            __('GeoLocation Settings', 'affiliate-hub'),
            array($this, 'geolocation_section_callback'),
            'affiliate_hub_settings'
        );
        
        // Add settings fields
        $this->add_settings_fields();
    }
    
    /**
     * Add settings fields
     */
    private function add_settings_fields() {
        // Link prefix field
        add_settings_field(
            'link_prefix',
            __('Link Prefix', 'affiliate-hub'),
            array($this, 'link_prefix_callback'),
            'affiliate_hub_settings',
            'affiliate_hub_general'
        );
        
        // Redirect type field
        add_settings_field(
            'redirect_type',
            __('Default Redirect Type', 'affiliate-hub'),
            array($this, 'redirect_type_callback'),
            'affiliate_hub_settings',
            'affiliate_hub_links'
        );
        
        // Nofollow field
        add_settings_field(
            'nofollow_links',
            __('Add nofollow to links', 'affiliate-hub'),
            array($this, 'nofollow_callback'),
            'affiliate_hub_settings',
            'affiliate_hub_links'
        );
        
        // Fast redirect field
        \add_settings_field(
            'enable_fast_redirect',
            \__('Enable Ultra-Fast Redirects', 'affiliate-hub'),
            array($this, 'fast_redirect_callback'),
            'affiliate_hub_settings',
            'affiliate_hub_performance'
        );
        
        // Sponsored field
        \add_settings_field(
            'sponsored_links',
            \__('Add sponsored to links', 'affiliate-hub'),
            array($this, 'sponsored_callback'),
            'affiliate_hub_settings',
            'affiliate_hub_links'
        );
        
        // Tracking field
        add_settings_field(
            'tracking_links',
            __('Add tracking to links', 'affiliate-hub'),
            array($this, 'tracking_callback'),
            'affiliate_hub_settings',
            'affiliate_hub_links'
        );
        
        // New window field
        add_settings_field(
            'open_new_window',
            __('Open links in new window', 'affiliate-hub'),
            array($this, 'new_window_callback'),
            'affiliate_hub_settings',
            'affiliate_hub_links'
        );
        
        // Enable stats field
        add_settings_field(
            'enable_stats',
            __('Enable Click Statistics', 'affiliate-hub'),
            array($this, 'enable_stats_callback'),
            'affiliate_hub_settings',
            'affiliate_hub_stats'
        );

        // Link Scanner visualization: disable animations
        register_setting(
            'affiliate_hub_settings',
            Constants::OPTION_LINK_SCANNER_DISABLE_ANIMATIONS,
            array(
                'sanitize_callback' => array($this, 'sanitize_checkbox'),
                'default' => false
            )
        );
        add_settings_field(
            'link_scanner_disable_animations',
            __('Disable Link Scanner Chart Animations', 'affiliate-hub'),
            array($this, 'link_scanner_disable_animations_callback'),
            'affiliate_hub_settings',
            'affiliate_hub_stats'
        );

        // Link Scanner visualization: debounce ms
        register_setting(
            'affiliate_hub_settings',
            Constants::OPTION_LINK_SCANNER_DEBOUNCE_MS,
            array(
                'sanitize_callback' => 'absint',
                'default' => 200
            )
        );
        
        // Link Scanner: whitelist
        register_setting(
            'affiliate_hub_settings',
            Constants::OPTION_LINK_SCANNER_WHITELIST,
            array(
                'sanitize_callback' => 'sanitize_text_field',
                'default' => ''
            )
        );
        
        // Link Scanner: blacklist
        register_setting(
            'affiliate_hub_settings',
            Constants::OPTION_LINK_SCANNER_BLACKLIST,
            array(
                'sanitize_callback' => 'sanitize_text_field',
                'default' => ''
            )
        );
        
        // Link Scanner: rate limit
        register_setting(
            'affiliate_hub_settings',
            Constants::OPTION_LINK_SCANNER_RATE,
            array(
                'sanitize_callback' => 'absint',
                'default' => 5
            )
        );
        
        // Link Scanner: disable animations
        register_setting(
            'affiliate_hub_settings',
            Constants::OPTION_LINK_SCANNER_DISABLE_ANIMATIONS,
            array(
                'sanitize_callback' => function($value) { return !empty($value) ? 1 : 0; },
                'default' => false
            )
        );
        
        add_settings_field(
            'link_scanner_whitelist',
            __('Whitelist (Allowed Domains)', 'affiliate-hub'),
            array($this, 'link_scanner_whitelist_callback'),
            'affiliate_hub_settings',
            'affiliate_hub_link_scanner'
        );
        
        add_settings_field(
            'link_scanner_blacklist',
            __('Blacklist (Excluded Domains)', 'affiliate-hub'),
            array($this, 'link_scanner_blacklist_callback'),
            'affiliate_hub_settings',
            'affiliate_hub_link_scanner'
        );
        
        add_settings_field(
            'link_scanner_rate',
            __('Scan Rate Limit (requests/second)', 'affiliate-hub'),
            array($this, 'link_scanner_rate_callback'),
            'affiliate_hub_settings',
            'affiliate_hub_link_scanner'
        );
        
        add_settings_field(
            'link_scanner_disable_animations',
            __('Chart Performance', 'affiliate-hub'),
            array($this, 'link_scanner_disable_animations_callback'),
            'affiliate_hub_settings',
            'affiliate_hub_link_scanner'
        );
        
        add_settings_field(
            'link_scanner_debounce_ms',
            __('UI Update Interval (ms)', 'affiliate-hub'),
            array($this, 'link_scanner_debounce_ms_callback'),
            'affiliate_hub_settings',
            'affiliate_hub_link_scanner'
        );
        
        // GeoLocation Settings
        add_settings_field(
            'enable_geolocation',
            __('Enable Geolocation', 'affiliate-hub'),
            array($this, 'enable_geolocation_callback'),
            'affiliate_hub_settings',
            'affiliate_hub_geolocation'
        );
        
        add_settings_field(
            'anonymize_ip',
            __('Anonymize IP Addresses (GDPR)', 'affiliate-hub'),
            array($this, 'anonymize_ip_callback'),
            'affiliate_hub_settings',
            'affiliate_hub_geolocation'
        );
        
        add_settings_field(
            'geo_primary_provider',
            __('Primary API Provider', 'affiliate-hub'),
            array($this, 'geo_primary_provider_callback'),
            'affiliate_hub_settings',
            'affiliate_hub_geolocation'
        );
        
        add_settings_field(
            'fallback_location',
            __('Fallback Location', 'affiliate-hub'),
            array($this, 'fallback_location_callback'),
            'affiliate_hub_settings',
            'affiliate_hub_geolocation'
        );
        
        add_settings_field(
            'maxmind_license_key',
            __('MaxMind License Key', 'affiliate-hub'),
            array($this, 'maxmind_license_key_callback'),
            'affiliate_hub_settings',
            'affiliate_hub_geolocation'
        );
        
        add_settings_field(
            'maxmind_status',
            __('MaxMind Status', 'affiliate-hub'),
            array($this, 'maxmind_status_callback'),
            'affiliate_hub_settings',
            'affiliate_hub_geolocation'
        );
        
        add_settings_field(
            'geo_test_tool',
            __('Test Geolocation', 'affiliate-hub'),
            array($this, 'geo_test_tool_callback'),
            'affiliate_hub_settings',
            'affiliate_hub_geolocation'
        );
    }
    
    /**
     * General section callback
     */
    public function general_section_callback() {
        echo '<p>' . esc_html__('Configure general plugin settings.', 'affiliate-hub') . '</p>';
    }
    
    /**
     * Links section callback
     */
    public function links_section_callback() {
        echo '<p>' . esc_html__('Configure default link behavior.', 'affiliate-hub') . '</p>';
    }
    
    /**
     * Stats section callback
     */
    public function stats_section_callback() {
        echo '<p>' . \esc_html__('Configure statistics tracking.', 'affiliate-hub') . '</p>';
    }
    
    /**
     * Performance section callback
     */
    public function performance_section_callback() {
        echo '<p>' . \esc_html__('Configure performance optimization settings for faster redirects.', 'affiliate-hub') . '</p>';
    }
    
    /**
     * Link Scanner section callback
     */
    public function link_scanner_section_callback() {
        echo '<p>' . \esc_html__('Configure Link Scanner behavior and performance settings.', 'affiliate-hub') . '</p>';
    }
    
    /**
     * GeoLocation section callback
     */
    public function geolocation_section_callback() {
        echo '<p>' . __('Configure IP geolocation for visitor tracking and analytics. This helps identify where your affiliate link clicks are coming from.', 'affiliate-hub') . '</p>';
        echo '<div class="notice notice-info inline">';
        echo '<p><strong>' . __('Recommended Setup:', 'affiliate-hub') . '</strong> ' . __('Use MaxMind GeoLite2 database for fastest and most accurate results, with API providers as fallback.', 'affiliate-hub') . '</p>';
        echo '<p><strong>' . __('Features:', 'affiliate-hub') . '</strong></p>';
        echo '<ul>';
        echo '<li>• ' . __('MaxMind GeoLite2 local database (PRIMARY - fastest & most accurate)', 'affiliate-hub') . '</li>';
        echo '<li>• ' . __('Multi-level caching for fast lookups', 'affiliate-hub') . '</li>';
        echo '<li>• ' . __('Multiple API providers with automatic failover', 'affiliate-hub') . '</li>';
        echo '<li>• ' . __('Background processing to avoid slowing down clicks', 'affiliate-hub') . '</li>';
        echo '<li>• ' . __('GDPR-compliant IP anonymization option', 'affiliate-hub') . '</li>';
        echo '</ul>';
        echo '</div>';
    }
    
    /**
     * Link prefix field callback
     */
    public function link_prefix_callback() {
        $value = get_option(Constants::OPTION_LINK_PREFIX, 'go');
        $field_id = Constants::OPTION_LINK_PREFIX;
        $base_url = home_url('/');
        
    echo '<input type="text" id="' . esc_attr($field_id) . '" name="' . esc_attr($field_id) . '" value="' . esc_attr($value) . '" class="regular-text" placeholder="go" />';
        
    echo '<p class="description">' . esc_html__('The prefix for your affiliate links. Links will look like:', 'affiliate-hub') . '</p>';
        echo '<p class="description" id="link-preview-container">';
        echo '<code id="link-preview">' . esc_html($base_url) . '<strong id="prefix-display">' . esc_html($value) . '</strong>/link-name/</code>';
        echo '</p>';
        
        // Add JavaScript for dynamic preview
        echo '<script type="text/javascript">
        document.addEventListener("DOMContentLoaded", function() {
            var prefixInput = document.getElementById("' . esc_js($field_id) . '");
            var prefixDisplay = document.getElementById("prefix-display");
            var linkPreview = document.getElementById("link-preview");
            var baseUrl = "' . esc_js($base_url) . '";
            
            function updatePreview() {
                var prefix = prefixInput.value.trim();
                if (prefix === "") {
                    // No prefix - show direct link
                    linkPreview.innerHTML = baseUrl + "link-name/";
                } else {
                    // With prefix - show bold prefix
                    linkPreview.innerHTML = baseUrl + "<strong>" + prefix + "</strong>/link-name/";
                }
            }
            
            // Update on input change
            prefixInput.addEventListener("input", updatePreview);
            prefixInput.addEventListener("keyup", updatePreview);
            prefixInput.addEventListener("change", updatePreview);
            
            // Initial update
            updatePreview();
        });
        </script>';
    }
    
    /**
     * Redirect type field callback
     */
    public function redirect_type_callback() {
        $value = get_option(Constants::OPTION_REDIRECT_TYPE, Constants::REDIRECT_301);
        $types = Constants::get_redirect_types();
        $field_id = Constants::OPTION_REDIRECT_TYPE;
        
    echo '<select id="' . esc_attr($field_id) . '" name="' . esc_attr($field_id) . '">';
        foreach ($types as $type => $label) {
            echo '<option value="' . esc_attr($type) . '"' . selected($value, $type, false) . '>' . esc_html($label) . '</option>';
        }
        echo '</select>';
    echo '<p class="description">' . esc_html__('Default redirect type for new affiliate links.', 'affiliate-hub') . '</p>';
    }
    
    /**
     * Nofollow field callback
     */
    public function nofollow_callback() {
        $value = get_option(Constants::OPTION_NOFOLLOW_LINKS, true);
        $field_id = Constants::OPTION_NOFOLLOW_LINKS;
    echo '<input type="checkbox" id="' . esc_attr($field_id) . '" name="' . esc_attr($field_id) . '" value="1"' . checked($value, true, false) . ' />';
    echo '<label for="' . esc_attr($field_id) . '">' . esc_html__('Add rel="nofollow" to affiliate links by default', 'affiliate-hub') . '</label>';
    }
    
    /**
     * Sponsored field callback
     */
    public function sponsored_callback() {
        $value = get_option(Constants::OPTION_SPONSORED_LINKS, false);
        $field_id = Constants::OPTION_SPONSORED_LINKS;
    echo '<input type="checkbox" id="' . esc_attr($field_id) . '" name="' . esc_attr($field_id) . '" value="1"' . checked($value, true, false) . ' />';
    echo '<label for="' . esc_attr($field_id) . '">' . esc_html__('Add rel="sponsored" to affiliate links by default', 'affiliate-hub') . '</label>';
    }
    
    /**
     * Tracking field callback
     */
    public function tracking_callback() {
        $value = get_option(Constants::OPTION_TRACKING_LINKS, false);
        $field_id = Constants::OPTION_TRACKING_LINKS;
    echo '<input type="checkbox" id="' . esc_attr($field_id) . '" name="' . esc_attr($field_id) . '" value="1"' . checked($value, true, false) . ' />';
    echo '<label for="' . esc_attr($field_id) . '">' . esc_html__('Add rel="ugc" (User Generated Content/Tracking) to affiliate links by default', 'affiliate-hub') . '</label>';
    }
    
    /**
     * New window field callback
     */
    public function new_window_callback() {
        $value = get_option(Constants::OPTION_OPEN_NEW_WINDOW, false);
        $field_id = Constants::OPTION_OPEN_NEW_WINDOW;
    echo '<input type="checkbox" id="' . esc_attr($field_id) . '" name="' . esc_attr($field_id) . '" value="1"' . checked($value, true, false) . ' />';
    echo '<label for="' . esc_attr($field_id) . '">' . esc_html__('Open affiliate links in new window by default', 'affiliate-hub') . '</label>';
    }
    
    /**
     * Enable stats field callback
     */
    public function enable_stats_callback() {
        $value = \get_option(Constants::OPTION_ENABLE_STATS, true);
        $field_id = Constants::OPTION_ENABLE_STATS;
    echo '<input type="checkbox" id="' . \esc_attr($field_id) . '" name="' . \esc_attr($field_id) . '" value="1"' . \checked($value, true, false) . ' />';
    echo '<label for="' . \esc_attr($field_id) . '">' . \esc_html__('Track click statistics for affiliate links', 'affiliate-hub') . '</label>';
    }
    
    /**
     * Fast redirect callback
     */
    public function fast_redirect_callback() {
        $value = \get_option(Constants::OPTION_ENABLE_FAST_REDIRECT, true);
        $field_id = Constants::OPTION_ENABLE_FAST_REDIRECT;
        echo '<input type="checkbox" id="' . \esc_attr($field_id) . '" name="' . \esc_attr($field_id) . '" value="1"' . \checked($value, true, false) . ' />';
        echo '<label for="' . \esc_attr($field_id) . '">' . \esc_html__('Enable ultra-fast redirects (5-15ms) - RECOMMENDED', 'affiliate-hub') . '</label>';
        echo '<p class="description">' . \esc_html__('Ultra-fast redirects with multi-layer tracking: JavaScript + AJAX + tracking pixel fallback. Works with all browsers, bots, and crawlers. Disable only if you experience conflicts with other plugins.', 'affiliate-hub') . '</p>';
    }
    
    /**
     * Sanitize redirect type
     */
    public function sanitize_redirect_type($value) {
        $valid_types = array_keys(Constants::get_redirect_types());
        return in_array($value, $valid_types) ? $value : Constants::REDIRECT_301;
    }
    
    /**
     * Sanitize checkbox values
     */
    public function sanitize_checkbox($value) {
        return !empty($value) ? true : false;
    }
    
    /**
     * Display settings page
     */
    public function display_settings_page() {
        // Check user capabilities
        if (!current_user_can('manage_options')) {
            return;
        }
        
        // Check if safe mode is active
        if (get_option('affiliate_hub_safe_mode', false)) {
            ?>
            <div class="wrap">
                <h1><?php echo esc_html(get_admin_page_title()); ?></h1>
                
                <div class="notice notice-warning">
                    <p>
                        <strong><?php esc_html_e('Safe Mode is Active', 'affiliate-hub'); ?></strong><br>
                        <?php esc_html_e('Settings are currently disabled because the plugin is running in safe mode. This usually happens due to memory constraints.', 'affiliate-hub'); ?>
                    </p>
                    <p>
                        <a href="<?php echo esc_url(add_query_arg('affiliate_hub_disable_safe_mode', 'true')); ?>" class="button button-primary">
                            <?php esc_html_e('Disable Safe Mode', 'affiliate-hub'); ?>
                        </a>
                    </p>
                </div>
            </div>
            <?php
            return;
        }
        
        // Check if settings were updated
        if (isset($_GET['settings-updated'])) {
            // Flush rewrite rules after saving
            flush_rewrite_rules();
            add_settings_error(
                'affiliate_hub_messages',
                'affiliate_hub_message',
                __('Settings saved successfully!', 'affiliate-hub'),
                'success'
            );
        }
        
        // Show error/update messages
        settings_errors('affiliate_hub_messages');
        ?>
        <div class="wrap">
            <h1><?php echo esc_html(get_admin_page_title()); ?></h1>
            
            <form method="post" action="options.php">
                <?php
                settings_fields('affiliate_hub_settings');
                do_settings_sections('affiliate_hub_settings');
                submit_button();
                ?>
            </form>
            
            <div class="affiliate-hub-settings-info">
                <h2><?php esc_html_e('Quick Start Guide', 'affiliate-hub'); ?></h2>
                <ol>
                    <li><?php esc_html_e('Configure your link prefix above (default: "go")', 'affiliate-hub'); ?></li>
                    <li><?php 
                        printf(
                            /* translators: %s: Link to "Affiliate Links > Add New" admin page */
                            esc_html__('Create your first affiliate link by going to %s', 'affiliate-hub'),
                            '<a href="' . esc_url(admin_url('post-new.php?post_type=' . Constants::POST_TYPE_AFFILIATE_LINK)) . '">Affiliate Links > Add New</a>'
                        );
                    ?></li>
                    <li><?php esc_html_e('Add the title and destination URL for your affiliate link', 'affiliate-hub'); ?></li>
                    <li><?php esc_html_e('Use the generated cloaked URL in your content', 'affiliate-hub'); ?></li>
                    <li><?php 
                        printf(
                            /* translators: %s: Link to Statistics admin page */
                            esc_html__('Monitor your link performance in %s', 'affiliate-hub'),
                            '<a href="' . esc_url(admin_url('admin.php?page=affiliate-hub-analytics')) . '">Analytics</a>'
                        );
                    ?></li>
                </ol>
            </div>
        </div>
        
        <style>
        .affiliate-hub-settings-info {
            background: #f9f9f9;
            border: 1px solid #e5e5e5;
            padding: 20px;
            margin-top: 20px;
            border-radius: 3px;
        }
        .affiliate-hub-settings-info h2 {
            margin-top: 0;
        }
        </style>
        <?php
    }
    
    /**
     * Enable Geolocation callback
     */
    public function enable_geolocation_callback() {
        $value = get_option(Constants::OPTION_ENABLE_GEOLOCATION, true);
        $field_id = Constants::OPTION_ENABLE_GEOLOCATION;
        echo '<input type="checkbox" id="' . esc_attr($field_id) . '" name="' . esc_attr($field_id) . '" value="1"' . checked($value, true, false) . ' />';
        echo '<label for="' . esc_attr($field_id) . '">' . __('Enable IP geolocation for visitor tracking', 'affiliate-hub') . '</label>';
        echo '<p class="description">' . __('Collect country, city, and other geographic data from visitor IP addresses.', 'affiliate-hub') . '</p>';
    }
    
    /**
     * Anonymize IP callback
     */
    public function anonymize_ip_callback() {
        $value = get_option(Constants::OPTION_ANONYMIZE_IP, false);
        $field_id = Constants::OPTION_ANONYMIZE_IP;
        echo '<input type="checkbox" id="' . esc_attr($field_id) . '" name="' . esc_attr($field_id) . '" value="1"' . checked($value, true, false) . ' />';
        echo '<label for="' . esc_attr($field_id) . '">' . __('Anonymize IP addresses (GDPR compliant)', 'affiliate-hub') . '</label>';
        echo '<p class="description">' . __('Removes last octet from IPv4 addresses and last 64 bits from IPv6 addresses before processing.', 'affiliate-hub') . '</p>';
    }
    
    /**
     * Primary provider callback
     */
    public function geo_primary_provider_callback() {
        $value = get_option('affiliate_hub_geo_primary_provider', 'ipapi_co');
        $field_id = 'affiliate_hub_geo_primary_provider';
        
        $providers = array(
            'ipapi_co' => 'IPApi.co (15k/month free, fast)',
            'ip_api' => 'IP-API.com (1k/min free)', 
            'ipinfo_io' => 'IPInfo.io (50k/month free)'
        );
        
        echo '<select id="' . esc_attr($field_id) . '" name="' . esc_attr($field_id) . '">';
        foreach ($providers as $provider_key => $provider_name) {
            echo '<option value="' . esc_attr($provider_key) . '"' . selected($value, $provider_key, false) . '>' . esc_html($provider_name) . '</option>';
        }
        echo '</select>';
        echo '<p class="description">' . __('Primary API provider for geolocation. Other providers will be used as fallbacks.', 'affiliate-hub') . '</p>';
    }
    
    /**
     * Fallback location callback
     */
    public function fallback_location_callback() {
        $country = get_option(Constants::OPTION_FALLBACK_COUNTRY, 'Poland');
        $city = get_option(Constants::OPTION_FALLBACK_CITY, 'Warsaw');
        
        echo '<label for="fallback_country">' . __('Country:', 'affiliate-hub') . '</label> ';
        echo '<input type="text" id="fallback_country" name="' . Constants::OPTION_FALLBACK_COUNTRY . '" value="' . esc_attr($country) . '" class="regular-text" />';
        echo '<br><br>';
        echo '<label for="fallback_city">' . __('City:', 'affiliate-hub') . '</label> ';
        echo '<input type="text" id="fallback_city" name="' . Constants::OPTION_FALLBACK_CITY . '" value="' . esc_attr($city) . '" class="regular-text" />';
        echo '<p class="description">' . __('Default location used for local/private IP addresses or when all API providers fail.', 'affiliate-hub') . '</p>';
    }
    
    /**
     * Geo test tool callback
     */
    public function geo_test_tool_callback() {
        echo '<div id="geo-test-container">';
        echo '<label for="test_ip">' . __('Test IP Address:', 'affiliate-hub') . '</label> ';
        echo '<input type="text" id="test_ip" value="8.8.8.8" class="regular-text" placeholder="8.8.8.8" />';
        echo '<button type="button" id="test-geo-btn" class="button">' . __('Test Lookup', 'affiliate-hub') . '</button>';
        echo '<div id="geo-test-results" style="margin-top: 15px;"></div>';
        echo '</div>';
        
        // Add JavaScript for testing
        ?>
        <script type="text/javascript">
        jQuery(document).ready(function($) {
            $('#test-geo-btn').on('click', function() {
                var testIp = $('#test_ip').val().trim();
                var $results = $('#geo-test-results');
                var $button = $(this);
                
                if (!testIp) {
                    $results.html('<div class="notice notice-error inline"><p>Please enter an IP address to test.</p></div>');
                    return;
                }
                
                $button.prop('disabled', true).text('Testing...');
                $results.html('<div class="notice notice-info inline"><p>Testing geolocation for IP: ' + testIp + '...</p></div>');
                
                $.ajax({
                    url: ajaxurl,
                    type: 'POST',
                    data: {
                        action: 'affiliate_hub_test_geo',
                        test_ip: testIp,
                        nonce: '<?php echo wp_create_nonce('affiliate_hub_admin_nonce'); ?>'
                    },
                    success: function(response) {
                        if (response.success) {
                            var data = response.data;
                            var html = '<div class="notice notice-success inline">';
                            html += '<p><strong>Geolocation Result:</strong></p>';
                            html += '<ul style="margin-left: 20px;">';
                            html += '<li><strong>Country:</strong> ' + (data.country || 'Unknown') + '</li>';
                            html += '<li><strong>Country Code:</strong> ' + (data.country_code || 'Unknown') + '</li>';
                            html += '<li><strong>City:</strong> ' + (data.city || 'Unknown') + '</li>';
                            html += '<li><strong>Provider:</strong> ' + (data.provider || 'Unknown') + '</li>';
                            html += '<li><strong>Cache Level:</strong> ' + (data.cache_level || 'Unknown') + '</li>';
                            html += '<li><strong>Response Time:</strong> ' + (data.response_time || 'Unknown') + '</li>';
                            html += '</ul>';
                            html += '</div>';
                            $results.html(html);
                        } else {
                            $results.html('<div class="notice notice-error inline"><p>Error: ' + (response.data || 'Unknown error') + '</p></div>');
                        }
                    },
                    error: function() {
                        $results.html('<div class="notice notice-error inline"><p>AJAX request failed.</p></div>');
                    },
                    complete: function() {
                        $button.prop('disabled', false).text('Test Lookup');
                    }
                });
            });
        });
        </script>
        <?php
        
        echo '<p class="description">' . __('Test the geolocation lookup system with any IP address.', 'affiliate-hub') . '</p>';
    }
    
    /**
     * MaxMind License Key callback
     */
    public function maxmind_license_key_callback() {
        $value = get_option('affiliate_hub_maxmind_license_key', '');
        echo '<input type="password" id="maxmind_license_key" name="affiliate_hub_maxmind_license_key" value="' . esc_attr($value) . '" class="regular-text" autocomplete="new-password" />';
        echo '<div class="maxmind-license-info">';
        echo '<p class="description">';
        echo __('Required for MaxMind GeoLite2 database access. ', 'affiliate-hub');
        echo '<strong>' . __('Free license available:', 'affiliate-hub') . '</strong> ';
        echo '<a href="https://www.maxmind.com/en/geolite2/signup" target="_blank">' . __('Register at maxmind.com', 'affiliate-hub') . '</a>';
        echo '</p>';
        echo '<p class="description">';
        echo __('After registration: Account → My License Keys → Generate new license key', 'affiliate-hub');
        echo '</p>';
        echo '</div>';
    }
    
    /**
     * MaxMind Status callback
     */
    public function maxmind_status_callback() {
        require_once AFFILIATE_HUB_PATH . 'includes/Modules/GeoLocation.php';
        $geo_module = new \AffiliateHub\Modules\GeoLocation();
        $geo_module->init();
        $status = $geo_module->get_maxmind_status();
        
        echo '<div class="maxmind-status-display">';
        
        if ($status['available']) {
            echo '<div class="notice notice-success inline">';
            echo '<p><span class="dashicons dashicons-yes-alt"></span> <strong>' . __('MaxMind Database Active', 'affiliate-hub') . '</strong></p>';
            echo '<ul>';
            echo '<li><strong>' . __('Database Size:', 'affiliate-hub') . '</strong> ' . esc_html($status['file_size']) . '</li>';
            echo '<li><strong>' . __('Last Updated:', 'affiliate-hub') . '</strong> ' . date_i18n(get_option('date_format'), $status['last_update']) . '</li>';
            echo '<li><strong>' . __('GeoIP2 Library:', 'affiliate-hub') . '</strong> ' . ($status['geoip2_available'] ? __('Available', 'affiliate-hub') : __('Missing', 'affiliate-hub')) . '</li>';
            if ($status['using_fallback']) {
                echo '<li><strong>' . __('Mode:', 'affiliate-hub') . '</strong> <span style="color: #d63638;">' . __('Fallback (API-based)', 'affiliate-hub') . '</span></li>';
            } else {
                echo '<li><strong>' . __('Mode:', 'affiliate-hub') . '</strong> <span style="color: #00a32a;">' . __('Native MaxMind', 'affiliate-hub') . '</span></li>';
            }
            echo '</ul>';
            
            if ($status['needs_update']) {
                echo '<p class="description" style="color: #d63638;">';
                echo '<span class="dashicons dashicons-warning"></span> ';
                echo __('Database is older than 2 weeks. Consider updating for better accuracy.', 'affiliate-hub');
                echo '</p>';
            }
            echo '</div>';
        } else {
            echo '<div class="notice notice-warning inline">';
            echo '<p><span class="dashicons dashicons-warning"></span> <strong>' . __('MaxMind Database Not Available', 'affiliate-hub') . '</strong></p>';
            echo '<p>' . esc_html($status['message']) . '</p>';
            
            if ($status['using_fallback']) {
                echo '<p class="description" style="color: #0073aa;">';
                echo '<span class="dashicons dashicons-info"></span> ';
                echo __('Currently using API fallback mode. Install composer and geoip2/geoip2 package for better performance.', 'affiliate-hub');
                echo '</p>';
            }
            
            if (!$status['geoip2_available']) {
                echo '<p class="description">';
                echo '<span class="dashicons dashicons-info"></span> ';
                echo __('GeoIP2 PHP library not detected. Using built-in fallback implementation.', 'affiliate-hub');
                echo '</p>';
            }
            echo '</div>';
        }
        
        echo '<button type="button" id="maxmind-download-btn" class="button">';
        echo __('Download/Update Database', 'affiliate-hub');
        echo '</button>';
        echo '<div id="maxmind-download-status"></div>';
        
        echo '</div>';
        
        // Add inline JavaScript for download button
        ?>
        <script type="text/javascript">
        jQuery(document).ready(function($) {
            $('#maxmind-download-btn').click(function() {
                var $button = $(this);
                var $status = $('#maxmind-download-status');
                
                $button.prop('disabled', true).text('<?php echo esc_js(__('Downloading...', 'affiliate-hub')); ?>');
                $status.html('<div class="notice notice-info inline"><p><?php echo esc_js(__('Downloading MaxMind database (~70MB)...', 'affiliate-hub')); ?></p></div>');
                
                $.ajax({
                    url: ajaxurl,
                    type: 'POST',
                    data: {
                        action: 'affiliate_hub_maxmind_download',
                        nonce: '<?php echo wp_create_nonce('affiliate_hub_admin_nonce'); ?>'
                    },
                    timeout: 120000, // 2 minutes timeout for large file
                    success: function(response) {
                        if (response.success) {
                            $status.html('<div class="notice notice-success inline"><p>' + response.data.message + '</p></div>');
                            // Refresh page to show updated status
                            setTimeout(function() { location.reload(); }, 2000);
                        } else {
                            $status.html('<div class="notice notice-error inline"><p>Error: ' + (response.data || 'Unknown error') + '</p></div>');
                        }
                    },
                    error: function() {
                        $status.html('<div class="notice notice-error inline"><p><?php echo esc_js(__('AJAX request failed. Check server logs.', 'affiliate-hub')); ?></p></div>');
                    },
                    complete: function() {
                        $button.prop('disabled', false).text('<?php echo esc_js(__('Download/Update Database', 'affiliate-hub')); ?>');
                    }
                });
            });
        });
        </script>
        <?php
    }
    
    /**
     * AJAX handler for MaxMind database download
     */
    public function ajax_maxmind_download() {
        check_ajax_referer('affiliate_hub_admin_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Insufficient permissions');
        }
        
        require_once AFFILIATE_HUB_PATH . 'includes/Modules/MaxMindManager.php';
        $maxmind_manager = new \AffiliateHub\Modules\MaxMindManager();
        
        $result = $maxmind_manager->update_database();
        
        if (is_wp_error($result)) {
            wp_send_json_error($result->get_error_message());
        } else {
            wp_send_json_success(array(
                'message' => __('MaxMind database downloaded successfully!', 'affiliate-hub')
            ));
        }
    }
}
