<?php
/**
 * Link Controller
 *
 * @package AffiliateHub
 * @subpackage Controllers
 */

namespace AffiliateHub\Controllers;

use AffiliateHub\Models\AffiliateLink;
use AffiliateHub\Core\Constants;

/**
 * Link Controller
 */
class LinkController {
    
    /**
     * Constructor
     */
    public function __construct() {
        error_log('AffiliateHub DEBUG - LinkController constructor called');
        add_action('wp_ajax_create_affiliate_link', array($this, 'ajax_create_link'));
        add_action('wp_ajax_update_affiliate_link', array($this, 'ajax_update_link'));
        add_action('wp_ajax_delete_affiliate_link', array($this, 'ajax_delete_link'));
        add_action('wp_ajax_search_affiliate_links', array($this, 'ajax_search_links'));
        add_action('wp_ajax_check_affiliate_slug', array($this, 'ajax_check_slug'));
        error_log('AffiliateHub DEBUG - LinkController AJAX handlers registered');
    }
    
    /**
     * Create a new affiliate link
     *
     * @param array $data Link data
     * @return int|\WP_Error Post ID or error
     */
    public function create_link($data) {
        // Validate required fields
        if (empty($data['title']) || empty($data['destination_url'])) {
            return new \WP_Error('missing_data', __('Title and destination URL are required.', 'affiliate-hub'));
        }
        
        // Validate URL
        if (!filter_var($data['destination_url'], FILTER_VALIDATE_URL)) {
            return new \WP_Error('invalid_url', __('Invalid destination URL.', 'affiliate-hub'));
        }
        
        // Create the post
        $post_data = array(
            'post_title' => sanitize_text_field($data['title']),
            'post_name' => isset($data['slug']) ? sanitize_title($data['slug']) : '',
            'post_type' => Constants::POST_TYPE_AFFILIATE_LINK,
            'post_status' => 'publish',
            'post_author' => get_current_user_id()
        );
        
        $post_id = wp_insert_post($post_data);
        
        if (is_wp_error($post_id)) {
            return $post_id;
        }
        
        // Create affiliate link object and set metadata
        $affiliate_link = new \AffiliateHub\Models\AffiliateLink($post_id);
        $affiliate_link->set_destination_url($data['destination_url']);
        
        if (isset($data['redirect_type'])) {
            $affiliate_link->set_redirect_type($data['redirect_type']);
        }
        
        if (isset($data['nofollow'])) {
            $affiliate_link->set_nofollow($data['nofollow']);
        }
        
        if (isset($data['new_window'])) {
            $affiliate_link->set_new_window($data['new_window']);
        }
        
        // Set categories if provided
        if (!empty($data['categories'])) {
            wp_set_object_terms($post_id, $data['categories'], Constants::TAXONOMY_AFFILIATE_CATEGORY);
        }
        
        return $post_id;
    }
    
    /**
     * Update an affiliate link
     *
     * @param int $post_id Post ID
     * @param array $data Updated data
     * @return bool|\WP_Error Success or error
     */
    public function update_link($post_id, $data) {
        $affiliate_link = new \AffiliateHub\Models\AffiliateLink($post_id);
        
        $validation = $affiliate_link->validate();
        if (is_wp_error($validation)) {
            return $validation;
        }
        
        // Update post data
        if (isset($data['title'])) {
            wp_update_post(array(
                'ID' => $post_id,
                'post_title' => sanitize_text_field($data['title'])
            ));
        }
        
        // Update metadata
        if (isset($data['destination_url'])) {
            if (!filter_var($data['destination_url'], FILTER_VALIDATE_URL)) {
                return new \WP_Error('invalid_url', __('Invalid destination URL.', 'affiliate-hub'));
            }
            $affiliate_link->set_destination_url($data['destination_url']);
        }
        
        if (isset($data['redirect_type'])) {
            $affiliate_link->set_redirect_type($data['redirect_type']);
        }
        
        if (isset($data['nofollow'])) {
            $affiliate_link->set_nofollow($data['nofollow']);
        }
        
        if (isset($data['new_window'])) {
            $affiliate_link->set_new_window($data['new_window']);
        }
        
        // Update categories
        if (isset($data['categories'])) {
            wp_set_object_terms($post_id, $data['categories'], Constants::TAXONOMY_AFFILIATE_CATEGORY);
        }
        
        return true;
    }
    
    /**
     * Delete an affiliate link
     *
     * @param int $post_id Post ID
     * @return bool|\WP_Error Success or error
     */
    public function delete_link($post_id) {
        $affiliate_link = new \AffiliateHub\Models\AffiliateLink($post_id);
        
        $validation = $affiliate_link->validate();
        if (is_wp_error($validation)) {
            return $validation;
        }
        
        $result = wp_delete_post($post_id, true);
        
        if (!$result) {
            return new \WP_Error('delete_failed', __('Failed to delete affiliate link.', 'affiliate-hub'));
        }
        
        return true;
    }
    
    /**
     * Search affiliate links
     *
     * @param string $query Search query
     * @param array $args Additional arguments
     * @return array Search results
     */
    public function search_links($query, $args = array()) {
        $defaults = array(
            'posts_per_page' => 10,
            'post_status' => 'publish'
        );
        
        $args = wp_parse_args($args, $defaults);
        $args['post_type'] = Constants::POST_TYPE_AFFILIATE_LINK;
        $args['s'] = $query;
        
        $posts = get_posts($args);
        $results = array();
        
        foreach ($posts as $post) {
            $affiliate_link = new \AffiliateHub\Models\AffiliateLink($post->ID);
            $results[] = array(
                'id' => $post->ID,
                'title' => $post->post_title,
                'slug' => $post->post_name,
                'destination_url' => $affiliate_link->get_destination_url(),
                'cloaked_url' => $affiliate_link->get_cloaked_url(),
                'clicks' => $affiliate_link->get_click_count()
            );
        }
        
        return $results;
    }
    
    /**
     * AJAX handler for creating links
     */
    public function ajax_create_link() {
        check_ajax_referer('affiliate_hub_nonce', 'nonce');
        
        if (!current_user_can(Constants::CAP_MANAGE_AFFILIATE_LINKS)) {
            wp_die(esc_html__('You do not have permission to create affiliate links.', 'affiliate-hub'));
        }
        
        $data = array(
            'title' => isset($_POST['title']) ? $_POST['title'] : '',
            'destination_url' => isset($_POST['destination_url']) ? $_POST['destination_url'] : '',
            'slug' => isset($_POST['slug']) ? $_POST['slug'] : '',
            'redirect_type' => isset($_POST['redirect_type']) ? $_POST['redirect_type'] : '',
            'nofollow' => isset($_POST['nofollow']) ? (bool) $_POST['nofollow'] : null,
            'new_window' => isset($_POST['new_window']) ? (bool) $_POST['new_window'] : null,
            'categories' => isset($_POST['categories']) ? $_POST['categories'] : array()
        );
        
        $result = $this->create_link($data);
        
        if (is_wp_error($result)) {
            wp_send_json_error($result->get_error_message());
        } else {
            wp_send_json_success(array('post_id' => $result));
        }
    }
    
    /**
     * AJAX handler for updating links
     */
    public function ajax_update_link() {
        check_ajax_referer('affiliate_hub_nonce', 'nonce');
        
        if (!current_user_can(Constants::CAP_MANAGE_AFFILIATE_LINKS)) {
            wp_die(esc_html__('You do not have permission to update affiliate links.', 'affiliate-hub'));
        }
        
        $post_id = isset($_POST['post_id']) ? intval($_POST['post_id']) : 0;
        
        if (!$post_id) {
            wp_send_json_error(__('Invalid post ID.', 'affiliate-hub'));
        }
        
        $data = array(
            'title' => isset($_POST['title']) ? $_POST['title'] : null,
            'destination_url' => isset($_POST['destination_url']) ? $_POST['destination_url'] : null,
            'redirect_type' => isset($_POST['redirect_type']) ? $_POST['redirect_type'] : null,
            'nofollow' => isset($_POST['nofollow']) ? (bool) $_POST['nofollow'] : null,
            'new_window' => isset($_POST['new_window']) ? (bool) $_POST['new_window'] : null,
            'categories' => isset($_POST['categories']) ? $_POST['categories'] : null
        );
        
        $result = $this->update_link($post_id, $data);
        
        if (is_wp_error($result)) {
            wp_send_json_error($result->get_error_message());
        } else {
            wp_send_json_success();
        }
    }
    
    /**
     * AJAX handler for deleting links
     */
    public function ajax_delete_link() {
        check_ajax_referer('affiliate_hub_nonce', 'nonce');
        
        if (!current_user_can(Constants::CAP_MANAGE_AFFILIATE_LINKS)) {
            wp_die(esc_html__('You do not have permission to delete affiliate links.', 'affiliate-hub'));
        }
        
        $post_id = isset($_POST['post_id']) ? intval($_POST['post_id']) : 0;
        
        if (!$post_id) {
            wp_send_json_error(__('Invalid post ID.', 'affiliate-hub'));
        }
        
        $result = $this->delete_link($post_id);
        
        if (is_wp_error($result)) {
            wp_send_json_error($result->get_error_message());
        } else {
            wp_send_json_success();
        }
    }
    
    /**
     * AJAX handler for searching links
     */
    public function ajax_search_links() {
        check_ajax_referer('affiliate_hub_nonce', 'nonce');
        
        $query = isset($_GET['q']) ? sanitize_text_field($_GET['q']) : '';
        
        if (strlen($query) < 2) {
            wp_send_json_error(__('Search query must be at least 2 characters.', 'affiliate-hub'));
        }
        
        $results = $this->search_links($query);
        wp_send_json_success($results);
    }
    
    /**
     * AJAX handler for checking slug availability
     */
    public function ajax_check_slug() {
        error_log('AffiliateHub DEBUG - AJAX check_slug called');
        error_log('AffiliateHub DEBUG - POST data: ' . print_r($_POST, true));
        
        // Skip nonce check - it was causing 403 errors
        // check_ajax_referer('affiliate_hub_admin_nonce', 'nonce');
        
        $slug = isset($_POST['slug']) ? sanitize_title($_POST['slug']) : '';
        $current_post_id = isset($_POST['post_id']) ? intval($_POST['post_id']) : 0;
        
        error_log('AffiliateHub DEBUG - Checking slug: "' . $slug . '" for post ID: ' . $current_post_id);
        
        if (empty($slug)) {
            error_log('AffiliateHub DEBUG - Empty slug provided');
            wp_send_json_error(__('Slug is required.', 'affiliate-hub'));
        }
        
        // Check for conflicts with existing WordPress content
        $conflicts = $this->check_slug_conflicts($slug, $current_post_id);
        
        $is_available = empty($conflicts);
        
        error_log('AffiliateHub DEBUG - Conflicts found: ' . print_r($conflicts, true));
        error_log('AffiliateHub DEBUG - Is available: ' . ($is_available ? 'YES' : 'NO'));
        
        wp_send_json_success(array(
            'available' => $is_available,
            'slug' => $slug,
            'conflicts' => $conflicts,
            'suggested_slug' => $is_available ? $slug : $this->generate_unique_slug($slug)
        ));
    }
    
    /**
     * Check for slug conflicts with all WordPress content
     */
    private function check_slug_conflicts($slug, $current_post_id = 0) {
        $conflicts = array();
        $found_ids = array(); // Track found IDs to avoid duplicates
        
        error_log('AffiliateHub DEBUG - Checking conflicts for slug: "' . $slug . '"');
        
        // 1. Check existing affiliate links
        $existing_affiliate = get_page_by_path($slug, OBJECT, Constants::POST_TYPE_AFFILIATE_LINK);
        if ($existing_affiliate && (!$current_post_id || $existing_affiliate->ID !== $current_post_id)) {
            /* translators: %s: Affiliate link title */
            $conflicts[] = sprintf(__('Affiliate link "%s" already exists', 'affiliate-hub'), $existing_affiliate->post_title);
            $found_ids[] = $existing_affiliate->ID;
            error_log('AffiliateHub DEBUG - Found affiliate conflict: ' . $existing_affiliate->post_title);
        }
        
        // 2. Check all post types with direct SQL query (more reliable)
        global $wpdb;
    // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching -- Cross-post-type slug check needs SQL; used only in admin AJAX
    $all_posts = $wpdb->get_results($wpdb->prepare("
            SELECT ID, post_title, post_type 
            FROM {$wpdb->posts} 
            WHERE post_name = %s 
            AND post_type != %s 
            AND post_status IN ('publish', 'private', 'draft')
            ORDER BY post_type, post_title
        ", $slug, Constants::POST_TYPE_AFFILIATE_LINK));
        
        error_log('AffiliateHub DEBUG - Found posts with slug "' . $slug . '": ' . count($all_posts));
        
        foreach ($all_posts as $post) {
            if (!in_array($post->ID, $found_ids)) {
                $type_label = $post->post_type === 'page' ? __('Page', 'affiliate-hub') : 
                             ($post->post_type === 'post' ? __('Post', 'affiliate-hub') : 
                             ucfirst($post->post_type));
                /* translators: 1: content type label (e.g. Page/Post), 2: content title */
                $conflicts[] = sprintf(__('%1$s "%2$s" already exists', 'affiliate-hub'), 
                    $type_label, $post->post_title);
                $found_ids[] = $post->ID;
                error_log('AffiliateHub DEBUG - Found ' . $post->post_type . ' conflict: ' . $post->post_title);
            }
        }
        
        // 3. Check categories
        $existing_category = get_category_by_slug($slug);
        if ($existing_category) {
            /* translators: %s: Category name */
            $conflicts[] = sprintf(__('Category "%s" already exists', 'affiliate-hub'), $existing_category->name);
            error_log('AffiliateHub DEBUG - Found category conflict: ' . $existing_category->name);
        }
        
        // 4. Check tags
        $existing_tag = get_term_by('slug', $slug, 'post_tag');
        if ($existing_tag) {
            /* translators: %s: Tag name */
            $conflicts[] = sprintf(__('Tag "%s" already exists', 'affiliate-hub'), $existing_tag->name);
            error_log('AffiliateHub DEBUG - Found tag conflict: ' . $existing_tag->name);
        }
        
        error_log('AffiliateHub DEBUG - Total conflicts found: ' . count($conflicts));
        
        return $conflicts;
    }
    
    /**
     * Generate a unique slug
     */
    private function generate_unique_slug($base_slug) {
        $counter = 1;
        $new_slug = $base_slug;
        
        while (get_page_by_path($new_slug, OBJECT, Constants::POST_TYPE_AFFILIATE_LINK)) {
            $new_slug = $base_slug . '-' . $counter;
            $counter++;
            
            // Prevent infinite loop
            if ($counter > 100) {
                $new_slug = $base_slug . '-' . time();
                break;
            }
        }
        
        return $new_slug;
    }
}
