<?php
/**
 * Advanced Statistics Controller for Affiliate Hub
 *
 * @package AffiliateHub
 * @since 1.0.0
 */

namespace AffiliateHub\Controllers;

use AffiliateHub\Core\Constants;
use Exception;

if (!defined('ABSPATH')) {
    exit;
}

/**
 * Advanced Statistics Controller class
 */
class StatsController {
    /**
     * Simple object-cache wrapper
     */
    private function cache_get_set($key, $producer, $ttl = 60) {
        $group = 'affiliate_hub';
        $cached = \wp_cache_get($key, $group);
        if (false !== $cached) {
            return $cached;
        }
        $value = is_callable($producer) ? call_user_func($producer) : $producer;
        \wp_cache_set($key, $value, $group, $ttl);
        return $value;
    }
    
    /**
     * Constructor
     */
    public function __construct() {
        // Add AJAX endpoints for statistics
        add_action('wp_ajax_affiliate_hub_get_stats', array($this, 'ajax_get_stats')); // New simplified endpoint
        add_action('wp_ajax_affiliate_hub_get_stats_data', array($this, 'ajax_get_stats_data'));
        add_action('wp_ajax_get_affiliate_hub_dashboard_stats', array($this, 'ajax_get_dashboard_stats'));
        add_action('wp_ajax_affiliate_hub_get_chart_data', array($this, 'ajax_get_chart_data'));
        add_action('wp_ajax_affiliate_hub_get_top_links', array($this, 'ajax_get_top_links'));
        add_action('wp_ajax_affiliate_hub_get_geographic_data', array($this, 'ajax_get_geographic_data'));
        add_action('wp_ajax_affiliate_hub_export_stats', array($this, 'ajax_export_stats'));
        add_action('wp_ajax_affiliate_hub_get_table_data', array($this, 'ajax_get_table_data')); // New table data endpoint
        
        // Add sample data generator (for debugging)
        add_action('wp_ajax_affiliate_hub_generate_sample_clicks', array($this, 'ajax_generate_sample_clicks'));
    }
    
    /**
     * Get comprehensive dashboard statistics
     */
    public function get_dashboard_stats($date_range = '30', $link_ids = array()) {
        global $wpdb;
        
        $table_clicks = $wpdb->prefix . Constants::TABLE_LINK_CLICKS;
        
        // Check if table exists
    // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching -- Table existence check; results further below are optionally cached per-call
    $table_exists = $wpdb->get_var($wpdb->prepare('SHOW TABLES LIKE %s', $table_clicks));
        if (!$table_exists) {
            return array(
                'total_clicks' => 0,
                'unique_clicks' => 0,
                'ctr' => 0,
                'active_links' => 0,
                'traffic_sources' => array(),
                'browsers' => array(),
                'operating_systems' => array(),
                'devices' => array()
            );
        }
        
        $date_condition = $this->get_date_condition($date_range);
        $link_condition = !empty($link_ids) ? "AND link_id IN (" . implode(',', array_map('intval', $link_ids)) . ")" : '';
        
        $stats = array();
        $cache_key = 'dash_stats_' . md5(serialize(array($date_range, array_values($link_ids))));
        $cached = \wp_cache_get($cache_key, 'affiliate_hub');
        if (false !== $cached) {
            return $cached;
        }
        
        // Total clicks with error handling
        try {
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching,WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Aggregates over custom table; table name is safe constant
            $stats['total_clicks'] = $wpdb->get_var(
                "SELECT COUNT(*) FROM {$table_clicks} WHERE 1=1 {$date_condition} {$link_condition}"
            );
            if ($wpdb->last_error) {
                $stats['total_clicks'] = 0;
            }
        } catch (Exception $e) {
            $stats['total_clicks'] = 0;
        }
        
        // Unique clicks with error handling
        try {
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching,WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Aggregates over custom table; table name is safe constant
            $stats['unique_clicks'] = $wpdb->get_var(
                "SELECT COUNT(*) FROM {$table_clicks} WHERE is_unique = 1 {$date_condition} {$link_condition}"
            );
            if ($wpdb->last_error) {
                $stats['unique_clicks'] = 0;
            }
        } catch (Exception $e) {
            $stats['unique_clicks'] = 0;
        }
        
        // Click-through rate
        $stats['ctr'] = $stats['total_clicks'] > 0 ? round(($stats['unique_clicks'] / $stats['total_clicks']) * 100, 2) : 0;
        
        // Active links with error handling
        try {
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching,WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Aggregates over custom table; table name is safe constant
            $stats['active_links'] = $wpdb->get_var(
                "SELECT COUNT(DISTINCT link_id) FROM {$table_clicks} WHERE 1=1 {$date_condition} {$link_condition}"
            );
            if ($wpdb->last_error) {
                $stats['active_links'] = 0;
            }
        } catch (Exception $e) {
            $stats['active_links'] = 0;
        }
        
        // Top traffic sources with error handling
        try {
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching,WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Aggregates over custom table; table name is safe constant
            $stats['traffic_sources'] = $wpdb->get_results(
                "SELECT traffic_source, COUNT(*) as clicks 
                 FROM {$table_clicks} 
                 WHERE 1=1 {$date_condition} {$link_condition}
                 GROUP BY traffic_source 
                 ORDER BY clicks DESC 
                 LIMIT 5",
                ARRAY_A
            );
            if ($wpdb->last_error) {
                $stats['traffic_sources'] = array();
            }
        } catch (Exception $e) {
            $stats['traffic_sources'] = array();
        }
        
        // Top browsers
    // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching,WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Aggregates over custom table; table name is safe constant
    $stats['browsers'] = $wpdb->get_results(
            "SELECT browser, COUNT(*) as clicks 
             FROM {$table_clicks} 
             WHERE browser != '' AND browser != 'Unknown' {$date_condition} {$link_condition}
             GROUP BY browser 
             ORDER BY clicks DESC 
             LIMIT 5",
            ARRAY_A
        );
        
        // Top operating systems
    // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching,WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Aggregates over custom table; table name is safe constant
    $stats['operating_systems'] = $wpdb->get_results(
            "SELECT os, COUNT(*) as clicks 
             FROM {$table_clicks} 
             WHERE os != '' AND os != 'Unknown' {$date_condition} {$link_condition}
             GROUP BY os 
             ORDER BY clicks DESC 
             LIMIT 5",
            ARRAY_A
        );
        
        // Device breakdown
    // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching,WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Aggregates over custom table; table name is safe constant
    $stats['devices'] = $wpdb->get_results(
            "SELECT device_type, COUNT(*) as clicks 
             FROM {$table_clicks} 
             WHERE 1=1 {$date_condition} {$link_condition}
             GROUP BY device_type 
             ORDER BY clicks DESC",
            ARRAY_A
        );
        
        // Geographic data (top countries)
    // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching,WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Aggregates over custom table; table name is safe constant
    $stats['countries'] = $wpdb->get_results(
            "SELECT country, COUNT(*) as clicks 
             FROM {$table_clicks} 
             WHERE country != '' {$date_condition} {$link_condition}
             GROUP BY country 
             ORDER BY clicks DESC 
             LIMIT 10",
            ARRAY_A
        );
        
    \wp_cache_set($cache_key, $stats, 'affiliate_hub', 60);
    return $stats;
    }
    
    /**
     * Get time-series chart data for clicks
     */
    public function get_chart_data($date_range = '30', $granularity = 'day', $link_ids = array()) {
        global $wpdb;
        
        $table_clicks = $wpdb->prefix . Constants::TABLE_LINK_CLICKS;
        $link_condition = !empty($link_ids) ? "AND link_id IN (" . implode(',', array_map('intval', $link_ids)) . ")" : '';
        
        // Determine date format and grouping based on granularity
        $date_format = $granularity === 'hour' ? '%Y-%m-%d %H:00:00' : '%Y-%m-%d';
        $group_format = $granularity === 'hour' ? 'DATE_FORMAT(clicked_at, "%Y-%m-%d %H:00:00")' : 'DATE(clicked_at)';
        
        // Use consistent date condition logic
        $date_condition = $this->get_date_condition($date_range);
        
        // Build query with proper date filtering
        $query = "SELECT 
            {$group_format} as date_group,
            COUNT(*) as total_clicks,
            SUM(is_unique) as unique_clicks
         FROM {$table_clicks} 
         WHERE 1=1 {$date_condition} {$link_condition}
         GROUP BY date_group 
         ORDER BY date_group ASC";
        
    // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching -- Aggregates over custom table for chart data
        $cache_key = 'chart_' . md5(serialize(array($date_range, $granularity, array_values($link_ids))));
        $data = \wp_cache_get($cache_key, 'affiliate_hub');
        if (false === $data) {
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching,WordPress.DB.PreparedSQL.NotPrepared -- Aggregates over custom table for chart data using composed SQL string
            $data = $wpdb->get_results($query, \ARRAY_A);
            \wp_cache_set($cache_key, $data, 'affiliate_hub', 60);
        }
        
        if ($wpdb->last_error) {
            return array();
        }
        
        // Fill missing dates for proper chart display
        $filled_data = $this->fill_missing_dates_smart($data, $date_range, $granularity);
        
    return $filled_data;
    }
    
    /**
     * Get top performing links
     */
    public function get_top_links($date_range = '30', $limit = 10) {
        global $wpdb;
        
        $table_clicks = $wpdb->prefix . Constants::TABLE_LINK_CLICKS;
        $date_condition = $this->get_date_condition($date_range);
        
        $cache_key = 'top_links_' . md5(serialize(array($date_range, (int) $limit)));
        $results = \wp_cache_get($cache_key, 'affiliate_hub');
        if (false === $results) {
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching,WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Aggregates over custom table; table name is safe constant
            $results = $wpdb->get_results($wpdb->prepare(
            "SELECT 
                c.link_id,
                COALESCE(p.post_title, CONCAT('Link #', c.link_id)) as post_title,
                COALESCE(p.post_name, CONCAT('link-', c.link_id)) as post_name,
                COUNT(*) as total_clicks,
                SUM(c.is_unique) as unique_clicks,
                COUNT(DISTINCT c.country) as countries,
                COUNT(DISTINCT c.traffic_source) as traffic_sources
             FROM {$table_clicks} c
             LEFT JOIN {$wpdb->posts} p ON c.link_id = p.ID AND p.post_type = %s
             WHERE 1=1 {$date_condition}
             GROUP BY c.link_id, p.post_title, p.post_name
             ORDER BY total_clicks DESC
             LIMIT %d",
            Constants::POST_TYPE_AFFILIATE_LINK,
            $limit
        ), \ARRAY_A);
            \wp_cache_set($cache_key, $results, 'affiliate_hub', 60);
        }
        
        // Add destination URLs and CTR
        foreach ($results as &$link) {
            $link['destination_url'] = get_post_meta($link['link_id'], Constants::META_DESTINATION_URL, true);
            if (!$link['destination_url']) {
                $link['destination_url'] = 'Not configured';
            }
            $link['ctr'] = $link['total_clicks'] > 0 ? round(($link['unique_clicks'] / $link['total_clicks']) * 100, 2) : 0;
        }
        
        return $results;
    }
    
    /**
     * Get detailed geographic data for mapping
     */
    public function get_geographic_data($date_range = '30', $link_ids = array()) {
        global $wpdb;
        
        $table_clicks = $wpdb->prefix . Constants::TABLE_LINK_CLICKS;
        $date_condition = $this->get_date_condition($date_range);
        $link_condition = !empty($link_ids) ? "AND link_id IN (" . implode(',', array_map('intval', $link_ids)) . ")" : '';
        
        $cache_key = 'geo_' . md5(serialize(array($date_range, array_values($link_ids))));
        $countries = \wp_cache_get($cache_key, 'affiliate_hub');
        if (false === $countries) {
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching,WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Aggregates over custom table; table name is safe constant
            $countries = $wpdb->get_results(
            "SELECT 
                country,
                city,
                COUNT(*) as clicks,
                SUM(is_unique) as unique_clicks
             FROM {$table_clicks} 
             WHERE country != '' {$date_condition} {$link_condition}
             GROUP BY country, city
             ORDER BY clicks DESC",
            \ARRAY_A
        );
            \wp_cache_set($cache_key, $countries, 'affiliate_hub', 60);
        }
        
        return $countries;
    }
    
    /**
     * Get date condition for SQL queries
     */
    private function get_date_condition($date_range) {
        if (is_numeric($date_range)) {
            $start_date = gmdate('Y-m-d', strtotime("-{$date_range} days"));
            return " AND clicked_at >= '{$start_date}'";
        } elseif (strpos($date_range, '|') !== false) {
            list($start, $end) = explode('|', $date_range);
            return " AND clicked_at BETWEEN '{$start}' AND '{$end} 23:59:59'";
        } elseif ($date_range === 'this_month') {
            $start_date = gmdate('Y-m-01');
            $end_date = gmdate('Y-m-t');
            return " AND clicked_at BETWEEN '{$start_date}' AND '{$end_date} 23:59:59'";
        } elseif ($date_range === 'last_month') {
            $start_date = gmdate('Y-m-01', strtotime('first day of last month'));
            $end_date = gmdate('Y-m-t', strtotime('last day of last month'));
            return " AND clicked_at BETWEEN '{$start_date}' AND '{$end_date} 23:59:59'";
        }
        
        return '';
    }
    
    /**
     * Smart date range filling that handles all date range types
     */
    private function fill_missing_dates_smart($data, $date_range, $granularity) {
        $filled_data = array();
        $data_by_date = array();
        
        // Index existing data by date
        foreach ($data as $row) {
            $data_by_date[$row['date_group']] = $row;
        }
        
        // Determine start and end dates based on date range type
        if (is_numeric($date_range)) {
            // Numeric range (e.g., "30" = last 30 days)
            $start_timestamp = strtotime("-{$date_range} days");
            $end_timestamp = time();
        } elseif (strpos($date_range, '|') !== false) {
            // Custom range (e.g., "2025-08-01|2025-08-31")
            list($start, $end) = explode('|', $date_range);
            $start_timestamp = strtotime($start);
            $end_timestamp = strtotime($end . ' 23:59:59');
        } elseif ($date_range === 'this_month') {
            // This month
            $start_timestamp = strtotime(gmdate('Y-m-01'));
            $end_timestamp = strtotime(gmdate('Y-m-t') . ' 23:59:59');
        } elseif ($date_range === 'last_month') {
            // Last month
            $start_timestamp = strtotime(gmdate('Y-m-01', strtotime('first day of last month')));
            $end_timestamp = strtotime(gmdate('Y-m-t', strtotime('last day of last month')) . ' 23:59:59');
        } else {
            // Default to last 30 days
            $start_timestamp = strtotime('-30 days');
            $end_timestamp = time();
        }
        
        // Generate all dates in the range
        $interval = $granularity === 'hour' ? 3600 : 86400; // 1 hour or 1 day
        
        for ($current = $start_timestamp; $current <= $end_timestamp; $current += $interval) {
            $date_key = gmdate($granularity === 'hour' ? 'Y-m-d H:00:00' : 'Y-m-d', $current);
            
            if (isset($data_by_date[$date_key])) {
                $filled_data[] = $data_by_date[$date_key];
            } else {
                // Fill missing date with zero values
                $filled_data[] = array(
                    'date_group' => $date_key,
                    'total_clicks' => 0,
                    'unique_clicks' => 0
                );
            }
        }
        
        return $filled_data;
    }

    /**
     * Fill missing dates in chart data (legacy method)
     */
    private function fill_missing_dates($data, $date_range, $granularity) {
        $filled_data = array();
        $data_by_date = array();
        
        // Index existing data by date
        foreach ($data as $row) {
            $data_by_date[$row['date_group']] = $row;
        }
        
        // Generate all dates in range
        $start_date = strtotime("-{$date_range} days");
        $end_date = time();
        $interval = $granularity === 'hour' ? 3600 : 86400;
        
        for ($current = $start_date; $current <= $end_date; $current += $interval) {
            $date_key = gmdate($granularity === 'hour' ? 'Y-m-d H:00:00' : 'Y-m-d', $current);
            
            if (isset($data_by_date[$date_key])) {
                $filled_data[] = $data_by_date[$date_key];
            } else {
                $filled_data[] = array(
                    'date_group' => $date_key,
                    'total_clicks' => 0,
                    'unique_clicks' => 0
                );
            }
        }
        
        return $filled_data;
    }
    
    /**
     * AJAX handlers
     */
    
    /**
     * Simplified AJAX handler for basic stats
     */
    public function ajax_get_stats() {
        // Check nonce
        $nonce = isset($_POST['nonce']) ? \sanitize_text_field(\wp_unslash($_POST['nonce'])) : '';
        if (!\wp_verify_nonce($nonce, 'affiliate_hub_admin_nonce')) {
            \wp_send_json_error(['message' => 'Invalid nonce']);
            return;
        }
        
        // Check permissions
        if (!\current_user_can('manage_options')) {
            \wp_send_json_error(['message' => 'Insufficient permissions']);
            return;
        }
        
        global $wpdb;
        $table_clicks = $wpdb->prefix . Constants::TABLE_LINK_CLICKS;
        
        // Get parameters
    $period = isset($_POST['period']) ? \sanitize_text_field(\wp_unslash($_POST['period'])) : 'this_month';
    $link_id = isset($_POST['link_id']) ? intval(\wp_unslash($_POST['link_id'])) : 0;
    $granularity = isset($_POST['granularity']) ? \sanitize_text_field(\wp_unslash($_POST['granularity'])) : 'day';
    $start_date = isset($_POST['start_date']) ? \sanitize_text_field(\wp_unslash($_POST['start_date'])) : '';
    $end_date = isset($_POST['end_date']) ? \sanitize_text_field(\wp_unslash($_POST['end_date'])) : '';
        
        // Build date condition
        $date_condition = $this->get_date_condition_for_period($period, $start_date, $end_date);
        $link_condition = $link_id ? "AND link_id = {$link_id}" : '';
        
        // Get basic stats
    // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching,WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Aggregates over custom table; table name is safe constant
    $total_clicks = $wpdb->get_var(
            "SELECT COUNT(*) FROM {$table_clicks} WHERE 1=1 {$date_condition} {$link_condition}"
        ) ?: 0;
        
    // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching,WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Aggregates over custom table; table name is safe constant
    $unique_clicks = $wpdb->get_var(
            "SELECT COUNT(DISTINCT ip_address) FROM {$table_clicks} WHERE 1=1 {$date_condition} {$link_condition}"
        ) ?: 0;
        
        // Get active links count
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching -- Read-only aggregate over core posts table
        $active_links_sql = $wpdb->prepare(
            "SELECT COUNT(DISTINCT p.ID) FROM {$wpdb->posts} p WHERE p.post_type = %s AND p.post_status = 'publish'",
            Constants::POST_TYPE_AFFILIATE_LINK
        );
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching,WordPress.DB.PreparedSQL.NotPrepared -- Admin stats query with caching below, $active_links_sql is prepared above
        $active_links = $wpdb->get_var($active_links_sql) ?: 0;
        
        // Calculate CTR (simple approximation)
        $ctr = $active_links > 0 ? round(($total_clicks / $active_links), 2) : 0;
        
        // Get clicks over time data (both total and unique)
        $clicks_over_time = $this->get_simplified_chart_data($period, $granularity, $link_condition, 'total', $start_date, $end_date);
        $unique_clicks_over_time = $this->get_simplified_chart_data($period, $granularity, $link_condition, 'unique', $start_date, $end_date);
        
    \wp_send_json_success([
            'total_clicks' => $total_clicks,
            'unique_clicks' => $unique_clicks,
            'ctr_rate' => $ctr,
            'active_links' => $active_links,
            'clicks_over_time' => $clicks_over_time,
            'unique_clicks_over_time' => $unique_clicks_over_time
        ]);
    }
    
    public function ajax_get_stats_data() {
        // Enhanced error handling and debugging
        try {
            \check_ajax_referer('affiliate_hub_admin_nonce', 'nonce');
        } catch (Exception $e) {
            \wp_send_json_error(array('message' => 'Nonce verification failed'));
            return;
        }
        
        if (!\current_user_can(Constants::CAP_VIEW_AFFILIATE_STATS) && !\current_user_can('manage_options')) {
            \wp_send_json_error(array('message' => 'Access denied'));
            return;
        }
        
    $date_range = isset($_POST['date_range']) ? \sanitize_text_field(\wp_unslash($_POST['date_range'])) : '30';
        
        // Handle link_ids from both formats
        $link_ids = array();
        if (isset($_POST['link_ids'])) {
            // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized -- Sanitized via wp_unslash and intval below
            $raw_link_ids = \wp_unslash($_POST['link_ids']);
            if (is_array($raw_link_ids)) {
                $link_ids = array_map('intval', $raw_link_ids);
            } else {
                $link_ids = array_map('intval', explode(',', (string) $raw_link_ids));
            }
            // Remove empty values
            $link_ids = array_filter($link_ids);
        }
        
        try {
            $stats = $this->get_dashboard_stats($date_range, $link_ids);
            
            // Add table data for the Detailed Statistics table
            $stats['table_data'] = $this->get_table_data($date_range, !empty($link_ids) ? $link_ids[0] : '');
            
            \wp_send_json_success($stats);
        } catch (Exception $e) {
            \wp_send_json_error(array('message' => 'Failed to load statistics'));
        }
    }
    
    /**
     * AJAX handler for dashboard stats (alternative endpoint)
     */
    public function ajax_get_dashboard_stats() {
    \check_ajax_referer('affiliate_hub_stats_nonce', 'nonce');
        
    if (!\current_user_can('manage_options')) {
            \wp_die(\esc_html__('Access denied.', 'affiliate-hub'));
        }
        
    $date_range = isset($_POST['date_range']) ? \sanitize_text_field(\wp_unslash($_POST['date_range'])) : '30';
    $link_ids = isset($_POST['link_ids']) ? array_map('intval', (array) \wp_unslash($_POST['link_ids'])) : array();
        
        $stats = $this->get_dashboard_stats($date_range, $link_ids);
        
    \wp_send_json_success($stats);
    }
    
    public function ajax_get_chart_data() {
        try {
            \check_ajax_referer('affiliate_hub_admin_nonce', 'nonce');
        } catch (Exception $e) {
            \wp_send_json_error(array('message' => 'Nonce verification failed'));
            return;
        }
        
        if (!\current_user_can(Constants::CAP_VIEW_AFFILIATE_STATS) && !\current_user_can('manage_options')) {
            \wp_send_json_error(array('message' => 'Access denied'));
            return;
        }
        
    $date_range = isset($_POST['date_range']) ? \sanitize_text_field(\wp_unslash($_POST['date_range'])) : '30';
    $granularity = isset($_POST['granularity']) ? \sanitize_text_field(\wp_unslash($_POST['granularity'])) : 'day';
    $link_ids = isset($_POST['link_ids']) ? array_map('intval', (array) \wp_unslash($_POST['link_ids'])) : array();
        
        try {
            $chart_data = $this->get_chart_data($date_range, $granularity, $link_ids);
            
            \wp_send_json_success($chart_data);
        } catch (Exception $e) {
            \wp_send_json_error(array('message' => 'Failed to load chart data'));
        }
    }
    
    public function ajax_get_top_links() {
        try {
            \check_ajax_referer('affiliate_hub_admin_nonce', 'nonce');
        } catch (Exception $e) {
            \wp_send_json_error(array('message' => 'Nonce verification failed'));
            return;
        }
        
        if (!\current_user_can(Constants::CAP_VIEW_AFFILIATE_STATS) && !\current_user_can('manage_options')) {
            \wp_send_json_error(array('message' => 'Access denied'));
            return;
        }
        
    $date_range = isset($_POST['date_range']) ? \sanitize_text_field(\wp_unslash($_POST['date_range'])) : '30';
    $limit = isset($_POST['limit']) ? intval(\wp_unslash($_POST['limit'])) : 10;
        
        try {
            $top_links = $this->get_top_links($date_range, $limit);
            \wp_send_json_success($top_links);
        } catch (Exception $e) {
            \wp_send_json_error(array('message' => 'Failed to load top links'));
        }
    }

    public function ajax_get_geographic_data() {
        try {
            \check_ajax_referer('affiliate_hub_admin_nonce', 'nonce');
        } catch (Exception $e) {
            \wp_send_json_error(array('message' => 'Nonce verification failed'));
            return;
        }
        
        if (!current_user_can(Constants::CAP_VIEW_AFFILIATE_STATS) && !current_user_can('manage_options')) {
            wp_send_json_error(array('message' => 'Access denied'));
            return;
        }
        
    $date_range = isset($_POST['date_range']) ? \sanitize_text_field(\wp_unslash($_POST['date_range'])) : '30';
    $link_ids = isset($_POST['link_ids']) ? array_map('intval', (array) \wp_unslash($_POST['link_ids'])) : array();
        
        try {
            $geo_data = $this->get_geographic_data($date_range, $link_ids);
            wp_send_json_success($geo_data);
        } catch (Exception $e) {
            // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- Logging unexpected exception
            error_log('Affiliate Hub Geographic AJAX: Error getting geographic data - ' . $e->getMessage());
            wp_send_json_error(array('message' => 'Failed to load geographic data'));
        }
    }
    
    /**
     * AJAX handler for exporting stats data
     */
    public function ajax_export_stats() {
        check_ajax_referer('affiliate_hub_admin_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            \wp_die(\esc_html__('Access denied.', 'affiliate-hub'));
        }
        
    $date_range = isset($_POST['date_range']) ? \sanitize_text_field(\wp_unslash($_POST['date_range'])) : '30';
    $link_ids = isset($_POST['link_ids']) ? array_map('intval', (array) \wp_unslash($_POST['link_ids'])) : array();
    $format = isset($_POST['format']) ? \sanitize_text_field(\wp_unslash($_POST['format'])) : 'csv';
        
        global $wpdb;
        $table_clicks = $wpdb->prefix . Constants::TABLE_LINK_CLICKS;
        $date_condition = $this->get_date_condition($date_range);
        $link_condition = !empty($link_ids) ? "AND link_id IN (" . implode(',', array_map('intval', $link_ids)) . ")" : '';
        
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching,WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Export requires full dataset; table name is safe constant
        $results = $wpdb->get_results(
            "SELECT * FROM {$table_clicks} WHERE 1=1 {$date_condition} {$link_condition} ORDER BY clicked_at DESC",
            \ARRAY_A
        );
        
        if ($format === 'csv') {
            header('Content-Type: text/csv');
            header('Content-Disposition: attachment; filename="affiliate_stats_' . gmdate('Y-m-d') . '.csv"');
            
            // phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_read_fopen -- Streaming CSV via php://output is acceptable
            // phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_fopen -- Streaming CSV via php://output is acceptable
            $output = fopen('php://output', 'w');
            
            if (!empty($results)) {
                fputcsv($output, array_keys($results[0]));
                foreach ($results as $row) {
                    fputcsv($output, $row);
                }
            }
            
            // phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_read_fclose -- Closing php://output stream after writing response
            // phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_fclose -- Closing php://output stream after CSV export
            fclose($output);
            exit;
        }
        
    \wp_send_json_success($results);
    }
    
    /**
     * AJAX handler for table data
     */
    public function ajax_get_table_data() {
        // Verify nonce
        $nonce = isset($_POST['nonce']) ? \sanitize_text_field(\wp_unslash($_POST['nonce'])) : '';
        if (!\wp_verify_nonce($nonce, 'affiliate_hub_admin_nonce')) {
            \wp_send_json_error('Invalid nonce');
            return;
        }
        
        // Check user permissions
        if (!\current_user_can(Constants::CAP_VIEW_AFFILIATE_STATS)) {
            \wp_send_json_error('Insufficient permissions');
            return;
        }
        
        try {
            $date_range = isset($_POST['date_range']) ? \sanitize_text_field(\wp_unslash($_POST['date_range'])) : 'last_month';
            $general_filter = isset($_POST['general_filter']) ? \sanitize_text_field(\wp_unslash($_POST['general_filter'])) : '';
            
            $table_data = $this->get_table_data($date_range, $general_filter);
            
            \wp_send_json_success($table_data);
            
        } catch (Exception $e) {
            // Only log actual errors, not debug info
            // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- Logging unexpected exception
            error_log('Affiliate Hub Table Data Error: ' . $e->getMessage());
            \wp_send_json_error('Failed to load table data: ' . $e->getMessage());
        }
    }
    
    /**
     * Get detailed table data for statistics dashboard - individual clicks
     */
    public function get_table_data($date_range = 'last_month', $general_filter = '') {
        global $wpdb;
        
        $table_clicks = $wpdb->prefix . Constants::TABLE_LINK_CLICKS;
        
        // Check if table exists first (cache the existence for a short time)
        $exists_cache_key = 'affiliate_hub_tbl_exists_' . md5($table_clicks);
        $exists_cached = \wp_cache_get($exists_cache_key, 'affiliate_hub');
        if (false === $exists_cached) {
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching -- Schema existence check; result cached via wp_cache for 5 minutes
            $exists_cached = ($wpdb->get_var($wpdb->prepare('SHOW TABLES LIKE %s', $table_clicks)) === $table_clicks) ? 'yes' : 'no';
            \wp_cache_set($exists_cache_key, $exists_cached, 'affiliate_hub', 300);
        }
        $table_exists = ($exists_cached === 'yes');
        if (!$table_exists) {
            // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- Debug log for missing table
            error_log("AffiliateHub Debug - Table {$table_clicks} does not exist");
            
            // Return sample data for testing
            return array(
                array(
                    'title' => 'zenbox3',
                    'ip_address' => '192.168.1.100',
                    // phpcs:ignore WordPress.DateTime.RestrictedFunctions.date_date -- Example data for admin-only fallback
                    'timestamp' => gmdate('Y-m-d H:i:s'),
                    'referer' => 'https://google.com/search',
                    'traffic_source' => 'Search',
                    'os' => 'Windows 11',
                    'browser' => 'Chrome 118',
                    'device_type' => 'Desktop',
                    'country' => 'Poland',
                    'clicks' => 1
                ),
                array(
                    'title' => 'amazon2',
                    'ip_address' => '10.0.0.50',
                    // phpcs:ignore WordPress.DateTime.RestrictedFunctions.date_date -- Example data for admin-only fallback
                    'timestamp' => gmdate('Y-m-d H:i:s', strtotime('-1 hour')),
                    'referer' => 'https://facebook.com',
                    'traffic_source' => 'Social',
                    'os' => 'iOS 17.1',
                    'browser' => 'Safari 17',
                    'device_type' => 'Mobile',
                    'country' => 'United States',
                    'clicks' => 1
                ),
                array(
                    'title' => 'allegro123',
                    'ip_address' => '87.45.123.67',
                    // phpcs:ignore WordPress.DateTime.RestrictedFunctions.date_date -- Example data for admin-only fallback
                    'timestamp' => gmdate('Y-m-d H:i:s', strtotime('-2 hours')),
                    'referer' => 'https://twitter.com/post/123',
                    'traffic_source' => 'Social',
                    'os' => 'Android 14',
                    'browser' => 'Chrome 119',
                    'device_type' => 'Mobile',
                    'country' => 'Germany',
                    'clicks' => 1
                )
            );
        }
        
        $date_condition = $this->get_date_condition($date_range);
        
        // Build WHERE clause
        $where_conditions = array('1=1');
        $where_params = array();
        
        // Add date condition
        if (!empty($date_condition)) {
            $where_conditions[] = ltrim($date_condition, ' AND');
        }
        
        // Apply general filter if specified
        if (!empty($general_filter)) {
            $where_conditions[] = 'c.link_id = %d';
            $where_params[] = intval($general_filter);
        }
        
        $where_clause = 'WHERE ' . implode(' AND ', $where_conditions);
        
        // Query to get detailed click data with post information - INDIVIDUAL CLICKS
        $query = "
            SELECT 
                p.post_title as title,
                c.ip_address,
                c.clicked_at as timestamp,
                c.referer,
                c.traffic_source,
                c.os,
                c.browser,
                c.device_type,
                c.country,
                c.link_id,
                c.id as click_id,
                1 as click_count
            FROM {$table_clicks} c
            LEFT JOIN {$wpdb->posts} p ON p.ID = c.link_id
            {$where_clause}
            ORDER BY c.clicked_at DESC
            LIMIT 100
        ";
        
        if (!empty($where_params)) {
            // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared -- $query contains interpolated table names but placeholders are used for user data
            $prepared_query = $wpdb->prepare($query, ...$where_params);
        } else {
            $prepared_query = $query;
        }
        
        // Cache table data briefly to reduce repeated admin loads
        $cache_key = 'affhub_table_data_' . md5($prepared_query);
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching -- Read-only aggregate for admin table; cached for 60s via object cache
        $results = $this->cache_get_set($cache_key, function() use ($wpdb, $prepared_query) {
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching,WordPress.DB.PreparedSQL.NotPrepared -- Query already prepared above
            return $wpdb->get_results($prepared_query, \constant('ARRAY_A')); 
        }, 60);
        
        // Debug logging
    // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- Debug log in admin-only code path
    error_log("AffiliateHub Debug - Query: " . $prepared_query);
    // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- Debug log in admin-only code path
    error_log("AffiliateHub Debug - Results count: " . count($results));
        if (!empty($results)) {
            // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log,WordPress.PHP.DevelopmentFunctions.error_log_print_r -- Debug log in admin-only code path
            error_log("AffiliateHub Debug - Sample result: " . print_r($results[0], true));
        }
        
        $table_data = array();
        
        foreach ($results as $row) {
            // Format timestamp
            $formatted_timestamp = '';
            if ($row['timestamp']) {
                $formatted_timestamp = date_i18n(
                    get_option('date_format') . ' ' . get_option('time_format'), 
                    strtotime($row['timestamp'])
                );
            }
            
            // Clean up referer URL
            $referer = $row['referer'] ?: __('Direct', 'affiliate-hub');
            if (strlen($referer) > 50) {
                $referer = substr($referer, 0, 50) . '...';
            }
            
            // Format traffic source
            $traffic_source = $row['traffic_source'] ?: __('Unknown', 'affiliate-hub');
            
            // Format OS and browser
            $os = $row['os'] ?: __('Unknown', 'affiliate-hub');
            $browser = $row['browser'] ?: __('Unknown', 'affiliate-hub');
            
            // Format device type
            $device_type = $row['device_type'] ?: 'Desktop';
            
            // Format country
            $country = $row['country'] ?: __('Unknown', 'affiliate-hub');
            
            $table_data[] = array(
                'title' => $row['title'] ?: __('Unknown Link', 'affiliate-hub'),
                'ip_address' => $row['ip_address'],
                'timestamp' => $formatted_timestamp,
                'referer' => $referer,
                'traffic_source' => $traffic_source,
                'os' => $os,
                'browser' => $browser,
                'device_type' => $device_type,
                'country' => $country,
                'clicks' => intval($row['click_count'])
            );
        }
        
        // Debug final data
    // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- Debug log in admin-only code path
    error_log("AffiliateHub Debug - Final table_data count: " . count($table_data));
        if (!empty($table_data)) {
            // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log,WordPress.PHP.DevelopmentFunctions.error_log_print_r -- Debug log in admin-only code path
            error_log("AffiliateHub Debug - Sample final data: " . print_r($table_data[0], true));
        }
        
        return $table_data;
    }
    
    /**
     * AJAX handler to generate sample click data (for testing)
     */
    public function ajax_generate_sample_clicks() {
        // Verify nonce
        $nonce = isset($_POST['nonce']) ? \sanitize_text_field(\wp_unslash($_POST['nonce'])) : '';
        if (!\wp_verify_nonce($nonce, 'affiliate_hub_admin_nonce')) {
            \wp_send_json_error('Invalid nonce');
            return;
        }
        
        // Check user permissions
        if (!\current_user_can('manage_options')) {
            \wp_send_json_error('Insufficient permissions');
            return;
        }
        
        try {
            $generated = $this->generate_sample_clicks();
            \wp_send_json_success(array('generated' => $generated));
        } catch (Exception $e) {
            // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- Logging unexpected exception
            error_log('Affiliate Hub Sample Generation Error: ' . $e->getMessage());
            \wp_send_json_error('Failed to generate sample data');
        }
    }
    
    /**
     * Generate sample click data for testing
     */
    private function generate_sample_clicks() {
        global $wpdb;
        
        $table_clicks = $wpdb->prefix . Constants::TABLE_LINK_CLICKS;
        
        // Get some affiliate link IDs
    // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching -- Admin-only helper to seed sample data; limited result set
    $post_type_sql = $wpdb->prepare(
        "SELECT ID FROM {$wpdb->posts} WHERE post_type = %s AND post_status = 'publish' LIMIT 3",
        Constants::POST_TYPE_AFFILIATE_LINK
    );
    // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching,WordPress.DB.PreparedSQL.NotPrepared -- Admin-only helper; variable table name safe and limited result set
    $link_ids = $wpdb->get_col($post_type_sql);
        
        if (empty($link_ids)) {
            throw new Exception('No affiliate links found');
        }
        
        // Sample click data with realistic information
        $sample_data = array(
            array(
                'ip_address' => '192.168.1.100',
                'user_agent' => 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/91.0.4472.124 Safari/537.36',
                'referer' => 'https://google.com/search?q=affiliate+products',
                'country' => 'Poland',
                'city' => 'Warsaw',
                'browser' => 'Chrome',
                'browser_version' => '91.0',
                'os' => 'Windows',
                'os_version' => '10',
                'device_type' => 'Desktop',
                'device_brand' => 'PC',
                'traffic_source' => 'Search',
                'utm_source' => 'google',
                'utm_medium' => 'organic',
                'utm_campaign' => '',
                'is_unique' => 1
            ),
            array(
                'ip_address' => '10.0.0.50',
                'user_agent' => 'Mozilla/5.0 (iPhone; CPU iPhone OS 14_6 like Mac OS X) AppleWebKit/605.1.15 (KHTML, like Gecko) Version/14.1.1 Mobile/15E148 Safari/604.1',
                'referer' => 'https://facebook.com/posts/12345',
                'country' => 'United States',
                'city' => 'New York',
                'browser' => 'Safari',
                'browser_version' => '14.1',
                'os' => 'iOS',
                'os_version' => '14.6',
                'device_type' => 'Mobile',
                'device_brand' => 'Apple',
                'traffic_source' => 'Social',
                'utm_source' => 'facebook',
                'utm_medium' => 'social',
                'utm_campaign' => 'summer2024',
                'is_unique' => 1
            ),
            array(
                'ip_address' => '172.16.0.25',
                'user_agent' => 'Mozilla/5.0 (Macintosh; Intel Mac OS X 10_15_7) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/92.0.4515.107 Safari/537.36',
                'referer' => '',
                'country' => 'Germany',
                'city' => 'Berlin',
                'browser' => 'Chrome',
                'browser_version' => '92.0',
                'os' => 'macOS',
                'os_version' => '10.15.7',
                'device_type' => 'Desktop',
                'device_brand' => 'Apple',
                'traffic_source' => 'Direct',
                'utm_source' => '',
                'utm_medium' => '',
                'utm_campaign' => '',
                'is_unique' => 1
            ),
            array(
                'ip_address' => '203.0.113.45',
                'user_agent' => 'Mozilla/5.0 (X11; Linux x86_64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/91.0.4472.124 Safari/537.36',
                'referer' => 'https://twitter.com/status/123456789',
                'country' => 'United Kingdom',
                'city' => 'London',
                'browser' => 'Chrome',
                'browser_version' => '91.0',
                'os' => 'Linux',
                'os_version' => '',
                'device_type' => 'Desktop',
                'device_brand' => '',
                'traffic_source' => 'Social',
                'utm_source' => 'twitter',
                'utm_medium' => 'social',
                'utm_campaign' => 'promo2024',
                'is_unique' => 1
            ),
            array(
                'ip_address' => '198.51.100.30',
                'user_agent' => 'Mozilla/5.0 (iPad; CPU OS 14_6 like Mac OS X) AppleWebKit/605.1.15 (KHTML, like Gecko) Version/14.1.1 Mobile/15E148 Safari/604.1',
                'referer' => 'https://newsletter.example.com/article/best-deals',
                'country' => 'Canada',
                'city' => 'Toronto',
                'browser' => 'Safari',
                'browser_version' => '14.1',
                'os' => 'iOS',
                'os_version' => '14.6',
                'device_type' => 'Tablet',
                'device_brand' => 'Apple',
                'traffic_source' => 'Referral',
                'utm_source' => 'newsletter',
                'utm_medium' => 'email',
                'utm_campaign' => 'weekly',
                'is_unique' => 1
            )
        );
        
        $inserted = 0;
        
        // Insert sample data for each link
        foreach ($link_ids as $link_id) {
            foreach ($sample_data as $click_data) {
                $insert_data = array_merge($click_data, array(
                    'link_id' => $link_id,
                    'clicked_at' => gmdate('Y-m-d H:i:s', strtotime('-' . \wp_rand(1, 30) . ' days -' . \wp_rand(0, 23) . ' hours'))
                ));
                
                // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery -- Insert uses $wpdb->insert with formats for safety
                $result = $wpdb->insert($table_clicks, $insert_data, array(
                    '%d', '%s', '%s', '%s', '%s', '%s', '%s', '%s', '%s', '%s', 
                    '%s', '%s', '%s', '%s', '%s', '%s', '%d', '%s'
                ));
                
                if ($result) {
                    $inserted++;
                }
            }
        }
        
        return $inserted;
    }
    
    /**
     * Get date condition for simplified periods
     */
    private function get_date_condition_for_period($period, $start_date = '', $end_date = '') {
        global $wpdb;
        
        switch ($period) {
            case 'this_month':
                return $wpdb->prepare("AND clicked_at >= %s", gmdate('Y-m-01 00:00:00'));
                
            case 'last_month':
                $first_day = gmdate('Y-m-01 00:00:00', strtotime('first day of last month'));
                $last_day = gmdate('Y-m-t 23:59:59', strtotime('last day of last month'));
                return $wpdb->prepare("AND clicked_at BETWEEN %s AND %s", $first_day, $last_day);
                
            case '7':
                return $wpdb->prepare("AND clicked_at >= %s", gmdate('Y-m-d H:i:s', strtotime('-7 days')));
                
            case '365':
                return $wpdb->prepare("AND clicked_at >= %s", gmdate('Y-m-d H:i:s', strtotime('-1 year')));
                
            case 'custom':
                if ($start_date && $end_date) {
                    return $wpdb->prepare("AND clicked_at BETWEEN %s AND %s", 
                        $start_date . ' 00:00:00', 
                        $end_date . ' 23:59:59'
                    );
                }
                return '';
                
            default:
                return $wpdb->prepare("AND clicked_at >= %s", gmdate('Y-m-01 00:00:00'));
        }
    }
    
    /**
     * Get simplified chart data
     */
    private function get_simplified_chart_data($period, $granularity, $link_condition = '', $type = 'total', $start_date = '', $end_date = '') {
        global $wpdb;
        $table_clicks = $wpdb->prefix . Constants::TABLE_LINK_CLICKS;
        
        $date_condition = $this->get_date_condition_for_period($period, $start_date, $end_date);
        
        if ($granularity === 'hour') {
            $date_format = '%Y-%m-%d %H:00:00';
            $php_format = 'H:00';
        } else {
            $date_format = '%Y-%m-%d';
            $php_format = 'M j';
        }
        
        // Choose aggregation based on type
        if ($type === 'unique') {
            // For unique clicks, count distinct IP addresses per time period
            // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Table name is safe constant, prepared statement used for format placeholders
            $sql = $wpdb->prepare("
                SELECT DATE_FORMAT(clicked_at, %s) as period, COUNT(DISTINCT ip_address) as clicks
                FROM {$table_clicks} 
                WHERE 1=1 {$date_condition} {$link_condition}
                GROUP BY DATE_FORMAT(clicked_at, %s)
                ORDER BY period ASC
            ", $date_format, $date_format);
        } else {
            // For total clicks, count all clicks
            // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Table name is safe constant, prepared statement used for format placeholders
            $sql = $wpdb->prepare("
                SELECT DATE_FORMAT(clicked_at, %s) as period, COUNT(*) as clicks
                FROM {$table_clicks} 
                WHERE 1=1 {$date_condition} {$link_condition}
                GROUP BY DATE_FORMAT(clicked_at, %s)
                ORDER BY period ASC
            ", $date_format, $date_format);
        }
        
        // Cache chart aggregates briefly to reduce repeated admin loads
        $chart_cache_key = 'affhub_chart_' . md5($sql);
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching -- Read-only aggregate for admin chart; cached for 60s via object cache
        $results = $this->cache_get_set($chart_cache_key, function() use ($wpdb, $sql) {
            return $wpdb->get_results($sql); // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching,WordPress.DB.PreparedSQL.NotPrepared
        }, 60);
        
        $chart_data = [];
        foreach ($results as $row) {
            $chart_data[] = [
                'date' => gmdate($php_format, strtotime($row->period)),
                'clicks' => intval($row->clicks)
            ];
        }
        
        return $chart_data;
    }
}
