<?php
/**
 * Plugin Activator
 *
 * @package AffiliateHub
 * @subpackage Core
 */

namespace AffiliateHub\Core;

use AffiliateHub\Core\Constants;

/**
 * Handles plugin activation
 */
class Activator {
    
    /**
     * Activate the plugin
     */
    public static function activate() {
        // Create database tables
        self::create_tables();
        
        // Set default options
        self::set_default_options();
        
        // Add capabilities
        self::add_capabilities();
        
        // Register post types and taxonomies before flushing rules
        if (class_exists('AffiliateHub\\Core\\Plugin')) {
            $plugin = \AffiliateHub\Core\Plugin::get_instance();
            $plugin->init_components();
        }
        
        // Flush rewrite rules
        flush_rewrite_rules();
        
        // Set activation flag
        update_option('affiliate_hub_activated', true);
    }
    
    /**
     * Create database tables
     */
    private static function create_tables() {
        global $wpdb;
        
        $charset_collate = $wpdb->get_charset_collate();
        
        // Link clicks table with enhanced analytics
        $table_clicks = $wpdb->prefix . Constants::TABLE_LINK_CLICKS;
        $sql_clicks = "CREATE TABLE $table_clicks (
            id bigint(20) NOT NULL AUTO_INCREMENT,
            link_id bigint(20) NOT NULL,
            ip_address varchar(45) NOT NULL,
            user_agent text,
            referer text,
            country varchar(100) DEFAULT '',
            country_code char(2) DEFAULT '',
            city varchar(100) DEFAULT '',
            latitude decimal(10,8) DEFAULT NULL,
            longitude decimal(11,8) DEFAULT NULL,
            timezone varchar(100) DEFAULT '',
            browser varchar(50) DEFAULT '',
            browser_version varchar(20) DEFAULT '',
            os varchar(50) DEFAULT '',
            os_version varchar(20) DEFAULT '',
            device_type varchar(20) DEFAULT 'Desktop',
            device_brand varchar(50) DEFAULT '',
            traffic_source varchar(50) DEFAULT '',
            utm_source varchar(100) DEFAULT '',
            utm_medium varchar(100) DEFAULT '',
            utm_campaign varchar(100) DEFAULT '',
            is_unique tinyint(1) DEFAULT 0,
            session_fingerprint varchar(64) DEFAULT '',
            is_unique_24h tinyint(1) DEFAULT 0,
            geo_provider varchar(20) DEFAULT '',
            geo_cached tinyint(1) DEFAULT 0,
            processing_time_ms int DEFAULT 0,
            clicked_at datetime DEFAULT CURRENT_TIMESTAMP,
            PRIMARY KEY (id),
            KEY link_id (link_id),
            KEY clicked_at (clicked_at),
            KEY country (country),
            KEY country_code (country_code),
            KEY browser (browser),
            KEY os (os),
            KEY device_type (device_type),
            KEY traffic_source (traffic_source),
            KEY is_unique (is_unique),
            KEY session_fingerprint (session_fingerprint),
            KEY is_unique_24h (is_unique_24h),
            KEY geo_provider (geo_provider),
            KEY latitude (latitude),
            KEY longitude (longitude)
        ) $charset_collate;";
        
        // Click meta table
        $table_meta = $wpdb->prefix . Constants::TABLE_CLICK_META;
        $sql_meta = "CREATE TABLE $table_meta (
            meta_id bigint(20) NOT NULL AUTO_INCREMENT,
            click_id bigint(20) NOT NULL,
            meta_key varchar(255) NOT NULL,
            meta_value longtext,
            PRIMARY KEY (meta_id),
            KEY click_id (click_id),
            KEY meta_key (meta_key)
        ) $charset_collate;";
        
        require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
        dbDelta($sql_clicks);
        dbDelta($sql_meta);

        // Link scanner tables
        $table_scans = $wpdb->prefix . Constants::TABLE_LINK_SCANS;
        $sql_scans = "CREATE TABLE $table_scans (
            id bigint(20) NOT NULL AUTO_INCREMENT,
            started_at datetime DEFAULT NULL,
            finished_at datetime DEFAULT NULL,
            status varchar(20) DEFAULT 'pending',
            total_urls int DEFAULT 0,
            processed int DEFAULT 0,
            created_by bigint(20) DEFAULT 0,
            options longtext,
            PRIMARY KEY (id),
            KEY status (status)
        ) $charset_collate;";

        $table_scanned = $wpdb->prefix . Constants::TABLE_SCANNED_LINKS;
        $sql_scanned = "CREATE TABLE $table_scanned (
            id bigint(20) NOT NULL AUTO_INCREMENT,
            scan_id bigint(20) NOT NULL,
            post_id bigint(20) DEFAULT 0,
            url text NOT NULL,
            anchor_text varchar(255) DEFAULT '',
            context_snippet text,
            status_code int DEFAULT 0,
            status varchar(20) DEFAULT 'pending',
            ignored tinyint(1) DEFAULT 0,
            note text DEFAULT NULL,
            final_url text,
            attempts int DEFAULT 0,
            last_checked datetime DEFAULT NULL,
            created_at datetime DEFAULT CURRENT_TIMESTAMP,
            PRIMARY KEY (id),
            KEY scan_id (scan_id),
            KEY status (status)
        ) $charset_collate;";

        dbDelta($sql_scans);
        dbDelta($sql_scanned);
        
        // GeoLocation cache table
        $table_geo_cache = $wpdb->prefix . 'affiliate_hub_geo_cache';
        $sql_geo_cache = "CREATE TABLE $table_geo_cache (
            id bigint(20) NOT NULL AUTO_INCREMENT,
            ip_address varchar(45) NOT NULL,
            country_name varchar(100) DEFAULT '',
            country_code char(2) DEFAULT '',
            city varchar(100) DEFAULT '',
            latitude decimal(10,8) DEFAULT NULL,
            longitude decimal(11,8) DEFAULT NULL,
            timezone varchar(100) DEFAULT '',
            provider_used varchar(50) DEFAULT '',
            cached_at datetime DEFAULT CURRENT_TIMESTAMP,
            expires_at datetime DEFAULT NULL,
            PRIMARY KEY (id),
            UNIQUE KEY ip_address (ip_address),
            KEY expires_at (expires_at),
            KEY cached_at (cached_at),
            KEY country_code (country_code)
        ) $charset_collate;";
        
        dbDelta($sql_geo_cache);
        
        // Keywords Autolinker cache table
        $table_autolinker = $wpdb->prefix . Constants::TABLE_AUTOLINKER_CACHE;
        $sql_autolinker = "CREATE TABLE $table_autolinker (
            id bigint(20) NOT NULL AUTO_INCREMENT,
            link_id bigint(20) NOT NULL,
            keyword varchar(255) NOT NULL,
            keyword_hash varchar(32) NOT NULL,
            link_url text NOT NULL,
            link_title varchar(255) DEFAULT '',
            priority int DEFAULT 0,
            case_sensitive tinyint(1) DEFAULT 0,
            keyword_limit int DEFAULT 0,
            created_at datetime DEFAULT CURRENT_TIMESTAMP,
            updated_at datetime DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
            PRIMARY KEY (id),
            UNIQUE KEY unique_link_keyword (link_id, keyword_hash),
            KEY keyword (keyword),
            KEY keyword_hash (keyword_hash),
            KEY priority (priority),
            KEY link_id (link_id)
        ) $charset_collate;";
        
        dbDelta($sql_autolinker);
        
        // After creating/updating tables, upgrade existing data structure
        self::upgrade_click_tracking_table();
    }

    /**
     * Upgrade existing click tracking table with Enhanced Analytics columns
     */
    private static function upgrade_click_tracking_table() {
        global $wpdb;
        
        $table_clicks = $wpdb->prefix . Constants::TABLE_LINK_CLICKS;
        
        // Check if new columns already exist
        $columns = $wpdb->get_col("SHOW COLUMNS FROM `$table_clicks`");
        $existing_columns = array_map('strtolower', $columns);
        
        $new_columns = [
            'session_fingerprint' => 'VARCHAR(64) NULL AFTER user_agent',
            'is_unique_24h' => 'TINYINT(1) DEFAULT 1 AFTER session_fingerprint',
            'geo_provider' => 'VARCHAR(20) NULL AFTER geo_data',
            'geo_cached' => 'TINYINT(1) DEFAULT 0 AFTER geo_provider',
            'processing_time_ms' => 'INT DEFAULT 0 AFTER geo_cached'
        ];
        
        $added_columns = [];
        
        foreach ($new_columns as $column_name => $column_def) {
            if (!in_array(strtolower($column_name), $existing_columns)) {
                $sql = "ALTER TABLE `$table_clicks` ADD COLUMN `$column_name` $column_def";
                $result = $wpdb->query($sql);
                
                if ($result !== false) {
                    $added_columns[] = $column_name;
                }
            }
        }
        
        // Add new indexes if columns were added
        if (!empty($added_columns)) {
            $indexes_to_add = [];
            
            if (in_array('session_fingerprint', $added_columns)) {
                $indexes_to_add[] = 'ADD INDEX idx_session_fingerprint (session_fingerprint)';
            }
            if (in_array('is_unique_24h', $added_columns)) {
                $indexes_to_add[] = 'ADD INDEX idx_unique_24h (is_unique_24h)';
            }
            if (in_array('geo_provider', $added_columns)) {
                $indexes_to_add[] = 'ADD INDEX idx_geo_provider (geo_provider)';
            }
            
            if (!empty($indexes_to_add)) {
                $sql_indexes = "ALTER TABLE `$table_clicks` " . implode(', ', $indexes_to_add);
                $wpdb->query($sql_indexes);
            }
            
            // Log successful upgrade
            error_log("AffiliateHub: Enhanced Click Analytics - Added columns: " . implode(', ', $added_columns));
        }
    }

    /**
     * Ensure tables exist (public wrapper to run create_tables via other hooks)
     */
    public static function ensure_tables() {
        // Run create_tables but capture whether we created tables or added columns
        global $wpdb;
        $before = $wpdb->get_col("SHOW TABLES LIKE '" . $wpdb->prefix . \AffiliateHub\Core\Constants::TABLE_SCANNED_LINKS . "'");
        $after = $before;
        $msg = '';
        $status = 'success';
        $details = '';

        try {
            self::create_tables();
            $after = $wpdb->get_col("SHOW TABLES LIKE '" . $wpdb->prefix . \AffiliateHub\Core\Constants::TABLE_SCANNED_LINKS . "'");

            if (empty($before) && !empty($after)) {
                $msg = 'Link Scanner tables created';
            } else {
                // Detect if columns were added by attempting to SHOW columns
                $added = array();
                // Use full table name (with prefix) directly — don't use prepared placeholders for identifiers
                $scanned_table = $wpdb->prefix . \AffiliateHub\Core\Constants::TABLE_SCANNED_LINKS;
                $col_ignored = $wpdb->get_var("SHOW COLUMNS FROM `" . $scanned_table . "` LIKE 'ignored'");
                if (empty($col_ignored)) {
                    $added[] = 'ignored';
                }
                $col_note = $wpdb->get_var("SHOW COLUMNS FROM `" . $scanned_table . "` LIKE 'note'");
                if (empty($col_note)) {
                    $added[] = 'note';
                }
                if (!empty($added)) {
                    $msg = 'Link Scanner updated: added columns ' . implode(', ', $added);
                }
            }
        } catch (\Exception $e) {
            $status = 'error';
            $msg = 'Link Scanner migration failed';
            $details = $e->getMessage();

            // Ensure logs directory exists and write error
            $log_dir = rtrim(AFFILIATE_HUB_PATH, DIRECTORY_SEPARATOR) . DIRECTORY_SEPARATOR . 'logs';
            if (!file_exists($log_dir)) {
                if (function_exists('wp_mkdir_p')) {
                    wp_mkdir_p($log_dir);
                } else {
                    @mkdir($log_dir, 0755, true);
                }
            }
            $log_file = $log_dir . DIRECTORY_SEPARATOR . 'migration.log';
            $log_line = date('Y-m-d H:i:s') . " - ERROR: " . $details . PHP_EOL;
            @file_put_contents($log_file, $log_line, FILE_APPEND | LOCK_EX);

            // Optional email notification if enabled
            if (get_option('affiliate_hub_migration_notify', false)) {
                $admin_email = get_option('admin_email');
                if (!empty($admin_email) && function_exists('wp_mail')) {
                    wp_mail($admin_email, 'Affiliate Hub migration error', $msg . "\n\n" . $details);
                }
            }
        }

        if (!empty($msg) || !empty($details)) {
            $history = get_option('affiliate_hub_migration_history', array());
            $entry = array(
                'time' => time(),
                'message' => $msg . (!empty($details) ? ': ' . $details : ''),
                'before' => !empty($before),
                'after' => !empty($after),
                'status' => $status,
                'details' => $details,
            );
            $history[] = $entry;
            // Keep only latest 50 entries
            if (count($history) > 50) {
                $history = array_slice($history, -50);
            }
            update_option('affiliate_hub_migration_history', $history);
            // Keep latest message too for quick notice
            update_option('affiliate_hub_migration_message', $msg . (!empty($details) ? ': ' . $details : ''));
            update_option('affiliate_hub_migration_time', time());
        }
    }
    
    /**
     * Set default options
     */
    private static function set_default_options() {
        $defaults = Constants::get_default_options();
        // Add link scanner defaults
        $defaults[Constants::OPTION_LINK_SCANNER_WHITELIST] = '';
        $defaults[Constants::OPTION_LINK_SCANNER_BLACKLIST] = '127.0.0.1,localhost,::1';
        $defaults[Constants::OPTION_LINK_SCANNER_RATE] = 5; // requests per second
        // Frontend/admin visualization defaults
        $defaults[Constants::OPTION_LINK_SCANNER_DISABLE_ANIMATIONS] = false;
        $defaults[Constants::OPTION_LINK_SCANNER_DEBOUNCE_MS] = 200; // milliseconds
        
        // Add Keywords Autolinker defaults
        $defaults[Constants::OPTION_AUTOLINKER_ENABLED] = false;
        $defaults[Constants::OPTION_AUTOLINKER_KEYWORD_LIMIT] = 3;
        $defaults[Constants::OPTION_AUTOLINKER_RANDOM_PLACEMENT] = false;
        $defaults[Constants::OPTION_AUTOLINKER_LINK_IN_HEADINGS] = false;
        $defaults[Constants::OPTION_AUTOLINKER_DISABLE_ARCHIVE] = true;
        $defaults[Constants::OPTION_AUTOLINKER_DISABLE_HOME] = false;
        $defaults[Constants::OPTION_AUTOLINKER_ENABLE_FEEDS] = false;
        $defaults[Constants::OPTION_AUTOLINKER_POST_TYPES] = array('post', 'page');
        
        foreach ($defaults as $option => $value) {
            if (get_option($option) === false) {
                update_option($option, $value);
            }
        }
        
        // Set version
        update_option(Constants::OPTION_VERSION, AFFILIATE_HUB_VERSION);
    }
    
    /**
     * Add custom capabilities
     */
    private static function add_capabilities() {
        $role = get_role('administrator');
        if ($role) {
            $role->add_cap(Constants::CAP_MANAGE_AFFILIATE_LINKS);
            $role->add_cap(Constants::CAP_VIEW_AFFILIATE_STATS);
            $role->add_cap('manage_categories'); // For managing affiliate categories
        }
        
        $role = get_role('editor');
        if ($role) {
            $role->add_cap(Constants::CAP_MANAGE_AFFILIATE_LINKS);
            $role->add_cap(Constants::CAP_VIEW_AFFILIATE_STATS);
            $role->add_cap('manage_categories'); // For managing affiliate categories
        }
    }
}
