<?php
/**
 * Admin functionality
 *
 * @package AffiliateHub
 * @subpackage Core
 */

namespace AffiliateHub\Core;

use AffiliateHub\Core\Constants;
use AffiliateHub\Admin\Settings;
use AffiliateHub\Admin\MetaBoxes;

/**
 * Admin class for handling admin-side functionality
 */
class Admin {
    
    /**
     * Settings instance
     *
     * @var Settings
     */
    private $settings;
    
    /**
     * MetaBoxes instance
     *
     * @var MetaBoxes
     */
    private $meta_boxes;
    
    /**
     * GeoLocationAdmin instance
     *
     * @var \AffiliateHub\Admin\GeoLocationAdmin
     */
    private $geo_admin;
    
    /**
     * Constructor
     */
    public function __construct() {
        $this->init();
    }
    
    /**
     * Initialize admin functionality
     */
    private function init() {
        // Show admin notices first
        add_action('admin_notices', array($this, 'show_admin_notices'));
        
        // Initialize components only if not in safe mode
        if (!get_option('affiliate_hub_safe_mode', false)) {
            if (class_exists('AffiliateHub\\Admin\\Settings')) {
                $this->settings = new \AffiliateHub\Admin\Settings();
            }
            if (class_exists('AffiliateHub\\Admin\\MetaBoxes')) {
                $this->meta_boxes = new \AffiliateHub\Admin\MetaBoxes();
            }
            if (class_exists('AffiliateHub\\Admin\\GeoLocationAdmin')) {
                $this->geo_admin = new \AffiliateHub\Admin\GeoLocationAdmin();
            }
            if (class_exists('AffiliateHub\\Admin\\DashboardWidget')) {
                new \AffiliateHub\Admin\DashboardWidget();
            }
            if (class_exists('AffiliateHub\\Admin\\ImportExport')) {
                new \AffiliateHub\Admin\ImportExport();
            }
            if (class_exists('AffiliateHub\\Admin\\AnalyticsDashboard')) {
                new \AffiliateHub\Admin\AnalyticsDashboard();
                error_log('✅ AnalyticsDashboard initialized in Admin.php');
            } else {
                error_log('❌ AnalyticsDashboard class not found in Admin.php!');
            }
        } else {
            // Minimal initialization in safe mode - admin components limited
        }
        
        // Add admin menu
        add_action('admin_menu', array($this, 'add_admin_menu'), 10);
        
        // Keep our menu active for related pages
        add_filter('parent_file', array($this, 'set_parent_menu'));
        add_filter('submenu_file', array($this, 'set_submenu'));
        
        // Additional hook for menu highlighting with higher priority
        add_action('admin_head', array($this, 'highlight_custom_menu'));
        
        // Handle safe mode toggle
        add_action('admin_init', array($this, 'handle_safe_mode_toggle'));
        
        // Add custom columns to post list
        add_filter('manage_' . Constants::POST_TYPE_AFFILIATE_LINK . '_posts_columns', array($this, 'custom_columns'));
        add_action('manage_' . Constants::POST_TYPE_AFFILIATE_LINK . '_posts_custom_column', array($this, 'custom_column_content'), 10, 2);
        add_filter('manage_edit-' . Constants::POST_TYPE_AFFILIATE_LINK . '_sortable_columns', array($this, 'sortable_columns'));
        
        // Block other plugins from adding columns with higher priority
        add_filter('manage_' . Constants::POST_TYPE_AFFILIATE_LINK . '_posts_columns', array($this, 'block_external_columns'), 999);
        
        // Alternative hook for debugging
        add_action('manage_affiliate_link_posts_custom_column', array($this, 'custom_column_content'), 10, 2);
        
        // Handle custom column sorting
        add_action('pre_get_posts', array($this, 'handle_custom_sorting'));
        
        // Add bulk actions
        add_filter('bulk_actions-edit-' . Constants::POST_TYPE_AFFILIATE_LINK, array($this, 'add_bulk_actions'));
        add_filter('handle_bulk_actions-edit-' . Constants::POST_TYPE_AFFILIATE_LINK, array($this, 'handle_bulk_actions'), 10, 3);
        
        // Add filters above the table
        add_action('restrict_manage_posts', array($this, 'add_table_filters'));
        add_filter('parse_query', array($this, 'handle_table_filters'));
        
        // Add pagination controls
        add_filter('edit_posts_per_page', array($this, 'custom_posts_per_page'), 10, 2);
        add_action('restrict_manage_posts', array($this, 'add_per_page_filter'), 5);
    }
    
    
    /**
     * Show admin notices
     */
    public function show_admin_notices() {
        // Show safe mode notice
        if (class_exists('AffiliateHub\\Core\\SafeMode') && \AffiliateHub\Core\SafeMode::is_active()) {
            echo '<div class="notice notice-warning"><p>';
            echo esc_html(\AffiliateHub\Core\SafeMode::get_status_message());
            echo ' <a href="' . esc_url(add_query_arg('affiliate_hub_disable_safe_mode', 'true')) . '">';
            echo esc_html__('Try to Disable Safe Mode', 'affiliate-hub') . '</a>';
            echo '</p></div>';
        }

        // Show one-time migration message if set
        $msg = get_option('affiliate_hub_migration_message', '');
        if (!empty($msg)) {
            echo '<div class="notice notice-success is-dismissible"><p>' . esc_html__('Affiliate Hub:', 'affiliate-hub') . ' ' . esc_html($msg) . '</p></div>';
            delete_option('affiliate_hub_migration_message');
            delete_option('affiliate_hub_migration_time');
        }
    }
    
    /**
     * Handle safe mode toggle
     */
    public function handle_safe_mode_toggle() {
        if (isset($_GET['affiliate_hub_disable_safe_mode'])) {
            if (!current_user_can('manage_options')) {
                return;
            }
            
            if (class_exists('AffiliateHub\\Core\\SafeMode')) {
                \AffiliateHub\Core\SafeMode::disable();
            } else {
                delete_option('affiliate_hub_safe_mode');
                update_option('affiliate_hub_enable_stats', true);
            }
            
            wp_redirect(remove_query_arg('affiliate_hub_disable_safe_mode'));
            exit;
        }
        
        // Force disable safe mode if memory usage is low
        if (get_option('affiliate_hub_safe_mode', false)) {
            $memory_limit = wp_convert_hr_to_bytes(ini_get('memory_limit'));
            $memory_usage = memory_get_usage(true);
            $available_memory = $memory_limit - $memory_usage;
            
            // If we have more than 128MB available, auto-disable safe mode
            if ($available_memory > 134217728) { // 128MB
                if (class_exists('AffiliateHub\\Core\\SafeMode')) {
                    \AffiliateHub\Core\SafeMode::disable();
                } else {
                    delete_option('affiliate_hub_safe_mode');
                    update_option('affiliate_hub_enable_stats', true);
                }
            }
        }
    }

    /**
     * Add admin menu with all functionality grouped under one main menu
     */
    public function add_admin_menu() {
        // Main menu page - redirect to affiliate links list
        add_menu_page(
            __('Affiliate Hub', 'affiliate-hub'),
            __('Affiliate Hub', 'affiliate-hub'),
            Constants::CAP_MANAGE_AFFILIATE_LINKS,
            'edit.php?post_type=' . Constants::POST_TYPE_AFFILIATE_LINK,
            '',
            'dashicons-admin-links',
            25
        );
        
        // Add New Link submenu
        add_submenu_page(
            'edit.php?post_type=' . Constants::POST_TYPE_AFFILIATE_LINK,
            __('Add New Link', 'affiliate-hub'),
            __('Add New Link', 'affiliate-hub'),
            Constants::CAP_MANAGE_AFFILIATE_LINKS,
            'post-new.php?post_type=' . Constants::POST_TYPE_AFFILIATE_LINK,
            ''
        );
        
        // Categories submenu
        add_submenu_page(
            'edit.php?post_type=' . Constants::POST_TYPE_AFFILIATE_LINK,
            __('Categories', 'affiliate-hub'),
            __('Categories', 'affiliate-hub'),
            Constants::CAP_MANAGE_AFFILIATE_LINKS,
            'edit-tags.php?taxonomy=' . Constants::TAXONOMY_AFFILIATE_CATEGORY . '&post_type=' . Constants::POST_TYPE_AFFILIATE_LINK,
            ''
        );
        
        // Tags submenu
        add_submenu_page(
            'edit.php?post_type=' . Constants::POST_TYPE_AFFILIATE_LINK,
            __('Tags', 'affiliate-hub'),
            __('Tags', 'affiliate-hub'),
            Constants::CAP_MANAGE_AFFILIATE_LINKS,
            'edit-tags.php?taxonomy=' . Constants::TAXONOMY_AFFILIATE_TAG . '&post_type=' . Constants::POST_TYPE_AFFILIATE_LINK,
            ''
        );

        // Link Scanner submenu (placed after Tags and before Analytics)
        if (class_exists('\AffiliateHub\Modules\LinkScanner')) {
            add_submenu_page(
                'edit.php?post_type=' . Constants::POST_TYPE_AFFILIATE_LINK,
                __('Link Scanner', 'affiliate-hub'),
                __('Link Scanner', 'affiliate-hub'),
                Constants::CAP_MANAGE_AFFILIATE_LINKS,
                'affiliate-hub-link-scanner',
                array($this, 'display_link_scanner_page')
            );
        }
        
        // Keywords Autolinker submenu
        if (class_exists('\AffiliateHub\Modules\KeywordsAutolinker')) {
            add_submenu_page(
                'edit.php?post_type=' . Constants::POST_TYPE_AFFILIATE_LINK,
                __('Keywords Autolinker', 'affiliate-hub'),
                __('Keywords Autolinker', 'affiliate-hub'),
                'manage_options',
                'affiliate-hub-autolinker',
                array($this, 'display_autolinker_page')
            );
        }
        
        // Enhanced Analytics submenu (only if enabled, not in safe mode, and user has permissions)
        $analytics_enabled = get_option(Constants::OPTION_ENABLE_STATS, true);
        $safe_mode = get_option('affiliate_hub_safe_mode', false);
        $user_can_view = current_user_can(Constants::CAP_VIEW_AFFILIATE_STATS);
        
        if ($analytics_enabled && !$safe_mode && $user_can_view && class_exists('AffiliateHub\\Admin\\AnalyticsDashboard')) {
            // Note: Analytics dashboard is registered separately by AnalyticsDashboard class
            // We keep this logic for permission checking
        }
        
        // GeoLocation submenu (if module is enabled)
        $geo_enabled = get_option(Constants::OPTION_ENABLE_GEOLOCATION, true);
        if ($geo_enabled && !$safe_mode && class_exists('AffiliateHub\\Admin\\GeoLocationAdmin')) {
            add_submenu_page(
                'edit.php?post_type=' . Constants::POST_TYPE_AFFILIATE_LINK,
                __('GeoLocation Analytics', 'affiliate-hub'),
                __('GeoLocation', 'affiliate-hub'),
                'manage_options',
                'affiliate-hub-geolocation',
                array($this, 'display_geolocation_page')
            );
        }
        
        // Settings submenu
        add_submenu_page(
            'edit.php?post_type=' . Constants::POST_TYPE_AFFILIATE_LINK,
            __('Settings', 'affiliate-hub'),
            __('Settings', 'affiliate-hub'),
            'manage_options',
            'affiliate-hub-settings',
            array($this, 'display_settings_page')
        );
    }
    
    
    /**
     * Display main admin page
     */
    public function display_main_page() {
        echo '<div class="wrap">';
        echo '<h1>' . esc_html__('Affiliate Hub', 'affiliate-hub') . '</h1>';
        echo '<p>' . esc_html__('Manage your affiliate links effectively.', 'affiliate-hub') . '</p>';
        
        // Show safe mode status
        if (get_option('affiliate_hub_safe_mode', false)) {
            echo '<div class="notice notice-warning inline"><p>';
            echo esc_html__('Plugin is running in safe mode. Some features may be disabled.', 'affiliate-hub');
            echo '</p></div>';
        }
        
        echo '<h2>' . esc_html__('Quick Stats', 'affiliate-hub') . '</h2>';
        
        // Get quick stats
        $total_links = wp_count_posts(Constants::POST_TYPE_AFFILIATE_LINK);
        $published_links = $total_links->publish ?? 0;
        
        echo '<div class="affiliate-hub-stats" style="display: flex; gap: 20px; margin: 20px 0;">';
        echo '<div class="stat-box" style="background: #fff; padding: 20px; border: 1px solid #ccd0d4; border-radius: 4px; text-align: center; min-width: 150px;">';
        echo '<h3 style="margin: 0; font-size: 2em; color: #0073aa;">' . intval($published_links) . '</h3>';
        echo '<p style="margin: 5px 0 0 0; color: #666;">' . esc_html__('Active Links', 'affiliate-hub') . '</p>';
        echo '</div>';
        echo '</div>';
        
        echo '<h2>' . esc_html__('Quick Actions', 'affiliate-hub') . '</h2>';
        echo '<p>';
        echo '<a href="' . esc_url(admin_url('post-new.php?post_type=' . Constants::POST_TYPE_AFFILIATE_LINK)) . '" class="button button-primary">';
        echo esc_html__('Add New Link', 'affiliate-hub') . '</a> ';
        echo '<a href="' . esc_url(admin_url('edit.php?post_type=' . Constants::POST_TYPE_AFFILIATE_LINK)) . '" class="button">';
        echo esc_html__('View All Links', 'affiliate-hub') . '</a> ';
        echo '<a href="' . esc_url(admin_url('admin.php?page=affiliate-hub-settings')) . '" class="button">';
        echo esc_html__('Settings', 'affiliate-hub') . '</a>';
        echo '</p>';
        
        // Recent links
        $recent_links = get_posts(array(
            'post_type' => Constants::POST_TYPE_AFFILIATE_LINK,
            'posts_per_page' => 5,
            'post_status' => 'publish',
            'orderby' => 'date',
            'order' => 'DESC'
        ));
        
        if ($recent_links) {
            echo '<h2>' . esc_html__('Recent Links', 'affiliate-hub') . '</h2>';
            echo '<table class="wp-list-table widefat fixed striped">';
            echo '<thead><tr>';
            echo '<th>' . esc_html__('Title', 'affiliate-hub') . '</th>';
            echo '<th>' . esc_html__('Destination URL', 'affiliate-hub') . '</th>';
            echo '<th>' . esc_html__('Clicks', 'affiliate-hub') . '</th>';
            echo '<th>' . esc_html__('Date', 'affiliate-hub') . '</th>';
            echo '</tr></thead><tbody>';
            
            foreach ($recent_links as $link) {
                $destination_url = get_post_meta($link->ID, Constants::META_DESTINATION_URL, true);
                $clicks = get_post_meta($link->ID, Constants::META_CLICK_COUNT, true);
                
                echo '<tr>';
                echo '<td><a href="' . esc_url(get_edit_post_link($link->ID)) . '">' . esc_html($link->post_title) . '</a></td>';
                echo '<td>' . ($destination_url ? '<a href="' . esc_url($destination_url) . '" target="_blank">' . esc_html($destination_url) . '</a>' : '—') . '</td>';
                echo '<td>' . intval($clicks) . '</td>';
                echo '<td>' . get_the_date('', $link) . '</td>';
                echo '</tr>';
            }
            
            echo '</tbody></table>';
        }
        
        echo '</div>';
    }
    
    /**
     * Display settings page
     */
    public function display_settings_page() {
        if ($this->settings && method_exists($this->settings, 'display_settings_page')) {
            $this->settings->display_settings_page();
            return;
        }

        // Simple fallback settings page
        if (!current_user_can('manage_options')) {
            wp_die(__('Unauthorized', 'affiliate-hub'));
        }

        if (isset($_POST['affiliate_hub_save_settings']) && check_admin_referer('affiliate_hub_save_settings')) {
            $notify = isset($_POST['affiliate_hub_migration_notify']) ? 1 : 0;
            update_option('affiliate_hub_migration_notify', $notify);
            echo '<div class="updated notice"><p>' . esc_html__('Settings saved.', 'affiliate-hub') . '</p></div>';
        }

        $notify = get_option('affiliate_hub_migration_notify', 0);
        echo '<div class="wrap">';
        echo '<h1>' . esc_html__('Affiliate Hub Settings', 'affiliate-hub') . '</h1>';
        echo '<form method="post">' . wp_nonce_field('affiliate_hub_save_settings') . '<table class="form-table"><tbody>';
        echo '<tr><th>' . esc_html__('Migration notifications', 'affiliate-hub') . '</th><td>';
        echo '<label><input type="checkbox" name="affiliate_hub_migration_notify" value="1"' . ($notify ? ' checked' : '') . '> ' . esc_html__('Email admin on migration errors', 'affiliate-hub') . '</label>';
        echo '</td></tr>';
        echo '</tbody></table>';
        echo '<p><button class="button button-primary" name="affiliate_hub_save_settings" type="submit">' . esc_html__('Save Settings', 'affiliate-hub') . '</button></p>';
        echo '</form></div>';
    }

    /**
     * Display the Link Scanner admin page
     */
    public function display_link_scanner_page() {
        if (class_exists('AffiliateHub\\Modules\\LinkScanner\\AdminPage')) {
            $admin = new \AffiliateHub\Modules\LinkScanner\AdminPage();
            $admin->render();
        } else {
            echo '<div class="wrap"><h1>' . esc_html__('Link Scanner', 'affiliate-hub') . '</h1><p>' . esc_html__('Link Scanner module is not available.', 'affiliate-hub') . '</p></div>';
        }
    }
    
    /**
     * Display the Keywords Autolinker admin page
     */
    public function display_autolinker_page() {
        if (class_exists('AffiliateHub\\Modules\\KeywordsAutolinker\\Settings')) {
            $settings = new \AffiliateHub\Modules\KeywordsAutolinker\Settings();
            $settings->render_settings_page();
        } else {
            echo '<div class="wrap"><h1>' . esc_html__('Keywords Autolinker', 'affiliate-hub') . '</h1><p>' . esc_html__('Keywords Autolinker module is not available.', 'affiliate-hub') . '</p></div>';
        }
    }
    
    /**
     * Custom columns for affiliate links list
     */
    public function custom_columns($columns) {
        // Define allowed columns - block all other plugins from adding columns
        $allowed_columns = array(
            'cb',
            'title', 
            'link_id',
            'cloaked_url',
            'destination_url',
            'taxonomy-' . Constants::TAXONOMY_AFFILIATE_CATEGORY,
            'taxonomy-' . Constants::TAXONOMY_AFFILIATE_TAG,
            'redirect_type',
            'clicks',
            'last_clicked',
            'date'
        );
        
        // Start with a clean slate - only use WordPress core columns as base
        $new_columns = array();
        
        // Keep checkbox and title at the beginning (these are WordPress core)
        if (isset($columns['cb'])) {
            $new_columns['cb'] = $columns['cb'];
        }
        if (isset($columns['title'])) {
            $new_columns['title'] = $columns['title'];
        }
        
        // Add our custom columns
        $new_columns['link_id'] = __('Link ID', 'affiliate-hub');
        $new_columns['cloaked_url'] = __('Cloaked URL', 'affiliate-hub');
        $new_columns['destination_url'] = __('Destination URL', 'affiliate-hub');
        
        // Add taxonomy columns only if they exist in original columns
        $category_key = 'taxonomy-' . Constants::TAXONOMY_AFFILIATE_CATEGORY;
        $tag_key = 'taxonomy-' . Constants::TAXONOMY_AFFILIATE_TAG;
        
        if (isset($columns[$category_key])) {
            $new_columns[$category_key] = $columns[$category_key];
        }
        
        if (isset($columns[$tag_key])) {
            $new_columns[$tag_key] = $columns[$tag_key];
        }
        
        // Add redirect type
        $new_columns['redirect_type'] = __('Redirect Type', 'affiliate-hub');
        
        // Add stats columns only if stats are enabled
        if (get_option(Constants::OPTION_ENABLE_STATS, true) && !get_option('affiliate_hub_safe_mode', false)) {
            $new_columns['clicks'] = __('Clicks', 'affiliate-hub');
            $new_columns['last_clicked'] = __('Last Clicked', 'affiliate-hub');
        }
        
        // Add date column at the very end (last column)
        if (isset($columns['date'])) {
            $new_columns['date'] = $columns['date'];
        }
        
        // Filter out any columns from other plugins that aren't in our allowed list
        $filtered_columns = array();
        foreach ($new_columns as $key => $label) {
            if (in_array($key, $allowed_columns)) {
                $filtered_columns[$key] = $label;
            }
        }
        
        return $filtered_columns;
    }
    
    /**
     * Custom column content
     */
    public function custom_column_content($column, $post_id) {
        switch ($column) {
            case 'link_id':
                echo '<strong>#' . esc_html($post_id) . '</strong>';
                break;
                
            case 'cloaked_url':
                try {
                    if (class_exists('AffiliateHub\\Models\\AffiliateLink')) {
                        $affiliate_link = new \AffiliateHub\Models\AffiliateLink($post_id);
                        $cloaked_url = $affiliate_link->get_cloaked_url();
                        echo '<a href="' . esc_url($cloaked_url) . '" target="_blank" rel="noopener" class="affiliate-url-link" title="' . esc_attr($cloaked_url) . '">';
                        echo esc_html($cloaked_url);
                        echo '</a>';
                        // Add copy button
                        echo '<button type="button" class="button-link copy-url-btn" data-url="' . esc_attr($cloaked_url) . '" title="' . esc_attr__('Copy URL', 'affiliate-hub') . '">';
                        echo '<span class="dashicons dashicons-clipboard"></span>';
                        echo '</button>';
                    } else {
                        echo '—';
                    }
                } catch (\Exception $e) {
                    echo '—';
                }
                break;
                
            case 'destination_url':
                $url = get_post_meta($post_id, Constants::META_DESTINATION_URL, true);
                if ($url) {
                    echo '<a href="' . esc_url($url) . '" target="_blank" rel="noopener" class="affiliate-url-link" title="' . esc_attr($url) . '">';
                    echo esc_html($url);
                    echo '</a>';
                    // Add copy button
                    echo '<button type="button" class="button-link copy-url-btn" data-url="' . esc_attr($url) . '" title="' . esc_attr__('Copy URL', 'affiliate-hub') . '">';
                    echo '<span class="dashicons dashicons-clipboard"></span>';
                    echo '</button>';
                } else {
                    echo '—';
                }
                break;
                
            case 'redirect_type':
                $redirect_type = get_post_meta($post_id, Constants::META_REDIRECT_TYPE, true);
                if (!$redirect_type) {
                    $redirect_type = get_option(Constants::OPTION_REDIRECT_TYPE, Constants::REDIRECT_301);
                }
                $redirect_types = Constants::get_redirect_types();
                $type_label = isset($redirect_types[$redirect_type]) ? $redirect_types[$redirect_type] : $redirect_type;
                
                echo '<span class="affiliate-hub-redirect-type redirect-' . esc_attr($redirect_type) . '">';
                echo esc_html($redirect_type);
                echo '</span>';
                break;
                
            case 'clicks':
                try {
                    if (class_exists('AffiliateHub\\Models\\AffiliateLink')) {
                        $affiliate_link = new \AffiliateHub\Models\AffiliateLink($post_id);
                        $total_clicks = $affiliate_link->get_click_count();
                        $unique_clicks = $affiliate_link->get_unique_clicks();
                        
                        $title = sprintf(
                            /* translators: 1: total clicks count, 2: unique clicks count */
                            __('%1$d kliknięć / %2$d unikalnych', 'affiliate-hub'),
                            $total_clicks,
                            $unique_clicks
                        );
                        
                        echo '<span class="affiliate-hub-clicks-cell" title="' . esc_attr($title) . '">';
                        echo esc_html($total_clicks . '/' . $unique_clicks);
                        echo '</span>';
                    } else {
                        $clicks = get_post_meta($post_id, Constants::META_CLICK_COUNT, true);
                        echo '<span class="affiliate-hub-clicks-cell">' . intval($clicks) . '</span>';
                    }
                } catch (\Exception $e) {
                    $clicks = get_post_meta($post_id, Constants::META_CLICK_COUNT, true);
                    echo '<span class="affiliate-hub-clicks-cell">' . intval($clicks) . '</span>';
                }
                break;
                
            case 'last_clicked':
                $last_clicked = get_post_meta($post_id, Constants::META_LAST_CLICKED, true);
                if ($last_clicked) {
                    echo esc_html(date_i18n(get_option('date_format') . ' ' . get_option('time_format'), strtotime($last_clicked)));
                } else {
                    echo '—';
                }
                break;
        }
    }
    
    /**
     * Legacy Statistics page - redirect to new Analytics dashboard
     * @deprecated Use AnalyticsDashboard instead
     */
    public function display_stats_page() {
        // Redirect to new Analytics dashboard
        $analytics_url = admin_url('admin.php?page=affiliate-hub-analytics');
        
        // Preserve link_id parameter if present
        if (isset($_GET['link_id'])) {
            $analytics_url = add_query_arg('link_id', intval($_GET['link_id']), $analytics_url);
        }
        
        wp_redirect($analytics_url);
        exit;
    }
    
    /**
     * Make columns sortable
     */
    public function sortable_columns($columns) {
        $columns['link_id'] = 'ID';
        $columns['clicks'] = 'clicks';
        $columns['last_clicked'] = 'last_clicked';
        $columns['categories'] = 'categories';
        $columns['redirect_type'] = 'redirect_type';
        return $columns;
    }
    
    /**
     * Set parent menu for our custom post type and taxonomy pages
     */
    public function set_parent_menu($parent_file) {
        global $current_screen;
        
        if (!$current_screen) {
            return $parent_file;
        }
        
        // Ensure our CPT and its taxonomies stay under our top-level menu
        $top_level_slug = 'edit.php?post_type=' . Constants::POST_TYPE_AFFILIATE_LINK;

        // For our post type pages
        if (!empty($current_screen->post_type) && $current_screen->post_type === Constants::POST_TYPE_AFFILIATE_LINK) {
            return $top_level_slug;
        }

        // For our taxonomy pages (categories and tags for affiliate links)
        if (!empty($current_screen->taxonomy) && (
            $current_screen->taxonomy === Constants::TAXONOMY_AFFILIATE_CATEGORY ||
            $current_screen->taxonomy === Constants::TAXONOMY_AFFILIATE_TAG
        )) {
            return $top_level_slug;
        }

        // For our custom admin pages
        if (isset($_GET['page'])) {
            $page = sanitize_text_field($_GET['page']);
            $our_pages = array(
                'affiliate-hub-link-scanner',
                'affiliate-hub-analytics', 
                'affiliate-hub-settings'
            );
            
            if (in_array($page, $our_pages)) {
                return $top_level_slug;
            }
        }

        // Also check by current screen base for hook_suffix based identification
        if (!empty($current_screen->base)) {
            $screen_mapping = array(
                'affiliate_link_page_affiliate-hub-link-scanner' => $top_level_slug,
                'affiliate_link_page_affiliate-hub-analytics' => $top_level_slug,
                'affiliate_link_page_affiliate-hub-settings' => $top_level_slug
            );
            
            if (isset($screen_mapping[$current_screen->base])) {
                return $screen_mapping[$current_screen->base];
            }
        }

        return $parent_file;
    }
    
    /**
     * Set submenu file to highlight correct submenu item
     */
    public function set_submenu($submenu_file) {
        global $current_screen, $pagenow;
        
        if (!$current_screen) {
            return $submenu_file;
        }

        // For taxonomy pages (Categories)
        if (!empty($current_screen->taxonomy) && $current_screen->taxonomy === Constants::TAXONOMY_AFFILIATE_CATEGORY) {
            return 'edit-tags.php?taxonomy=' . Constants::TAXONOMY_AFFILIATE_CATEGORY . '&post_type=' . Constants::POST_TYPE_AFFILIATE_LINK;
        }

        // For taxonomy pages (Tags)
        if (!empty($current_screen->taxonomy) && $current_screen->taxonomy === Constants::TAXONOMY_AFFILIATE_TAG) {
            return 'edit-tags.php?taxonomy=' . Constants::TAXONOMY_AFFILIATE_TAG . '&post_type=' . Constants::POST_TYPE_AFFILIATE_LINK;
        }
        
        // For post list pages
        if (!empty($current_screen->post_type) && $current_screen->post_type === Constants::POST_TYPE_AFFILIATE_LINK && $pagenow === 'edit.php') {
            return 'edit.php?post_type=' . Constants::POST_TYPE_AFFILIATE_LINK;
        }
        
        // For new post pages
        if (!empty($current_screen->post_type) && $current_screen->post_type === Constants::POST_TYPE_AFFILIATE_LINK && $pagenow === 'post-new.php') {
            return 'post-new.php?post_type=' . Constants::POST_TYPE_AFFILIATE_LINK;
        }

        // For our custom admin pages - highlight the correct submenu item
        if (isset($_GET['page'])) {
            $page = sanitize_text_field($_GET['page']);
            
            // Map page to correct submenu file
            $page_mapping = array(
                'affiliate-hub-link-scanner' => 'affiliate-hub-link-scanner',
                'affiliate-hub-analytics' => 'affiliate-hub-analytics',
                'affiliate-hub-settings' => 'affiliate-hub-settings'
            );
            
            if (isset($page_mapping[$page])) {
                return $page_mapping[$page];
            }
        }

        // Also check by current screen base for hook_suffix based identification
        if (!empty($current_screen->base)) {
            $screen_mapping = array(
                'affiliate_link_page_affiliate-hub-link-scanner' => 'affiliate-hub-link-scanner',
                'affiliate_link_page_affiliate-hub-analytics' => 'affiliate-hub-analytics',
                'affiliate_link_page_affiliate-hub-settings' => 'affiliate-hub-settings'
            );
            
            if (isset($screen_mapping[$current_screen->base])) {
                return $screen_mapping[$current_screen->base];
            }
        }
        
        return $submenu_file;
    }
    
    /**
     * Handle custom column sorting
     */
    public function handle_custom_sorting($query) {
        if (!is_admin() || !$query->is_main_query()) {
            return;
        }
        
        if ($query->get('post_type') !== Constants::POST_TYPE_AFFILIATE_LINK) {
            return;
        }
        
        $orderby = $query->get('orderby');
        
        switch ($orderby) {
            case 'clicks':
                $query->set('meta_key', Constants::META_CLICK_COUNT);
                $query->set('orderby', 'meta_value_num');
                break;
                
            case 'last_clicked':
                $query->set('meta_key', Constants::META_LAST_CLICKED);
                $query->set('orderby', 'meta_value');
                break;
                
            case 'redirect_type':
                $query->set('meta_key', Constants::META_REDIRECT_TYPE);
                $query->set('orderby', 'meta_value');
                break;
                
            case 'categories':
                $query->set('orderby', 'title');
                break;
        }
    }
    
    /**
     * Add custom bulk actions
     */
    public function add_bulk_actions($bulk_actions) {
    $bulk_actions['reset_stats'] = __('Reset Statistics', 'affiliate-hub');
    $bulk_actions['enable_nofollow'] = __('Enable Nofollow', 'affiliate-hub');
    $bulk_actions['disable_nofollow'] = __('Disable Nofollow', 'affiliate-hub');
    $bulk_actions['set_301_redirect'] = __('Set 301 Redirect', 'affiliate-hub');
    $bulk_actions['set_302_redirect'] = __('Set 302 Redirect', 'affiliate-hub');
        return $bulk_actions;
    }
    
    /**
     * Handle custom bulk actions
     */
    public function handle_bulk_actions($redirect_to, $doaction, $post_ids) {
        if (!current_user_can(Constants::CAP_MANAGE_AFFILIATE_LINKS)) {
            return $redirect_to;
        }
        
        $processed = 0;
        
        switch ($doaction) {
            case 'reset_stats':
                foreach ($post_ids as $post_id) {
                    delete_post_meta($post_id, Constants::META_CLICK_COUNT);
                    delete_post_meta($post_id, Constants::META_UNIQUE_CLICKS);
                    delete_post_meta($post_id, Constants::META_LAST_CLICKED);
                    
                    // Also remove click records from database
                    global $wpdb;
                    $table_clicks = $wpdb->prefix . Constants::TABLE_LINK_CLICKS;
                    // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching -- Simple table existence check in admin filter
                    if ($wpdb->get_var("SHOW TABLES LIKE '$table_clicks'") === $table_clicks) {
                        $wpdb->delete($table_clicks, array('link_id' => $post_id), array('%d'));
                    }
                    
                    $processed++;
                }
                $redirect_to = add_query_arg('bulk_reset_stats', $processed, $redirect_to);
                break;
                
            case 'enable_nofollow':
                foreach ($post_ids as $post_id) {
                    update_post_meta($post_id, Constants::META_NOFOLLOW, true);
                    $processed++;
                }
                $redirect_to = add_query_arg('bulk_nofollow_enabled', $processed, $redirect_to);
                break;
                
            case 'disable_nofollow':
                foreach ($post_ids as $post_id) {
                    update_post_meta($post_id, Constants::META_NOFOLLOW, false);
                    $processed++;
                }
                $redirect_to = add_query_arg('bulk_nofollow_disabled', $processed, $redirect_to);
                break;
                
            case 'set_301_redirect':
                foreach ($post_ids as $post_id) {
                    update_post_meta($post_id, Constants::META_REDIRECT_TYPE, Constants::REDIRECT_301);
                    $processed++;
                }
                $redirect_to = add_query_arg('bulk_redirect_301', $processed, $redirect_to);
                break;
                
            case 'set_302_redirect':
                foreach ($post_ids as $post_id) {
                    update_post_meta($post_id, Constants::META_REDIRECT_TYPE, Constants::REDIRECT_302);
                    $processed++;
                }
                $redirect_to = add_query_arg('bulk_redirect_302', $processed, $redirect_to);
                break;
        }
        
        return $redirect_to;
    }
    
    /**
     * Add filters above the table
     */
    public function add_table_filters() {
        $screen = get_current_screen();
        
        if ($screen && $screen->post_type === Constants::POST_TYPE_AFFILIATE_LINK) {
            // Category filter
            $categories = get_terms(array(
                'taxonomy' => Constants::TAXONOMY_AFFILIATE_CATEGORY,
                'hide_empty' => true
            ));
            
            if (!is_wp_error($categories) && !empty($categories)) {
                echo '<select name="filter_by_category" id="filter-by-category">';
                echo '<option value="">' . \esc_html__('All Categories', 'affiliate-hub') . '</option>';
                
                $selected_category = isset($_GET['filter_by_category']) ? $_GET['filter_by_category'] : '';
                
                foreach ($categories as $category) {
                    $count = \number_format_i18n((int) $category->count);
                    printf(
                        '<option value="%s" %s>%s (%s)</option>',
                        \esc_attr($category->slug),
                        \selected($selected_category, $category->slug, false),
                        \esc_html($category->name),
                        \esc_html($count)
                    );
                }
                echo '</select>';
            }
            
            // Tags filter
            $tags = get_terms(array(
                'taxonomy' => Constants::TAXONOMY_AFFILIATE_TAG,
                'hide_empty' => true
            ));
            
            if (!is_wp_error($tags) && !empty($tags)) {
                echo '<select name="filter_by_tag" id="filter-by-tag">';
                echo '<option value="">' . \esc_html__('All Tags', 'affiliate-hub') . '</option>';
                
                $selected_tag = isset($_GET['filter_by_tag']) ? $_GET['filter_by_tag'] : '';
                
                foreach ($tags as $tag) {
                    $count = \number_format_i18n((int) $tag->count);
                    printf(
                        '<option value="%s" %s>%s (%s)</option>',
                        \esc_attr($tag->slug),
                        \selected($selected_tag, $tag->slug, false),
                        \esc_html($tag->name),
                        \esc_html($count)
                    );
                }
                echo '</select>';
            }
            
            // Redirect type filter
            $redirect_types = Constants::get_redirect_types();
            echo '<select name="filter_by_redirect_type" id="filter-by-redirect-type">';
            echo '<option value="">' . \esc_html__('All Redirect Types', 'affiliate-hub') . '</option>';
            
            $selected_redirect = isset($_GET['filter_by_redirect_type']) ? $_GET['filter_by_redirect_type'] : '';
            
            foreach ($redirect_types as $type_value => $type_label) {
                printf(
                    '<option value="%s" %s>%s</option>',
                    esc_attr($type_value),
                    selected($selected_redirect, $type_value, false),
                    esc_html($type_label)
                );
            }
            echo '</select>';
            
            // Add some styling
            echo '<style>
                #filter-by-category, #filter-by-tag, #filter-by-redirect-type {
                    margin-left: 5px;
                    margin-right: 5px;
                }
                .copy-url-btn {
                    margin-left: 5px;
                    padding: 2px 4px;
                    border: none;
                    background: transparent;
                    cursor: pointer;
                    color: #666;
                    text-decoration: none;
                }
                .copy-url-btn:hover {
                    color: #0073aa;
                }
                .copy-url-btn .dashicons {
                    width: 16px;
                    height: 16px;
                    font-size: 16px;
                }
                .affiliate-cloaked-url {
                    display: inline-block;
                    max-width: 200px;
                    overflow: hidden;
                    text-overflow: ellipsis;
                    white-space: nowrap;
                    vertical-align: middle;
                }
                .affiliate-url-link {
                    display: inline-block;
                    max-width: 200px;
                    overflow: hidden;
                    text-overflow: ellipsis;
                    white-space: nowrap;
                    vertical-align: middle;
                    position: relative;
                }
                .affiliate-url-link:hover {
                    overflow: visible;
                    white-space: normal;
                    z-index: 999;
                }
                .column-link_id {
                    width: 80px;
                }
                .column-cloaked_url {
                    width: 250px;
                }
                .column-destination_url {
                    width: 250px;
                }
                .column-redirect_type {
                    width: 100px;
                }
                .affiliate-hub-redirect-type {
                    display: inline-block;
                    padding: 2px 6px;
                    border-radius: 3px;
                    font-size: 11px;
                    font-weight: bold;
                    text-transform: uppercase;
                }
                .redirect-301 {
                    background: #d1ecf1;
                    color: #0c5460;
                }
                .redirect-302 {
                    background: #fff3cd;
                    color: #856404;
                }
                .redirect-307 {
                    background: #f8d7da;
                    color: #721c24;
                }
            </style>';
            
            // Add JavaScript for copy functionality
            echo '<script>
                jQuery(document).ready(function($) {
                    $(".copy-url-btn").on("click", function(e) {
                        e.preventDefault();
                        var url = $(this).data("url");
                        if (url) {
                            navigator.clipboard.writeText(url).then(function() {
                                // Show success feedback
                                var btn = $(e.target).closest(".copy-url-btn");
                                var originalTitle = btn.attr("title");
                                btn.attr("title", "' . \esc_js(\__('Copied!', 'affiliate-hub')) . '");
                                setTimeout(function() {
                                    btn.attr("title", originalTitle);
                                }, 2000);
                            }).catch(function() {
                                // Fallback for older browsers
                                var textArea = document.createElement("textarea");
                                textArea.value = url;
                                document.body.appendChild(textArea);
                                textArea.select();
                                document.execCommand("copy");
                                document.body.removeChild(textArea);
                            });
                        }
                    });
                });
            </script>';
        }
    }
    
    /**
     * Handle table filters
     */
    public function handle_table_filters($query) {
        global $pagenow;
        
        if (!is_admin() || $pagenow !== 'edit.php') {
            return;
        }
        
        if (!isset($_GET['post_type']) || $_GET['post_type'] !== Constants::POST_TYPE_AFFILIATE_LINK) {
            return;
        }
        
        // Handle category filter
        if (!empty($_GET['filter_by_category'])) {
            $query->query_vars['tax_query'][] = array(
                'taxonomy' => Constants::TAXONOMY_AFFILIATE_CATEGORY,
                'field' => 'slug',
                'terms' => sanitize_text_field($_GET['filter_by_category'])
            );
        }
        
        // Handle tag filter
        if (!empty($_GET['filter_by_tag'])) {
            $query->query_vars['tax_query'][] = array(
                'taxonomy' => Constants::TAXONOMY_AFFILIATE_TAG,
                'field' => 'slug',
                'terms' => sanitize_text_field($_GET['filter_by_tag'])
            );
        }
        
        // Handle redirect type filter
        if (!empty($_GET['filter_by_redirect_type'])) {
            $redirect_type = sanitize_text_field($_GET['filter_by_redirect_type']);
            
            $query->query_vars['meta_query'][] = array(
                'relation' => 'OR',
                array(
                    'key' => Constants::META_REDIRECT_TYPE,
                    'value' => $redirect_type,
                    'compare' => '='
                ),
                array(
                    'key' => Constants::META_REDIRECT_TYPE,
                    'compare' => 'NOT EXISTS',
                    'value' => get_option(Constants::OPTION_REDIRECT_TYPE, Constants::REDIRECT_301) === $redirect_type ? 'match_default' : 'no_match'
                )
            );
            
            // If filtering by default redirect type, also include posts without meta
            if (get_option(Constants::OPTION_REDIRECT_TYPE, Constants::REDIRECT_301) === $redirect_type) {
                $query->query_vars['meta_query'][count($query->query_vars['meta_query']) - 1][1]['compare'] = 'NOT EXISTS';
            }
        }
    }
    
    /**
     * Add per page filter dropdown
     */
    public function add_per_page_filter() {
        global $typenow;
        
        if ($typenow === Constants::POST_TYPE_AFFILIATE_LINK) {
            $current_per_page = $this->get_current_per_page();
            $per_page_options = array(10, 20, 50, 100);
            
            echo '<select name="per_page" id="posts-per-page-select" style="float: none; margin-right: 5px;">';
            echo '<option value="">' . \esc_html__('Items per page', 'affiliate-hub') . '</option>';
            
            foreach ($per_page_options as $option) {
                printf(
                    '<option value="%s" %s>%s %s</option>',
                    \esc_attr((string) $option),
                    \selected($current_per_page, $option, false),
                    \esc_html((string) $option),
                    \esc_html__('items', 'affiliate-hub')
                );
            }
            
            echo '</select>';
            
            // Add JavaScript for automatic form submission
            echo '<script type="text/javascript">
                jQuery(document).ready(function($) {
                    $("#posts-per-page-select").on("change", function() {
                        var perPage = $(this).val();
                        var url = new URL(window.location.href);
                        if (perPage) {
                            url.searchParams.set("per_page", perPage);
                        } else {
                            url.searchParams.delete("per_page");
                        }
                        url.searchParams.delete("paged"); // Reset to first page
                        window.location.href = url.toString();
                    });
                });
            </script>';
        }
    }
    
    /**
     * Get current posts per page setting
     */
    private function get_current_per_page() {
        if (isset($_GET['per_page']) && is_numeric($_GET['per_page'])) {
            return intval($_GET['per_page']);
        }
        return 50; // Default
    }
    
    /**
     * Filter posts per page based on user selection
     */
    public function custom_posts_per_page($per_page, $post_type) {
        if ($post_type === Constants::POST_TYPE_AFFILIATE_LINK) {
            $custom_per_page = $this->get_current_per_page();
            if (in_array($custom_per_page, array(10, 20, 50, 100))) {
                return $custom_per_page;
            }
        }
        return $per_page;
    }
    
    /**
     * Force highlight custom menu items using JavaScript as fallback
     */
    public function highlight_custom_menu() {
        if (!isset($_GET['page'])) {
            return;
        }
        
        $page = sanitize_text_field($_GET['page']);
        $our_pages = array(
            'affiliate-hub-link-scanner',
            'affiliate-hub-analytics', 
            'affiliate-hub-settings'
        );
        
        if (!in_array($page, $our_pages)) {
            return;
        }
        
        // Add viewport meta tag for responsive design on admin pages
        if ($page === 'affiliate-hub-link-scanner') {
            echo '<meta name="viewport" content="width=device-width, initial-scale=1.0, user-scalable=yes">';
        }
        
        ?>
        <script type="text/javascript">
        jQuery(document).ready(function($) {
            // Remove current classes from all menu items
            $('#adminmenu .wp-has-submenu').removeClass('wp-has-current-submenu wp-menu-open current');
            $('#adminmenu .wp-submenu li').removeClass('current');
            
            // Find and highlight the correct parent menu
            var $parentMenu = $('#adminmenu a[href="edit.php?post_type=affiliate_link"]').closest('.wp-has-submenu');
            $parentMenu.addClass('wp-has-current-submenu wp-menu-open current');
            
            // Find and highlight the correct submenu item
            var currentPage = '<?php echo esc_js($page); ?>';
            var $submenuItem = $('#adminmenu a[href*="page=' + currentPage + '"]').closest('li');
            $submenuItem.addClass('current');
            
            // Also highlight the parent link
            $('#adminmenu a[href="edit.php?post_type=affiliate_link"]').parent().addClass('current');
        });
        </script>
        <?php
    }
    
    /**
     * Block external plugins from adding columns to affiliate links table
     * This method runs with high priority (999) to override other plugins
     */
    public function block_external_columns($columns) {
        // Define our allowed columns only
        $allowed_columns = array(
            'cb' => isset($columns['cb']) ? $columns['cb'] : '',
            'title' => isset($columns['title']) ? $columns['title'] : '',
            'link_id' => __('Link ID', 'affiliate-hub'),
            'cloaked_url' => __('Cloaked URL', 'affiliate-hub'),
            'destination_url' => __('Destination URL', 'affiliate-hub'),
            'redirect_type' => __('Redirect Type', 'affiliate-hub')
        );
        
        // Add taxonomy columns if they exist
        $category_key = 'taxonomy-' . Constants::TAXONOMY_AFFILIATE_CATEGORY;
        $tag_key = 'taxonomy-' . Constants::TAXONOMY_AFFILIATE_TAG;
        
        if (isset($columns[$category_key])) {
            $allowed_columns[$category_key] = $columns[$category_key];
        }
        
        if (isset($columns[$tag_key])) {
            $allowed_columns[$tag_key] = $columns[$tag_key];
        }
        
        // Add stats columns only if enabled
        if (get_option(Constants::OPTION_ENABLE_STATS, true) && !get_option('affiliate_hub_safe_mode', false)) {
            $allowed_columns['clicks'] = __('Clicks', 'affiliate-hub');
            $allowed_columns['last_clicked'] = __('Last Clicked', 'affiliate-hub');
        }
        
        // Add date column at the very end (last column)
        if (isset($columns['date'])) {
            $allowed_columns['date'] = $columns['date'];
        }
        
        // Log blocked columns for debugging (only in debug mode)
        if (defined('WP_DEBUG') && WP_DEBUG) {
            $blocked = array_diff_key($columns, $allowed_columns);
            if (!empty($blocked)) {
                error_log('Affiliate Hub: Blocked external columns from other plugins: ' . implode(', ', array_keys($blocked)));
            }
        }
        
        return $allowed_columns;
    }
    
    /**
     * Display GeoLocation page
     */
    public function display_geolocation_page() {
        error_log('display_geolocation_page called');
        error_log('Current screen: ' . get_current_screen()->id);
        
        if (isset($this->geo_admin)) {
            error_log('Using existing geo_admin instance');
            $this->geo_admin->render_admin_page();
        } elseif (class_exists('AffiliateHub\\Admin\\GeoLocationAdmin')) {
            error_log('Creating new GeoLocationAdmin instance');
            $geo_admin = new \AffiliateHub\Admin\GeoLocationAdmin();
            $geo_admin->render_admin_page();
        } else {
            error_log('GeoLocationAdmin class not found');
            echo '<div class="wrap">';
            echo '<h1>' . esc_html__('GeoLocation Analytics', 'affiliate-hub') . '</h1>';
            echo '<div class="notice notice-error"><p>' . esc_html__('GeoLocation module is not available.', 'affiliate-hub') . '</p></div>';
            echo '</div>';
        }
    }
}
