<?php
namespace AffiliateHub\Core;

/**
 * Ultra-fast redirect system with asynchronous tracking
 * 
 * This system provides fastest possible redirects while maintaining
 * full tracking capabilities through JavaScript callbacks
 */
class FastRedirect {
    
    /**
     * Initialize fast redirect system
     */
    public function init() {
        // Ultra-early hook for fastest redirects
        \add_action('init', array($this, 'early_redirect'), 1);
        
        // AJAX endpoint for asynchronous tracking
        \add_action('wp_ajax_affiliate_fast_track', array($this, 'ajax_track_click'));
        \add_action('wp_ajax_nopriv_affiliate_fast_track', array($this, 'ajax_track_click'));
        
        // Add tracking pixel generator
        \add_action('wp_ajax_affiliate_tracking_pixel', array($this, 'serve_tracking_pixel'));
        \add_action('wp_ajax_nopriv_affiliate_tracking_pixel', array($this, 'serve_tracking_pixel'));
    }
    
    /**
     * Ultra-fast redirect handler
     * Executes before WordPress loads most systems
     */
    public function early_redirect() {
        // Check if fast redirect is enabled (make it default)
        if (!\get_option(Constants::OPTION_ENABLE_FAST_REDIRECT, true)) { // Changed default to true
            return;
        }
        
        // Only run on frontend
        if (\is_admin()) {
            return;
        }
        
        // Get current URL path
        $request_uri = $_SERVER['REQUEST_URI'] ?? '';
        $path = trim(parse_url($request_uri, PHP_URL_PATH), '/');
        
        if (empty($path)) {
            return;
        }
        
        // Debug logging
        error_log("FastRedirect: Checking path: '$path'");
        
        // Also log to plugin directory for easier debugging
        $debug_file = dirname(__FILE__, 3) . '/debug.log';
        file_put_contents($debug_file, date('Y-m-d H:i:s') . " - FastRedirect: Checking path: '$path'\n", FILE_APPEND);
        
        // Check if this matches any affiliate link pattern
        $affiliate_link = $this->find_fast_redirect_link($path);
        
        if (!$affiliate_link) {
            error_log("FastRedirect: No affiliate link found for path: '$path'");
            return;
        }
        
        error_log("FastRedirect: Found affiliate link ID: " . $affiliate_link->ID);
        
        // Get destination URL
        $destination_url = \get_post_meta($affiliate_link->ID, Constants::META_DESTINATION_URL, true);
        
        if (!$destination_url || !filter_var($destination_url, FILTER_VALIDATE_URL)) {
            error_log("FastRedirect: Invalid destination URL for link ID: " . $affiliate_link->ID);
            return;
        }
        
        error_log("FastRedirect: Valid destination URL: " . $destination_url);
        
        // Get redirect type
        $redirect_type = \get_post_meta($affiliate_link->ID, Constants::META_REDIRECT_TYPE, true);
        if (!$redirect_type) {
            $redirect_type = \get_option(Constants::OPTION_REDIRECT_TYPE, Constants::REDIRECT_301);
        }
        
        // Generate tracking data for JavaScript callback
        $tracking_data = $this->prepare_tracking_data($affiliate_link->ID);
        error_log("FastRedirect: Prepared tracking data for link ID: " . $affiliate_link->ID);
        
        // Multi-layer tracking approach
        $tracking_methods = $this->setup_tracking($affiliate_link->ID, $tracking_data);
        error_log("FastRedirect: Setup tracking methods: " . print_r($tracking_methods, true));
        
        // IMMEDIATE TRACKING: Record click before redirect to ensure it's captured
        $this->record_immediate_click($affiliate_link->ID, $tracking_data);
        
        // Ultra-fast redirect with embedded tracking
        if (!headers_sent()) {
            error_log("FastRedirect: Redirecting to: " . $destination_url);
            header('Location: ' . $destination_url, true, intval($redirect_type));
            
            // Output minimal tracking HTML for maximum compatibility
            echo $this->get_comprehensive_tracking_html($affiliate_link->ID, $tracking_data, $tracking_methods);
            exit;
        }
    }
    
    /**
     * Setup multi-layer tracking for maximum compatibility
     */
    private function setup_tracking($link_id, $tracking_data) {
        $methods = array();
        
        // Method 1: Cookie for JavaScript
        if (!headers_sent()) {
            setcookie(
                'affiliate_track_' . $link_id,
                json_encode($tracking_data),
                time() + 300,
                '/',
                '',
                false,
                false
            );
            $methods[] = 'cookie';
        }
        
        // Method 2: Always prepare tracking pixel
        $methods[] = 'pixel';
        
        // Method 3: Always prepare AJAX fallback
        $methods[] = 'ajax';
        
        return $methods;
    }
    
    /**
     * Generate comprehensive tracking HTML with multiple fallback methods
     */
    private function get_comprehensive_tracking_html($link_id, $tracking_data, $methods) {
        $ajax_url = \admin_url('admin-ajax.php');
        $pixel_url = $ajax_url . '?' . http_build_query(array(
            'action' => 'affiliate_tracking_pixel',
            'link_id' => $link_id,
            'data' => base64_encode(json_encode($tracking_data))
        ));
        
        return '<!DOCTYPE html><html><head><meta charset="utf-8"></head><body style="margin:0;padding:0;">
        
        <!-- Method 1: Immediate pixel tracking (works for bots/crawlers) -->
        <img src="' . \esc_url($pixel_url) . '" width="1" height="1" style="display:none;" alt="">
        
        <!-- Method 2: JavaScript tracking with multiple fallbacks -->
        <script>
        (function() {
            var tracked = false;
            var trackData = ' . json_encode($tracking_data) . ';
            
            // Function to send tracking
            function sendTrack(method) {
                if (tracked) return;
                
                try {
                    var xhr = new XMLHttpRequest();
                    xhr.open("POST", "' . $ajax_url . '", true);
                    xhr.setRequestHeader("Content-Type", "application/x-www-form-urlencoded");
                    xhr.timeout = 1000; // 1 second timeout
                    
                    var params = "action=affiliate_fast_track&method=" + method + "&" + 
                        Object.keys(trackData).map(k => k + "=" + encodeURIComponent(trackData[k])).join("&");
                    
                    xhr.onload = function() {
                        if (xhr.status === 200) {
                            tracked = true;
                        }
                    };
                    
                    xhr.send(params);
                } catch(e) {
                    // Silent fail
                }
            }
            
            // Immediate tracking attempt
            sendTrack("immediate");
            
            // Fallback after short delay
            setTimeout(function() {
                if (!tracked) sendTrack("delayed");
            }, 100);
            
            // Final fallback on page unload
            window.addEventListener("beforeunload", function() {
                if (!tracked) sendTrack("unload");
            });
            
        })();
        </script>
        
        </body></html>';
    }
    
    /**
     * Find affiliate link for fast redirect
     * Uses optimized caching for speed
     */
    private function find_fast_redirect_link($path) {
        global $wpdb;
        
        error_log("FastRedirect: Looking for path: '$path'");
        
        // Try cache first
        $cache_key = 'affiliate_fast_redirect_' . md5($path);
        $cached_link = \wp_cache_get($cache_key, 'affiliate_hub_fast');
        
        if ($cached_link !== false) {
            error_log("FastRedirect: Found cached result for: '$path'");
            return $cached_link ?: null;
        }
        
        // Look for exact match with full slug
        $link = $wpdb->get_row($wpdb->prepare("
            SELECT p.ID, p.post_title
            FROM {$wpdb->posts} p
            INNER JOIN {$wpdb->postmeta} pm ON p.ID = pm.post_id
            WHERE p.post_type = %s 
            AND p.post_status = 'publish'
            AND pm.meta_key = '_affiliate_full_slug'
            AND pm.meta_value = %s
            LIMIT 1
        ", Constants::POST_TYPE_AFFILIATE_LINK, $path));
        
        error_log("FastRedirect: SQL result for full slug lookup: " . ($link ? "Found ID {$link->ID}" : "Not found"));
        
        if (!$link) {
            // Debug: Check if any affiliate links exist at all
            $all_links = $wpdb->get_results($wpdb->prepare("
                SELECT p.ID, p.post_title, p.post_name
                FROM {$wpdb->posts} p
                WHERE p.post_type = %s 
                AND p.post_status = 'publish'
                LIMIT 5
            ", Constants::POST_TYPE_AFFILIATE_LINK), ARRAY_A);
            
            error_log("FastRedirect: Found " . count($all_links) . " affiliate links total");
            if (!empty($all_links)) {
                error_log("FastRedirect: Sample links: " . print_r($all_links, true));
            }
            
            // Check if any have full_slug meta
            $links_with_slug = $wpdb->get_results($wpdb->prepare("
                SELECT p.ID, p.post_title, pm.meta_value as full_slug
                FROM {$wpdb->posts} p
                INNER JOIN {$wpdb->postmeta} pm ON p.ID = pm.post_id
                WHERE p.post_type = %s 
                AND p.post_status = 'publish'
                AND pm.meta_key = '_affiliate_full_slug'
                LIMIT 5
            ", Constants::POST_TYPE_AFFILIATE_LINK), ARRAY_A);
            
            error_log("FastRedirect: Found " . count($links_with_slug) . " links with _affiliate_full_slug meta");
            if (!empty($links_with_slug)) {
                error_log("FastRedirect: Links with slug: " . print_r($links_with_slug, true));
            }
        }
        
        if (!$link) {
            // Try with old-style prefix matching
            $parts = explode('/', $path);
            if (count($parts) >= 2) {
                $potential_prefix = $parts[0];
                $potential_slug = $parts[1];
                
                $link = $wpdb->get_row($wpdb->prepare("
                    SELECT p.ID, p.post_title
                    FROM {$wpdb->posts} p
                    WHERE p.post_type = %s 
                    AND p.post_status = 'publish'
                    AND p.post_name = %s
                    LIMIT 1
                ", Constants::POST_TYPE_AFFILIATE_LINK, $potential_slug));
            }
        }
        
        // Cache result (including null results)
        \wp_cache_set($cache_key, $link, 'affiliate_hub_fast', 300);
        
        return $link;
    }
    
    /**
     * Prepare tracking data for asynchronous collection
     */
    private function prepare_tracking_data($link_id) {
        return array(
            'link_id' => $link_id,
            'timestamp' => time(),
            'ip' => $this->get_client_ip(),
            'user_agent' => substr($_SERVER['HTTP_USER_AGENT'] ?? '', 0, 1000),
            'referer' => substr($_SERVER['HTTP_REFERER'] ?? '', 0, 500),
            'request_uri' => $_SERVER['REQUEST_URI'] ?? '',
            'nonce' => \wp_create_nonce('affiliate_fast_track_' . $link_id)
        );
    }
    
    /**
     * Get tracking pixel HTML for backup tracking
     */
    private function get_tracking_pixel_html($link_id, $tracking_data) {
        $tracking_url = \admin_url('admin-ajax.php') . '?' . http_build_query(array(
            'action' => 'affiliate_tracking_pixel',
            'link_id' => $link_id,
            'data' => base64_encode(json_encode($tracking_data))
        ));
        
        return '<!DOCTYPE html><html><head><meta charset="utf-8"></head><body>' .
               '<img src="' . \esc_url($tracking_url) . '" width="1" height="1" style="display:none;" alt="">' .
               '<script>
                try {
                    var trackData = ' . json_encode($tracking_data) . ';
                    var xhr = new XMLHttpRequest();
                    xhr.open("POST", "' . \admin_url('admin-ajax.php') . '", true);
                    xhr.setRequestHeader("Content-Type", "application/x-www-form-urlencoded");
                    xhr.send("action=affiliate_fast_track&" + Object.keys(trackData).map(k => k + "=" + encodeURIComponent(trackData[k])).join("&"));
                } catch(e) {}
                </script></body></html>';
    }
    
    /**
     * AJAX handler for asynchronous click tracking
     */
    public function ajax_track_click() {
        error_log("FastRedirect: AJAX track click called");
        
        // Get tracking data
        $link_id = intval($_POST['link_id'] ?? 0);
        
        if (!$link_id) {
            error_log("FastRedirect: Invalid link ID");
            \wp_send_json_error('Invalid link ID');
        }
        
        error_log("FastRedirect: Tracking click for link ID: $link_id");
        
        // Verify nonce
        $nonce = $_POST['nonce'] ?? '';
        if (!\wp_verify_nonce($nonce, 'affiliate_fast_track_' . $link_id)) {
            \wp_send_json_error('Invalid nonce');
        }
        
        // Track the click using existing system
        try {
            // Use EnhancedClickTracker if available
            if (class_exists('AffiliateHub\\Modules\\EnhancedClickTracker')) {
                error_log("FastRedirect: Using EnhancedClickTracker");
                $tracker = \AffiliateHub\Modules\EnhancedClickTracker::get_instance();
                
                // Prepare context from POST data
                $context = array(
                    'source' => 'fast_redirect_ajax',
                    'utm_source' => $_POST['utm_source'] ?? '',
                    'utm_medium' => $_POST['utm_medium'] ?? '',
                    'utm_campaign' => $_POST['utm_campaign'] ?? ''
                );
                
                $click_id = $tracker->track_click($link_id, $context);
                
                if ($click_id) {
                    error_log("FastRedirect: EnhancedClickTracker recorded click ID: $click_id");
                    \wp_send_json_success('Click tracked via EnhancedClickTracker');
                } else {
                    error_log("FastRedirect: EnhancedClickTracker failed");
                    \wp_send_json_error('EnhancedClickTracker failed');
                }
            } else {
                error_log("FastRedirect: Falling back to basic tracking");
                // Fallback to basic tracking
                $affiliate_link = new \AffiliateHub\Models\AffiliateLink($link_id);
            
                // Use existing tracking method but with provided data
                global $wpdb;
                $table_name = $wpdb->prefix . Constants::TABLE_LINK_CLICKS;
                
                // Check if table exists
                if ($wpdb->get_var("SHOW TABLES LIKE '$table_name'") !== $table_name) {
                    \wp_send_json_error('Tracking table not found');
                }
                
                // Insert tracking data
                $result = $wpdb->insert(
                    $table_name,
                    array(
                        'link_id' => $link_id,
                        'ip_address' => \sanitize_text_field($_POST['ip'] ?? ''),
                        'user_agent' => \sanitize_text_field(substr($_POST['user_agent'] ?? '', 0, 1000)),
                        'referer' => \esc_url_raw(substr($_POST['referer'] ?? '', 0, 500)),
                        'clicked_at' => \current_time('mysql')
                    ),
                    array('%d', '%s', '%s', '%s', '%s')
                );
                
                if ($result !== false) {
                    // Update click counts
                    $affiliate_link->increment_clicks();
                    \wp_send_json_success('Click tracked');
                } else {
                    \wp_send_json_error('Database error');
                }
            }
            
        } catch (Exception $e) {
            \wp_send_json_error('Tracking failed: ' . $e->getMessage());
        }
    }
    
    /**
     * Serve tracking pixel for backup tracking
     */
    public function serve_tracking_pixel() {
        $link_id = intval($_GET['link_id'] ?? 0);
        $data = json_decode(base64_decode($_GET['data'] ?? ''), true);
        
        if ($link_id && is_array($data)) {
            // Track using pixel method
            $this->track_pixel_click($link_id, $data);
        }
        
        // Serve 1x1 transparent pixel
        header('Content-Type: image/png');
        header('Cache-Control: no-cache, no-store, must-revalidate');
        header('Expires: Thu, 01 Jan 1970 00:00:00 GMT');
        
        // 1x1 transparent PNG in base64
        echo base64_decode('iVBORw0KGgoAAAANSUhEUgAAAAEAAAABCAYAAAAfFcSJAAAADUlEQVR42mP8/5+hHgAHggJ/PchI7wAAAABJRU5ErkJggg==');
        exit;
    }
    
    /**
     * Track click via pixel method
     */
    private function track_pixel_click($link_id, $data) {
        global $wpdb;
        $table_name = $wpdb->prefix . Constants::TABLE_LINK_CLICKS;
        
        if ($wpdb->get_var("SHOW TABLES LIKE '$table_name'") !== $table_name) {
            return false;
        }
        
        $wpdb->insert(
            $table_name,
            array(
                'link_id' => $link_id,
                'ip_address' => \sanitize_text_field($data['ip'] ?? ''),
                'user_agent' => \sanitize_text_field(substr($data['user_agent'] ?? '', 0, 1000)),
                'referer' => \esc_url_raw(substr($data['referer'] ?? '', 0, 500)),
                'clicked_at' => \current_time('mysql')
            ),
            array('%d', '%s', '%s', '%s', '%s')
        );
        
        // Update click counts
        try {
            $affiliate_link = new \AffiliateHub\Models\AffiliateLink($link_id);
            $affiliate_link->increment_clicks();
        } catch (Exception $e) {
            // Silent fail
        }
    }
    
    /**
     * Get client IP address
     */
    private function get_client_ip() {
        $ip_keys = array('HTTP_CLIENT_IP', 'HTTP_X_FORWARDED_FOR', 'REMOTE_ADDR');
        
        foreach ($ip_keys as $key) {
            if (array_key_exists($key, $_SERVER) === true) {
                foreach (explode(',', $_SERVER[$key]) as $ip) {
                    $ip = trim($ip);
                    
                    if (filter_var($ip, FILTER_VALIDATE_IP, 
                        FILTER_FLAG_NO_PRIV_RANGE | FILTER_FLAG_NO_RES_RANGE) !== false) {
                        return $ip;
                    }
                }
            }
        }
        
        return $_SERVER['REMOTE_ADDR'] ?? '0.0.0.0';
    }
    
    /**
     * Record click immediately before redirect to ensure tracking
     */
    private function record_immediate_click($link_id, $tracking_data) {
        try {
            error_log("FastRedirect: Recording immediate click for link ID: $link_id");
            
            // Use EnhancedClickTracker if available
            if (class_exists('AffiliateHub\\Modules\\EnhancedClickTracker')) {
                $tracker = \AffiliateHub\Modules\EnhancedClickTracker::get_instance();
                
                $context = array(
                    'source' => 'fast_redirect_immediate',
                    'utm_source' => $_GET['utm_source'] ?? '',
                    'utm_medium' => $_GET['utm_medium'] ?? '',
                    'utm_campaign' => $_GET['utm_campaign'] ?? ''
                );
                
                $result = $tracker->track_click($link_id, $context);
                
                if ($result['success'] ?? false) {
                    $click_id = $result['click_id'] ?? null;
                    error_log("FastRedirect: Immediate tracking successful - Click ID: $click_id");
                } else {
                    error_log("FastRedirect: Immediate tracking failed - " . ($result['error'] ?? 'Unknown error'));
                }
            } else {
                error_log("FastRedirect: EnhancedClickTracker not available for immediate tracking");
            }
            
        } catch (\Exception $e) {
            error_log("FastRedirect: Error in immediate tracking: " . $e->getMessage());
        }
    }
}
