<?php
/**
 * Main Plugin Class
 *
 * @package AffiliateHub
 * @subpackage Core
 */

namespace AffiliateHub\Core;

use AffiliateHub\Core\Constants;
use AffiliateHub\Core\I18n;
use AffiliateHub\Core\Admin;
use AffiliateHub\Models\AffiliateLink;
use AffiliateHub\Controllers\LinkController;
use AffiliateHub\Admin\AnalyticsDashboard;

/**
 * Main plugin class - Singleton pattern
 */
class Plugin {
    
    /**
     * Single instance of the plugin
     */
    private static $instance = null;
    
    /**
     * Plugin version
     */
    public $version;
    
    /**
     * Module manager instance
     */
    public $modules;
    
    /**
     * Fast redirect system instance
     */
    public $fast_redirect;
    
    /**
     * Admin instance
     */
    public $admin;
    
    /**
     * Get single instance
     */
    public static function get_instance() {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    /**
     * Constructor
     */
    private function __construct() {
        $this->version = defined('AFFILIATE_HUB_VERSION') ? AFFILIATE_HUB_VERSION : '1.0.0';
        $this->init();
    }
    
    /**
     * Initialize the plugin
     */
    private function init() {
        // Load text domain
    \add_action('init', array($this, 'load_textdomain'));
        
        // Initialize core components
    \add_action('init', array($this, 'init_components'), 5);
        
        // Initialize admin
        if (\is_admin()) {
            \add_action('init', array($this, 'init_admin'), 10);
        }
        
        // Initialize frontend
    \add_action('init', array($this, 'init_frontend'), 15);
        
        // Initialize fast redirect system
        $this->fast_redirect = new FastRedirect();
        $this->fast_redirect->init();
        
        // Add rewrite rules
    \add_action('init', array($this, 'add_rewrite_rules'), 20);
        
        // Handle template redirect (only if fast redirect is disabled)
    \add_action('template_redirect', array($this, 'handle_redirect_conditional'));
        
        // Custom permalink structure for affiliate links
    \add_filter('post_type_link', array($this, 'custom_post_type_link'), 10, 2);
        
        // Flush rewrite rules when affiliate link is saved
    \add_action('save_post', array($this, 'maybe_flush_rewrite_rules'), 10, 3);
        
        // Also flush on plugin activation to ensure our rules are active
    \register_activation_hook(AFFILIATE_HUB_FILE, 'flush_rewrite_rules');
        
        // Enqueue scripts and styles
    \add_action('wp_enqueue_scripts', array($this, 'enqueue_frontend_assets'));
    \add_action('admin_enqueue_scripts', array($this, 'enqueue_admin_assets'));
    }
    
    /**
     * Load plugin text domain
     */
    public function load_textdomain() {
    // Since WordPress 4.6, .org-hosted plugins auto-load translations.
    // Kept intentionally empty to satisfy Plugin Check guidance.
    // For non-.org installs, I18n::load_plugin_textdomain() can be wired if needed.
    }
    
    /**
     * Initialize core components
     */
    public function init_components() {
        // Check if safe mode should be enabled
        if (class_exists('AffiliateHub\\Core\\SafeMode')) {
            if (\AffiliateHub\Core\SafeMode::should_enable()) {
                \AffiliateHub\Core\SafeMode::enable();
            }
        }
        
        // Register custom post type and taxonomy
        $this->register_post_types();
        $this->register_taxonomies();
        
        // Initialize controllers only if not in safe mode
    if (!\get_option('affiliate_hub_safe_mode', false)) {
            if (class_exists('AffiliateHub\\Controllers\\LinkController')) {
                new \AffiliateHub\Controllers\LinkController();
            }
            // Note: AnalyticsDashboard is initialized in Admin.php
            
            // Initialize module manager if class exists
            if (class_exists('AffiliateHub\\Modules\\ModuleManager')) {
                $this->modules = new \AffiliateHub\Modules\ModuleManager();
            }
        }
    }
    
    /**
     * Initialize admin components
     */
    public function init_admin() {
        if (class_exists('AffiliateHub\\Core\\Admin')) {
            $this->admin = new \AffiliateHub\Core\Admin();
        }
    }
    
    /**
     * Initialize frontend components
     */
    public function init_frontend() {
        // Initialize click tracking for frontend (always active unless explicitly disabled)
        if (\get_option(Constants::OPTION_ENABLE_STATS, true) && !\is_admin()) {
            if (class_exists('AffiliateHub\\Modules\\ClickTracker')) {
                $click_tracker = new \AffiliateHub\Modules\ClickTracker();
                $click_tracker->init();
            }
        }
    }
    
    /**
     * Register custom post types
     */
    private function register_post_types() {
        // Get the current link prefix for rewrite slug
        $link_prefix = \get_option('affiliate_hub_link_prefix', 'go');
        
        // Register affiliate link post type
        \register_post_type(Constants::POST_TYPE_AFFILIATE_LINK, array(
            'labels' => array(
                'name' => \__('Affiliate Links', 'affiliate-hub'),
                'singular_name' => \__('Affiliate Link', 'affiliate-hub'),
                'add_new' => \__('Add New', 'affiliate-hub'),
                'add_new_item' => \__('Add New Link', 'affiliate-hub'),
                'edit_item' => \__('Edit Link', 'affiliate-hub'),
                'new_item' => \__('New Link', 'affiliate-hub'),
                'view_item' => \__('View Link', 'affiliate-hub'),
                'view_items' => \__('View Links', 'affiliate-hub'),
                'search_items' => \__('Search Links', 'affiliate-hub'),
                'not_found' => \__('No links found.', 'affiliate-hub'),
                'not_found_in_trash' => \__('No links found in Trash.', 'affiliate-hub'),
            ),
            'public' => false,
            'publicly_queryable' => true,
            'show_ui' => true,
            'show_in_menu' => false,
            'show_in_admin_bar' => true,
            'query_var' => true,
            'rewrite' => array('slug' => $link_prefix, 'with_front' => false),
            'capability_type' => 'post',
            'has_archive' => false,
            'hierarchical' => false,
            'menu_position' => 25,
            'menu_icon' => 'dashicons-admin-links',
            'supports' => array('title', 'custom-fields'),
            'taxonomies' => array(Constants::TAXONOMY_AFFILIATE_CATEGORY),
            'show_in_rest' => false
        ));
    }
    
    /**
     * Register taxonomies
     */
    private function register_taxonomies() {
        // Register affiliate link categories
        \register_taxonomy(Constants::TAXONOMY_AFFILIATE_CATEGORY, Constants::POST_TYPE_AFFILIATE_LINK, array(
            'labels' => array(
                'name' => \__('Link Categories', 'affiliate-hub'),
                'singular_name' => \__('Link Category', 'affiliate-hub'),
                'search_items' => \__('Search Categories', 'affiliate-hub'),
                'all_items' => \__('All Categories', 'affiliate-hub'),
                'parent_item' => \__('Parent Category', 'affiliate-hub'),
                'parent_item_colon' => \__('Parent Category:', 'affiliate-hub'),
                'edit_item' => \__('Edit Category', 'affiliate-hub'),
                'update_item' => \__('Update Category', 'affiliate-hub'),
                'add_new_item' => \__('Add New Category', 'affiliate-hub'),
                'new_item_name' => \__('New Category Name', 'affiliate-hub'),
                'menu_name' => \__('Categories', 'affiliate-hub')
            ),
            'hierarchical' => true,
            'show_ui' => true,
            'show_admin_column' => true,
            'query_var' => true,
            'rewrite' => array('slug' => 'affiliate-category'),
            'show_in_rest' => true,
            'public' => false,
            'publicly_queryable' => false,
            'show_in_nav_menus' => false,
            'show_tagcloud' => false,
            'capabilities' => array(
                'manage_terms' => 'manage_categories',
                'edit_terms'   => 'manage_categories',
                'delete_terms' => 'manage_categories',
                'assign_terms' => 'edit_posts',
            ),
        ));
        
        // Register affiliate link tags
        \register_taxonomy(Constants::TAXONOMY_AFFILIATE_TAG, Constants::POST_TYPE_AFFILIATE_LINK, array(
            'labels' => array(
                'name' => \__('Link Tags', 'affiliate-hub'),
                'singular_name' => \__('Link Tag', 'affiliate-hub'),
                'search_items' => \__('Search Tags', 'affiliate-hub'),
                'all_items' => \__('All Tags', 'affiliate-hub'),
                'edit_item' => \__('Edit Tag', 'affiliate-hub'),
                'update_item' => \__('Update Tag', 'affiliate-hub'),
                'add_new_item' => \__('Add New Tag', 'affiliate-hub'),
                'new_item_name' => \__('New Tag Name', 'affiliate-hub'),
                'menu_name' => \__('Tags', 'affiliate-hub'),
                'popular_items' => \__('Popular Tags', 'affiliate-hub'),
                'separate_items_with_commas' => \__('Separate tags with commas', 'affiliate-hub'),
                'add_or_remove_items' => \__('Add or remove tags', 'affiliate-hub'),
                'choose_from_most_used' => \__('Choose from the most used tags', 'affiliate-hub'),
                'not_found' => \__('No tags found.', 'affiliate-hub'),
            ),
            'hierarchical' => false,
            'show_ui' => true,
            'show_admin_column' => true,
            'query_var' => true,
            'rewrite' => array('slug' => 'affiliate-tag'),
            'show_in_rest' => true,
            'public' => false,
            'publicly_queryable' => false,
            'show_in_nav_menus' => false,
            'show_tagcloud' => true,
            'capabilities' => array(
                'manage_terms' => 'manage_categories',
                'edit_terms'   => 'manage_categories',
                'delete_terms' => 'manage_categories',
                'assign_terms' => 'edit_posts',
            ),
        ));
    }
    
    /**
     * Add rewrite rules for affiliate links
     */
    public function add_rewrite_rules() {
        // Keep the original rule for backward compatibility with global prefix
        $link_prefix = \get_option('affiliate_hub_link_prefix', 'go');
        if (!empty($link_prefix)) {
            \add_rewrite_rule(
                '^' . $link_prefix . '/([^/]+)/?$',
                'index.php?post_type=' . Constants::POST_TYPE_AFFILIATE_LINK . '&name=$matches[1]',
                'top'
            );
        }
        
        // Add specific rewrite rules for existing affiliate links
        $this->add_specific_affiliate_rules();
        
        // Add rewrite rule for preview links (for WordPress "Preview Changes" button)
        \add_rewrite_rule(
            '^$', // Match empty path (home page with query vars)
            'index.php',
            'top'
        );
        
        // Universal fallback rule for new affiliate links - with exclusions to prevent conflicts
        \add_rewrite_rule(
            '^(?!wp-admin|wp-content|wp-includes|wp-json|xmlrpc|robots\.txt|sitemap|feed|admin|login|register|dashboard|contact|donate|about|privacy|terms)([a-zA-Z0-9\-_/]+)/?$',
            'index.php?affiliate_full_path=$matches[1]',
            'bottom'
        );
        
        // Add query vars to WordPress (only once)
    \add_filter('query_vars', function($vars) {
            $vars[] = 'affiliate_full_path';
            $vars[] = 'affiliate_link';
            return $vars;
        });
    }
    
    /**
     * Add specific rewrite rules for existing affiliate links
     */
    private function add_specific_affiliate_rules() {
        global $wpdb;
        
        // Get all published affiliate links with their full slugs (cached to avoid repeated scans)
        $cache_key = 'ah_specific_rules_links';
        $affiliate_links = \wp_cache_get($cache_key, 'affiliate_hub');
        if (false === $affiliate_links) {
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching -- Read-only scan to build rewrite rules; cached below
            $affiliate_links = $wpdb->get_results($wpdb->prepare("
            SELECT p.ID, pm.meta_value as full_slug
            FROM {$wpdb->posts} p
            INNER JOIN {$wpdb->postmeta} pm ON p.ID = pm.post_id
            WHERE pm.meta_key = '_affiliate_full_slug'
            AND p.post_type = %s 
            AND p.post_status = 'publish'
            AND pm.meta_value != ''
        ", Constants::POST_TYPE_AFFILIATE_LINK));
            \wp_cache_set($cache_key, $affiliate_links, 'affiliate_hub', 300);
        }
        
        // Add a specific rule for each affiliate link
        foreach ($affiliate_links as $link) {
            $full_slug = trim($link->full_slug, '/');
            if ($full_slug) {
                // Escape special regex characters
                $escaped_slug = preg_quote($full_slug, '/');
                
                \add_rewrite_rule(
                    '^' . $escaped_slug . '/?$',
                    'index.php?affiliate_full_path=' . urlencode($full_slug),
                    'top'
                );
            }
        }
    }
    
    /**
     * Conditional redirect handler - only runs if fast redirect is disabled
     */
    public function handle_redirect_conditional() {
        // Skip if fast redirect is enabled (it handles redirects earlier)
        if (\get_option(Constants::OPTION_ENABLE_FAST_REDIRECT, true)) {
            return;
        }
        
        // Use the original redirect logic
        $this->handle_redirect();
    }
    
    /**
     * Handle affiliate link redirects (original method)
     */
    public function handle_redirect() {
        global $wp_query;
        
        $affiliate_full_path = \get_query_var('affiliate_full_path');
        
        // Check for full path custom routing first
        if ($affiliate_full_path) {
            // IMPORTANT: Check if this is an existing WordPress page/post first
            $existing_page = \get_page_by_path($affiliate_full_path);
            $existing_post = \get_page_by_path($affiliate_full_path, OBJECT, 'post');
            
            if ($existing_page || $existing_post) {
                return; // Let WordPress handle its own content
            }
            
            // Look for a link with this full path
            $post = $this->find_link_by_full_path($affiliate_full_path);
            
            if ($post && class_exists('AffiliateHub\\Models\\AffiliateLink')) {
                try {
                    $affiliate_link = new \AffiliateHub\Models\AffiliateLink($post->ID);
                    $affiliate_link->redirect();
                    return; // Stop processing
                } catch (\Exception $e) {
                    // Don't die here, let WordPress handle normally
                }
            } else {
                // CRITICAL: Set 404 if no affiliate link AND no WordPress page found
                $wp_query->set_404();
                \status_header(404);
                return;
            }
        }
        
        // Handle standard WordPress post type routing (including preview)
        else if (\is_singular(Constants::POST_TYPE_AFFILIATE_LINK)) {
            $post = \get_queried_object();
            
            if ($post && class_exists('AffiliateHub\\Models\\AffiliateLink')) {
                try {
                    // Validate post ID before creating model
                    if (!$post->ID || $post->post_type !== Constants::POST_TYPE_AFFILIATE_LINK) {
                        \wp_die(\esc_html__('Invalid affiliate link.', 'affiliate-hub'), \esc_html__('Not Found', 'affiliate-hub'), array('response' => 404));
                        return;
                    }
                    
                    $affiliate_link = new \AffiliateHub\Models\AffiliateLink($post->ID);
                    $affiliate_link->redirect();
                } catch (\InvalidArgumentException $e) {
                    // Invalid link arguments
                    \wp_die(\esc_html__('Affiliate link not found.', 'affiliate-hub'), \esc_html__('Not Found', 'affiliate-hub'), array('response' => 404));
                } catch (\Exception $e) {
                    // Other errors
                    \wp_die(\esc_html__('Affiliate link error.', 'affiliate-hub'), \esc_html__('Error', 'affiliate-hub'), array('response' => 500));
                }
            }
        }
    }
    
    /**
     * Find affiliate link by full path
     */
    private function find_link_by_full_path($full_path) {
        global $wpdb;
        
        // Clean the path
        $full_path = trim($full_path, '/');
        
        // First try to find by full slug meta field (short cache)
        $cache_key = 'ah_find_full_path_' . md5($full_path);
        $post_id = \wp_cache_get($cache_key, 'affiliate_hub');
        if (false === $post_id) {
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching -- Targeted lookup on indexed meta
            $post_id = $wpdb->get_var($wpdb->prepare("
            SELECT p.ID 
            FROM {$wpdb->posts} p
            INNER JOIN {$wpdb->postmeta} pm ON p.ID = pm.post_id
            WHERE pm.meta_key = '_affiliate_full_slug'
            AND pm.meta_value = %s
            AND p.post_type = %s 
            AND p.post_status = 'publish'
        ", $full_path, Constants::POST_TYPE_AFFILIATE_LINK));
            if ($post_id) { \wp_cache_set($cache_key, $post_id, 'affiliate_hub', 60); }
        }
        
        if ($post_id) {
            return \get_post($post_id);
        }
        
        // Try to find by exact match as post_name (for simple slugs without prefix)
    // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching -- Targeted fallback lookup
    $post_id = $wpdb->get_var($wpdb->prepare("
            SELECT p.ID 
            FROM {$wpdb->posts} p
            LEFT JOIN {$wpdb->postmeta} pm ON p.ID = pm.post_id AND pm.meta_key = '_affiliate_custom_prefix'
            WHERE p.post_name = %s 
            AND p.post_type = %s 
            AND p.post_status = 'publish'
            AND (pm.meta_value = '' OR pm.meta_value IS NULL)
        ", $full_path, Constants::POST_TYPE_AFFILIATE_LINK));
        
        if ($post_id) {
            return \get_post($post_id);
        }
        
        return null;
    }
    
    /**
     * Enqueue frontend assets
     */
    public function enqueue_frontend_assets() {
        // Frontend assets are handled by ClickTracker module
        // to avoid conflicts and duplication
    }
    
    /**
     * Enqueue admin assets
     */
    public function enqueue_admin_assets($hook) {
        // Only load on affiliate link pages to avoid conflicts
        $allowed_hooks = array(
            'post.php',
            'post-new.php',
            'edit.php',
            'affiliate-hub_page_affiliate-hub-analytics',
            'toplevel_page_affiliate-hub'
        );
        
        // Check if current page should load admin assets
        $load_assets = false;
        
        // Check hook
        foreach ($allowed_hooks as $allowed_hook) {
            if (strpos($hook, $allowed_hook) !== false) {
                $load_assets = true;
                break;
            }
        }
        
        // Check if we're editing affiliate links
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Read-only admin screen detection
        if (isset($_GET['post_type'])) {
            $pt = \sanitize_text_field(\wp_unslash($_GET['post_type']));
            if ($pt === Constants::POST_TYPE_AFFILIATE_LINK) {
                $load_assets = true;
            }
        }
        
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Read-only admin screen detection
        if (isset($_GET['post'])) {
            $post_id = (int) \absint($_GET['post']);
            $post_type = $post_id ? \get_post_type($post_id) : '';
            if ($post_type === Constants::POST_TYPE_AFFILIATE_LINK) {
                $load_assets = true;
            }
        }
        
        if (!$load_assets) {
            return; // Don't load on other admin pages
        }
        
        \wp_enqueue_script(
            'affiliate-hub-admin',
            AFFILIATE_HUB_URL . 'assets/js/admin.js',
            array('jquery'),
            $this->version,
            true
        );
        
        \wp_localize_script('affiliate-hub-admin', 'affiliateHubAdmin', array(
            'ajaxUrl' => \admin_url('admin-ajax.php'),
            'nonce' => \wp_create_nonce('affiliate_hub_admin_nonce'),
            'homeUrl' => \trailingslashit(\home_url()),
            'linkPrefix' => \get_option(Constants::OPTION_LINK_PREFIX, 'go'),
            'postType' => Constants::POST_TYPE_AFFILIATE_LINK,
            // Translatable strings
            'testing' => \__('Testing...', 'affiliate-hub'),
            'working' => \__('Working!', 'affiliate-hub'),
            'error' => \__('Error', 'affiliate-hub'),
            'test_link' => \__('Test Link', 'affiliate-hub'),
            'copied' => \__('Copied!', 'affiliate-hub'),
            'copy_link' => \__('Copy Link', 'affiliate-hub'),
            'confirm_delete' => \__('Are you sure you want to delete the selected links?', 'affiliate-hub'),
            'clicks' => \__('Clicks', 'affiliate-hub'),
            'invalidUrl' => \__('Please enter a valid destination URL first.', 'affiliate-hub')
        ));
        
        \wp_enqueue_style(
            'affiliate-hub-admin',
            AFFILIATE_HUB_URL . 'assets/css/admin.css',
            array(),
            $this->version
        );
    }
    
    /**
     * Flush rewrite rules when affiliate link is saved or published
     * 
     * @param int $post_id Post ID
     * @param WP_Post $post Post object
     * @param bool $update Whether this is an update or a new post
     */
    public function maybe_flush_rewrite_rules($post_id, $post, $update) {
        // Only for affiliate links
        if ($post->post_type !== Constants::POST_TYPE_AFFILIATE_LINK) {
            return;
        }
        
        // Only if post is published or being published
        if ($post->post_status !== 'publish') {
            return;
        }
        
        // Avoid infinite loops and auto-saves
    if (\wp_is_post_revision($post_id) || \wp_is_post_autosave($post_id)) {
            return;
        }
        
        // Force complete rewrite rules regeneration
        $this->regenerate_affiliate_rewrite_rules();
    }
    
    /**
     * Regenerate all affiliate rewrite rules
     * This ensures new links have their rules immediately
     */
    public function regenerate_affiliate_rewrite_rules() {
        // Remove all existing rewrite rules
        global $wp_rewrite;
        $wp_rewrite->flush_rules(false);
        
        // Re-add our rewrite rules (this will call add_rewrite_rules again)
        $this->add_rewrite_rules();
        
        // Final flush to commit changes
    \flush_rewrite_rules(false);
    }
    
    /**
     * Custom permalink structure for affiliate links
     * This ensures WordPress generates the correct URLs for our affiliate links
     * 
     * @param string $post_link The post's permalink
     * @param WP_Post $post The post object
     * @return string Modified permalink
     */
    public function custom_post_type_link($post_link, $post) {
        if ($post->post_type !== Constants::POST_TYPE_AFFILIATE_LINK) {
            return $post_link;
        }
        
        // Get the full slug from meta
    $full_slug = \get_post_meta($post->ID, '_affiliate_full_slug', true);
        
        if ($full_slug) {
            // Use the full slug as stored in meta
            return \trailingslashit(\home_url()) . trim($full_slug, '/') . '/';
        }
        
        // Fallback to post_name if no full slug
    return \trailingslashit(\home_url()) . $post->post_name . '/';
    }
}