<?php
/**
 * Emergency Safe Mode Functions
 * 
 * @package AffiliateHub
 * @subpackage Core
 */

namespace AffiliateHub\Core;

/**
 * Emergency safe mode functionality
 */
class SafeMode {
    
    /**
     * Check if safe mode should be enabled
     * 
     * @return bool
     */
    public static function should_enable() {
        // Check memory
        $memory_limit = wp_convert_hr_to_bytes(ini_get('memory_limit'));
        $memory_usage = memory_get_usage(true);
        $available_memory = $memory_limit - $memory_usage;
        
        // Enable safe mode if less than 64MB available
        if ($available_memory < 67108864) {
            return true;
        }
        
        // Check if manually enabled
        if (get_option('affiliate_hub_safe_mode', false)) {
            return true;
        }
        
        return false;
    }
    
    /**
     * Enable safe mode
     */
    public static function enable() {
        update_option('affiliate_hub_safe_mode', true);
        
        // Disable heavy features
        update_option('affiliate_hub_enable_stats', false);
    }
    
    /**
     * Disable safe mode
     */
    public static function disable() {
        delete_option('affiliate_hub_safe_mode');
        
        // Re-enable features
        update_option('affiliate_hub_enable_stats', true);
    }
    
    /**
     * Check if safe mode is active
     * 
     * @return bool
     */
    public static function is_active() {
        return get_option('affiliate_hub_safe_mode', false);
    }
    
    /**
     * Get safe mode status message
     * 
     * @return string
     */
    public static function get_status_message() {
        if (self::is_active()) {
            $memory_limit = ini_get('memory_limit');
            $memory_usage = size_format(memory_get_usage(true));
            $peak_usage = size_format(memory_get_peak_usage(true));
            
            return sprintf(
                /* translators: 1: memory limit, 2: current memory usage, 3: peak memory usage */
                __('Affiliate Hub is running in safe mode due to memory constraints. Memory limit: %1$s, Current usage: %2$s, Peak usage: %3$s. Some features are disabled.', 'affiliate-hub'),
                $memory_limit,
                $memory_usage,
                $peak_usage
            );
        }
        
        return '';
    }
    
    /**
     * Get memory info for diagnostics
     * 
     * @return array Memory statistics
     */
    public static function get_memory_info() {
        $memory_limit = wp_convert_hr_to_bytes(ini_get('memory_limit'));
        $memory_usage = memory_get_usage(true);
        $peak_usage = memory_get_peak_usage(true);
        $available_memory = $memory_limit - $memory_usage;
        
        return array(
            'limit' => $memory_limit,
            'usage' => $memory_usage,
            'peak' => $peak_usage,
            'available' => $available_memory,
            'usage_percentage' => ($memory_usage / $memory_limit) * 100,
            'safe_mode_needed' => $available_memory < 67108864 // 64MB
        );
    }
}
