<?php
/**
 * Affiliate Link Model
 *
 * @package AffiliateHub
 * @subpackage Models
 */

namespace AffiliateHub\Models;

use AffiliateHub\Core\Constants;

/**
 * Affiliate Link Model
 */
class AffiliateLink {
    
    /**
     * Post ID
     *
     * @var int
     */
    private $id;
    
    /**
     * Post object
     *
     * @var \WP_Post
     */
    private $post;
    
    /**
     * Constructor
     *
     * @param int $id Post ID
     */
    public function __construct($id) {
        $this->id = intval($id);
        
        // Validate ID first to prevent unnecessary queries
        if ($this->id <= 0) {
            throw new \InvalidArgumentException('Invalid post ID');
        }
        
        $this->post = get_post($this->id);
        
        // Validate post exists and is correct type
        if (!$this->post || $this->post->post_type !== Constants::POST_TYPE_AFFILIATE_LINK) {
            throw new \InvalidArgumentException('Invalid affiliate link post');
        }
    }
    
    /**
     * Get the post ID
     *
     * @return int
     */
    public function get_id() {
        return $this->id;
    }
    
    /**
     * Get the post object
     *
     * @return \WP_Post|null
     */
    public function get_post() {
        return $this->post;
    }
    
    /**
     * Get the title
     *
     * @return string
     */
    public function get_title() {
        return $this->post ? $this->post->post_title : '';
    }
    
    /**
     * Get the slug
     *
     * @return string
     */
    public function get_slug() {
        return $this->post ? $this->post->post_name : '';
    }
    
    /**
     * Get the destination URL
     *
     * @return string
     */
    public function get_destination_url() {
        return get_post_meta($this->id, Constants::META_DESTINATION_URL, true);
    }
    
    /**
     * Set the destination URL
     *
     * @param string $url
     */
    public function set_destination_url($url) {
        update_post_meta($this->id, Constants::META_DESTINATION_URL, esc_url_raw($url));
    }
    
    /**
     * Get the cloaked URL
     *
     * @return string
     */
    public function get_cloaked_url() {
        // Check if this link has a full slug saved (new system)
        $full_slug = get_post_meta($this->id, '_affiliate_full_slug', true);
        
        if (!empty($full_slug)) {
            // Use the full slug path directly
            return home_url($full_slug . '/');
        }
        
        // Fallback to old system
        // Check if this link has a custom prefix setting
        $custom_prefix = get_post_meta($this->id, '_affiliate_custom_prefix', true);
        $meta_all = get_post_meta($this->id, '_affiliate_custom_prefix');
        $has_custom_setting = !empty($meta_all); // Meta key exists
        
        if ($has_custom_setting) {
            // This link was saved with custom prefix setting
            if ($custom_prefix === '') {
                // No prefix - direct URL
                return home_url($this->get_slug() . '/');
            } else {
                // Custom prefix
                return home_url($custom_prefix . '/' . $this->get_slug() . '/');
            }
        } else {
            // Use global prefix (backward compatibility)
            $link_prefix = get_option(Constants::OPTION_LINK_PREFIX, 'go');
            return home_url($link_prefix . '/' . $this->get_slug() . '/');
        }
    }
    
    /**
     * Get redirect type
     *
     * @return string
     */
    public function get_redirect_type() {
        $redirect_type = get_post_meta($this->id, Constants::META_REDIRECT_TYPE, true);
        return $redirect_type ?: get_option(Constants::OPTION_REDIRECT_TYPE, Constants::REDIRECT_301);
    }
    
    /**
     * Set redirect type
     *
     * @param string $type
     */
    public function set_redirect_type($type) {
        if (in_array($type, array_keys(Constants::get_redirect_types()))) {
            update_post_meta($this->id, Constants::META_REDIRECT_TYPE, $type);
        }
    }
    
    /**
     * Check if link should have nofollow
     *
     * @return bool
     */
    public function is_nofollow() {
        $nofollow = get_post_meta($this->id, Constants::META_NOFOLLOW, true);
        if ($nofollow === '') {
            return get_option(Constants::OPTION_NOFOLLOW_LINKS, true);
        }
        return (bool) $nofollow;
    }
    
    /**
     * Set nofollow
     *
     * @param bool $nofollow
     */
    public function set_nofollow($nofollow) {
        update_post_meta($this->id, Constants::META_NOFOLLOW, (bool) $nofollow);
    }
    
    /**
     * Check if link should have sponsored
     *
     * @return bool
     */
    public function is_sponsored() {
        $sponsored = get_post_meta($this->id, Constants::META_SPONSORED, true);
        if ($sponsored === '') {
            return get_option(Constants::OPTION_SPONSORED_LINKS, false);
        }
        return (bool) $sponsored;
    }
    
    /**
     * Set sponsored
     *
     * @param bool $sponsored
     */
    public function set_sponsored($sponsored) {
        update_post_meta($this->id, Constants::META_SPONSORED, (bool) $sponsored);
    }
    
    /**
     * Check if link should have tracking (ugc)
     *
     * @return bool
     */
    public function is_tracking() {
        $tracking = get_post_meta($this->id, Constants::META_TRACKING, true);
        if ($tracking === '') {
            return get_option(Constants::OPTION_TRACKING_LINKS, false);
        }
        return (bool) $tracking;
    }
    
    /**
     * Set tracking
     *
     * @param bool $tracking
     */
    public function set_tracking($tracking) {
        update_post_meta($this->id, Constants::META_TRACKING, (bool) $tracking);
    }
    
    /**
     * Check if link should open in new window
     *
     * @return bool
     */
    public function is_new_window() {
        $new_window = get_post_meta($this->id, Constants::META_NEW_WINDOW, true);
        if ($new_window === '') {
            return get_option(Constants::OPTION_OPEN_NEW_WINDOW, false);
        }
        return (bool) $new_window;
    }
    
    /**
     * Set new window
     *
     * @param bool $new_window
     */
    public function set_new_window($new_window) {
        update_post_meta($this->id, Constants::META_NEW_WINDOW, (bool) $new_window);
    }
    
    /**
     * Get click count
     *
     * @return int
     */
    public function get_click_count() {
        return (int) get_post_meta($this->id, Constants::META_CLICK_COUNT, true);
    }
    
    /**
     * Get unique click count
     *
     * @return int
     */
    public function get_unique_clicks() {
        global $wpdb;
        
        $table_name = $wpdb->prefix . Constants::TABLE_LINK_CLICKS;
        
        // Check if table exists
    // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching -- Table existence check for optional analytics table
    if ($wpdb->get_var("SHOW TABLES LIKE '$table_name'") !== $table_name) {
            return 0;
        }
        
    // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching -- Aggregate from custom analytics table
    $unique_clicks = $wpdb->get_var($wpdb->prepare(
            "SELECT COUNT(DISTINCT ip_address) FROM $table_name WHERE link_id = %d",
            $this->id
        ));
        
        return (int) $unique_clicks;
    }
    
    /**
     * Get categories for this link
     *
     * @return array
     */
    public function get_categories() {
        $terms = wp_get_post_terms($this->id, Constants::TAXONOMY_AFFILIATE_CATEGORY);
        return is_array($terms) ? $terms : array();
    }
    
    /**
     * Get categories as comma-separated string
     *
     * @return string
     */
    public function get_categories_string() {
        $categories = $this->get_categories();
        if (empty($categories)) {
            return '—';
        }
        
        $names = wp_list_pluck($categories, 'name');
        return implode(', ', $names);
    }
    
    /**
     * Increment click count
     */
    public function increment_clicks() {
        $current_count = $this->get_click_count();
        update_post_meta($this->id, Constants::META_CLICK_COUNT, $current_count + 1);
        update_post_meta($this->id, Constants::META_LAST_CLICKED, current_time('mysql'));
    }
    
    /**
     * Get last clicked date
     *
     * @return string
     */
    public function get_last_clicked() {
        return get_post_meta($this->id, Constants::META_LAST_CLICKED, true);
    }
    
    /**
     * Perform the redirect
     */
    public function redirect() {
        $destination_url = $this->get_destination_url();
        
        if (!$destination_url) {
            wp_die(esc_html__('Invalid affiliate link.', 'affiliate-hub'));
        }
        
        // Track the click if stats are enabled and memory is sufficient
        if (get_option(Constants::OPTION_ENABLE_STATS, true)) {
            $memory_limit = wp_convert_hr_to_bytes(ini_get('memory_limit'));
            $memory_usage = memory_get_usage(true);
            
            // Only track if we have sufficient memory (at least 32MB free)
            if (($memory_limit - $memory_usage) > 33554432) {
                try {
                    $this->track_click();
                } catch (\Exception $e) {
                    // Continue with redirect even if tracking fails
                }
            }
        }
        
        // Do the redirect
        $redirect_type = $this->get_redirect_type();
        wp_redirect($destination_url, intval($redirect_type));
        exit;
    }
    
    /**
     * Track a click using Enhanced Click Tracker
     */
    private function track_click() {
        try {
            // Use Enhanced Click Tracker for comprehensive analytics
            if (class_exists('\AffiliateHub\Modules\EnhancedClickTracker')) {
                $tracker = \AffiliateHub\Modules\EnhancedClickTracker::get_instance();
                
                $context = [
                    'source' => 'AffiliateLink',
                    'method' => 'redirect',
                    'post_title' => $this->get_title()
                ];
                
                $result = $tracker->track_click($this->id, $context);
                
                if ($result['success']) {
                    // Update click count in post meta
                    $this->increment_clicks();
                    return true;
                }
                
                error_log("AffiliateLink: Enhanced tracking failed - " . ($result['error'] ?? 'Unknown error'));
            }
            
            // Fallback to basic tracking if Enhanced tracker not available
            return $this->track_click_basic();
            
        } catch (\Exception $e) {
            error_log("AffiliateLink: Click tracking error - " . $e->getMessage());
            return $this->track_click_basic();
        }
    }
    
    /**
     * Basic click tracking (fallback method)
     */
    private function track_click_basic() {
        global $wpdb;
        
        // Check if table exists before trying to insert
        $table_name = $wpdb->prefix . Constants::TABLE_LINK_CLICKS;
        
    // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching -- Table existence check for optional analytics table
    if ($wpdb->get_var("SHOW TABLES LIKE '$table_name'") !== $table_name) {
            return false;
        }
        
        // Prepare data with limits to prevent memory issues
        $user_agent = '';
        if (isset($_SERVER['HTTP_USER_AGENT'])) {
            $user_agent = substr(sanitize_text_field($_SERVER['HTTP_USER_AGENT']), 0, 1000);
        }
        
        $referer = '';
        if (isset($_SERVER['HTTP_REFERER'])) {
            $referer = substr(esc_url_raw($_SERVER['HTTP_REFERER']), 0, 500);
        }
        
    // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery -- Insert via $wpdb->insert with explicit formats
    $result = $wpdb->insert(
            $table_name,
            array(
                'link_id' => $this->id,
                'ip_address' => $this->get_client_ip(),
                'user_agent' => $user_agent,
                'referer' => $referer,
                'clicked_at' => current_time('mysql')
            ),
            array('%d', '%s', '%s', '%s', '%s')
        );
        
        if ($result === false) {
            return false;
        }
        
        // Update click count
        $this->increment_clicks();
        
        return true;
    }
    
    /**
     * Get client IP address
     *
     * @return string
     */
    private function get_client_ip() {
        $ip_keys = array('HTTP_CLIENT_IP', 'HTTP_X_FORWARDED_FOR', 'REMOTE_ADDR');
        
        foreach ($ip_keys as $key) {
            if (array_key_exists($key, $_SERVER) === true) {
                foreach (explode(',', $_SERVER[$key]) as $ip) {
                    $ip = trim($ip);
                    
                    if (filter_var($ip, FILTER_VALIDATE_IP, 
                        FILTER_FLAG_NO_PRIV_RANGE | FILTER_FLAG_NO_RES_RANGE) !== false) {
                        return $ip;
                    }
                }
            }
        }
        
        return isset($_SERVER['REMOTE_ADDR']) ? $_SERVER['REMOTE_ADDR'] : '0.0.0.0';
    }
    
    /**
     * Validate the affiliate link
     *
     * @return bool|\WP_Error
     */
    public function validate() {
        if (!$this->post) {
            return new \WP_Error('invalid_post', __('Invalid post.', 'affiliate-hub'));
        }
        
        if ($this->post->post_type !== Constants::POST_TYPE_AFFILIATE_LINK) {
            return new \WP_Error('invalid_post_type', __('Invalid post type.', 'affiliate-hub'));
        }
        
        $destination_url = $this->get_destination_url();
        if (!$destination_url) {
            return new \WP_Error('missing_destination', __('Destination URL is required.', 'affiliate-hub'));
        }
        
        if (!filter_var($destination_url, FILTER_VALIDATE_URL)) {
            return new \WP_Error('invalid_url', __('Invalid destination URL.', 'affiliate-hub'));
        }
        
        return true;
    }
}
