<?php
/**
 * Click Tracker Module - ULTRA SIMPLE VERSION
 * Testing with hardcoded values first to isolate the problem
 *
 * @package AffiliateHub
 * @subpackage Modules
 */

namespace AffiliateHub\Modules;

use AffiliateHub\Core\Constants;

class ClickTracker implements ModuleInterface {
    
    public function init() {
        if (!get_option(Constants::OPTION_ENABLE_STATS, true)) {
            return;
        }
        
        add_action('wp_enqueue_scripts', array($this, 'enqueue_tracking_script'));
        add_action('wp_ajax_affiliate_hub_track_click', array($this, 'ajax_track_click'));
        add_action('wp_ajax_nopriv_affiliate_hub_track_click', array($this, 'ajax_track_click'));
        add_filter('the_content', array($this, 'add_click_tracking_to_content'));
        
        // Add test AJAX handler
        add_action('wp_ajax_affiliate_hub_test', array($this, 'ajax_test'));
        add_action('wp_ajax_nopriv_affiliate_hub_test', array($this, 'ajax_test'));
    }
    
    public function get_info() {
        return array(
            'name' => __('Click Tracker', 'affiliate-hub'),
            'description' => __('Ultra simple click tracker for debugging.', 'affiliate-hub'),
            'version' => '3.0.0',
            'author' => 'Affiliate Hub'
        );
    }
    
    public function is_enabled() {
        return get_option(Constants::OPTION_ENABLE_STATS, true);
    }
    
    public function enqueue_tracking_script() {
        wp_enqueue_script(
            'affiliate-hub-tracker',
            AFFILIATE_HUB_URL . 'assets/js/click-tracker.js',
            array('jquery'),
            AFFILIATE_HUB_VERSION,
            true
        );
        
        wp_localize_script('affiliate-hub-tracker', 'affiliateHubTracker', array(
            'ajax_url' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('affiliate_hub_tracker_nonce'),
            'linkPrefix' => get_option(Constants::OPTION_LINK_PREFIX, 'go')
        ));
    }
    
    public function add_click_tracking_to_content($content) {
        $link_prefix = get_option(Constants::OPTION_LINK_PREFIX, 'go');
        $home_url = home_url();
        
        $pattern = '#<a([^>]*?)href=["\'](' . preg_quote($home_url, '#') . '/' . preg_quote($link_prefix, '#') . '/[^"\']+)["\']([^>]*?)>(.*?)</a>#i';
        
        return preg_replace_callback($pattern, function($matches) {
            $before_href = $matches[1];
            $url = $matches[2];
            $after_href = $matches[3];
            $link_text = $matches[4];
            
            $link_prefix = get_option(Constants::OPTION_LINK_PREFIX, 'go');
            $pattern = '#/' . preg_quote($link_prefix, '#') . '/([^/]+)/?#';
            
            if (preg_match($pattern, $url, $slug_matches)) {
                $slug = $slug_matches[1];
                $post = get_page_by_path($slug, OBJECT, Constants::POST_TYPE_AFFILIATE_LINK);
                
                if ($post) {
                    $tracking_attr = ' data-affiliate-track="' . esc_attr($post->ID) . '"';
                    return '<a' . $before_href . 'href="' . esc_url($url) . '"' . $after_href . $tracking_attr . '>' . $link_text . '</a>';
                }
            }
            
            return $matches[0];
        }, $content);
    }
    
    public function ajax_track_click() {
        check_ajax_referer('affiliate_hub_tracker_nonce', 'nonce');
        
        $link_id = isset($_POST['link_id']) ? intval($_POST['link_id']) : 0;
        $slug = isset($_POST['slug']) ? sanitize_text_field($_POST['slug']) : '';
        
        // Get post
        $post = null;
        if ($link_id) {
            $post = get_post($link_id);
        } elseif ($slug) {
            $post = get_page_by_path($slug, OBJECT, Constants::POST_TYPE_AFFILIATE_LINK);
            if ($post) {
                $link_id = $post->ID;
            }
        }
        
        if (!$post || $post->post_type !== Constants::POST_TYPE_AFFILIATE_LINK) {
            wp_send_json_error('Invalid affiliate link');
        }
        
        // Track the click
        $result = $this->track_click($link_id);
        
        if ($result) {
            wp_send_json_success('Click tracked successfully');
        } else {
            wp_send_json_error('Failed to track click');
        }
    }
    
    private function track_click($link_id) {
        try {
            // Use Enhanced Click Tracker for comprehensive analytics
            if (class_exists('\AffiliateHub\Modules\EnhancedClickTracker')) {
                $tracker = \AffiliateHub\Modules\EnhancedClickTracker::get_instance();
                
                $context = [
                    'source' => 'ClickTracker',
                    'method' => 'ajax',
                    'utm_source' => $_GET['utm_source'] ?? '',
                    'utm_medium' => $_GET['utm_medium'] ?? '',
                    'utm_campaign' => $_GET['utm_campaign'] ?? ''
                ];
                
                $result = $tracker->track_click($link_id, $context);
                
                if ($result['success']) {
                    return true;
                }
                
                error_log("ClickTracker: Enhanced tracking failed - " . ($result['error'] ?? 'Unknown error'));
            }
            
            // Fallback to basic tracking if Enhanced tracker not available
            return $this->track_click_basic($link_id);
            
        } catch (\Exception $e) {
            error_log("ClickTracker: Click tracking error - " . $e->getMessage());
            return $this->track_click_basic($link_id);
        }
    }
    
    private function track_click_basic($link_id) {
        global $wpdb;
        
        $table_name = $wpdb->prefix . Constants::TABLE_LINK_CLICKS;
        
        // Get basic data
        $ip = $this->get_real_ip();
        $user_agent = $_SERVER['HTTP_USER_AGENT'] ?? '';
        $referer = $_SERVER['HTTP_REFERER'] ?? '';
        
        // Prepare basic click data
        $click_data = array(
            'link_id' => $link_id,
            'ip_address' => $ip,
            'user_agent' => $user_agent,
            'referer' => $referer,
            'is_unique' => 1,
            'clicked_at' => current_time('mysql')
        );
        
        // Apply geolocation filter (if GeoLocation module is active)
        $click_data = apply_filters('affiliate_hub_before_track_click', $click_data, $ip);
        
        // Prepare database insert with all fields
        $insert_data = array_merge($click_data, array(
            'country' => $click_data['country'] ?? '',
            'country_code' => $click_data['country_code'] ?? '',
            'city' => $click_data['city'] ?? '',
            'latitude' => $click_data['latitude'] ?? null,
            'longitude' => $click_data['longitude'] ?? null,
            'timezone' => $click_data['timezone'] ?? '',
            'browser' => $this->detect_browser($user_agent)['browser'] ?? '',
            'browser_version' => $this->detect_browser($user_agent)['version'] ?? '',
            'operating_system' => $this->detect_browser($user_agent)['os'] ?? '',
            'os_version' => '',
            'device_type' => $this->detect_device_type($user_agent),
            'device_brand' => '',
            'traffic_source' => $this->detect_traffic_source($referer),
            'utm_source' => $_GET['utm_source'] ?? '',
            'utm_medium' => $_GET['utm_medium'] ?? '',
            'utm_campaign' => $_GET['utm_campaign'] ?? ''
        ));
    
    // Insert to database
    // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery -- Insert via $wpdb->insert with explicit formats
    $result = $wpdb->insert($table_name, $insert_data, array(
            '%d', '%s', '%s', '%s', '%s', '%s', '%s', '%f', '%f', '%s', '%s', 
            '%s', '%s', '%s', '%s', '%s', '%s', '%s', '%s', '%s', '%d', '%s'
        ));
        
        if ($result) {
            $this->update_click_counts($link_id);
        }
        
        return $result !== false;
    }
    
    /**
     * Detect browser from user agent
     */
    private function detect_browser($user_agent) {
        if (empty($user_agent)) {
            return array('browser' => 'Unknown', 'version' => '', 'os' => 'Unknown');
        }
        
        // Detect browser
        if (strpos($user_agent, 'Chrome') !== false && strpos($user_agent, 'Edg') === false) {
            $browser = 'Chrome';
            preg_match('/Chrome\/([0-9.]+)/', $user_agent, $matches);
            $version = $matches[1] ?? '';
        } elseif (strpos($user_agent, 'Firefox') !== false) {
            $browser = 'Firefox';
            preg_match('/Firefox\/([0-9.]+)/', $user_agent, $matches);
            $version = $matches[1] ?? '';
        } elseif (strpos($user_agent, 'Safari') !== false && strpos($user_agent, 'Chrome') === false) {
            $browser = 'Safari';
            $version = '';
        } elseif (strpos($user_agent, 'Edg') !== false) {
            $browser = 'Edge';
            $version = '';
        } else {
            $browser = 'Unknown';
            $version = '';
        }
        
        // Detect OS
        if (strpos($user_agent, 'Windows') !== false) {
            $os = 'Windows';
        } elseif (strpos($user_agent, 'Mac') !== false) {
            $os = 'macOS';
        } elseif (strpos($user_agent, 'Android') !== false) {
            $os = 'Android';
        } elseif (strpos($user_agent, 'Linux') !== false) {
            $os = 'Linux';
        } else {
            $os = 'Unknown';
        }
        
        return array(
            'browser' => $browser,
            'version' => $version,
            'os' => $os
        );
    }
    
    /**
     * Detect device type from user agent
     */
    private function detect_device_type($user_agent) {
        if (empty($user_agent)) {
            return 'Desktop';
        }
        
        if (strpos($user_agent, 'Mobile') !== false || strpos($user_agent, 'Android') !== false) {
            return 'Mobile';
        } elseif (strpos($user_agent, 'Tablet') !== false || strpos($user_agent, 'iPad') !== false) {
            return 'Tablet';
        } else {
            return 'Desktop';
        }
    }
    
    /**
     * Detect traffic source from referer
     */
    private function detect_traffic_source($referer) {
        if (empty($referer)) {
            return 'Direct';
        }
        
        $domain = parse_url($referer, PHP_URL_HOST);
        
        if (strpos($domain, 'google') !== false) {
            return 'Google';
        } elseif (strpos($domain, 'facebook') !== false) {
            return 'Facebook';
        } elseif (strpos($domain, 'twitter') !== false) {
            return 'Twitter';
        } elseif (strpos($domain, 'linkedin') !== false) {
            return 'LinkedIn';
        } else {
            return 'Referral';
        }
    }
    
    /**
     * PHASE 1: Return hardcoded test values first
     * Once this works, we'll implement real detection
     */
    private function get_test_analytics($ip, $user_agent, $referer) {
        // STEP 1: Return hardcoded values to test database insertion
        $hardcoded_data = array(
            'country' => 'Poland',
            'city' => 'Warsaw', 
            'browser' => 'Chrome',
            'browser_version' => '128.0.0',
            'os' => 'Windows',
            'os_version' => '10',
            'device_type' => 'Desktop',
            'device_brand' => '',
            'traffic_source' => 'Direct',
            'utm_source' => '',
            'utm_medium' => '',
            'utm_campaign' => ''
        );
        
        // STEP 2: Try simple browser detection
        $browser_data = $this->simple_browser_detect($user_agent);
        $hardcoded_data['browser'] = $browser_data['browser'];
        $hardcoded_data['browser_version'] = $browser_data['version'];
        $hardcoded_data['os'] = $browser_data['os'];
        
        // STEP 3: Try simple location detection
        $location_data = $this->simple_location_detect($ip);
        $hardcoded_data['country'] = $location_data['country'];
        $hardcoded_data['city'] = $location_data['city'];
        
        return $hardcoded_data;
    }
    
    /**
     * Super simple browser detection
     */
    private function simple_browser_detect($user_agent) {
        $result = array(
            'browser' => 'Unknown',
            'version' => '',
            'os' => 'Unknown'
        );
        
        if (empty($user_agent)) {
            return $result;
        }
        
        // Detect browser
        if (strpos($user_agent, 'Chrome') !== false && strpos($user_agent, 'Edg') === false) {
            $result['browser'] = 'Chrome';
            if (preg_match('/Chrome\/([0-9.]+)/', $user_agent, $matches)) {
                $result['version'] = $matches[1];
            }
        } elseif (strpos($user_agent, 'Firefox') !== false) {
            $result['browser'] = 'Firefox';
            if (preg_match('/Firefox\/([0-9.]+)/', $user_agent, $matches)) {
                $result['version'] = $matches[1];
            }
        } elseif (strpos($user_agent, 'Safari') !== false && strpos($user_agent, 'Chrome') === false) {
            $result['browser'] = 'Safari';
        } elseif (strpos($user_agent, 'Edg') !== false) {
            $result['browser'] = 'Edge';
        }
        
        // Detect OS
        if (strpos($user_agent, 'Windows') !== false) {
            $result['os'] = 'Windows';
        } elseif (strpos($user_agent, 'Mac') !== false) {
            $result['os'] = 'macOS';
        } elseif (strpos($user_agent, 'Android') !== false) {
            $result['os'] = 'Android';
        } elseif (strpos($user_agent, 'Linux') !== false) {
            $result['os'] = 'Linux';
        }
        
        return $result;
    }
    
    /**
     * Super simple location detection
     */
    private function simple_location_detect($ip) {
        // For local IPs, return test values
        if ($ip === '127.0.0.1' || strpos($ip, '192.168.') === 0 || strpos($ip, '10.') === 0) {
            return array('country' => 'Poland', 'city' => 'Warsaw');
        }
        
        // Try simple API call
        $api_url = "http://ip-api.com/json/$ip?fields=status,country,city";
        
        $response = wp_remote_get($api_url, array('timeout' => 3));
        
        if (is_wp_error($response)) {
            return array('country' => 'Poland', 'city' => 'Warsaw'); // Fallback
        }
        
        $body = wp_remote_retrieve_body($response);
        
        $data = json_decode($body, true);
        
        if ($data && $data['status'] === 'success') {
            $result = array(
                'country' => $data['country'] ?? 'Poland',
                'city' => $data['city'] ?? 'Warsaw'
            );
            return $result;
        }
        
        return array('country' => 'Poland', 'city' => 'Warsaw');
    }
    
    /**
     * Simple IP detection
     */
    private function get_real_ip() {
        // Check common headers
        $ip_headers = array(
            'HTTP_X_FORWARDED_FOR',
            'HTTP_CLIENT_IP', 
            'HTTP_CF_CONNECTING_IP',
            'HTTP_X_REAL_IP',
            'REMOTE_ADDR'
        );
        
        foreach ($ip_headers as $header) {
            if (!empty($_SERVER[$header])) {
                $ip = trim(explode(',', $_SERVER[$header])[0]);
                if (filter_var($ip, FILTER_VALIDATE_IP)) {
                    return $ip;
                }
            }
        }
        
        $fallback = '127.0.0.1';
        return $fallback;
    }
    
    /**
     * Test AJAX handler to check if AJAX works at all
     */
    public function ajax_test() {
        wp_send_json_success('Test AJAX works!');
    }
    
    private function update_click_counts($link_id) {
        $current = get_post_meta($link_id, Constants::META_CLICK_COUNT, true);
        update_post_meta($link_id, Constants::META_CLICK_COUNT, intval($current) + 1);
        
        $current_unique = get_post_meta($link_id, Constants::META_UNIQUE_CLICKS, true);
        update_post_meta($link_id, Constants::META_UNIQUE_CLICKS, intval($current_unique) + 1);
        
        update_post_meta($link_id, Constants::META_LAST_CLICKED, current_time('mysql'));

        // Invalidate quick stats caches so dashboard updates promptly
        if (function_exists('delete_transient')) {
            \delete_transient('affiliate_hub_quick_stats');
            \delete_transient('affiliate_hub_top_links_5');
        }
    }
}
