<?php
namespace AffiliateHub\Modules;

use AffiliateHub\Core\Constants;
use AffiliateHub\Modules\UserAgentParser;

/**
 * Enhanced Click Tracker with comprehensive analytics
 * 
 * Features:
 * - 24-hour unique visitor fingerprinting
 * - Session management
 * - Cache-first geolocation
 * - Browser/OS detection
 * - Performance monitoring
 * - Unified tracking across all entry points
 */
class EnhancedClickTracker {
    
    private static $instance = null;
    private $geo_module = null;
    private $user_agent_parser = null;
    
    /**
     * Singleton instance
     */
    public static function get_instance() {
        if (self::$instance === null) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    private function __construct() {
        $this->init_dependencies();
    }
    
    /**
     * Initialize required modules
     */
    private function init_dependencies() {
        // Load GeoLocation module if available
        if (class_exists('\AffiliateHub\Modules\GeoLocation')) {
            try {
                // GeoLocation is not a singleton, create new instance
                $this->geo_module = new \AffiliateHub\Modules\GeoLocation();
                
                // Initialize if needed (GeoLocation implements ModuleInterface)
                if (method_exists($this->geo_module, 'is_enabled') && $this->geo_module->is_enabled()) {
                    if (method_exists($this->geo_module, 'init')) {
                        $this->geo_module->init();
                    }
                }
            } catch (\Exception $e) {
                error_log("EnhancedClickTracker: Failed to initialize GeoLocation - " . $e->getMessage());
                $this->geo_module = null;
            }
        }
        
        // Initialize User-Agent parser
        try {
            $this->user_agent_parser = new UserAgentParser();
            error_log("EnhancedClickTracker: UserAgentParser initialized successfully");
        } catch (\Exception $e) {
            error_log("EnhancedClickTracker: Failed to initialize UserAgentParser - " . $e->getMessage());
            $this->user_agent_parser = null;
        }
    }
    
    /**
     * Main tracking method - unified entry point
     * 
     * @param int $link_id Affiliate link ID
     * @param array $context Additional context data
     * @return array Tracking result with analytics data
     */
    public function track_click($link_id, $context = []) {
        $start_time = microtime(true);
        
        try {
            // Generate session fingerprint for 24h uniqueness
            $fingerprint = $this->generate_session_fingerprint($link_id);
            
            // Check if this is a unique click within 24h
            $is_unique_24h = $this->is_unique_within_24h($fingerprint);
            
            // Collect click data
            $click_data = $this->collect_click_data($link_id, $fingerprint, $is_unique_24h, $context);
            
            // Store click record
            $click_id = $this->store_click_record($click_data);
            
            // Update meta fields if click was successfully stored
            if ($click_id) {
                $this->update_link_meta_fields($link_id);
            }
            
            // Calculate processing time
            $processing_time = round((microtime(true) - $start_time) * 1000, 2);
            
            // Update processing time in database
            if ($click_id) {
                $this->update_processing_time($click_id, $processing_time);
            }
            
            return [
                'success' => true,
                'click_id' => $click_id,
                'is_unique_24h' => $is_unique_24h,
                'processing_time_ms' => $processing_time,
                'session_fingerprint' => $fingerprint,
                'geo_data' => $click_data['geo_data'] ?? null
            ];
            
        } catch (\Exception $e) {
            error_log("EnhancedClickTracker Error: " . $e->getMessage());
            
            return [
                'success' => false,
                'error' => $e->getMessage(),
                'processing_time_ms' => round((microtime(true) - $start_time) * 1000, 2)
            ];
        }
    }
    
    /**
     * Generate unique session fingerprint for 24h tracking
     * 
     * @param int $link_id
     * @return string SHA256 hash
     */
    private function generate_session_fingerprint($link_id) {
        $ip_address = $this->get_client_ip();
        $user_agent = $_SERVER['HTTP_USER_AGENT'] ?? '';
        $date = date('Y-m-d'); // 24h window
        
        $fingerprint_data = $ip_address . '|' . $user_agent . '|' . $link_id . '|' . $date;
        
        return hash('sha256', $fingerprint_data);
    }
    
    /**
     * Check if click is unique within 24h window
     * 
     * @param string $fingerprint
     * @return bool
     */
    private function is_unique_within_24h($fingerprint) {
        global $wpdb;
        
        $table_clicks = $wpdb->prefix . Constants::TABLE_LINK_CLICKS;
        
        $existing = $wpdb->get_var($wpdb->prepare(
            "SELECT id FROM $table_clicks 
             WHERE session_fingerprint = %s 
             AND clicked_at >= DATE_SUB(NOW(), INTERVAL 24 HOUR)
             LIMIT 1",
            $fingerprint
        ));
        
        return empty($existing);
    }
    
    /**
     * Collect comprehensive click data
     * 
     * @param int $link_id
     * @param string $fingerprint
     * @param bool $is_unique_24h
     * @param array $context
     * @return array
     */
    private function collect_click_data($link_id, $fingerprint, $is_unique_24h, $context) {
        $ip_address = $this->get_client_ip();
        $user_agent = $_SERVER['HTTP_USER_AGENT'] ?? '';
        $referer = $_SERVER['HTTP_REFERER'] ?? '';
        
        // Parse User-Agent for browser/OS info
        $ua_data = [];
        if ($this->user_agent_parser) {
            try {
                $ua_data = $this->user_agent_parser->parse($user_agent);
                error_log("EnhancedClickTracker: UA parsed successfully - Browser: " . ($ua_data['browser'] ?? 'null'));
            } catch (\Exception $e) {
                error_log("EnhancedClickTracker: UserAgent parsing failed - " . $e->getMessage());
                $ua_data = [
                    'browser' => 'Unknown',
                    'browser_version' => '',
                    'os' => 'Unknown',
                    'os_version' => '',
                    'device_type' => 'Desktop'
                ];
            }
        } else {
            error_log("EnhancedClickTracker: UserAgentParser not initialized - setting defaults");
            $ua_data = [
                'browser' => 'Unknown',
                'browser_version' => '',
                'os' => 'Unknown', 
                'os_version' => '',
                'device_type' => 'Desktop'
            ];
        }
        
        // Fallback: simple browser detection if UserAgentParser fails
        if (empty($ua_data['browser']) || $ua_data['browser'] === 'Unknown') {
            error_log("EnhancedClickTracker: Fallback browser detection for UA: " . substr($user_agent, 0, 100));
            $ua_data['browser'] = $this->simple_browser_detect($user_agent);
        }
        
        // Get geolocation data (cache-first approach)
        $geo_data = $this->get_geo_data($ip_address);
        $geo_info = $geo_data['data'] ? json_decode($geo_data['data'], true) : [];
        
        // UTM parameters from context or GET
        $utm_source = $context['utm_source'] ?? $_GET['utm_source'] ?? '';
        $utm_medium = $context['utm_medium'] ?? $_GET['utm_medium'] ?? '';
        $utm_campaign = $context['utm_campaign'] ?? $_GET['utm_campaign'] ?? '';
        
        // Traffic source detection
        $traffic_source = $this->detect_traffic_source($referer, $utm_source);
        
        // Debug browser data
        error_log("EnhancedClickTracker: Browser data before save - Browser: '" . ($ua_data['browser'] ?? 'null') . "', User Agent: '" . substr($user_agent, 0, 100) . "'");
        
        return [
            'link_id' => (int) $link_id,
            'ip_address' => $ip_address,
            'user_agent' => $user_agent,
            'referer' => $referer,
            'country' => $geo_info['country'] ?? '',
            'country_code' => $geo_info['country_code'] ?? '',
            'city' => $geo_info['city'] ?? '',
            'latitude' => !empty($geo_info['latitude']) ? (float) $geo_info['latitude'] : null,
            'longitude' => !empty($geo_info['longitude']) ? (float) $geo_info['longitude'] : null,
            'timezone' => $geo_info['timezone'] ?? '',
            'browser' => $ua_data['browser'] ?? '',
            'browser_version' => $ua_data['browser_version'] ?? '',
            'os' => $ua_data['os'] ?? '',
            'os_version' => $ua_data['os_version'] ?? '',
            'device_type' => $ua_data['device_type'] ?? 'Desktop',
            'device_brand' => '', // Could be enhanced in future
            'traffic_source' => $traffic_source,
            'utm_source' => $utm_source,
            'utm_medium' => $utm_medium,
            'utm_campaign' => $utm_campaign,
            'is_unique' => $is_unique_24h ? 1 : 0, // Legacy column
            'session_fingerprint' => $fingerprint,
            'is_unique_24h' => $is_unique_24h ? 1 : 0,
            'geo_provider' => $geo_data['provider'] ?? '',
            'geo_cached' => $geo_data['cached'] ? 1 : 0,
            'processing_time_ms' => 0, // Will be updated later
            'clicked_at' => current_time('mysql')
        ];
    }
    
    /**
     * Detect traffic source from referrer and UTM
     * 
     * @param string $referer
     * @param string $utm_source
     * @return string
     */
    private function detect_traffic_source($referer, $utm_source) {
        // UTM source takes priority
        if (!empty($utm_source)) {
            return 'utm_' . sanitize_key($utm_source);
        }
        
        // No referrer = direct traffic
        if (empty($referer)) {
            return 'direct';
        }
        
        // Parse referrer domain
        $referer_host = parse_url($referer, PHP_URL_HOST);
        $site_host = parse_url(home_url(), PHP_URL_HOST);
        
        // Internal referrer
        if ($referer_host === $site_host) {
            return 'internal';
        }
        
        // Common search engines
        $search_engines = [
            'google' => 'google',
            'bing' => 'bing',
            'yahoo' => 'yahoo',
            'duckduckgo' => 'duckduckgo',
            'baidu' => 'baidu'
        ];
        
        foreach ($search_engines as $engine => $name) {
            if (strpos($referer_host, $engine) !== false) {
                return 'search_' . $name;
            }
        }
        
        // Social media
        $social_networks = [
            'facebook' => 'facebook',
            'twitter' => 'twitter',
            'linkedin' => 'linkedin',
            'instagram' => 'instagram',
            'youtube' => 'youtube',
            'tiktok' => 'tiktok'
        ];
        
        foreach ($social_networks as $network => $name) {
            if (strpos($referer_host, $network) !== false) {
                return 'social_' . $name;
            }
        }
        
        // Generic external referrer
        return 'referral';
    }
    
    /**
     * Store click record in database
     * 
     * @param array $click_data
     * @return int|false Click ID or false on failure
     */
    private function store_click_record($click_data) {
        global $wpdb;
        
        $table_clicks = $wpdb->prefix . Constants::TABLE_LINK_CLICKS;
        
        // Debug browser data being stored
        error_log("EnhancedClickTracker: Storing browser data - Browser: '" . ($click_data['browser'] ?? 'null') . "', Device: '" . ($click_data['device_type'] ?? 'null') . "'");
        
        $result = $wpdb->insert($table_clicks, $click_data);
        
        if ($result !== false) {
            return $wpdb->insert_id;
        }
        
        return false;
    }
    
    /**
     * Update processing time for click record
     * 
     * @param int $click_id
     * @param float $processing_time_ms
     */
    private function update_processing_time($click_id, $processing_time_ms) {
        global $wpdb;
        
        $table_clicks = $wpdb->prefix . Constants::TABLE_LINK_CLICKS;
        
        $wpdb->update(
            $table_clicks,
            ['processing_time_ms' => $processing_time_ms],
            ['id' => $click_id],
            ['%f'],
            ['%d']
        );
    }
    
    /**
     * Get geolocation data with cache-first approach
     * 
     * @param string $ip_address
     * @return array
     */
    private function get_geo_data($ip_address) {
        if (!$this->geo_module) {
            return [
                'data' => null,
                'provider' => null,
                'cached' => false
            ];
        }
        
        try {
            // Use the correct method name from GeoLocation module
            $geo_result = $this->geo_module->get_location_data($ip_address, 'enhanced_tracking');
            
            // GeoLocation returns data directly, not nested in 'data' key
            $geo_data = [
                'country' => $geo_result['country'] ?? '',
                'country_code' => $geo_result['country_code'] ?? '',
                'city' => $geo_result['city'] ?? '',
                'latitude' => $geo_result['latitude'] ?? null,
                'longitude' => $geo_result['longitude'] ?? null,
                'timezone' => $geo_result['timezone'] ?? ''
            ];
            
            return [
                'data' => wp_json_encode($geo_data),
                'provider' => $geo_result['provider'] ?? null,
                'cached' => $geo_result['cached'] ?? false
            ];
            
        } catch (\Exception $e) {
            error_log("EnhancedClickTracker Geo Error: " . $e->getMessage());
            
            return [
                'data' => null,
                'provider' => 'error',
                'cached' => false
            ];
        }
    }
    
    /**
     * Get client IP address with proxy support
     * 
     * @return string
     */
    private function get_client_ip() {
        $headers = [
            'HTTP_CF_CONNECTING_IP',     // Cloudflare
            'HTTP_X_REAL_IP',            // Nginx proxy
            'HTTP_X_FORWARDED_FOR',      // Load balancer/proxy
            'HTTP_X_FORWARDED',          // Proxy
            'HTTP_X_CLUSTER_CLIENT_IP',  // Cluster
            'HTTP_FORWARDED_FOR',        // Proxy
            'HTTP_FORWARDED',            // Proxy
            'REMOTE_ADDR'                // Standard
        ];
        
        foreach ($headers as $header) {
            if (!empty($_SERVER[$header])) {
                $ip = $_SERVER[$header];
                
                // Handle comma-separated IPs (X-Forwarded-For)
                if (strpos($ip, ',') !== false) {
                    $ip = trim(explode(',', $ip)[0]);
                }
                
                // Validate IP
                if (filter_var($ip, FILTER_VALIDATE_IP, FILTER_FLAG_NO_PRIV_RANGE | FILTER_FLAG_NO_RES_RANGE)) {
                    return $ip;
                }
            }
        }
        
        // Fallback to REMOTE_ADDR even if private
        return $_SERVER['REMOTE_ADDR'] ?? '0.0.0.0';
    }
    
    /**
     * Get analytics summary
     * 
     * @param array $filters
     * @return array
     */
    public function get_analytics_summary($filters = []) {
        global $wpdb;
        
        $table_clicks = $wpdb->prefix . Constants::TABLE_LINK_CLICKS;
        
        // Build WHERE clause from filters
        $where_conditions = ['1=1'];
        $where_values = [];
        
        if (!empty($filters['link_id'])) {
            $where_conditions[] = 'link_id = %d';
            $where_values[] = $filters['link_id'];
        }
        
        if (!empty($filters['date_from'])) {
            $where_conditions[] = 'clicked_at >= %s';
            $where_values[] = $filters['date_from'];
        }
        
        if (!empty($filters['date_to'])) {
            $where_conditions[] = 'clicked_at <= %s';
            $where_values[] = $filters['date_to'];
        }
        
        if (!empty($filters['unique_only'])) {
            $where_conditions[] = 'is_unique_24h = 1';
        }
        
        $where_clause = implode(' AND ', $where_conditions);
        
        $sql = "SELECT 
                    COUNT(*) as total_clicks,
                    COUNT(CASE WHEN is_unique_24h = 1 THEN 1 END) as unique_clicks,
                    COUNT(DISTINCT session_fingerprint) as unique_sessions,
                    AVG(processing_time_ms) as avg_processing_time,
                    COUNT(CASE WHEN geo_cached = 1 THEN 1 END) as geo_cache_hits,
                    COUNT(CASE WHEN geo_provider IS NOT NULL THEN 1 END) as geo_success_rate
                FROM $table_clicks 
                WHERE $where_clause";
        
        if (!empty($where_values)) {
            $sql = $wpdb->prepare($sql, $where_values);
        }
        
        return $wpdb->get_row($sql, ARRAY_A) ?: [];
    }
    
    /**
     * Get click trends data with proper filtering
     * 
     * @param array $filters
     * @return array
     */
    public function get_click_trends($filters = []) {
        global $wpdb;
        
        try {
            // Log the filters received
            error_log('=== GET_CLICK_TRENDS CALLED ===');
            error_log('Filters received: ' . print_r($filters, true));
            
            $table_clicks = $wpdb->prefix . Constants::TABLE_LINK_CLICKS;
            
            // Build WHERE clause from filters
            $where_conditions = ['1=1'];
            
            if (!empty($filters['link_id'])) {
                $where_conditions[] = 'link_id = ' . intval($filters['link_id']);
                error_log('Added link_id filter: ' . intval($filters['link_id']));
            }
            
            if (!empty($filters['date_from'])) {
                $date_from = sanitize_text_field($filters['date_from']);
                $where_conditions[] = "clicked_at >= '$date_from'";
                error_log('Added date_from filter: ' . $date_from);
            }
            
            if (!empty($filters['date_to'])) {
                $date_to = sanitize_text_field($filters['date_to']);
                $where_conditions[] = "clicked_at <= '$date_to'";
                error_log('Added date_to filter: ' . $date_to);
            }
            
            if (!empty($filters['unique_only'])) {
                $where_conditions[] = 'is_unique_24h = 1';
                error_log('Added unique_only filter');
            }
            
            // Fallback to days filter if no date range specified
            if (empty($filters['date_from']) && empty($filters['date_to'])) {
                $days = $filters['days'] ?? 30;
                $where_conditions[] = "clicked_at >= DATE_SUB(NOW(), INTERVAL $days DAY)";
                error_log('Using fallback days filter: ' . $days);
            }
            
            $where_clause = implode(' AND ', $where_conditions);
            
            error_log('Final WHERE clause: ' . $where_clause);
            
            $sql = "
                SELECT 
                    DATE(clicked_at) as date,
                    COUNT(*) as total_clicks,
                    COUNT(CASE WHEN is_unique_24h = 1 THEN 1 END) as unique_clicks
                FROM $table_clicks 
                WHERE $where_clause
                GROUP BY DATE(clicked_at)
                ORDER BY date ASC
            ";
            
            error_log('Final SQL: ' . $sql);
            $result = $wpdb->get_results($sql, ARRAY_A) ?: [];
            error_log('Trends result count: ' . count($result));
            error_log('Trends result sample: ' . print_r(array_slice($result, 0, 3), true));
            
            return $result;
            
        } catch (\Exception $e) {
            error_log('Exception in get_click_trends: ' . $e->getMessage());
            return [];
        } catch (\Error $e) {
            error_log('Fatal error in get_click_trends: ' . $e->getMessage());
            return [];
        }
    }
    
    /**
     * Simple browser detection fallback
     * 
     * @param string $user_agent
     * @return string
     */
    private function simple_browser_detect($user_agent) {
        if (empty($user_agent)) {
            return 'Unknown';
        }
        
        // Simple detection patterns
        if (strpos($user_agent, 'Edg/') !== false) {
            return 'Edge';
        } elseif (strpos($user_agent, 'Chrome/') !== false) {
            return 'Chrome';
        } elseif (strpos($user_agent, 'Firefox/') !== false) {
            return 'Firefox';
        } elseif (strpos($user_agent, 'Safari/') !== false && strpos($user_agent, 'Chrome') === false) {
            return 'Safari';
        } elseif (strpos($user_agent, 'Opera') !== false || strpos($user_agent, 'OPR/') !== false) {
            return 'Opera';
        } elseif (strpos($user_agent, 'MSIE') !== false || strpos($user_agent, 'Trident/') !== false) {
            return 'Internet Explorer';
        }
        
        return 'Unknown';
    }
    
    /**
     * Update affiliate link meta fields (total clicks count and last clicked)
     * 
     * @param int $link_id
     */
    private function update_link_meta_fields($link_id) {
        try {
            // Get current click count from meta
            $current_count = (int) get_post_meta($link_id, Constants::META_CLICK_COUNT, true);
            
            // Increment click count
            $new_count = $current_count + 1;
            update_post_meta($link_id, Constants::META_CLICK_COUNT, $new_count);
            
            // Update last clicked timestamp
            update_post_meta($link_id, Constants::META_LAST_CLICKED, current_time('mysql'));
            
            error_log("EnhancedClickTracker: Updated meta fields for link $link_id - Total clicks: $new_count");
            
        } catch (\Exception $e) {
            error_log("EnhancedClickTracker: Failed to update meta fields for link $link_id - " . $e->getMessage());
        }
    }
}
