<?php
/**
 * GeoIP2 Auto-installer
 * 
 * One-click installer for geoip2/geoip2 package without requiring composer
 * Downloads and installs GeoIP2 library automatically for WordPress users
 *
 * @package AffiliateHub
 * @subpackage Modules
 */

namespace AffiliateHub\Modules;

class GeoIP2AutoInstaller {
    
    private $vendor_dir;
    private $temp_dir;
    private $github_api_url = 'https://api.github.com/repos/maxmind/GeoIP2-php';
    private $backup_dir;
    
    public function __construct() {
        $upload_dir = \wp_upload_dir();
        $this->vendor_dir = AFFILIATE_HUB_PATH . 'vendor/';
        $this->temp_dir = $upload_dir['basedir'] . '/affiliate-hub/temp/';
        $this->backup_dir = $upload_dir['basedir'] . '/affiliate-hub/backups/';
        
        $this->ensure_directories();
    }
    
    /**
     * Check if native GeoIP2 is installed
     */
    public function is_native_installed() {
        $autoload_file = $this->vendor_dir . 'autoload.php';
        $reader_file = $this->vendor_dir . 'geoip2/Database/Reader.php';
        
        if (!file_exists($autoload_file) || !file_exists($reader_file)) {
            return false;
        }
        
        // Load autoloader if not already loaded
        $this->load_autoloader();
        
        // Check if class is available after loading autoloader
        return class_exists('GeoIp2\Database\Reader');
    }
    
    /**
     * Load GeoIP2 autoloader
     */
    public function load_autoloader() {
        $autoload_file = $this->vendor_dir . 'autoload.php';
        
        if (file_exists($autoload_file) && !class_exists('GeoIp2\Database\Reader')) {
            require_once $autoload_file;
        }
    }
    
    /**
     * Get installation status with details
     */
    public function get_status() {
        $native_installed = $this->is_native_installed();
        $fallback_available = class_exists('GeoIp2_Database_Reader');
        
        $status = array(
            'native_installed' => $native_installed,
            'fallback_available' => $fallback_available,
            'current_mode' => $native_installed ? 'native' : 'fallback',
            'vendor_dir_writable' => is_writable(dirname($this->vendor_dir)),
            'temp_dir_writable' => is_writable(dirname($this->temp_dir)),
        );
        
        if ($native_installed) {
            $status['version'] = $this->get_installed_version();
            $status['install_date'] = $this->get_install_date();
        }
        
        return $status;
    }
    
    /**
     * Install GeoIP2 library automatically
     */
    public function install() {
        try {
            error_log('GeoIP2AutoInstaller: Starting installation process');
            
            // Step 1: Create backup if exists
            if ($this->is_native_installed()) {
                error_log('GeoIP2AutoInstaller: Creating backup of existing installation');
                $this->create_backup();
            }
            
            // Step 2: Get latest release info from GitHub
            error_log('GeoIP2AutoInstaller: Fetching latest release info from GitHub');
            $release_info = $this->get_latest_release();
            if (!$release_info) {
                throw new \Exception(\__('Failed to fetch latest GeoIP2 release information', 'affiliate-hub'));
            }
            error_log('GeoIP2AutoInstaller: Latest release: ' . $release_info['tag_name']);
            
            // Step 3: Download package
            error_log('GeoIP2AutoInstaller: Downloading package');
            $download_url = $release_info['zipball_url'];
            $zip_file = $this->download_package($download_url);
            if (!$zip_file) {
                throw new \Exception(\__('Failed to download GeoIP2 package', 'affiliate-hub'));
            }
            error_log('GeoIP2AutoInstaller: Package downloaded to: ' . $zip_file);
            
            // Step 4: Extract and install
            error_log('GeoIP2AutoInstaller: Extracting and installing');
            $this->extract_and_install($zip_file, $release_info['tag_name']);
            
            // Step 4.5: Install MaxMind DB Reader dependency separately
            error_log('GeoIP2AutoInstaller: Installing MaxMind DB Reader dependency');
            $this->install_maxmind_db_reader();
            
            // Step 5: Verify installation
            error_log('GeoIP2AutoInstaller: Verifying installation');
            $verification_result = $this->verify_installation();
            if (!$verification_result) {
                // Get detailed verification info for debugging
                $debug_info = $this->get_verification_debug_info();
                $error_message = \__('Installation verification failed', 'affiliate-hub') . '. Debug: ' . json_encode($debug_info);
                throw new \Exception($error_message);
            }
            
            // Step 6: Cleanup
            error_log('GeoIP2AutoInstaller: Cleaning up temp files');
            $this->cleanup_temp_files();
            
            // Step 7: Update install metadata
            error_log('GeoIP2AutoInstaller: Updating install metadata');
            $this->update_install_metadata($release_info);
            
            return array(
                'success' => true,
                'message' => \__('GeoIP2 library installed successfully!', 'affiliate-hub'),
                'version' => $release_info['tag_name'],
                'mode' => 'native'
            );
            
        } catch (\Exception $e) {
            error_log('GeoIP2AutoInstaller install exception: ' . $e->getMessage());
            error_log('GeoIP2AutoInstaller install trace: ' . $e->getTraceAsString());
            
            // Rollback on error
            $this->rollback_from_backup();
            $this->cleanup_temp_files();
            
            return array(
                'success' => false,
                'message' => $e->getMessage(),
                'mode' => 'fallback',
                'debug_info' => array(
                    'exception_type' => get_class($e),
                    'file' => $e->getFile(),
                    'line' => $e->getLine()
                )
            );
        } catch (\Error $e) {
            error_log('GeoIP2AutoInstaller install PHP error: ' . $e->getMessage());
            error_log('GeoIP2AutoInstaller install error trace: ' . $e->getTraceAsString());
            
            // Rollback on error
            $this->rollback_from_backup();
            $this->cleanup_temp_files();
            
            return array(
                'success' => false,
                'message' => 'PHP Error: ' . $e->getMessage(),
                'mode' => 'fallback',
                'debug_info' => array(
                    'error_type' => get_class($e),
                    'file' => $e->getFile(),
                    'line' => $e->getLine()
                )
            );
        }
    }
    
    /**
     * Get latest release from GitHub API
     */
    private function get_latest_release() {
        $response = \wp_remote_get($this->github_api_url . '/releases/latest', array(
            'timeout' => 15,
            'headers' => array(
                'User-Agent' => 'AffiliateHub-WordPress-Plugin',
                'Accept' => 'application/vnd.github.v3+json'
            )
        ));
        
        if (\is_wp_error($response)) {
            return false;
        }
        
        $body = \wp_remote_retrieve_body($response);
        $data = json_decode($body, true);
        
        if (!$data || !isset($data['tag_name'])) {
            return false;
        }
        
        return $data;
    }
    
    /**
     * Download package from GitHub
     */
    private function download_package($url, $filename = null) {
        if (!$filename) {
            $filename = 'geoip2-' . time() . '.zip';
        }
        $temp_file = $this->temp_dir . $filename;
        
        $response = \wp_remote_get($url, array(
            'timeout' => 60,
            'stream' => true,
            'filename' => $temp_file,
            'headers' => array(
                'User-Agent' => 'AffiliateHub-WordPress-Plugin'
            )
        ));
        
        if (\is_wp_error($response) || \wp_remote_retrieve_response_code($response) !== 200) {
            return false;
        }
        
        return file_exists($temp_file) ? $temp_file : false;
    }
    
    /**
     * Extract and install the package
     */
    private function extract_and_install($zip_file, $version) {
        if (!class_exists('ZipArchive')) {
            throw new \Exception(\__('ZipArchive class not available. Cannot extract package.', 'affiliate-hub'));
        }
        
        $zip = new \ZipArchive();
        $extract_path = $this->temp_dir . 'extract/';
        
        if ($zip->open($zip_file) !== TRUE) {
            throw new \Exception(\__('Failed to open downloaded package', 'affiliate-hub'));
        }
        
        // Extract to temp directory
        if (!$zip->extractTo($extract_path)) {
            $zip->close();
            throw new \Exception(\__('Failed to extract package', 'affiliate-hub'));
        }
        $zip->close();
        
        // Find extracted directory (GitHub creates folder like maxmind-GeoIP2-php-abc123)
        $extracted_dirs = glob($extract_path . 'maxmind-GeoIP2-php-*');
        if (empty($extracted_dirs)) {
            throw new \Exception(\__('Extracted package structure not recognized', 'affiliate-hub'));
        }
        
        $source_dir = $extracted_dirs[0];
        
        // Copy necessary files to vendor directory
        $this->copy_geoip2_files($source_dir);
        
        return true;
    }
    
    /**
     * Copy GeoIP2 files to vendor directory
     */
    private function copy_geoip2_files($source_dir) {
        error_log('GeoIP2AutoInstaller: Copying files from source: ' . $source_dir);
        
        // Create vendor structure
        $geoip2_target = $this->vendor_dir . 'geoip2/geoip2/';
        $maxmind_target = $this->vendor_dir . 'maxmind-db/reader/';
        
        \wp_mkdir_p($geoip2_target);
        \wp_mkdir_p($maxmind_target);
        
        error_log('GeoIP2AutoInstaller: Target directories created - geoip2: ' . $geoip2_target . ', maxmind: ' . $maxmind_target);
        
        // Copy GeoIP2 src directory
        $geoip2_src = $source_dir . '/src/';
        if (is_dir($geoip2_src)) {
            error_log('GeoIP2AutoInstaller: Copying GeoIP2 src from: ' . $geoip2_src);
            $this->recursive_copy($geoip2_src, $geoip2_target . 'src/');
        } else {
            error_log('GeoIP2AutoInstaller: GeoIP2 src directory not found: ' . $geoip2_src);
        }
        
        // Look for MaxMind\Db dependency
        $maxmind_db_source = $source_dir . '/vendor/maxmind/db-reader/src/';
        error_log('GeoIP2AutoInstaller: Looking for MaxMind dependency at: ' . $maxmind_db_source);
        
        if (is_dir($maxmind_db_source)) {
            error_log('GeoIP2AutoInstaller: Found MaxMind dependency, copying to: ' . $maxmind_target . 'src/');
            $this->recursive_copy($maxmind_db_source, $maxmind_target . 'src/');
        } else {
            error_log('GeoIP2AutoInstaller: MaxMind dependency not found at primary location, trying alternative');
            
            // Try alternative location
            $alt_source = $source_dir . '/vendor/maxmind-db/reader/src/';
            error_log('GeoIP2AutoInstaller: Trying alternative location: ' . $alt_source);
            
            if (is_dir($alt_source)) {
                error_log('GeoIP2AutoInstaller: Found MaxMind dependency at alternative location, copying');
                $this->recursive_copy($alt_source, $maxmind_target . 'src/');
            } else {
                error_log('GeoIP2AutoInstaller: MaxMind dependency not found at alternative location either');
                
                // List what's actually in vendor directory
                $vendor_path = $source_dir . '/vendor/';
                if (is_dir($vendor_path)) {
                    $vendor_contents = scandir($vendor_path);
                    error_log('GeoIP2AutoInstaller: Vendor directory contents: ' . json_encode($vendor_contents));
                    
                    // Try to find maxmind directory recursively
                    foreach ($vendor_contents as $item) {
                        if ($item != '.' && $item != '..' && is_dir($vendor_path . $item)) {
                            $item_contents = scandir($vendor_path . $item);
                            error_log('GeoIP2AutoInstaller: Contents of ' . $item . ': ' . json_encode($item_contents));
                        }
                    }
                } else {
                    error_log('GeoIP2AutoInstaller: Vendor directory does not exist: ' . $vendor_path);
                }
            }
        }
        
        // Create autoloader
        error_log('GeoIP2AutoInstaller: Creating autoloader');
        $this->create_autoloader();
    }
    
    /**
     * Recursively copy directory
     */
    private function recursive_copy($source, $destination) {
        if (!is_dir($source)) {
            return false;
        }
        
        \wp_mkdir_p($destination);
        
        $directory = opendir($source);
        while (($file = readdir($directory)) !== false) {
            if ($file == '.' || $file == '..') {
                continue;
            }
            
            $source_path = $source . $file;
            $dest_path = $destination . $file;
            
            if (is_dir($source_path)) {
                $this->recursive_copy($source_path . '/', $dest_path . '/');
            } else {
                copy($source_path, $dest_path);
            }
        }
        closedir($directory);
        
        return true;
    }
    
    /**
     * Create autoloader for installed packages
     */
    private function create_autoloader() {
        $autoloader_content = '<?php
/**
 * Auto-generated autoloader for GeoIP2 library
 * Created by AffiliateHub Auto-installer
 */

spl_autoload_register(function ($class) {
    $base_dir = __DIR__ . \'/\';
    $prefix_map = array(
        \'GeoIp2\\\\\' => $base_dir . \'geoip2/geoip2/src/\',
        \'MaxMind\\\\Db\\\\\' => $base_dir . \'maxmind-db/reader/src/\'
    );
    
    foreach ($prefix_map as $prefix => $dir) {
        if (strpos($class, $prefix) === 0) {
            $relative_class = substr($class, strlen($prefix));
            $file = $dir . str_replace(\'\\\\\', \'/\', $relative_class) . \'.php\';
            
            if (file_exists($file)) {
                require_once $file;
                return true;
            }
        }
    }
    
    return false;
});
';
        
        file_put_contents($this->vendor_dir . 'autoload.php', $autoloader_content);
    }
    
    /**
     * Verify installation success
     */
    private function verify_installation() {
        error_log('GeoIP2AutoInstaller: Starting installation verification');
        
        // Check if vendor directory exists
        if (!is_dir($this->vendor_dir)) {
            error_log('GeoIP2AutoInstaller: Vendor directory does not exist: ' . $this->vendor_dir);
            return false;
        }
        
        // Check if classes already exist to avoid redeclaration
        if (class_exists('GeoIp2\\Database\\Reader')) {
            error_log('GeoIP2AutoInstaller: GeoIP2 Reader class already exists - verification passed');
            return true;
        }
        
        // Clear any existing autoloader registrations to avoid conflicts
        $autoload_file = $this->vendor_dir . 'autoload.php';
        if (file_exists($autoload_file)) {
            error_log('GeoIP2AutoInstaller: Loading fresh autoloader');
            include_once $autoload_file; // Use include_once to avoid redeclaration
        } else {
            error_log('GeoIP2AutoInstaller: Autoloader file not found');
            return false;
        }
        
        // Don't manually include class files - let autoloader handle it
        // Force class loading through autoloader by checking class existence
        if (class_exists('GeoIp2\\Database\\Reader')) {
            error_log('GeoIP2AutoInstaller: GeoIP2 Reader class loaded successfully');
            return true;
        }
        
        // Test if classes are available 
        $maxmind_exists = class_exists('MaxMind\\Db\\Reader');
        error_log('GeoIP2AutoInstaller: MaxMind\\Db\\Reader class available: ' . ($maxmind_exists ? 'YES' : 'NO'));
        
        // Check if required files exist
        $reader_file = $this->vendor_dir . 'geoip2/geoip2/src/Database/Reader.php';
        $maxmind_file = $this->vendor_dir . 'maxmind-db/reader/src/MaxMind/Db/Reader.php';
        
        $files_exist = file_exists($reader_file) && file_exists($maxmind_file);
        error_log('GeoIP2AutoInstaller: Required files exist: ' . ($files_exist ? 'YES' : 'NO'));
        
        // Consider installation successful if GeoIP2 class exists or files exist
        $success = class_exists('GeoIp2\\Database\\Reader') || $files_exist;
        
        error_log('GeoIP2AutoInstaller: Installation verification result: ' . ($success ? 'SUCCESS' : 'FAILED'));
        
        return $success;
    }
    
    /**
     * Get detailed verification debug information
     */
    private function get_verification_debug_info() {
        $info = array();
        
        // Check vendor directory
        $info['vendor_dir_exists'] = is_dir($this->vendor_dir);
        $info['vendor_dir_path'] = $this->vendor_dir;
        
        // Check autoloader
        $autoload_file = $this->vendor_dir . 'autoload.php';
        $info['autoloader_exists'] = file_exists($autoload_file);
        $info['autoloader_path'] = $autoload_file;
        
        // Check specific files
        $reader_file = $this->vendor_dir . 'geoip2/geoip2/src/Database/Reader.php';
        $maxmind_file = $this->vendor_dir . 'maxmind-db/reader/src/MaxMind/Db/Reader.php';
        
        $info['reader_file_exists'] = file_exists($reader_file);
        $info['reader_file_path'] = $reader_file;
        $info['maxmind_file_exists'] = file_exists($maxmind_file);
        $info['maxmind_file_path'] = $maxmind_file;
        
        // Check classes
        $info['reader_class_exists'] = class_exists('GeoIp2\Database\Reader');
        $info['maxmind_class_exists'] = class_exists('MaxMind\Db\Reader');
        
        // List vendor directory contents
        if (is_dir($this->vendor_dir)) {
            $info['vendor_contents'] = scandir($this->vendor_dir);
            
            // Check maxmind-db directory structure
            $maxmind_db_dir = $this->vendor_dir . 'maxmind-db/';
            if (is_dir($maxmind_db_dir)) {
                $info['maxmind_db_contents'] = $this->scan_directory_recursive($maxmind_db_dir, 3); // 3 levels deep
            } else {
                $info['maxmind_db_contents'] = 'Directory does not exist';
            }
        } else {
            $info['vendor_contents'] = 'Directory does not exist';
        }
        
        return $info;
    }
    
    /**
     * Recursively scan directory structure for debugging
     */
    private function scan_directory_recursive($dir, $max_depth = 2, $current_depth = 0) {
        if ($current_depth >= $max_depth || !is_dir($dir)) {
            return 'Max depth reached or not a directory';
        }
        
        $result = array();
        $items = scandir($dir);
        
        foreach ($items as $item) {
            if ($item == '.' || $item == '..') continue;
            
            $path = $dir . $item;
            if (is_dir($path)) {
                $result[$item] = $this->scan_directory_recursive($path . '/', $max_depth, $current_depth + 1);
            } else {
                $result[] = $item;
            }
        }
        
        return $result;
    }
    
    /**
     * Install MaxMind DB Reader dependency separately
     */
    private function install_maxmind_db_reader() {
        $maxmind_db_api_url = 'https://api.github.com/repos/maxmind/MaxMind-DB-Reader-php';
        
        // Get latest release
        $response = \wp_remote_get($maxmind_db_api_url . '/releases/latest', array(
            'timeout' => 30,
            'user-agent' => 'AffiliateHub-GeoIP2-AutoInstaller/1.0'
        ));
        
        if (\is_wp_error($response) || \wp_remote_retrieve_response_code($response) !== 200) {
            throw new \Exception(\__('Failed to fetch MaxMind DB Reader release information', 'affiliate-hub'));
        }
        
        $release_data = json_decode(\wp_remote_retrieve_body($response), true);
        if (!$release_data || !isset($release_data['zipball_url'])) {
            throw new \Exception(\__('Invalid MaxMind DB Reader release data', 'affiliate-hub'));
        }
        
        // Download MaxMind DB Reader
        $zip_file = $this->download_package($release_data['zipball_url'], 'maxmind-db-reader.zip');
        if (!$zip_file) {
            throw new \Exception(\__('Failed to download MaxMind DB Reader package', 'affiliate-hub'));
        }
        
        // Extract MaxMind DB Reader
        if (!class_exists('ZipArchive')) {
            throw new \Exception(\__('ZipArchive class not available for MaxMind DB Reader extraction', 'affiliate-hub'));
        }
        
        $zip = new \ZipArchive();
        $extract_path = $this->temp_dir . 'maxmind-db-extract/';
        
        if ($zip->open($zip_file) !== TRUE) {
            throw new \Exception(\__('Failed to open MaxMind DB Reader package', 'affiliate-hub'));
        }
        
        if (!$zip->extractTo($extract_path)) {
            $zip->close();
            throw new \Exception(\__('Failed to extract MaxMind DB Reader package', 'affiliate-hub'));
        }
        $zip->close();
        
        // Find extracted directory
        $extracted_dirs = glob($extract_path . 'maxmind-MaxMind-DB-Reader-php-*');
        if (empty($extracted_dirs)) {
            throw new \Exception(\__('MaxMind DB Reader package structure not recognized', 'affiliate-hub'));
        }
        
        $source_dir = $extracted_dirs[0];
        $maxmind_target = $this->vendor_dir . 'maxmind-db/reader/';
        
        // Copy MaxMind DB Reader src files
        $src_path = $source_dir . '/src/';
        if (is_dir($src_path)) {
            \wp_mkdir_p($maxmind_target);
            $this->recursive_copy($src_path, $maxmind_target . 'src/');
        } else {
            throw new \Exception(\__('MaxMind DB Reader src directory not found', 'affiliate-hub'));
        }
        
        return true;
    }
    
    /**
     * Create backup of existing installation
     */
    private function create_backup() {
        if (!is_dir($this->vendor_dir)) {
            return;
        }
        
        $backup_name = 'vendor_backup_' . date('Y-m-d_H-i-s');
        $backup_path = $this->backup_dir . $backup_name;
        
        $this->recursive_copy($this->vendor_dir, $backup_path . '/');
        
        // Store backup reference
        \update_option('affiliate_hub_geoip2_backup', $backup_path);
    }
    
    /**
     * Rollback from backup
     */
    private function rollback_from_backup() {
        $backup_path = \get_option('affiliate_hub_geoip2_backup');
        if (!$backup_path || !is_dir($backup_path)) {
            return false;
        }
        
        // Remove failed installation
        if (is_dir($this->vendor_dir)) {
            $this->recursive_delete($this->vendor_dir);
        }
        
        // Restore backup
        $this->recursive_copy($backup_path . '/', $this->vendor_dir);
        
        return true;
    }
    
    /**
     * Recursively delete directory
     */
    private function recursive_delete($dir) {
        if (!is_dir($dir)) {
            return;
        }
        
        $files = array_diff(scandir($dir), array('.', '..'));
        foreach ($files as $file) {
            $path = $dir . '/' . $file;
            is_dir($path) ? $this->recursive_delete($path) : unlink($path);
        }
        rmdir($dir);
    }
    
    /**
     * Cleanup temporary files
     */
    private function cleanup_temp_files() {
        if (is_dir($this->temp_dir)) {
            $this->recursive_delete($this->temp_dir);
        }
    }
    
    /**
     * Update installation metadata
     */
    private function update_install_metadata($release_info) {
        \update_option('affiliate_hub_geoip2_version', $release_info['tag_name']);
        \update_option('affiliate_hub_geoip2_install_date', \current_time('timestamp'));
        \update_option('affiliate_hub_geoip2_install_method', 'auto-installer');
    }
    
    /**
     * Get installed version
     */
    private function get_installed_version() {
        return \get_option('affiliate_hub_geoip2_version', 'unknown');
    }
    
    /**
     * Get install date
     */
    private function get_install_date() {
        return \get_option('affiliate_hub_geoip2_install_date', 0);
    }
    
    /**
     * Ensure required directories exist
     */
    private function ensure_directories() {
        \wp_mkdir_p($this->temp_dir);
        \wp_mkdir_p($this->backup_dir);
        \wp_mkdir_p(dirname($this->vendor_dir));
    }
    
    /**
     * Uninstall GeoIP2 (revert to fallback)
     */
    public function uninstall() {
        if (is_dir($this->vendor_dir)) {
            $this->recursive_delete($this->vendor_dir);
        }
        
        // Clean options
        \delete_option('affiliate_hub_geoip2_version');
        \delete_option('affiliate_hub_geoip2_install_date');
        \delete_option('affiliate_hub_geoip2_install_method');
        \delete_option('affiliate_hub_geoip2_backup');
        
        return array(
            'success' => true,
            'message' => \__('GeoIP2 library uninstalled. Reverted to fallback mode.', 'affiliate-hub')
        );
    }
}
