<?php
/**
 * Fallback GeoIP2 Reader for WordPress
 * 
 * Simple implementation that provides GeoIP2\Database\Reader interface
 * but uses API calls when MaxMind database is not available
 */

if (!class_exists('GeoIp2\\Database\\Reader') && !class_exists('GeoIp2_Database_Reader')) {
    
    // Create GeoIP2 namespace classes for compatibility
    class GeoIp2_Database_Reader {
        private $fallback_mode = true;
        
        public function __construct($database_path) {
            // Check if database exists
            if (file_exists($database_path)) {
                // In a real implementation, we would use the actual MaxMind reader here
                // For now, we'll use fallback mode
                $this->fallback_mode = true;
            }
        }
        
        public function city($ip_address) {
            if ($this->fallback_mode) {
                return $this->getFallbackCity($ip_address);
            }
            
            // Real MaxMind implementation would go here
            return $this->getFallbackCity($ip_address);
        }
        
        private function getFallbackCity($ip_address) {
            // Use IP-API.com as fallback (free, no key required)
            $response = wp_remote_get("http://ip-api.com/json/{$ip_address}?fields=status,country,countryCode,city,lat,lon,timezone", array(
                'timeout' => 5,
                'headers' => array('User-Agent' => 'AffiliateHub-WordPress-Plugin')
            ));
            
            if (is_wp_error($response)) {
                return $this->getEmptyCity();
            }
            
            $data = json_decode(wp_remote_retrieve_body($response), true);
            
            if (!$data || $data['status'] !== 'success') {
                return $this->getEmptyCity();
            }
            
            return new GeoIp2_Model_City($data);
        }
        
        private function getEmptyCity() {
            return new GeoIp2_Model_City(array(
                'country' => 'Unknown',
                'countryCode' => 'XX',
                'city' => 'Unknown',
                'lat' => 0,
                'lon' => 0,
                'timezone' => 'UTC'
            ));
        }
    }
    
    class GeoIp2_Model_City {
        public $country;
        public $city;
        public $location;
        
        public function __construct($data) {
            $this->country = new GeoIp2_Record_Country($data);
            $this->city = new GeoIp2_Record_City($data);
            $this->location = new GeoIp2_Record_Location($data);
        }
    }
    
    class GeoIp2_Record_Country {
        public $name;
        public $isoCode;
        
        public function __construct($data) {
            $this->name = isset($data['country']) ? $data['country'] : 'Unknown';
            $this->isoCode = isset($data['countryCode']) ? $data['countryCode'] : 'XX';
        }
    }
    
    class GeoIp2_Record_City {
        public $name;
        
        public function __construct($data) {
            $this->name = isset($data['city']) ? $data['city'] : 'Unknown';
        }
    }
    
    class GeoIp2_Record_Location {
        public $latitude;
        public $longitude;
        public $timeZone;
        
        public function __construct($data) {
            $this->latitude = isset($data['lat']) ? (float)$data['lat'] : 0;
            $this->longitude = isset($data['lon']) ? (float)$data['lon'] : 0;
            $this->timeZone = isset($data['timezone']) ? $data['timezone'] : 'UTC';
        }
    }
    
    // Create class aliases to match real GeoIP2 namespace only if they don't exist
    if (!class_exists('GeoIp2\\Database\\Reader')) {
        class_alias('GeoIp2_Database_Reader', 'GeoIp2\\Database\\Reader');
    }
    if (!class_exists('GeoIp2\\Model\\City')) {
        class_alias('GeoIp2_Model_City', 'GeoIp2\\Model\\City');
    }
    if (!class_exists('GeoIp2\\Record\\Country')) {
        class_alias('GeoIp2_Record_Country', 'GeoIp2\\Record\\Country');
    }
    if (!class_exists('GeoIp2\\Record\\City')) {
        class_alias('GeoIp2_Record_City', 'GeoIp2\\Record\\City');
    }
    if (!class_exists('GeoIp2\\Record\\Location')) {
        class_alias('GeoIp2_Record_Location', 'GeoIp2\\Record\\Location');
    }
}
