<?php
/**
 * GeoIP Performance Monitor
 * 
 * Monitors and compares performance between fallback and native modes
 * Provides metrics and benchmarks for admin dashboard
 *
 * @package AffiliateHub
 * @subpackage Modules
 */

namespace AffiliateHub\Modules;

class GeoIPPerformanceMonitor {
    
    private $metrics_option = 'affiliate_hub_geoip_performance_metrics';
    private $benchmark_ips = array('8.8.8.8', '1.1.1.1', '208.67.222.222', '76.76.19.19');
    
    /**
     * Record lookup performance
     */
    public function record_lookup($ip_address, $mode, $response_time_ms, $provider = 'unknown') {
        $metrics = get_option($this->metrics_option, array(
            'total_lookups' => 0,
            'fallback_lookups' => 0,
            'native_lookups' => 0,
            'fallback_avg_time' => 0,
            'native_avg_time' => 0,
            'recent_lookups' => array()
        ));
        
        // Update counters
        $metrics['total_lookups']++;
        if ($mode === 'fallback') {
            $metrics['fallback_lookups']++;
            $metrics['fallback_avg_time'] = $this->update_average(
                $metrics['fallback_avg_time'], 
                $metrics['fallback_lookups'], 
                $response_time_ms
            );
        } else {
            $metrics['native_lookups']++;
            $metrics['native_avg_time'] = $this->update_average(
                $metrics['native_avg_time'], 
                $metrics['native_lookups'], 
                $response_time_ms
            );
        }
        
        // Store recent lookup (keep last 100)
        $metrics['recent_lookups'][] = array(
            'ip' => $this->anonymize_ip($ip_address),
            'mode' => $mode,
            'time_ms' => $response_time_ms,
            'provider' => $provider,
            'timestamp' => current_time('timestamp')
        );
        
        // Keep only last 100 lookups
        if (count($metrics['recent_lookups']) > 100) {
            $metrics['recent_lookups'] = array_slice($metrics['recent_lookups'], -100);
        }
        
        update_option($this->metrics_option, $metrics);
    }
    
    /**
     * Get performance metrics
     */
    public function get_metrics() {
        $metrics = get_option($this->metrics_option, array(
            'total_lookups' => 0,
            'fallback_lookups' => 0,
            'native_lookups' => 0,
            'fallback_avg_time' => 0,
            'native_avg_time' => 0,
            'recent_lookups' => array()
        ));
        
        // Calculate performance improvement
        $improvement = 0;
        if ($metrics['fallback_avg_time'] > 0 && $metrics['native_avg_time'] > 0) {
            $improvement = (($metrics['fallback_avg_time'] - $metrics['native_avg_time']) / $metrics['fallback_avg_time']) * 100;
        }
        
        $metrics['performance_improvement'] = round($improvement, 1);
        $metrics['last_updated'] = current_time('timestamp');
        
        return $metrics;
    }
    
    /**
     * Run benchmark test
     */
    public function run_benchmark($mode = 'both') {
        $results = array(
            'fallback' => array(),
            'native' => array(),
            'timestamp' => current_time('timestamp')
        );
        
        foreach ($this->benchmark_ips as $ip) {
            if ($mode === 'fallback' || $mode === 'both') {
                $results['fallback'][$ip] = $this->benchmark_lookup($ip, 'fallback');
            }
            
            if ($mode === 'native' || $mode === 'both') {
                $results['native'][$ip] = $this->benchmark_lookup($ip, 'native');
            }
            
            // Small delay between tests
            usleep(100000); // 0.1 second
        }
        
        // Calculate averages
        $results['fallback_avg'] = $this->calculate_average($results['fallback']);
        $results['native_avg'] = $this->calculate_average($results['native']);
        
        if ($results['fallback_avg'] > 0 && $results['native_avg'] > 0) {
            $results['improvement'] = round((($results['fallback_avg'] - $results['native_avg']) / $results['fallback_avg']) * 100, 1);
        } else {
            $results['improvement'] = 0;
        }
        
        // Store benchmark results
        update_option('affiliate_hub_geoip_last_benchmark', $results);
        
        return $results;
    }
    
    /**
     * Benchmark single lookup
     */
    private function benchmark_lookup($ip_address, $mode) {
        $start_time = microtime(true);
        
        try {
            if ($mode === 'fallback') {
                $this->test_fallback_lookup($ip_address);
            } else {
                $this->test_native_lookup($ip_address);
            }
            
            $end_time = microtime(true);
            $response_time = ($end_time - $start_time) * 1000; // Convert to ms
            
            return array(
                'success' => true,
                'time_ms' => round($response_time, 2),
                'error' => null
            );
            
        } catch (\Exception $e) {
            $end_time = microtime(true);
            $response_time = ($end_time - $start_time) * 1000;
            
            return array(
                'success' => false,
                'time_ms' => round($response_time, 2),
                'error' => $e->getMessage()
            );
        }
    }
    
    /**
     * Test fallback lookup
     */
    private function test_fallback_lookup($ip_address) {
        // Use our fallback implementation
        if (class_exists('GeoIp2_Database_Reader')) {
            $reader = new \GeoIp2_Database_Reader('dummy_path');
            $reader->city($ip_address);
        } else {
            // Direct API call for testing
            $response = wp_remote_get("http://ip-api.com/json/{$ip_address}?fields=status,country", array(
                'timeout' => 5
            ));
            if (is_wp_error($response)) {
                throw new \Exception('API request failed');
            }
        }
    }
    
    /**
     * Test native lookup
     */
    private function test_native_lookup($ip_address) {
        // Test native GeoIP2 if available
        if (class_exists('GeoIp2\Database\Reader')) {
            $geo_module = new GeoLocation();
            
            // Try to get MaxMind database path
            $maxmind_manager = new MaxMindManager();
            if (!$maxmind_manager->is_database_available()) {
                throw new \Exception('MaxMind database not available');
            }
            
            $db_path = $maxmind_manager->get_database_path();
            $reader = new \GeoIp2\Database\Reader($db_path);
            $reader->city($ip_address);
        } else {
            throw new \Exception('Native GeoIP2 not available');
        }
    }
    
    /**
     * Calculate average response time
     */
    private function calculate_average($results) {
        $times = array();
        foreach ($results as $result) {
            if (is_array($result) && isset($result['time_ms'])) {
                $times[] = $result['time_ms'];
            }
        }
        
        return empty($times) ? 0 : round(array_sum($times) / count($times), 2);
    }
    
    /**
     * Update running average
     */
    private function update_average($current_avg, $count, $new_value) {
        if ($count <= 1) {
            return $new_value;
        }
        
        return (($current_avg * ($count - 1)) + $new_value) / $count;
    }
    
    /**
     * Anonymize IP address for privacy
     */
    private function anonymize_ip($ip) {
        // Remove last octet for IPv4, last 4 groups for IPv6
        if (strpos($ip, ':') === false) {
            // IPv4
            $parts = explode('.', $ip);
            if (count($parts) === 4) {
                $parts[3] = 'xxx';
                return implode('.', $parts);
            }
        } else {
            // IPv6 - simplified anonymization
            $parts = explode(':', $ip);
            if (count($parts) > 4) {
                return implode(':', array_slice($parts, 0, 4)) . ':xxxx:xxxx:xxxx:xxxx';
            }
        }
        
        return 'xxx.xxx.xxx.xxx'; // Fallback
    }
    
    /**
     * Reset all metrics
     */
    public function reset_metrics() {
        delete_option($this->metrics_option);
        delete_option('affiliate_hub_geoip_last_benchmark');
        
        return array(
            'success' => true,
            'message' => __('Performance metrics reset successfully', 'affiliate-hub')
        );
    }
    
    /**
     * Get performance summary for dashboard
     */
    public function get_performance_summary() {
        $metrics = $this->get_metrics();
        $last_benchmark = get_option('affiliate_hub_geoip_last_benchmark', array());
        
        return array(
            'total_lookups' => $metrics['total_lookups'],
            'fallback_avg_time' => $metrics['fallback_avg_time'],
            'native_avg_time' => $metrics['native_avg_time'],
            'performance_improvement' => $metrics['performance_improvement'],
            'last_benchmark' => $last_benchmark,
            'recommendation' => $this->get_performance_recommendation($metrics)
        );
    }
    
    /**
     * Get performance recommendation
     */
    private function get_performance_recommendation($metrics) {
        if ($metrics['native_lookups'] > 0 && $metrics['fallback_lookups'] > 0) {
            if ($metrics['performance_improvement'] > 80) {
                return array(
                    'type' => 'excellent',
                    'message' => __('Excellent! Native mode is significantly faster than fallback.', 'affiliate-hub')
                );
            } elseif ($metrics['performance_improvement'] > 50) {
                return array(
                    'type' => 'good',
                    'message' => __('Good performance improvement with native mode.', 'affiliate-hub')
                );
            } else {
                return array(
                    'type' => 'marginal',
                    'message' => __('Marginal improvement. Check database and server configuration.', 'affiliate-hub')
                );
            }
        } elseif ($metrics['fallback_lookups'] > 0) {
            return array(
                'type' => 'install_native',
                'message' => __('Install native GeoIP2 for up to 90% performance improvement.', 'affiliate-hub')
            );
        }
        
        return array(
            'type' => 'no_data',
            'message' => __('Insufficient data for performance recommendation.', 'affiliate-hub')
        );
    }
}
