<?php
/**
 * Advanced GeoLocation Module for Affiliate Hub
 * 
 * Multi-pro        // Admin hooks
        add_action('wp_ajax_affiliate_hub_test_geo', array($this, 'ajax_test_geolocation'));
        add_action('wp_ajax_affiliate_hub_bulk_enrich_old_data', array($this, 'ajax_bulk_enrich_old_data'));
        add_action('wp_ajax_affiliate_hub_cleanup_cache', array($this, 'ajax_cleanup_cache'));er, multi-cache system for IP geolocation
 * Phases: Cache-first → Database → Local GeoIP → API Fallback
 *
 * @package AffiliateHub
 * @subpackage Modules
 * @since 2.0.0
 */

namespace AffiliateHub\Modules;

use AffiliateHub\Core\Constants;

class GeoLocation implements ModuleInterface {
    
    /**
     * API Providers configuration
     */
    private $providers = array(
        'ipapi_co' => array(
            'name' => 'IPApi.co',
            'url' => 'https://ipapi.co/{ip}/json/',
            'free_limit' => 15000, // per month
            'timeout' => 2,
            'fields' => array('country_name', 'city', 'latitude', 'longitude', 'timezone', 'country_code')
        ),
        'ip_api' => array(
            'name' => 'IP-API.com', 
            'url' => 'http://ip-api.com/json/{ip}?fields=status,country,countryCode,city,lat,lon,timezone',
            'free_limit' => 1000, // per minute
            'timeout' => 2,
            'fields' => array('country', 'city', 'lat', 'lon', 'timezone', 'countryCode')
        ),
        'ipinfo_io' => array(
            'name' => 'IPInfo.io',
            'url' => 'https://ipinfo.io/{ip}/json',
            'free_limit' => 50000, // per month
            'timeout' => 3,
            'fields' => array('country', 'city', 'loc', 'timezone')
        )
    );
    
    /**
     * Cache TTL configuration
     */
    private $cache_ttl = array(
        'object' => 3600,        // 1 hour - WordPress object cache
        'database' => 604800,    // 7 days - custom table
        'transient' => 86400,    // 24 hours - WordPress transients
        'file' => 604800         // 7 days - file system cache for MaxMind results
    );
    
    /**
     * MaxMind Manager instance
     */
    private $maxmind_manager;
    private $last_maxmind_check;
    private $auto_installer;
    private $performance_monitor;
    
    /**
     * Rate limiting per provider
     */
    private $rate_limits = array(
        'ipapi_co' => array('per_hour' => 1000),
        'ip_api' => array('per_minute' => 60),
        'ipinfo_io' => array('per_hour' => 2000)
    );
    
    /**
     * Initialize GeoLocation module
     */
    public function init() {
        if (!$this->is_enabled()) {
            return;
        }
        
        // Initialize MaxMind Manager
        require_once AFFILIATE_HUB_PATH . 'includes/Modules/MaxMindManager.php';
        require_once AFFILIATE_HUB_PATH . 'includes/Modules/GeoIP2Fallback.php';
        require_once AFFILIATE_HUB_PATH . 'includes/Modules/GeoIP2AutoInstaller.php';
        require_once AFFILIATE_HUB_PATH . 'includes/Modules/GeoIPPerformanceMonitor.php';
        
        $this->maxmind_manager = new \AffiliateHub\Modules\MaxMindManager();
        $this->auto_installer = new \AffiliateHub\Modules\GeoIP2AutoInstaller();
        $this->performance_monitor = new \AffiliateHub\Modules\GeoIPPerformanceMonitor();
        
        // Hook into click tracking
        add_filter('affiliate_hub_before_track_click', array($this, 'enrich_click_data'), 10, 2);
        
        // Background processing
        add_action('affiliate_hub_enrich_geo', array($this, 'background_enrich_click'), 10, 2);
        
        // MaxMind auto-update (every 2 weeks)
        add_action('affiliate_hub_maxmind_update', array($this, 'auto_update_maxmind'));
        if (!wp_next_scheduled('affiliate_hub_maxmind_update')) {
            wp_schedule_event(time(), 'biweekly', 'affiliate_hub_maxmind_update');
        }
        
        // Add custom cron schedule
        add_filter('cron_schedules', array($this, 'add_cron_schedules'));
        
        add_action('affiliate_hub_geo_cleanup', array($this, 'cleanup_expired_cache'));
        
        // Admin hooks
        add_action('wp_ajax_affiliate_hub_test_geo', array($this, 'ajax_test_geolocation'));
        add_action('wp_ajax_affiliate_hub_bulk_enrich', array($this, 'ajax_bulk_enrich_old_data'));
        add_action('wp_ajax_affiliate_hub_cleanup_cache', array($this, 'ajax_cleanup_cache'));
        
        // Cron jobs
        if (!wp_next_scheduled('affiliate_hub_geo_cleanup')) {
            wp_schedule_event(time(), 'daily', 'affiliate_hub_geo_cleanup');
        }
        
        // Create database tables on init
        $this->maybe_create_tables();
    }
    
    public function get_info() {
        return array(
            'name' => __('Advanced GeoLocation', 'affiliate-hub'),
            'description' => __('Multi-provider IP geolocation with intelligent caching and background processing.', 'affiliate-hub'),
            'version' => '2.0.0',
            'author' => 'Affiliate Hub',
            'requires' => array('curl', 'json'),
            'features' => array(
                'Multi-level caching',
                'Background processing', 
                'Rate limiting',
                'Multiple API providers',
                'Local GeoIP database support'
            )
        );
    }
    
    public function is_enabled() {
        return get_option(Constants::OPTION_ENABLE_GEOLOCATION, true) && 
               get_option(Constants::OPTION_ENABLE_STATS, true);
    }
    
    /**
     * Main geolocation function - multi-level cache approach
     */
    public function get_location_data($ip_address, $context = 'tracking') {
        $start_time = microtime(true);
        $lookup_mode = 'fallback';
        $provider = 'unknown';
        
        // Validate IP
        if (!$this->is_valid_ip($ip_address)) {
            return $this->get_fallback_location($ip_address);
        }
        
        // Anonymize IP if GDPR mode enabled
        if (get_option(Constants::OPTION_ANONYMIZE_IP, false)) {
            $ip_address = $this->anonymize_ip($ip_address);
        }
        
        // Level 1: Object Cache (fastest - ~0.001s)
        $cache_key = 'geo_' . md5($ip_address);
        $cached = wp_cache_get($cache_key, 'affiliate_hub_geo');
        if ($cached !== false && $this->is_cache_valid($cached)) {
            $this->log_cache_hit('object', $ip_address);
            return $cached;
        }
        
        // Level 2: Database Cache (~0.01s)
        $db_cached = $this->get_from_database_cache($ip_address);
        if ($db_cached && $this->is_cache_valid($db_cached)) {
            // Store in object cache for next time
            wp_cache_set($cache_key, $db_cached, 'affiliate_hub_geo', $this->cache_ttl['object']);
            $this->log_cache_hit('database', $ip_address);
            return $db_cached;
        }
        
        // Level 3: Transient Cache (~0.05s)
        $transient_key = 'affiliate_geo_' . md5($ip_address);
        $transient_cached = get_transient($transient_key);
        if ($transient_cached && $this->is_cache_valid($transient_cached)) {
            $this->store_in_caches($ip_address, $transient_cached, 'transient');
            $this->log_cache_hit('transient', $ip_address);
            return $transient_cached;
        }
        
        // Level 4: Local MaxMind GeoIP Database (PRIMARY) (~0.1-0.5s)
        $local_geo = $this->get_from_local_geoip($ip_address);
        if ($local_geo) {
            $this->store_in_caches($ip_address, $local_geo, 'local_geoip');
            $this->log_cache_hit('local_geoip', $ip_address);
            $this->log_provider_usage('maxmind_local', 'success');
            $lookup_mode = 'native';
            $provider = 'maxmind_local';
            return $this->log_performance_and_return($ip_address, $local_geo, $start_time, $lookup_mode, $provider);
        }
        
        // Level 5: File Cache (~0.1s) - for MaxMind results caching
        $file_cached = $this->get_from_file_cache($ip_address);
        if ($file_cached && $this->is_cache_valid($file_cached)) {
            $this->store_in_caches($ip_address, $file_cached, 'file');
            $this->log_cache_hit('file', $ip_address);
            return $file_cached;
        }
        
        // Level 6: API Providers (FALLBACK) (~1-3s) - Background processing for non-critical contexts
        if ($context === 'background' || $context === 'admin_test') {
            $api_result = $this->get_from_api_providers($ip_address);
            $lookup_mode = 'fallback';
            $provider = $api_result['provider'] ?? 'api_provider';
            return $this->log_performance_and_return($ip_address, $api_result, $start_time, $lookup_mode, $provider);
        } else {
            // Schedule background enrichment
            wp_schedule_single_event(time() + 30, 'affiliate_hub_enrich_geo', array($ip_address, $context));
            $fallback = $this->get_fallback_location($ip_address);
            return $this->log_performance_and_return($ip_address, $fallback, $start_time, 'fallback', 'static');
        }
    }
    
    /**
     * Enhanced click data with geolocation
     */
    public function enrich_click_data($click_data, $ip_address) {
        $geo_data = $this->get_location_data($ip_address, 'tracking');
        
        return array_merge($click_data, array(
            'country' => $geo_data['country'] ?? '',
            'country_code' => $geo_data['country_code'] ?? '',
            'city' => $geo_data['city'] ?? '',
            'latitude' => $geo_data['latitude'] ?? null,
            'longitude' => $geo_data['longitude'] ?? null,
            'timezone' => $geo_data['timezone'] ?? '',
            'geo_provider' => $geo_data['provider'] ?? 'fallback',
            'geo_cached' => $geo_data['cached'] ?? false
        ));
    }
    
    /**
     * Get data from database cache
     */
    private function get_from_database_cache($ip_address) {
        global $wpdb;
        
        $table_cache = $wpdb->prefix . 'affiliate_hub_geo_cache';
        
        // Check if table exists
        if ($wpdb->get_var("SHOW TABLES LIKE '$table_cache'") !== $table_cache) {
            return false;
        }
        
        $result = $wpdb->get_row($wpdb->prepare(
            "SELECT * FROM $table_cache WHERE ip_address = %s AND expires_at > NOW() ORDER BY cached_at DESC LIMIT 1",
            $ip_address
        ), ARRAY_A);
        
        if (!$result) {
            return false;
        }
        
        return array(
            'country' => $result['country_name'],
            'country_code' => $result['country_code'],
            'city' => $result['city'],
            'latitude' => $result['latitude'],
            'longitude' => $result['longitude'],
            'timezone' => $result['timezone'],
            'provider' => $result['provider_used'],
            'cached' => true,
            'cache_level' => 'database',
            'cached_at' => $result['cached_at']
        );
    }
    
    /**
     * Get location from API providers with smart failover
     */
    private function get_from_api_providers($ip_address) {
        $preferred_provider = get_option('affiliate_hub_geo_primary_provider', 'ipapi_co');
        
        // Try primary provider first
        if (isset($this->providers[$preferred_provider])) {
            $result = $this->query_provider($preferred_provider, $ip_address);
            if ($result) {
                $this->store_in_caches($ip_address, $result, $preferred_provider);
                return $result;
            }
        }
        
        // Try other providers as fallback
        foreach ($this->providers as $provider_key => $provider_config) {
            if ($provider_key === $preferred_provider) continue; // Already tried
            
            if (!$this->is_rate_limited($provider_key)) {
                $result = $this->query_provider($provider_key, $ip_address);
                if ($result) {
                    $this->store_in_caches($ip_address, $result, $provider_key);
                    $this->log_provider_usage($provider_key, 'success');
                    return $result;
                }
                $this->log_provider_usage($provider_key, 'failed');
            }
        }
        
        // All providers failed
        $fallback = $this->get_fallback_location($ip_address);
        $this->store_in_caches($ip_address, $fallback, 'fallback');
        return $fallback;
    }
    
    /**
     * Query specific API provider
     */
    private function query_provider($provider_key, $ip_address) {
        $provider = $this->providers[$provider_key];
        $url = str_replace('{ip}', $ip_address, $provider['url']);
        
        $response = wp_remote_get($url, array(
            'timeout' => $provider['timeout'],
            'user-agent' => 'AffiliateHub/' . AFFILIATE_HUB_VERSION,
            'headers' => $this->get_provider_headers($provider_key)
        ));
        
        if (is_wp_error($response)) {
            $this->log_error("Provider $provider_key failed for IP $ip_address: " . $response->get_error_message());
            return false;
        }
        
        $body = wp_remote_retrieve_body($response);
        $data = json_decode($body, true);
        
        if (!$data) {
            $this->log_error("Provider $provider_key returned invalid JSON for IP $ip_address");
            return false;
        }
        
        return $this->normalize_provider_response($provider_key, $data, $ip_address);
    }
    
    /**
     * Normalize different provider responses to unified format
     */
    private function normalize_provider_response($provider_key, $data, $ip_address) {
        $normalized = array(
            'provider' => $provider_key,
            'cached' => false,
            'cache_level' => 'api',
            'cached_at' => current_time('mysql')
        );
        
        switch ($provider_key) {
            case 'ipapi_co':
                $normalized['country'] = $data['country_name'] ?? '';
                $normalized['country_code'] = $data['country_code'] ?? '';
                $normalized['city'] = $data['city'] ?? '';
                $normalized['latitude'] = $data['latitude'] ?? null;
                $normalized['longitude'] = $data['longitude'] ?? null;
                $normalized['timezone'] = $data['timezone'] ?? '';
                break;
                
            case 'ip_api':
                if (($data['status'] ?? '') !== 'success') {
                    return false;
                }
                $normalized['country'] = $data['country'] ?? '';
                $normalized['country_code'] = $data['countryCode'] ?? '';
                $normalized['city'] = $data['city'] ?? '';
                $normalized['latitude'] = $data['lat'] ?? null;
                $normalized['longitude'] = $data['lon'] ?? null;
                $normalized['timezone'] = $data['timezone'] ?? '';
                break;
                
            case 'ipinfo_io':
                $normalized['country'] = $data['country'] ?? '';
                $normalized['country_code'] = $data['country'] ?? '';
                $normalized['city'] = $data['city'] ?? '';
                $normalized['timezone'] = $data['timezone'] ?? '';
                
                // Parse "lat,lon" format
                if (isset($data['loc']) && strpos($data['loc'], ',') !== false) {
                    list($lat, $lon) = explode(',', $data['loc']);
                    $normalized['latitude'] = floatval($lat);
                    $normalized['longitude'] = floatval($lon);
                }
                break;
        }
        
        // Validate required fields
        if (empty($normalized['country']) && empty($normalized['city'])) {
            return false;
        }
        
        return $normalized;
    }
    
    /**
     * Store data in multiple cache levels
     */
    private function store_in_caches($ip_address, $data, $source) {
        // Object cache
        $cache_key = 'geo_' . md5($ip_address);
        wp_cache_set($cache_key, $data, 'affiliate_hub_geo', $this->cache_ttl['object']);
        
        // Transient cache
        $transient_key = 'affiliate_geo_' . md5($ip_address);
        set_transient($transient_key, $data, $this->cache_ttl['transient']);
        
        // Database cache
        $this->store_in_database_cache($ip_address, $data);
        
        // Log caching activity
        $this->log_cache_store($ip_address, $source);
    }
    
    /**
     * Store in database cache table
     */
    private function store_in_database_cache($ip_address, $data) {
        global $wpdb;
        
        $table_cache = $wpdb->prefix . 'affiliate_hub_geo_cache';
        
        // Check if table exists
        if ($wpdb->get_var("SHOW TABLES LIKE '$table_cache'") !== $table_cache) {
            $this->maybe_create_tables();
        }
        
        $wpdb->replace(
            $table_cache,
            array(
                'ip_address' => $ip_address,
                'country_name' => $data['country'] ?? '',
                'country_code' => $data['country_code'] ?? '',
                'city' => $data['city'] ?? '',
                'latitude' => $data['latitude'] ?? null,
                'longitude' => $data['longitude'] ?? null,
                'timezone' => $data['timezone'] ?? '',
                'provider_used' => $data['provider'] ?? 'unknown',
                'cached_at' => current_time('mysql'),
                'expires_at' => date('Y-m-d H:i:s', time() + $this->cache_ttl['database'])
            ),
            array('%s', '%s', '%s', '%s', '%f', '%f', '%s', '%s', '%s', '%s')
        );
    }
    
    /**
     * Get fallback location for invalid/local IPs
     */
    private function get_fallback_location($ip_address) {
        // Default fallback based on site settings or server location
        $fallback_country = get_option('affiliate_hub_fallback_country', 'Poland');
        $fallback_city = get_option('affiliate_hub_fallback_city', 'Warsaw');
        
        // Special handling for local IPs
        if ($this->is_local_ip($ip_address)) {
            return array(
                'country' => $fallback_country,
                'country_code' => $this->get_country_code($fallback_country),
                'city' => $fallback_city,
                'latitude' => null,
                'longitude' => null,
                'timezone' => get_option('timezone_string', 'Europe/Warsaw'),
                'provider' => 'local_fallback',
                'cached' => false,
                'cache_level' => 'fallback'
            );
        }
        
        return array(
            'country' => $fallback_country,
            'country_code' => $this->get_country_code($fallback_country),
            'city' => $fallback_city,
            'latitude' => null,
            'longitude' => null,
            'timezone' => '',
            'provider' => 'fallback',
            'cached' => false,
            'cache_level' => 'fallback'
        );
    }
    
    /**
     * Validate IP address
     */
    private function is_valid_ip($ip) {
        return filter_var($ip, FILTER_VALIDATE_IP, FILTER_FLAG_NO_PRIV_RANGE | FILTER_FLAG_NO_RES_RANGE) !== false;
    }
    
    /**
     * Check if IP is local/private
     */
    private function is_local_ip($ip) {
        return !filter_var($ip, FILTER_VALIDATE_IP, FILTER_FLAG_NO_PRIV_RANGE | FILTER_FLAG_NO_RES_RANGE);
    }
    
    /**
     * Anonymize IP address (GDPR compliance)
     */
    private function anonymize_ip($ip) {
        if (filter_var($ip, FILTER_VALIDATE_IP, FILTER_FLAG_IPV4)) {
            // IPv4: remove last octet
            $parts = explode('.', $ip);
            $parts[3] = '0';
            return implode('.', $parts);
        } elseif (filter_var($ip, FILTER_VALIDATE_IP, FILTER_FLAG_IPV6)) {
            // IPv6: remove last 64 bits
            $parts = explode(':', $ip);
            for ($i = 4; $i < 8; $i++) {
                $parts[$i] = '0';
            }
            return implode(':', $parts);
        }
        
        return $ip;
    }
    
    /**
     * Check if cache data is still valid
     */
    private function is_cache_valid($cache_data) {
        if (!isset($cache_data['cached_at'])) {
            return false;
        }
        
        $cached_time = strtotime($cache_data['cached_at']);
        $cache_level = $cache_data['cache_level'] ?? 'object';
        $ttl = $this->cache_ttl[$cache_level] ?? $this->cache_ttl['object'];
        
        return (time() - $cached_time) < $ttl;
    }
    
    /**
     * Background enrichment of click data
     */
    public function background_enrich_click($ip_address, $context = 'background') {
        $geo_data = $this->get_from_api_providers($ip_address);
        
        if ($geo_data) {
            // Update any existing clicks with this IP that don't have geo data
            $this->update_existing_clicks_with_geo($ip_address, $geo_data);
        }
    }
    
    /**
     * Update existing clicks with geo data
     */
    private function update_existing_clicks_with_geo($ip_address, $geo_data) {
        global $wpdb;
        
        $table_clicks = $wpdb->prefix . Constants::TABLE_LINK_CLICKS;
        
        $updated = $wpdb->update(
            $table_clicks,
            array(
                'country' => $geo_data['country'] ?? '',
                'city' => $geo_data['city'] ?? '',
                'latitude' => $geo_data['latitude'] ?? null,
                'longitude' => $geo_data['longitude'] ?? null,
                'timezone' => $geo_data['timezone'] ?? ''
            ),
            array(
                'ip_address' => $ip_address
            ),
            array('%s', '%s', '%f', '%f', '%s'),
            array('%s')
        );
        
        if ($updated) {
            $this->log_info("Updated $updated clicks with geo data for IP: $ip_address");
        }
    }
    
    /**
     * Create necessary database tables
     */
    private function maybe_create_tables() {
        global $wpdb;
        
        $charset_collate = $wpdb->get_charset_collate();
        $table_cache = $wpdb->prefix . 'affiliate_hub_geo_cache';
        
        // Check if table already exists
        if ($wpdb->get_var("SHOW TABLES LIKE '$table_cache'") === $table_cache) {
            return;
        }
        
        $sql = "CREATE TABLE $table_cache (
            id bigint(20) NOT NULL AUTO_INCREMENT,
            ip_address varchar(45) NOT NULL,
            country_name varchar(100) DEFAULT '',
            country_code char(2) DEFAULT '',
            city varchar(100) DEFAULT '',
            latitude decimal(10,8) DEFAULT NULL,
            longitude decimal(11,8) DEFAULT NULL,
            timezone varchar(100) DEFAULT '',
            provider_used varchar(50) DEFAULT '',
            cached_at datetime DEFAULT CURRENT_TIMESTAMP,
            expires_at datetime DEFAULT NULL,
            PRIMARY KEY (id),
            UNIQUE KEY ip_address (ip_address),
            KEY expires_at (expires_at),
            KEY cached_at (cached_at),
            KEY country_code (country_code)
        ) $charset_collate;";
        
        require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
        dbDelta($sql);
        
        // Also need to extend the clicks table with new geo fields
        $this->maybe_extend_clicks_table();
    }
    
    /**
     * Extend clicks table with additional geo fields
     */
    private function maybe_extend_clicks_table() {
        global $wpdb;
        
        $table_clicks = $wpdb->prefix . Constants::TABLE_LINK_CLICKS;
        
        // Check if new columns exist
        $columns = $wpdb->get_results("SHOW COLUMNS FROM $table_clicks LIKE 'latitude'");
        
        if (empty($columns)) {
            // Add new geo columns
            $wpdb->query("ALTER TABLE $table_clicks 
                ADD COLUMN latitude decimal(10,8) DEFAULT NULL AFTER city,
                ADD COLUMN longitude decimal(11,8) DEFAULT NULL AFTER latitude,
                ADD COLUMN timezone varchar(100) DEFAULT '' AFTER longitude,
                ADD COLUMN country_code char(2) DEFAULT '' AFTER country,
                ADD INDEX idx_latitude (latitude),
                ADD INDEX idx_longitude (longitude),
                ADD INDEX idx_country_code (country_code)
            ");
        }
    }
    
    /**
     * Cleanup expired cache entries
     */
    public function cleanup_expired_cache() {
        global $wpdb;
        
        $table_cache = $wpdb->prefix . 'affiliate_hub_geo_cache';
        
        if ($wpdb->get_var("SHOW TABLES LIKE '$table_cache'") === $table_cache) {
            $deleted = $wpdb->query("DELETE FROM $table_cache WHERE expires_at < NOW()");
            $this->log_info("Cleaned up $deleted expired geo cache entries");
        }
    }
    
    /**
     * Check if provider is rate limited
     */
    private function is_rate_limited($provider_key) {
        if (!isset($this->rate_limits[$provider_key])) {
            return false;
        }
        
        $limits = $this->rate_limits[$provider_key];
        $transient_key = "affiliate_geo_rate_$provider_key";
        $current_usage = get_transient($transient_key) ?: array();
        
        foreach ($limits as $period => $limit) {
            $period_key = date('Y-m-d-H') . ($period === 'per_minute' ? '-' . date('i') : '');
            $usage = $current_usage[$period_key] ?? 0;
            
            if ($usage >= $limit) {
                return true;
            }
        }
        
        return false;
    }
    
    /**
     * Log provider usage for rate limiting
     */
    private function log_provider_usage($provider_key, $status) {
        $transient_key = "affiliate_geo_rate_$provider_key";
        $current_usage = get_transient($transient_key) ?: array();
        
        // Log for both minute and hour
        $minute_key = date('Y-m-d-H-i');
        $hour_key = date('Y-m-d-H');
        
        $current_usage[$minute_key] = ($current_usage[$minute_key] ?? 0) + 1;
        $current_usage[$hour_key] = ($current_usage[$hour_key] ?? 0) + 1;
        
        // Clean old entries (keep only last 2 hours)
        $cutoff = date('Y-m-d-H', time() - 7200);
        foreach ($current_usage as $key => $value) {
            if ($key < $cutoff) {
                unset($current_usage[$key]);
            }
        }
        
        set_transient($transient_key, $current_usage, 7200); // 2 hours
        
        // Also log to stats
        $this->increment_provider_stats($provider_key, $status);
    }
    
    /**
     * Get provider headers (API keys, etc)
     */
    private function get_provider_headers($provider_key) {
        $headers = array();
        
        // Add API keys if configured
        $api_key = get_option("affiliate_hub_geo_api_key_$provider_key", '');
        if ($api_key) {
            switch ($provider_key) {
                case 'ipapi_co':
                    $headers['Authorization'] = 'Bearer ' . $api_key;
                    break;
                case 'ipinfo_io':
                    $headers['Authorization'] = 'Bearer ' . $api_key;
                    break;
            }
        }
        
        return $headers;
    }
    
    /**
     * PHASE 3: Local MaxMind GeoIP support (PRIMARY)
     */
    private function get_from_local_geoip($ip_address) {
        // Check if MaxMind is available and database exists
        if (!$this->maxmind_manager || !$this->maxmind_manager->is_database_available()) {
            $this->maybe_update_maxmind_database();
            if (!$this->maxmind_manager || !$this->maxmind_manager->is_database_available()) {
                return false;
            }
        }
        
        $geoip_path = $this->maxmind_manager->get_database_path();
        
        // Check if GeoIP2 library is available
        if (!$this->is_geoip2_available()) {
            return false;
        }
        
        // Load autoloader for native GeoIP2 if available
        if ($this->auto_installer) {
            $this->auto_installer->load_autoloader();
        }
        
        try {
            $reader = new \GeoIp2\Database\Reader($geoip_path);
            $record = $reader->city($ip_address);
            
            return array(
                'country' => $record->country->name ?: '',
                'country_code' => $record->country->isoCode ?: '',
                'city' => $record->city->name ?: '',
                'latitude' => $record->location->latitude ?: 0,
                'longitude' => $record->location->longitude ?: 0,
                'timezone' => $record->location->timeZone ?: '',
                'provider' => 'maxmind_local',
                'cached' => false,
                'cache_level' => 'local_geoip'
            );
            
        } catch (Exception $e) {
            $this->log_error("MaxMind GeoIP lookup failed for $ip_address: " . $e->getMessage());
            $this->log_provider_usage('maxmind_local', 'failed');
            return false;
        }
    }
    
    /**
     * Get file cache (for MaxMind database results)
     */
    private function get_from_file_cache($ip_address) {
        $cache_dir = $this->get_cache_directory();
        $cache_file = $cache_dir . '/' . md5($ip_address) . '.json';
        
        if (!file_exists($cache_file)) {
            return false;
        }
        
        $content = file_get_contents($cache_file);
        $data = json_decode($content, true);
        
        if (!$data || !$this->is_cache_valid($data)) {
            unlink($cache_file);
            return false;
        }
        
        return $data;
    }
    
    /**
     * Get cache directory path
     */
    private function get_cache_directory() {
        $upload_dir = wp_upload_dir();
        $cache_dir = $upload_dir['basedir'] . '/affiliate-hub/geo-cache';
        
        if (!file_exists($cache_dir)) {
            wp_mkdir_p($cache_dir);
            // Add .htaccess for security
            file_put_contents($cache_dir . '/.htaccess', "Deny from all\n");
        }
        
        return $cache_dir;
    }
    
    /**
     * Get MaxMind database path
     */
    private function get_geoip_database_path() {
        $upload_dir = wp_upload_dir();
        return $upload_dir['basedir'] . '/affiliate-hub/geoip/GeoLite2-City.mmdb';
    }
    
    /**
     * Get country code from country name
     */
    private function get_country_code($country_name) {
        $country_codes = array(
            'Poland' => 'PL',
            'United States' => 'US',
            'Germany' => 'DE',
            'United Kingdom' => 'GB',
            'France' => 'FR',
            'Spain' => 'ES',
            'Italy' => 'IT',
            'Canada' => 'CA',
            'Australia' => 'AU',
            'Japan' => 'JP'
        );
        
        return $country_codes[$country_name] ?? '';
    }
    
    /**
     * Logging methods
     */
    private function log_cache_hit($level, $ip) {
        if (defined('WP_DEBUG') && WP_DEBUG) {
            error_log("AffiliateHub GeoLocation: Cache HIT ($level) for IP: $ip");
        }
    }
    
    private function log_cache_store($ip, $source) {
        if (defined('WP_DEBUG') && WP_DEBUG) {
            error_log("AffiliateHub GeoLocation: Stored cache for IP: $ip from source: $source");
        }
    }
    
    private function log_error($message) {
        if (defined('WP_DEBUG') && WP_DEBUG) {
            error_log("AffiliateHub GeoLocation ERROR: $message");
        }
    }
    
    private function log_info($message) {
        if (defined('WP_DEBUG') && WP_DEBUG) {
            error_log("AffiliateHub GeoLocation INFO: $message");
        }
    }
    
    /**
     * Increment provider statistics
     */
    private function increment_provider_stats($provider_key, $status) {
        error_log("DEBUG: increment_provider_stats called for provider: $provider_key, status: $status");
        
        $stats_key = "affiliate_geo_stats_$provider_key";
        $stats = get_option($stats_key, array(
            'total_requests' => 0,
            'successful_requests' => 0,
            'failed_requests' => 0,
            'last_used' => ''
        ));
        
        $stats['total_requests']++;
        if ($status === 'success') {
            $stats['successful_requests']++;
        } else {
            $stats['failed_requests']++;
        }
        $stats['last_used'] = current_time('mysql');
        
        $result = update_option($stats_key, $stats);
        error_log("DEBUG: Stats updated for $provider_key: " . json_encode($stats) . ", update_result: " . ($result ? 'SUCCESS' : 'FAILED'));
    }
    
    /**
     * AJAX: Test geolocation functionality
     */
    public function ajax_test_geolocation() {
        check_ajax_referer('affiliate_hub_admin_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Insufficient permissions');
        }
        
        $test_ip = sanitize_text_field($_POST['test_ip'] ?? '8.8.8.8');
        
        $start_time = microtime(true);
        $result = $this->get_location_data($test_ip, 'admin_test');
        $end_time = microtime(true);
        
        $result['response_time'] = round(($end_time - $start_time) * 1000, 2) . 'ms';
        
        wp_send_json_success($result);
    }
    
    /**
     * AJAX: Bulk enrich old data without geo information
     */
    public function ajax_bulk_enrich_old_data() {
        \check_ajax_referer('affiliate_hub_admin_nonce', 'nonce');
        
        if (!\current_user_can('manage_options')) {
            \wp_send_json_error('Insufficient permissions');
        }
        
        global $wpdb;
        $table_clicks = $wpdb->prefix . Constants::TABLE_LINK_CLICKS;
        
        // Get IPs without geo data (limit to avoid timeout)
        $ips = $wpdb->get_col("
            SELECT DISTINCT ip_address 
            FROM $table_clicks 
            WHERE (country = '' OR country IS NULL) 
            AND ip_address != '' 
            LIMIT 50
        ");
        
        $processed = 0;
        foreach ($ips as $ip) {
            \wp_schedule_single_event(time() + ($processed * 2), 'affiliate_hub_enrich_geo', array($ip, 'bulk_enrich'));
            $processed++;
        }
        
        \wp_send_json_success(array(
            'message' => sprintf(\__('Scheduled %d IPs for background geolocation enrichment.', 'affiliate-hub'), $processed),
            'processed' => $processed
        ));
    }
    
    /**
     * AJAX: Cleanup expired cache
     */
    public function ajax_cleanup_cache() {
        \check_ajax_referer('affiliate_hub_admin_nonce', 'nonce');
        
        if (!\current_user_can('manage_options')) {
            \wp_send_json_error('Insufficient permissions');
        }
        
        global $wpdb;
        $table_cache = $wpdb->prefix . 'affiliate_hub_geo_cache';
        
        if ($wpdb->get_var("SHOW TABLES LIKE '$table_cache'") === $table_cache) {
            $deleted = $wpdb->query("DELETE FROM $table_cache WHERE expires_at < NOW()");
            
            \wp_send_json_success(array(
                'message' => sprintf(\__('Cleaned up %d expired cache entries.', 'affiliate-hub'), $deleted),
                'deleted' => $deleted
            ));
        } else {
            \wp_send_json_error(\__('Cache table not found.', 'affiliate-hub'));
        }
    }
    
    /**
     * Get module statistics for admin dashboard
     */
    public function get_statistics() {
        global $wpdb;
        
        $stats = array();
        
        // Cache statistics
        $table_cache = $wpdb->prefix . 'affiliate_hub_geo_cache';
        if ($wpdb->get_var("SHOW TABLES LIKE '$table_cache'") === $table_cache) {
            $stats['total_cached_ips'] = $wpdb->get_var("SELECT COUNT(*) FROM $table_cache");
            $stats['expired_cache_entries'] = $wpdb->get_var("SELECT COUNT(*) FROM $table_cache WHERE expires_at < NOW()");
        }
        
        // Provider statistics
        $stats['providers'] = array();
        
        // Add MaxMind local provider
        $maxmind_stats = get_option("affiliate_geo_stats_maxmind_local", array());
        $stats['providers']['maxmind_local'] = array_merge(
            array(
                'name' => 'MaxMind Local',
                'url' => 'Local GeoLite2 Database',
                'free_limit' => 'Unlimited',
                'timeout' => 0.1,
                'fields' => array('country_name', 'country_code', 'city', 'latitude', 'longitude', 'timezone')
            ), 
            $maxmind_stats
        );
        
        // Add external API providers
        foreach ($this->providers as $provider_key => $provider_config) {
            $provider_stats = get_option("affiliate_geo_stats_$provider_key", array());
            $stats['providers'][$provider_key] = array_merge($provider_config, $provider_stats);
        }
        
        // Clicks with geo data
        $table_clicks = $wpdb->prefix . Constants::TABLE_LINK_CLICKS;
        $stats['clicks_with_geo'] = $wpdb->get_var("SELECT COUNT(*) FROM $table_clicks WHERE country != ''");
        $stats['clicks_without_geo'] = $wpdb->get_var("SELECT COUNT(*) FROM $table_clicks WHERE (country = '' OR country IS NULL)");
        
        return $stats;
    }
    
    /**
     * Add custom cron schedules
     */
    public function add_cron_schedules($schedules) {
        $schedules['biweekly'] = array(
            'interval' => 14 * DAY_IN_SECONDS,
            'display'  => __('Every 2 weeks', 'affiliate-hub')
        );
        return $schedules;
    }
    
    /**
     * Auto-update MaxMind database
     */
    public function auto_update_maxmind() {
        if (!$this->maxmind_manager) {
            return;
        }
        
        $result = $this->maxmind_manager->update_database();
        if (is_wp_error($result)) {
            $this->log_error('MaxMind auto-update failed: ' . $result->get_error_message());
        } else {
            $this->log_error('MaxMind database updated successfully');
        }
    }
    
    /**
     * Maybe update MaxMind database if missing
     */
    private function maybe_update_maxmind_database() {
        if (!$this->maxmind_manager) {
            return;
        }
        
        // Check only once per hour to avoid spam
        $last_check = get_transient('affiliate_hub_maxmind_check');
        if ($last_check) {
            return;
        }
        
        set_transient('affiliate_hub_maxmind_check', time(), HOUR_IN_SECONDS);
        
        if (!$this->maxmind_manager->is_database_available()) {
            $result = $this->maxmind_manager->update_database();
            if (is_wp_error($result)) {
                $this->log_error('MaxMind database download failed: ' . $result->get_error_message());
            }
        }
    }
    
    /**
     * Check if GeoIP2 library is available
     */
    private function is_geoip2_available() {
        // Check if classes already exist
        if (class_exists('GeoIp2\Database\Reader') || class_exists('GeoIp2_Database_Reader')) {
            return true;
        }
        
        // Try to load autoloader if available
        if ($this->auto_installer) {
            $this->auto_installer->load_autoloader();
            return class_exists('GeoIp2\Database\Reader') || class_exists('GeoIp2_Database_Reader');
        }
        
        // Our fallback implementation is always available
        return class_exists('GeoIp2_Database_Reader');
    }
    
    /**
     * Get MaxMind database status for admin
     */
    public function get_maxmind_status() {
        if (!$this->maxmind_manager) {
            return array(
                'available' => false,
                'message' => __('MaxMind Manager not initialized', 'affiliate-hub'),
                'geoip2_available' => $this->is_geoip2_available(),
                'using_fallback' => true
            );
        }
        
        $is_available = $this->maxmind_manager->is_database_available();
        $last_update = $this->maxmind_manager->get_last_update();
        $geoip2_native = class_exists('GeoIp2\Database\Reader') && !class_exists('GeoIp2_Database_Reader');
        
        return array(
            'available' => $is_available,
            'last_update' => $last_update,
            'file_size' => $is_available ? size_format(filesize($this->maxmind_manager->get_database_path())) : 0,
            'needs_update' => $last_update && (time() - $last_update) > (14 * DAY_IN_SECONDS),
            'geoip2_available' => $this->is_geoip2_available(),
            'geoip2_native' => $geoip2_native,
            'using_fallback' => !$geoip2_native,
            'message' => $is_available ? 
                __('MaxMind database ready', 'affiliate-hub') : 
                ($this->is_geoip2_available() ? __('Using API fallback mode', 'affiliate-hub') : __('MaxMind database not available', 'affiliate-hub'))
        );
    }
    
    /**
     * Log performance metrics and return result
     */
    private function log_performance_and_return($ip_address, $result, $start_time, $mode, $provider) {
        $end_time = microtime(true);
        $response_time_ms = ($end_time - $start_time) * 1000;
        
        // Log performance if monitor is available
        if ($this->performance_monitor) {
            $this->performance_monitor->record_lookup($ip_address, $mode, $response_time_ms, $provider);
        }
        
        return $result;
    }
    
    /**
     * Get auto-installer instance
     */
    public function get_auto_installer() {
        if (null === $this->auto_installer) {
            require_once AFFILIATE_HUB_PATH . 'includes/Modules/GeoIP2AutoInstaller.php';
            $this->auto_installer = new \AffiliateHub\Modules\GeoIP2AutoInstaller();
        }
        return $this->auto_installer;
    }
    
    /**
     * Get performance monitor instance  
     */
    public function get_performance_monitor() {
        if (null === $this->performance_monitor) {
            require_once AFFILIATE_HUB_PATH . 'includes/Modules/GeoIPPerformanceMonitor.php';
            $this->performance_monitor = new \AffiliateHub\Modules\GeoIPPerformanceMonitor();
        }
        return $this->performance_monitor;
    }
}
