<?php
/**
 * Keywords Autolinker Module
 *
 * @package AffiliateHub
 * @subpackage Modules
 */

namespace AffiliateHub\Modules;

use AffiliateHub\Core\Constants;

/**
 * Keywords Autolinker Module
 * 
 * Automatically converts keywords in post content to affiliate links
 */
class KeywordsAutolinker implements ModuleInterface {
    
    /**
     * Cache instance
     *
     * @var KeywordsAutolinker\Cache
     */
    private $cache;
    
    /**
     * Processor instance
     *
     * @var KeywordsAutolinker\Processor  
     */
    private $processor;
    
    /**
     * Settings instance
     *
     * @var KeywordsAutolinker\Settings
     */
    private $settings;
    
    /**
     * Initialize the module
     */
    public function init() {
        // Initialize components (always needed for admin)
        $this->cache = new KeywordsAutolinker\Cache();
        $this->processor = new KeywordsAutolinker\Processor($this->cache);
        $this->settings = new KeywordsAutolinker\Settings();
        
        // Only enable content processing if autolinker is enabled
        if ($this->is_enabled()) {
            // Hook into content processing
            \add_filter('the_content', array($this->processor, 'process_content'), 999);
        }
        
        // Hook into save post to update cache (always needed)
        \add_action('save_post', array($this, 'on_save_post'), 10, 3);
        
        // Hook into settings changes
        \add_action('updated_option', array($this, 'on_settings_updated'), 10, 3);
        
        // Add admin AJAX handlers
        \add_action('wp_ajax_affiliate_hub_rebuild_autolinker_cache', array($this, 'ajax_rebuild_cache'));
        \add_action('wp_ajax_affiliate_hub_get_autolinker_stats', array($this, 'ajax_get_stats'));
        \add_action('wp_ajax_affiliate_hub_check_keyword_conflict', array($this, 'ajax_check_keyword_conflict'));
        
        // Add metabox integration
        \add_action('add_meta_boxes', array($this, 'add_autolinker_metabox'));
        
        // Initialize admin components
        if (\is_admin()) {
            $this->init_admin();
        }
    }
    
    /**
     * Get module information
     *
     * @return array
     */
    public function get_info() {
        return array(
            'name' => __('Keywords Autolinker', 'affiliate-hub'),
            'description' => __('Automatically converts keywords in content to affiliate links based on predefined keyword lists for each affiliate link.', 'affiliate-hub'),
            'version' => '1.0.0',
            'author' => 'Affiliate Hub'
        );
    }
    
    /**
     * Check if module is enabled
     *
     * @return bool
     */
    public function is_enabled() {
        return (bool) get_option(Constants::OPTION_AUTOLINKER_ENABLED, false);
    }
    
    /**
     * Initialize admin components
     */
    private function init_admin() {
        // No longer need to add settings page here - handled by Admin.php
        
        // Ensure cache has data on admin (rebuild if empty or on request)
        if (\is_admin()) {
            $cached_keywords = $this->cache->get_cached_keywords();
            if (empty($cached_keywords)) {
                error_log('🔧 Cache is empty - rebuilding...');
                $this->cache->rebuild_all_cache();
                // Verify rebuild worked
                $cached_keywords = $this->cache->get_cached_keywords();
                error_log('🔧 After rebuild: ' . count($cached_keywords) . ' keywords in cache');
            } else {
                error_log('🔧 Cache has ' . count($cached_keywords) . ' keywords');
            }
        }
        
        // Enqueue admin assets
        \add_action('admin_enqueue_scripts', array($this, 'enqueue_admin_assets'));
    }
    
    /**
     * Handle post save to update cache
     *
     * @param int $post_id Post ID
     * @param WP_Post $post Post object
     * @param bool $update Whether this is an update
     */
    public function on_save_post($post_id, $post, $update) {
        // Only process affiliate links
        if ($post->post_type !== Constants::POST_TYPE_AFFILIATE_LINK) {
            return;
        }
        
        // Skip autosaves and revisions
        if (wp_is_post_autosave($post_id) || wp_is_post_revision($post_id)) {
            return;
        }
        
        // Update cache for this link
        $this->cache->update_link_cache($post_id);
        
        // Clear processed content cache
        $this->cache->clear_content_cache();
    }
    
    /**
     * Handle settings updates
     *
     * @param string $option Option name
     * @param mixed $old_value Old value
     * @param mixed $new_value New value
     */
    public function on_settings_updated($option, $old_value, $new_value) {
        // Clear cache if autolinker settings changed
        $autolinker_options = array(
            Constants::OPTION_AUTOLINKER_ENABLED,
            Constants::OPTION_AUTOLINKER_KEYWORD_LIMIT,
            Constants::OPTION_AUTOLINKER_RANDOM_PLACEMENT,
            Constants::OPTION_AUTOLINKER_LINK_IN_HEADINGS,
            Constants::OPTION_AUTOLINKER_DISABLE_ARCHIVE,
            Constants::OPTION_AUTOLINKER_DISABLE_HOME,
            Constants::OPTION_AUTOLINKER_ENABLE_FEEDS,
            Constants::OPTION_AUTOLINKER_POST_TYPES
        );
        
        if (in_array($option, $autolinker_options)) {
            $this->cache->clear_all_cache();
        }
    }
    
    /**
     * Add autolinker metabox to affiliate link edit screen
     */
    public function add_autolinker_metabox() {
        \add_meta_box(
            'affiliate-hub-autolinker-keywords',
            \__('Autolink Keywords', 'affiliate-hub'),
            array($this, 'render_keywords_metabox'),
            Constants::POST_TYPE_AFFILIATE_LINK,
            'normal',
            'default'
        );
    }
    
    /**
     * Render keywords metabox
     *
     * @param WP_Post $post Post object
     */
    public function render_keywords_metabox($post) {
        // Get existing keywords
        $keywords = \get_post_meta($post->ID, Constants::META_AUTOLINK_KEYWORDS, true);
        $keyword_limit = \get_post_meta($post->ID, Constants::META_AUTOLINK_KEYWORD_LIMIT, true);
        
        // Default values
        $keywords = !empty($keywords) ? $keywords : array();
        $keyword_limit = !empty($keyword_limit) ? intval($keyword_limit) : 0;
        
        // Nonce for security
        \wp_nonce_field('affiliate_hub_autolinker_keywords_nonce', 'affiliate_hub_autolinker_keywords_nonce');
        
        ?>
        <div class="affiliate-hub-autolinker-keywords">
            <p class="description">
                <?php esc_html_e('Enter a list of keywords you wish to automatically link with this affiliate link (case-insensitive).', 'affiliate-hub'); ?>
            </p>
            
            <table class="form-table">
                <tr>
                    <th scope="row">
                        <label for="autolink_keywords"><?php esc_html_e('Keywords', 'affiliate-hub'); ?></label>
                    </th>
                    <td>
                        <div id="autolink-keywords-container">
                            <?php if (!empty($keywords)): ?>
                                <?php foreach ($keywords as $index => $keyword): ?>
                                    <div class="keyword-tag">
                                        <div class="keyword-input-wrapper">
                                            <input type="text" name="autolink_keywords[]" value="<?php echo \esc_attr($keyword); ?>" placeholder="<?php \esc_attr_e('Enter keyword', 'affiliate-hub'); ?>" />
                                            <button type="button" class="button remove-keyword">×</button>
                                        </div>
                                        <div class="keyword-status-message" style="display: none;"></div>
                                    </div>
                                <?php endforeach; ?>
                            <?php else: ?>
                                <div class="keyword-tag">
                                    <div class="keyword-input-wrapper">
                                        <input type="text" name="autolink_keywords[]" value="" placeholder="<?php \esc_attr_e('Enter keyword', 'affiliate-hub'); ?>" />
                                        <button type="button" class="button remove-keyword">×</button>
                                    </div>
                                    <div class="keyword-status-message" style="display: none;"></div>
                                </div>
                            <?php endif; ?>
                        </div>
                        <p>
                            <button type="button" id="add-keyword" class="button"><?php esc_html_e('Add Keyword', 'affiliate-hub'); ?></button>
                        </p>
                        <p class="description">
                            <?php esc_html_e('Note: Place your keywords in order of precedence. Type your entries rather than copy/pasting to eliminate encoding issues.', 'affiliate-hub'); ?>
                        </p>
                    </td>
                </tr>
                <tr>
                    <th scope="row">
                        <label for="autolink_keyword_limit"><?php esc_html_e('Limit (per keyword)', 'affiliate-hub'); ?></label>
                    </th>
                    <td>
                        <input type="number" id="autolink_keyword_limit" name="autolink_keyword_limit" value="<?php echo esc_attr($keyword_limit); ?>" min="0" max="100" class="small-text" />
                        <p class="description">
                            <?php esc_html_e('Maximum number of links per keyword (0 = use global setting). If you have 2 keywords and set limit to 3, you\'ll have maximum 6 links total.', 'affiliate-hub'); ?>
                        </p>
                    </td>
                </tr>
            </table>
        </div>
        <?php
    }
    
    /**
     * AJAX handler to rebuild cache
     */
    public function ajax_rebuild_cache() {
        \check_ajax_referer('affiliate_hub_admin_nonce', 'nonce');
        
        if (!\current_user_can('manage_options')) {
            \wp_die(\__('Insufficient permissions', 'affiliate-hub'));
        }
        
        try {
            $this->cache->rebuild_all_cache();
            
            \wp_send_json_success(array(
                'message' => \__('Autolinker cache rebuilt successfully.', 'affiliate-hub')
            ));
        } catch (\Exception $e) {
            \wp_send_json_error(array(
                'message' => sprintf(\__('Error rebuilding cache: %s', 'affiliate-hub'), $e->getMessage())
            ));
        }
    }
    
    /**
     * AJAX handler to get autolinker stats
     */
    public function ajax_get_stats() {
        \check_ajax_referer('affiliate_hub_admin_nonce', 'nonce');
        
        if (!\current_user_can('manage_options')) {
            \wp_die(\__('Insufficient permissions', 'affiliate-hub'));
        }
        
        $stats = $this->cache->get_cache_stats();
        
        \wp_send_json_success($stats);
    }
    
    /**
     * Enqueue admin assets
     *
     * @param string $hook Current admin page hook
     */
    public function enqueue_admin_assets($hook) {
        // Check if assets are already loaded to prevent duplicates
        if (\wp_script_is('affiliate-hub-autolinker-admin', 'enqueued')) {
            return;
        }
        
        // Debug: Log hook and context
        error_log('🔧 enqueue_admin_assets called - Hook: ' . $hook);
        error_log('🔧 GET params: ' . json_encode($_GET));
        global $typenow;
        error_log('🔧 typenow: ' . $typenow);
        
        // Only load on affiliate link edit pages and settings
        $load_assets = false;
        
        if (strpos($hook, 'affiliate-hub') !== false) {
            $load_assets = true;
        }
        
        // Check for existing post edit
        if (isset($_GET['post'])) {
            $post_id = intval($_GET['post']);
            $post_type = \get_post_type($post_id);
            if ($post_type === Constants::POST_TYPE_AFFILIATE_LINK) {
                $load_assets = true;
            }
        }
        
        // Check for new post creation
        if (isset($_GET['post_type']) && $_GET['post_type'] === Constants::POST_TYPE_AFFILIATE_LINK) {
            $load_assets = true;
        }
        
        // Also check if we're on post-new.php for affiliate links
        global $typenow;
        if ($typenow === Constants::POST_TYPE_AFFILIATE_LINK) {
            $load_assets = true;
        }
        
        if (!$load_assets) {
            error_log('🔧 Autolinker assets NOT loaded - Hook: ' . $hook . ', GET params: ' . json_encode($_GET) . ', typenow: ' . $typenow);
            return;
        }
        
        error_log('🔧 Autolinker assets LOADED - Hook: ' . $hook . ', GET params: ' . json_encode($_GET) . ', typenow: ' . $typenow);
        
        \wp_enqueue_script('jquery-ui-sortable');
        
        \wp_enqueue_style(
            'affiliate-hub-autolinker-admin',
            AFFILIATE_HUB_URL . 'assets/css/autolinker-admin.css',
            array(),
            AFFILIATE_HUB_VERSION
        );
        
        \wp_enqueue_script(
            'affiliate-hub-autolinker-admin',
            AFFILIATE_HUB_URL . 'assets/js/autolinker-admin-clean.js',
            array('jquery', 'jquery-ui-sortable'),
            AFFILIATE_HUB_VERSION,
            true
        );
        
        \wp_localize_script('affiliate-hub-autolinker-admin', 'affiliateHubAutolinker', array(
            'ajaxUrl' => \admin_url('admin-ajax.php'),
            'nonce' => \wp_create_nonce('affiliate_hub_admin_nonce'),
            'strings' => array(
                'confirmRebuildCache' => \__('Are you sure you want to rebuild the autolinker cache? This will clear all cached data.', 'affiliate-hub'),
                'rebuildingCache' => \__('Rebuilding cache...', 'affiliate-hub'),
                'cacheRebuilt' => \__('Cache rebuilt successfully!', 'affiliate-hub'),
                'errorRebuildingCache' => \__('Error rebuilding cache. Please try again.', 'affiliate-hub'),
                'enterKeyword' => \__('Enter keyword', 'affiliate-hub')
            )
        ));
    }
    
    /**
     * Get cache instance
     *
     * @return KeywordsAutolinker\Cache
     */
    public function get_cache() {
        return $this->cache;
    }
    
    /**
     * Get processor instance
     *
     * @return KeywordsAutolinker\Processor
     */
    public function get_processor() {
        return $this->processor;
    }
    
    /**
     * Get settings instance
     *
     * @return KeywordsAutolinker\Settings
     */
    public function get_settings() {
        return $this->settings;
    }
    
    /**
     * AJAX handler to check for keyword conflicts
     */
    public function ajax_check_keyword_conflict() {
        \check_ajax_referer('affiliate_hub_admin_nonce', 'nonce');
        
        if (!\current_user_can('manage_options')) {
            \wp_die(\__('Insufficient permissions', 'affiliate-hub'));
        }
        
        $keyword = \sanitize_text_field($_POST['keyword'] ?? '');
        $current_post_id = intval($_POST['post_id'] ?? 0);
        
        // Debug logging
        error_log('🔧 AJAX keyword check: ' . $keyword . ' for post ' . $current_post_id);
        
        if (empty($keyword)) {
            \wp_send_json_error(['message' => \__('Keyword cannot be empty', 'affiliate-hub')]);
        }
        
        // Check if keyword exists in cache (case-insensitive)
        $existing_keywords = $this->cache->get_cached_keywords();
        
        // Debug: Log what we're checking
        error_log('🔧 Checking keyword "' . $keyword . '" against ' . count($existing_keywords) . ' cached keywords');
        error_log('🔧 Current post ID: ' . $current_post_id);
        
        $found_conflict = false;
        foreach ($existing_keywords as $cached_keyword) {
            // Use case-insensitive comparison
            $keyword_lower = strtolower(trim($keyword));
            $cached_lower = strtolower(trim($cached_keyword['keyword']));
            
            error_log('🔧 Comparing: "' . $keyword_lower . '" vs "' . $cached_lower . '" from post ' . $cached_keyword['link_id']);
            
            // Check if keywords match and it's a different post (use link_id not post_id!)
            if ($keyword_lower === $cached_lower && intval($cached_keyword['link_id']) !== intval($current_post_id)) {
                error_log('🔧 CONFLICT DETECTED! Keyword exists in post ' . $cached_keyword['link_id']);
                
                // Get link details
                $existing_post = \get_post($cached_keyword['link_id']);
                if ($existing_post) {
                    \wp_send_json_success([
                        'conflict' => true,
                        'existing_link_id' => $cached_keyword['link_id'],
                        'existing_link_name' => $existing_post->post_title,
                        'existing_link_edit_url' => \admin_url('post.php?post=' . $cached_keyword['link_id'] . '&action=edit'),
                        'message' => sprintf(
                            'RYZYKO! Używane w: %s (ID: %d)',
                            $existing_post->post_title,
                            $cached_keyword['link_id']
                        )
                    ]);
                }
                break; // Found conflict, no need to continue
            }
        }
        
        // No conflict found
        error_log('🔧 No conflict found - keyword is available');
        
        // DEBUG: Send cache contents to JavaScript for inspection
        $debug_cache = array_map(function($item) {
            return [
                'keyword' => $item['keyword'],
                'post_id' => $item['post_id'],
                'priority' => $item['priority'] ?? 0
            ];
        }, $existing_keywords);
        
        \wp_send_json_success([
            'conflict' => false,
            'message' => 'Dostępne - możesz użyć tego słowa kluczowego',
            'debug_cache' => $debug_cache,
            'debug_searched_keyword' => $keyword,
            'debug_current_post' => $current_post_id
        ]);
    }
}
