<?php
/**
 * Keywords Autolinker Cache Management
 *
 * @package AffiliateHub
 * @subpackage Modules\KeywordsAutolinker
 */

namespace AffiliateHub\Modules\KeywordsAutolinker;

use AffiliateHub\Core\Constants;

/**
 * Cache Management for Keywords Autolinker
 */
class Cache {
    
    /**
     * Cache table name
     *
     * @var string
     */
    private $table_name;
    
    /**
     * Constructor
     */
    public function __construct() {
        global $wpdb;
        $this->table_name = $wpdb->prefix . Constants::TABLE_AUTOLINKER_CACHE;
    }
    
    /**
     * Rebuild entire cache
     *
     * @return bool Success status
     */
    public function rebuild_all_cache() {
        global $wpdb;
        
        try {
            // Clear existing cache
            $wpdb->query("TRUNCATE TABLE {$this->table_name}");
            
            // Get all affiliate links (including drafts and pending)
            $affiliate_links = get_posts(array(
                'post_type' => Constants::POST_TYPE_AFFILIATE_LINK,
                'post_status' => array('publish', 'draft', 'pending', 'private'),
                'posts_per_page' => -1,
                'fields' => 'ids'
            ));
            
            $total_inserted = 0;
            
            foreach ($affiliate_links as $link_id) {
                $inserted = $this->update_link_cache($link_id);
                $total_inserted += $inserted;
            }
            
            // Update cache timestamp
            update_option('affiliate_hub_autolinker_cache_built', current_time('timestamp'));
            
            return true;
            
        } catch (Exception $e) {
            error_log('Affiliate Hub Autolinker Cache Error: ' . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Update cache for specific link
     *
     * @param int $link_id Affiliate link ID
     * @return int Number of keywords inserted
     */
    public function update_link_cache($link_id) {
        global $wpdb;
        
        // Remove existing cache for this link
        $wpdb->delete($this->table_name, array('link_id' => $link_id), array('%d'));
        
        // Get link data
        $post = get_post($link_id);
        if (!$post || $post->post_type !== Constants::POST_TYPE_AFFILIATE_LINK) {
            return 0;
        }
        
        // Accept any post status (publish, draft, pending, private)
        if (!in_array($post->post_status, array('publish', 'draft', 'pending', 'private'))) {
            return 0;
        }
        
        // Get keywords and settings
        $keywords = get_post_meta($link_id, Constants::META_AUTOLINK_KEYWORDS, true);
        $keyword_limit = get_post_meta($link_id, Constants::META_AUTOLINK_KEYWORD_LIMIT, true);
        
        if (empty($keywords) || !is_array($keywords)) {
            return 0;
        }
        
        // Get link URL
        $link_prefix = get_option(Constants::OPTION_LINK_PREFIX, 'go');
        $link_url = trailingslashit(home_url()) . trailingslashit($link_prefix) . $post->post_name;
        
        $inserted_count = 0;
        $priority = 0;
        
        foreach ($keywords as $keyword) {
            $keyword = trim($keyword);
            if (empty($keyword)) {
                continue;
            }
            
            // Create hash for uniqueness check
            $keyword_hash = md5(strtolower($keyword));
            
            $data = array(
                'link_id' => $link_id,
                'keyword' => $keyword,
                'keyword_hash' => $keyword_hash,
                'link_url' => $link_url,
                'link_title' => $post->post_title,
                'priority' => $priority,
                'case_sensitive' => 0, // Always case-insensitive per requirements
                'keyword_limit' => intval($keyword_limit),
                'created_at' => current_time('mysql'),
                'updated_at' => current_time('mysql')
            );
            
            $inserted = $wpdb->insert($this->table_name, $data, array(
                '%d', '%s', '%s', '%s', '%s', '%d', '%d', '%d', '%s', '%s'
            ));
            
            if ($inserted) {
                $inserted_count++;
            }
            
            $priority++;
        }
        
        return $inserted_count;
    }
    
    /**
     * Get cached keywords for processing
     *
     * @param array $options Processing options
     * @return array Cached keywords data
     */
    public function get_cached_keywords($options = array()) {
        global $wpdb;
        
        $cache_key = 'affiliate_hub_autolinker_keywords_' . md5(serialize($options));
        $cached = wp_cache_get($cache_key, 'affiliate_hub_autolinker');
        
        if (false !== $cached) {
            return $cached;
        }
        
        // Build query
        $sql = "SELECT * FROM {$this->table_name} WHERE 1=1";
        $sql .= " ORDER BY CHAR_LENGTH(keyword) DESC, priority ASC";
        
        $results = $wpdb->get_results($sql, ARRAY_A);
        
        if (!$results) {
            return array();
        }
        
        // Return flat array that Processor expects
        $keywords_data = array();
        
        foreach ($results as $row) {
            $keywords_data[] = array(
                'keyword' => $row['keyword'],
                'link_id' => $row['link_id'],
                'limit' => $row['keyword_limit'], // Use 'limit' key as expected by Processor
                'link_url' => $row['link_url'],
                'link_title' => $row['link_title'],
                'priority' => $row['priority']
            );
        }
        
        // Cache for 1 hour
        wp_cache_set($cache_key, $keywords_data, 'affiliate_hub_autolinker', 3600);
        
        return $keywords_data;
    }
    
    /**
     * Clear all cache
     */
    public function clear_all_cache() {
        // Clear database cache
        wp_cache_flush_group('affiliate_hub_autolinker');
        
        // Clear content cache
        $this->clear_content_cache();
    }
    
    /**
     * Clear processed content cache
     */
    public function clear_content_cache() {
        // Clear object cache for processed content
        wp_cache_flush_group('affiliate_hub_content');
        
        // Clear any transients
        delete_transient('affiliate_hub_autolinker_content_cache');
    }
    
    /**
     * Get cache statistics
     *
     * @return array Cache stats
     */
    public function get_cache_stats() {
        global $wpdb;
        
        $stats = array();
        
        // Total cached keywords
        $stats['total_keywords'] = $wpdb->get_var("SELECT COUNT(*) FROM {$this->table_name}");
        
        // Total unique keywords
        $stats['unique_keywords'] = $wpdb->get_var("SELECT COUNT(DISTINCT keyword_hash) FROM {$this->table_name}");
        
        // Total links with keywords
        $stats['links_with_keywords'] = $wpdb->get_var("SELECT COUNT(DISTINCT link_id) FROM {$this->table_name}");
        
        // Last cache rebuild
        $last_built = get_option('affiliate_hub_autolinker_cache_built', 0);
        $stats['last_built'] = $last_built ? date_i18n(get_option('date_format') . ' ' . get_option('time_format'), $last_built) : __('Never', 'affiliate-hub');
        
        // Cache size (approximate)
        $cache_size = $wpdb->get_var("SELECT ROUND(((data_length + index_length) / 1024 / 1024), 2) AS 'MB' FROM information_schema.TABLES WHERE table_schema = DATABASE() AND table_name = '{$this->table_name}'");
        $stats['cache_size'] = $cache_size ? $cache_size . ' MB' : __('Unknown', 'affiliate-hub');
        
        // Check if cache needs rebuilding
        $stats['needs_rebuild'] = $this->needs_rebuild();
        
        return $stats;
    }
    
    /**
     * Check if cache needs rebuilding
     *
     * @return bool Whether cache needs rebuilding
     */
    public function needs_rebuild() {
        global $wpdb;
        
        // Check if cache table exists and has data
        $cache_count = $wpdb->get_var("SELECT COUNT(*) FROM {$this->table_name}");
        
        if ($cache_count == 0) {
            return true;
        }
        
        // Check if there are affiliate links without cache entries
        $links_count = $wpdb->get_var("
            SELECT COUNT(p.ID) 
            FROM {$wpdb->posts} p
            LEFT JOIN {$wpdb->postmeta} pm ON p.ID = pm.post_id AND pm.meta_key = '" . Constants::META_AUTOLINK_KEYWORDS . "'
            LEFT JOIN {$this->table_name} c ON p.ID = c.link_id
            WHERE p.post_type = '" . Constants::POST_TYPE_AFFILIATE_LINK . "'
            AND p.post_status = 'publish'
            AND pm.meta_value IS NOT NULL 
            AND pm.meta_value != ''
            AND pm.meta_value != 'a:0:{}'
            AND c.link_id IS NULL
        ");
        
        return $links_count > 0;
    }
    
    /**
     * Clean up orphaned cache entries
     *
     * @return int Number of entries cleaned
     */
    public function cleanup_cache() {
        global $wpdb;
        
        $deleted = $wpdb->query("
            DELETE c FROM {$this->table_name} c
            LEFT JOIN {$wpdb->posts} p ON c.link_id = p.ID
            WHERE p.ID IS NULL 
            OR p.post_type != '" . Constants::POST_TYPE_AFFILIATE_LINK . "'
            OR p.post_status != 'publish'
        ");
        
        return $deleted !== false ? $deleted : 0;
    }
}
