<?php
/**
 * Keywords Autolinker Content Processor
 *
 * @package AffiliateHub
 * @subpackage Modules\KeywordsAutolinker
 */

namespace AffiliateHub\Modules\KeywordsAutolinker;

use AffiliateHub\Core\Constants;

/**
 * Content Processor for Keywords Autolinker
 */
class Processor {
    
    /**
     * Cache instance
     *
     * @var Cache
     */
    private $cache;
    
    /**
     * Link counters for current content
     *
     * @var array
     */
    private $link_counters = array();
    
    /**
     * Protected elements during content processing
     *
     * @var array
     */
    private $protected_elements = array();
    
    /**
     * Constructor
     *
     * @param Cache $cache Cache instance
     */
    public function __construct($cache) {
        $this->cache = $cache;
    }
    
    /**
     * Initialize hooks
     */
    public function init() {
        // Hook into content filtering with high priority
        add_filter('the_content', array($this, 'process_content'), 999);
    }
    
    /**
     * Process content to add affiliate links
     *
     * @param string $content Post content
     * @return string Processed content
     */
    public function process_content($content) {
        // Reset protected elements for this processing cycle
        $this->protected_elements = array();
        $this->link_counters = array();
        
        // Skip if module is not enabled
        if (!get_option(Constants::OPTION_AUTOLINKER_ENABLED, false)) {
            return $content;
        }
        
        // Skip if content is empty
        if (empty($content)) {
            return $content;
        }
        
        // Skip based on context
        if (!$this->should_process_current_context()) {
            return $content;
        }
        
        // Skip if processing feeds and feeds are disabled
        if (is_feed() && !get_option(Constants::OPTION_AUTOLINKER_ENABLE_FEEDS, false)) {
            return $content;
        }
        
        // Check post type restrictions
        $current_post_type = get_post_type();
        $allowed_post_types = get_option(Constants::OPTION_AUTOLINKER_POST_TYPES, array('post', 'page'));
        if ($current_post_type && !in_array($current_post_type, $allowed_post_types)) {
            return $content;
        }
        
        // Try to get cached version first
        $cache_key = 'autolinker_content_' . md5($content);
        $cached_content = wp_cache_get($cache_key, 'affiliate_hub_content');
        if ($cached_content !== false) {
            return $cached_content;
        }
        
        // Get cached keywords
        $cached_keywords = $this->cache->get_cached_keywords();
        if (empty($cached_keywords)) {
            return $content;
        }
        
        // Protect existing elements
        $protected_content = $this->protect_existing_elements($content);
        
        // Process keywords
        $processed_content = $this->process_keywords($protected_content, $cached_keywords);
        
        // Restore protected elements
        $final_content = $this->restore_protected_elements($processed_content);
        
        // Cache the result
        wp_cache_set($cache_key, $final_content, 'affiliate_hub_content', 1800);
        
        return $final_content;
    }
    
    /**
     * Check if we should process content in current context
     *
     * @return bool
     */
    private function should_process_current_context() {
        // Skip in admin
        if (is_admin()) {
            return false;
        }
        
        // Skip on login/register pages
        global $pagenow;
        if (in_array($pagenow, array('wp-login.php', 'wp-register.php'))) {
            return false;
        }
        
        // Skip on home page if disabled
        if (is_home() && get_option(Constants::OPTION_AUTOLINKER_DISABLE_HOME, false)) {
            return false;
        }
        
        // Skip on archive pages if disabled
        if ((is_archive() || is_category() || is_tag() || is_date() || is_author()) &&
            get_option(Constants::OPTION_AUTOLINKER_DISABLE_ARCHIVE, false)) {
            return false;
        }
        
        return true;
    }
    
    /**
     * Process keywords in content
     *
     * @param string $content Content to process
     * @param array $cached_keywords Cached keyword data
     * @return string Processed content
     */
    private function process_keywords($content, $cached_keywords) {
        $processed_content = $content;
        $global_limit = \get_option(Constants::OPTION_AUTOLINKER_KEYWORD_LIMIT, 0);
        $random_placement = \get_option(Constants::OPTION_AUTOLINKER_RANDOM_PLACEMENT, false);
        
        foreach ($cached_keywords as $keyword_data) {
            // Validate required keys exist
            if (!isset($keyword_data['keyword'], $keyword_data['link_id'])) {
                continue;
            }
            
            $keyword = $keyword_data['keyword'];
            $link_id = $keyword_data['link_id'];
            $limit = isset($keyword_data['limit']) && $keyword_data['limit'] > 0 ? $keyword_data['limit'] : $global_limit;
            
            // Skip if we've already processed this keyword enough times (unless limit is 0 = unlimited)
            if ($limit > 0 && isset($this->link_counters[$link_id]) && $this->link_counters[$link_id] >= $limit) {
                continue;
            }
            
            // Find keyword occurrences
            $occurrences = $this->find_keyword_occurrences($processed_content, $keyword);
            
            if (empty($occurrences)) {
                continue;
            }
            
            // Randomize occurrences if random placement is enabled
            if ($random_placement) {
                shuffle($occurrences);
            }
            
            // Process occurrences up to the limit (0 = unlimited)
            $processed = 0;
            if ($limit == 0) {
                // Unlimited - process all occurrences
                $current_limit = PHP_INT_MAX;
            } else {
                $current_limit = $limit - (isset($this->link_counters[$link_id]) ? $this->link_counters[$link_id] : 0);
            }
            
            // Process occurrences from end to beginning to maintain position accuracy
            $occurrences = array_reverse($occurrences);
            
            foreach ($occurrences as $occurrence) {
                if ($processed >= $current_limit) {
                    break;
                }
                
                // Use the occurrence position directly (since we're going reverse, positions are still valid)
                $current_pos = $occurrence['position'];
                
                // Make sure the text at this position still matches what we expect
                $actual_text = substr($processed_content, $current_pos, $occurrence['length']);
                if (strtolower($actual_text) !== strtolower($occurrence['full_match'])) {
                    continue; // Text at this position has changed, skip
                }
                
                // Extract context around the match for protection check
                $before_match = substr($processed_content, 0, $current_pos);
                $after_match = substr($processed_content, $current_pos + $occurrence['length']);
                
                // Skip if this keyword is inside a link or other protected element
                if ($this->is_inside_protected_element($before_match, $after_match)) {
                    continue;
                }
                
                // Create the replacement link
                $link_html = $this->create_affiliate_link($occurrence['full_match'], $link_id);
                
                // Replace this specific occurrence
                $processed_content = substr_replace(
                    $processed_content, 
                    $link_html, 
                    $current_pos, 
                    $occurrence['length']
                );
                
                $processed++;
                if (!isset($this->link_counters[$link_id])) {
                    $this->link_counters[$link_id] = 0;
                }
                $this->link_counters[$link_id]++;
            }
        }
        
        return $processed_content;
    }
    
    /**
     * Find keyword occurrences in content
     *
     * @param string $content Content to search
     * @param string $keyword Keyword to find
     * @return array Array of occurrences with position and context
     */
    private function find_keyword_occurrences($content, $keyword) {
        $occurrences = array();
        $escaped_keyword = preg_quote($keyword, '/');
        
        // Use word boundaries to ensure we match whole words
        $pattern = '/\b' . $escaped_keyword . '\b/i';
        
        preg_match_all($pattern, $content, $matches, PREG_OFFSET_CAPTURE);
        
        foreach ($matches[0] as $match) {
            $occurrences[] = array(
                'full_match' => $match[0],
                'position' => $match[1],
                'length' => strlen($match[0])
            );
        }
        
        return $occurrences;
    }
    
    /**
     * Protect existing links and HTML elements from being processed
     *
     * @param string $content Content to protect
     * @return string Protected content
     */
    private function protect_existing_elements($content) {
        $protect_patterns = array(
            // Links - improved pattern to catch all link variations
            '/<a\b[^>]*>.*?<\/a>/is',
            // Images
            '/<img[^>]*>/is',
            // Script tags
            '/<script\b[^>]*>.*?<\/script>/is',
            // Style tags
            '/<style\b[^>]*>.*?<\/style>/is',
            // Code blocks
            '/<code\b[^>]*>.*?<\/code>/is',
            '/<pre\b[^>]*>.*?<\/pre>/is',
            // Shortcodes
            '/\[.*?\]/',
            // HTML comments
            '/<!--.*?-->/s',
        );
        
        // Optionally protect heading tags
        if (!\get_option(Constants::OPTION_AUTOLINKER_LINK_IN_HEADINGS, false)) {
            $protect_patterns[] = '/<h[1-6]\b[^>]*>.*?<\/h[1-6]>/is';
        }
        
        $protected_content = $content;
        $placeholder_counter = 0;
        
        foreach ($protect_patterns as $pattern) {
            $protected_content = preg_replace_callback($pattern, function($matches) use (&$placeholder_counter) {
                $placeholder = '{{AFFILIATE_HUB_PROTECTED_' . $placeholder_counter . '}}';
                $this->protected_elements[$placeholder] = $matches[0];
                $placeholder_counter++;
                return $placeholder;
            }, $protected_content);
        }
        
        return $protected_content;
    }
    
    /**
     * Restore protected elements after processing
     *
     * @param string $content Processed content
     * @return string Content with restored elements
     */
    private function restore_protected_elements($content) {
        $restored_content = $content;
        
        foreach ($this->protected_elements as $placeholder => $original) {
            $restored_content = str_replace($placeholder, $original, $restored_content);
        }
        
        return $restored_content;
    }
    
    /**
     * Create affiliate link HTML
     *
     * @param string $anchor_text Text for the link
     * @param int $link_id Affiliate link post ID
     * @return string Link HTML
     */
    private function create_affiliate_link($anchor_text, $link_id) {
        // Get link data
        $destination_url = \get_post_meta($link_id, Constants::META_DESTINATION_URL, true);
        $cloaked_url = \get_permalink($link_id);
        $title = \get_the_title($link_id);
        
        if (empty($destination_url)) {
            return $anchor_text; // Return original text if no URL
        }
        
        // Determine which URL to use
        $url = $cloaked_url ?: $destination_url;
        
        // Build link attributes
        $attributes = array(
            'href' => \esc_url($url),
            'title' => \esc_attr($title),
            'class' => 'affiliate-link autolinker-link',
            'rel' => 'nofollow noopener',
            'target' => '_blank'
        );
        
        // Allow filtering of attributes
        $attributes = \apply_filters('affiliate_hub_autolinker_link_attributes', $attributes, $link_id, $anchor_text);
        
        // Build link HTML
        $link_html = '<a';
        foreach ($attributes as $attr => $value) {
            if (!empty($value)) {
                $link_html .= ' ' . $attr . '="' . $value . '"';
            }
        }
        $link_html .= '>' . \esc_html($anchor_text) . '</a>';
        
        // Track click if click tracking is enabled
        \do_action('affiliate_hub_autolinker_link_created', $link_id, $url, $anchor_text);
        
        return $link_html;
    }
    
    /**
     * Get link counters (for debugging/stats)
     *
     * @return array
     */
    public function get_link_counters() {
        return $this->link_counters;
    }
    
    /**
     * Check if a keyword occurrence is inside a protected element
     *
     * @param string $before_match Content before the match
     * @param string $after_match Content after the match
     * @return bool True if inside protected element
     */
    private function is_inside_protected_element($before_match, $after_match) {
        // Check if we're inside a link tag
        $open_link_count = substr_count($before_match, '<a ') + substr_count($before_match, '<a>');
        $close_link_count = substr_count($before_match, '</a>');
        
        if ($open_link_count > $close_link_count) {
            return true; // We're inside an unclosed link
        }
        
        // Check if we're inside other protected elements
        $protected_tags = array('script', 'style', 'code', 'pre');
        foreach ($protected_tags as $tag) {
            $open_count = substr_count($before_match, '<' . $tag);
            $close_count = substr_count($before_match, '</' . $tag . '>');
            if ($open_count > $close_count) {
                return true;
            }
        }
        
        // Check for placeholders from protected elements
        if (strpos($before_match, '{{AFFILIATE_HUB_PROTECTED_') !== false) {
            $last_placeholder_start = strrpos($before_match, '{{AFFILIATE_HUB_PROTECTED_');
            $last_placeholder_end = strpos($before_match, '}}', $last_placeholder_start);
            if ($last_placeholder_end === false) {
                return true; // We're inside an incomplete placeholder
            }
        }
        
        return false;
    }
}
