<?php
/**
 * Keywords Autolinker Content Processor
 *
 * @package AffiliateHub
 * @subpackage Modules\KeywordsAutolinker
 */

namespace AffiliateHub\Modules\KeywordsAutolinker;

use AffiliateHub\Core\Constants;

/**
 * Content Processor for Keywords Autolinker
 */
class Processor {
    
    /**
     * Cache instance
     *
     * @var Cache
     */
    private $cache;
    
    /**
     * Link counters for current content
     *
     * @var array
     */
    private $link_counters = array();
    
    /**
     * Protected elements during content processing
     *
     * @var array
     */
    private $protected_elements = array();
    
    /**
     * Constructor
     *
     * @param Cache $cache Cache instance
     */
    public function __construct($cache) {
        $this->cache = $cache;
    }
    
    /**
     * Initialize hooks
     */
    public function init() {
        // Hook into content filtering with high priority
        add_filter('the_content', array($this, 'process_content'), 999);
    }
    
    /**
     * Process content to add affiliate links
     *
     * @param string $content Post content
     * @return string Processed content
     */
    public function process_content($content) {
        // Reset protected elements for this processing cycle
        $this->protected_elements = array();
        $this->link_counters = array();
        
        // Skip if module is not enabled
        if (!get_option(Constants::OPTION_AUTOLINKER_ENABLED, false)) {
            return $content;
        }

namespace AffiliateHub\Modules\KeywordsAutolinker;

use AffiliateHub\Core\Constants;

/**
 * Content Processor for Keywords Autolinker
 */
class Processor {
    
    /**
     * Cache instance
     *
     * @var Cache
     */
    private $cache;
    
    /**
     * Link counters for current content
     *
     * @var array
     */
    private $link_counters = array();
    
    /**
     * Protected elements during content processing
     *
     * @var array
     */
    private $protected_elements = array();
    
    /**
     * Constructor
     *
     * @param Cache $cache Cache instance
     */
    public function __construct(Cache $cache) {
        $this->cache = $cache;
    }
    
    /**
     * Process content and add autolinks
     *
     * @param string $content Post content
     * @return string Processed content
     */
    public function process_content($content) {
        // Skip if module is not enabled
        if (!get_option(Constants::OPTION_AUTOLINKER_ENABLED, false)) {
            return $content;
        }
        
        // Skip if content is empty
        if (empty($content)) {
            return $content;
        }
        
        // Skip based on context
        if (!$this->should_process_current_context()) {
            return $content;
        }
        
        // Skip if processing feeds and feeds are disabled
        if (is_feed() && !get_option(Constants::OPTION_AUTOLINKER_ENABLE_FEEDS, false)) {
            return $content;
        }
        
        // Check if current post type is enabled
        $current_post_type = get_post_type();
        $enabled_post_types = get_option(Constants::OPTION_AUTOLINKER_POST_TYPES, array('post', 'page'));
        
        if (!in_array($current_post_type, $enabled_post_types)) {
            return $content;
        }
        
        // Get cache key for this content
        $cache_key = 'processed_content_' . md5($content . serialize(get_option(Constants::OPTION_AUTOLINKER_POST_TYPES)));
        $cached_content = wp_cache_get($cache_key, 'affiliate_hub_content');
        
        if (false !== $cached_content) {
            return $cached_content;
        }
        
        // Reset counters for this content
        $this->link_counters = array();
        
        // Get keywords from cache
        $keywords_data = $this->cache->get_cached_keywords();
        
        if (empty($keywords_data)) {
            return $content;
        }
        
        // Process content
        $processed_content = $this->process_keywords($content, $keywords_data);
        
        // Cache processed content for 30 minutes
        wp_cache_set($cache_key, $processed_content, 'affiliate_hub_content', 1800);
        
        return $processed_content;
    }
    
    /**
     * Check if current context should be processed
     *
     * @return bool Whether to process content
     */
    private function should_process_current_context() {
        // Skip admin area
        if (is_admin()) {
            return false;
        }
        
        // Skip REST API and AJAX requests
        if (defined('REST_REQUEST') || defined('DOING_AJAX')) {
            return false;
        }
        
        // Skip if disabled on home page
        if (is_home() && get_option(Constants::OPTION_AUTOLINKER_DISABLE_HOME, false)) {
            return false;
        }
        
        // Skip if disabled on archive pages
        if ((is_archive() || is_category() || is_tag() || is_date() || is_author()) && 
            get_option(Constants::OPTION_AUTOLINKER_DISABLE_ARCHIVE, false)) {
            return false;
        }
        
        return true;
    }
    
    /**
     * Process keywords in content
     *
     * @param string $content Original content
     * @param array $keywords_data Keywords data from cache
     * @return string Processed content
     */
    private function process_keywords($content, $keywords_data) {
        // Don't process if content has no text
        if (strlen(strip_tags($content)) < 10) {
            return $content;
        }
        
        // Protect existing links and certain HTML elements
        $protected_content = $this->protect_existing_elements($content);
        
        // Sort keywords by length (longest first) for better matching
        uksort($keywords_data, function($a, $b) {
            return strlen($b) - strlen($a);
        });
        
        $global_limit = get_option(Constants::OPTION_AUTOLINKER_KEYWORD_LIMIT, 0);
        $random_placement = get_option(Constants::OPTION_AUTOLINKER_RANDOM_PLACEMENT, false);
        
        foreach ($keywords_data as $keyword_lower => $data) {
            $keyword = $data['keyword'];
            $links = $data['links'];
            
            // Skip empty keywords
            if (empty($keyword)) {
                continue;
            }
            
            // Find all occurrences of this keyword
            $occurrences = $this->find_keyword_occurrences($protected_content, $keyword);
            
            if (empty($occurrences)) {
                continue;
            }
            
            // Randomize occurrences if random placement is enabled
            if ($random_placement) {
                shuffle($occurrences);
            }
            
            // Process each link for this keyword
            foreach ($links as $link_data) {
                $link_id = $link_data['link_id'];
                $link_url = $link_data['link_url'];
                $link_title = $link_data['link_title'];
                $keyword_limit = $link_data['keyword_limit'] > 0 ? $link_data['keyword_limit'] : $global_limit;
                
                // Check if we've reached the limit for this keyword/link combination
                $counter_key = $keyword_lower . '_' . $link_id;
                if (!isset($this->link_counters[$counter_key])) {
                    $this->link_counters[$counter_key] = 0;
                }
                
                if ($keyword_limit > 0 && $this->link_counters[$counter_key] >= $keyword_limit) {
                    continue;
                }
                
                // Process occurrences for this link
                foreach ($occurrences as $occurrence) {
                    if ($keyword_limit > 0 && $this->link_counters[$counter_key] >= $keyword_limit) {
                        break;
                    }
                    
                    // Check if this occurrence is still available (not already replaced)
                    if (strpos($protected_content, $occurrence['full_match']) === false) {
                        continue;
                    }
                    
                    // Create the affiliate link
                    $affiliate_link = $this->create_affiliate_link($occurrence['keyword'], $link_url, $link_title, $link_id);
                    
                    // Replace first occurrence
                    $protected_content = $this->replace_first_occurrence($protected_content, $occurrence['full_match'], $affiliate_link);
                    
                    // Increment counter
                    $this->link_counters[$counter_key]++;
                }
            }
        }
        
        // Restore protected elements
        $final_content = $this->restore_protected_elements($protected_content);
        
        return $final_content;
    }
    
    /**
     * Protect existing links and HTML elements from processing
     *
     * @param string $content Original content
     * @return string Content with protected elements
     */
    private function protect_existing_elements($content) {
        // Patterns to protect
        $protect_patterns = array(
            // Links
            '/<a\b[^>]*>.*?<\/a>/is',
            // Code blocks
            '/<code\b[^>]*>.*?<\/code>/is',
            '/<pre\b[^>]*>.*?<\/pre>/is',
            // Scripts and styles
            '/<script\b[^>]*>.*?<\/script>/is',
            '/<style\b[^>]*>.*?<\/style>/is',
            // Images
            '/<img\b[^>]*>/is',
            // Input fields
            '/<input\b[^>]*>/is',
            '/<textarea\b[^>]*>.*?<\/textarea>/is',
            // Shortcodes
            '/\[[^\]]*\]/is'
        );
        
        // Optionally protect heading tags
        if (!get_option(Constants::OPTION_AUTOLINKER_LINK_IN_HEADINGS, false)) {
            $protect_patterns[] = '/<h[1-6]\b[^>]*>.*?<\/h[1-6]>/is';
        }
        
        $protected_content = $content;
        $this->protected_elements = array();
        $placeholder_counter = 0;
        
        foreach ($protect_patterns as $pattern) {
            $protected_content = preg_replace_callback($pattern, function($matches) use (&$placeholder_counter) {
                $placeholder = '{{AFFILIATE_HUB_PROTECTED_' . $placeholder_counter . '}}';
                $this->protected_elements[$placeholder] = $matches[0];
                $placeholder_counter++;
                return $placeholder;
            }, $protected_content);
        }
        
        return $protected_content;
    }
    
    /**
     * Restore protected elements after processing
     *
     * @param string $content Content with placeholders
     * @return string Content with restored elements
     */
    private function restore_protected_elements($content) {
        if (empty($this->protected_elements)) {
            return $content;
        }
        
        return str_replace(array_keys($this->protected_elements), array_values($this->protected_elements), $content);
    }
    
    /**
     * Find keyword occurrences in content
     *
     * @param string $content Content to search
     * @param string $keyword Keyword to find
     * @return array Occurrences with context
     */
    private function find_keyword_occurrences($content, $keyword) {
        $occurrences = array();
        
        // Create word boundary pattern for whole word matching
        $escaped_keyword = preg_quote($keyword, '/');
        $pattern = '/\b' . $escaped_keyword . '\b/ui'; // u flag for unicode, i for case-insensitive
        
        preg_match_all($pattern, $content, $matches, PREG_OFFSET_CAPTURE);
        
        foreach ($matches[0] as $match) {
            $occurrences[] = array(
                'keyword' => $match[0],
                'position' => $match[1],
                'full_match' => $match[0]
            );
        }
        
        return $occurrences;
    }
    
    /**
     * Create affiliate link HTML
     *
     * @param string $anchor_text Anchor text
     * @param string $url Link URL
     * @param string $title Link title
     * @param int $link_id Link ID for tracking
     * @return string HTML link
     */
    private function create_affiliate_link($anchor_text, $url, $title, $link_id) {
        $attributes = array(
            'href' => esc_url($url),
            'title' => esc_attr($title),
            'data-affiliate-id' => intval($link_id),
            'class' => 'affiliate-link autolinker-link'
        );
        
        // Add rel attributes based on settings
        $rel_attrs = array();
        
        if (get_option(Constants::OPTION_NOFOLLOW_LINKS, true)) {
            $rel_attrs[] = 'nofollow';
        }
        
        if (get_option(Constants::OPTION_SPONSORED_LINKS, false)) {
            $rel_attrs[] = 'sponsored';
        }
        
        if (!empty($rel_attrs)) {
            $attributes['rel'] = implode(' ', $rel_attrs);
        }
        
        // Add target="_blank" if enabled
        if (get_option(Constants::OPTION_OPEN_NEW_WINDOW, false)) {
            $attributes['target'] = '_blank';
        }
        
        // Build link HTML
        $link_html = '<a';
        foreach ($attributes as $attr => $value) {
            $link_html .= ' ' . $attr . '="' . $value . '"';
        }
        $link_html .= '>' . esc_html($anchor_text) . '</a>';
        
        return $link_html;
    }
    
    /**
     * Replace first occurrence of text
     *
     * @param string $content Content
     * @param string $search Search string
     * @param string $replace Replacement string
     * @return string Modified content
     */
    private function replace_first_occurrence($content, $search, $replace) {
        $pos = strpos($content, $search);
        if ($pos !== false) {
            return substr_replace($content, $replace, $pos, strlen($search));
        }
        return $content;
    }
    
    /**
     * Get processing statistics
     *
     * @return array Processing stats
     */
    public function get_processing_stats() {
        return array(
            'links_created' => array_sum($this->link_counters),
            'keywords_processed' => count($this->link_counters),
            'counters' => $this->link_counters
        );
    }
}
