<?php
/**
 * Keywords Autolinker Settings
 *
 * @package AffiliateHub
 * @subpackage Modules\KeywordsAutolinker
 */

namespace AffiliateHub\Modules\KeywordsAutolinker;

use AffiliateHub\Core\Constants;

/**
 * Settings Management for Keywords Autolinker
 */
class Settings {
    
    /**
     * Cache instance
     *
     * @var Cache
     */
    private $cache;
    
    /**
     * Constructor
     */
    public function __construct() {
        $this->cache = new Cache();
        
        // Initialize settings
        add_action('admin_init', array($this, 'init_settings'));
        
        // Handle form submission
        add_action('admin_init', array($this, 'handle_form_submission'));
        
        // Enqueue admin scripts and styles
        add_action('admin_enqueue_scripts', array($this, 'admin_enqueue_scripts'));
    }
    
    /**
     * Enqueue admin scripts and styles
     *
     * @param string $hook_suffix The current admin page hook suffix
     */
    public function admin_enqueue_scripts($hook_suffix) {
        // Only load on autolinker settings page
        if ($hook_suffix !== 'affiliate-links_page_affiliate-hub-autolinker') {
            return;
        }
        
        wp_enqueue_style(
            'affiliate-hub-autolinker-admin',
            AFFILIATE_HUB_URL . 'assets/css/autolinker-admin.css',
            array(),
            AFFILIATE_HUB_VERSION
        );
        
        wp_enqueue_script(
            'affiliate-hub-autolinker-admin',
            AFFILIATE_HUB_URL . 'assets/js/autolinker-admin.js',
            array('jquery', 'jquery-ui-sortable'),
            AFFILIATE_HUB_VERSION,
            true
        );
        
        // Localize script for AJAX
        wp_localize_script('affiliate-hub-autolinker-admin', 'affiliateHubAutolinker', array(
            'ajaxUrl' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('affiliate_hub_autolinker_nonce'),
            'strings' => array(
                'confirmRebuild' => __('This will rebuild the entire autolinker cache. Continue?', 'affiliate-hub'),
                'confirmClear' => __('This will clear all cached keyword data. Continue?', 'affiliate-hub'),
                'confirmCleanup' => __('This will remove cache entries for deleted links. Continue?', 'affiliate-hub'),
                'rebuildingCache' => __('Rebuilding cache...', 'affiliate-hub'),
                'clearingCache' => __('Clearing cache...', 'affiliate-hub'),
                'cleaningCache' => __('Cleaning cache...', 'affiliate-hub'),
                'cacheRebuilt' => __('Cache rebuilt successfully', 'affiliate-hub'),
                'cacheCleared' => __('Cache cleared successfully', 'affiliate-hub'),
                'cacheCleaned' => __('Cache cleaned successfully', 'affiliate-hub'),
                'error' => __('Error occurred', 'affiliate-hub'),
                'errorRebuildingCache' => __('Error rebuilding cache', 'affiliate-hub')
            )
        ));
    }
    
    /**
     * Add settings page to admin menu
     * Note: This is now handled by Admin.php directly
     */
    public function add_settings_page() {
        // This method is kept for compatibility but functionality moved to Admin.php
        // The menu is now added in Admin.php::add_admin_menu()
    }
    
    /**
     * Initialize settings
     */
    public function init_settings() {
        // Register the settings group first
        register_setting(
            'affiliate_hub_autolinker_settings', // Option group
            'affiliate_hub_autolinker_settings', // Option name
            array($this, 'sanitize_settings')
        );
        
        // Add settings sections
        add_settings_section(
            'affiliate_hub_autolinker_general',
            __('General Settings', 'affiliate-hub'),
            array($this, 'general_section_callback'),
            'affiliate_hub_autolinker_settings'
        );
        
        add_settings_section(
            'affiliate_hub_autolinker_behavior',
            __('Behavior Settings', 'affiliate-hub'),
            array($this, 'behavior_section_callback'),
            'affiliate_hub_autolinker_settings'
        );
        
        add_settings_section(
            'affiliate_hub_autolinker_placement',
            __('Placement Settings', 'affiliate-hub'),
            array($this, 'placement_section_callback'),
            'affiliate_hub_autolinker_settings'
        );
        
        // Add settings fields
        $this->add_settings_fields();
    }
    
    /**
     * Add settings fields
     */
    private function add_settings_fields() {
        // General Settings
        add_settings_field(
            'autolinker_enabled',
            __('Enable Keywords Autolinker', 'affiliate-hub'),
            array($this, 'enabled_callback'),
            'affiliate_hub_autolinker_settings',
            'affiliate_hub_autolinker_general'
        );
        
        add_settings_field(
            'autolinker_keyword_limit',
            __('Global Keyword Limit', 'affiliate-hub'),
            array($this, 'keyword_limit_callback'),
            'affiliate_hub_autolinker_settings',
            'affiliate_hub_autolinker_general'
        );
        
        add_settings_field(
            'autolinker_post_types',
            __('Enabled Post Types', 'affiliate-hub'),
            array($this, 'post_types_callback'),
            'affiliate_hub_autolinker_settings',
            'affiliate_hub_autolinker_general'
        );
        
        // Behavior Settings
        add_settings_field(
            'autolinker_random_placement',
            __('Random Placement', 'affiliate-hub'),
            array($this, 'random_placement_callback'),
            'affiliate_hub_autolinker_settings',
            'affiliate_hub_autolinker_behavior'
        );
        
        add_settings_field(
            'autolinker_link_in_headings',
            __('Link in Headings', 'affiliate-hub'),
            array($this, 'link_in_headings_callback'),
            'affiliate_hub_autolinker_settings',
            'affiliate_hub_autolinker_behavior'
        );
        
        // Placement Settings
        add_settings_field(
            'autolinker_disable_archive',
            __('Disable on Archives', 'affiliate-hub'),
            array($this, 'disable_archive_callback'),
            'affiliate_hub_autolinker_settings',
            'affiliate_hub_autolinker_placement'
        );
        
        add_settings_field(
            'autolinker_disable_home',
            __('Disable on Homepage', 'affiliate-hub'),
            array($this, 'disable_home_callback'),
            'affiliate_hub_autolinker_settings',
            'affiliate_hub_autolinker_placement'
        );
        
        add_settings_field(
            'autolinker_enable_feeds',
            __('Enable in Feeds', 'affiliate-hub'),
            array($this, 'enable_feeds_callback'),
            'affiliate_hub_autolinker_settings',
            'affiliate_hub_autolinker_placement'
        );
    }
    
    /**
     * General settings section callback
     */
    public function general_section_callback() {
        echo '<p>' . esc_html__('Configure the basic settings for the Keywords Autolinker.', 'affiliate-hub') . '</p>';
    }
    
    /**
     * Behavior settings section callback
     */
    public function behavior_section_callback() {
        echo '<p>' . esc_html__('Control how the autolinker processes content.', 'affiliate-hub') . '</p>';
    }
    
    /**
     * Placement settings section callback
     */
    public function placement_section_callback() {
        echo '<p>' . esc_html__('Configure where autolinks should appear on your site.', 'affiliate-hub') . '</p>';
    }
    
    /**
     * Render the settings page
     */
    public function render_settings_page() {
        $stats = $this->cache->get_cache_stats();
        ?>
        <div class="wrap">
            <h1><?php echo esc_html(get_admin_page_title()); ?></h1>
            
            <?php if (isset($_GET['settings-updated'])): ?>
                <div class="notice notice-success is-dismissible">
                    <p><?php esc_html_e('Settings saved successfully!', 'affiliate-hub'); ?></p>
                </div>
            <?php endif; ?>
            
            <?php if (!get_option(Constants::OPTION_AUTOLINKER_ENABLED, false)): ?>
                <div class="notice notice-warning">
                    <p><?php esc_html_e('Keywords Autolinker is currently disabled. Enable it in the General Settings section below.', 'affiliate-hub'); ?></p>
                </div>
            <?php endif; ?>
            
            <div class="affiliate-hub-settings-container">
                <div class="affiliate-hub-settings-main">
                    <form method="post" action="">
                        <?php wp_nonce_field('affiliate_hub_autolinker_settings_nonce', 'affiliate_hub_autolinker_settings_nonce'); ?>
                        <?php settings_fields('affiliate_hub_autolinker_settings'); ?>
                        <?php do_settings_sections('affiliate_hub_autolinker_settings'); ?>
                        <?php submit_button(); ?>
                    </form>
                </div>
                
                <div class="affiliate-hub-settings-sidebar">
                    <?php $this->render_cache_stats(); ?>
                    <?php $this->render_help(); ?>
                </div>
            </div>
        </div>
        
        <style>
        .affiliate-hub-settings-container {
            display: flex;
            gap: 20px;
        }
        .affiliate-hub-settings-main {
            flex: 2;
            min-width: 0;
        }
        .affiliate-hub-settings-sidebar {
            flex: 1;
            min-width: 280px;
        }
        .affiliate-hub-info-box {
            background: #fff;
            border: 1px solid #ccd0d4;
            border-radius: 4px;
            padding: 16px;
            margin-bottom: 20px;
        }
        .affiliate-hub-info-box h3 {
            margin-top: 0;
            margin-bottom: 12px;
        }
        .affiliate-hub-cache-stats .stat-item {
            display: flex;
            justify-content: space-between;
            margin-bottom: 8px;
        }
        .affiliate-hub-cache-actions {
            margin-top: 16px;
        }
        .affiliate-hub-cache-actions .button {
            margin-right: 8px;
            margin-bottom: 8px;
        }
        </style>
        <?php
    }
    
    /**
     * Render cache statistics box
     */
    private function render_cache_stats() {
        $stats = $this->cache->get_cache_stats();
        ?>
        <div class="affiliate-hub-info-box">
            <h3><?php esc_html_e('Cache Statistics', 'affiliate-hub'); ?></h3>
            <div class="affiliate-hub-cache-stats">
                <div class="stat-item">
                    <span><?php esc_html_e('Total Keywords:', 'affiliate-hub'); ?></span>
                    <strong><?php echo esc_html($stats['total_keywords']); ?></strong>
                </div>
                <div class="stat-item">
                    <span><?php esc_html_e('Unique Keywords:', 'affiliate-hub'); ?></span>
                    <strong><?php echo esc_html($stats['unique_keywords']); ?></strong>
                </div>
                <div class="stat-item">
                    <span><?php esc_html_e('Links with Keywords:', 'affiliate-hub'); ?></span>
                    <strong><?php echo esc_html($stats['links_with_keywords']); ?></strong>
                </div>
                <div class="stat-item">
                    <span><?php esc_html_e('Last Rebuilt:', 'affiliate-hub'); ?></span>
                    <strong><?php echo esc_html($stats['last_built']); ?></strong>
                </div>
                <div class="stat-item">
                    <span><?php esc_html_e('Cache Size:', 'affiliate-hub'); ?></span>
                    <strong><?php echo esc_html($stats['cache_size']); ?></strong>
                </div>
            </div>
            
            <?php if ($stats['needs_rebuild']): ?>
                <div class="notice notice-warning inline" style="margin: 12px 0;">
                    <p><?php esc_html_e('Cache may be outdated and should be rebuilt.', 'affiliate-hub'); ?></p>
                </div>
            <?php endif; ?>
            
            <div class="affiliate-hub-cache-actions">
                <button type="button" 
                        class="button button-primary" 
                        data-action="rebuild_cache"
                        data-nonce="<?php echo wp_create_nonce('affiliate_hub_autolinker_nonce'); ?>">
                        <?php esc_html_e('Rebuild Cache', 'affiliate-hub'); ?>
                </button>
                
                <button type="button" 
                        class="button button-secondary" 
                        data-action="clear_cache"
                        data-nonce="<?php echo wp_create_nonce('affiliate_hub_autolinker_nonce'); ?>">
                        <?php esc_html_e('Clear Cache', 'affiliate-hub'); ?>
                </button>
                
                <button type="button" 
                        class="button button-secondary" 
                        data-action="cleanup_cache"
                        data-nonce="<?php echo wp_create_nonce('affiliate_hub_autolinker_nonce'); ?>">
                        <?php esc_html_e('Cleanup Cache', 'affiliate-hub'); ?>
                </button>
            </div>
        </div>
        <?php
    }
    
    /**
     * Render help box
     */
    private function render_help() {
        ?>
        <div class="affiliate-hub-info-box">
            <h3><?php esc_html_e('How to Use', 'affiliate-hub'); ?></h3>
            <ol>
                <li><?php esc_html_e('Enable the Keywords Autolinker module above', 'affiliate-hub'); ?></li>
                <li><?php esc_html_e('Edit your affiliate links and add keywords in the "Autolink Keywords" meta box', 'affiliate-hub'); ?></li>
                <li><?php esc_html_e('Set keyword limits per link or use the global setting', 'affiliate-hub'); ?></li>
                <li><?php esc_html_e('Configure behavior settings as needed', 'affiliate-hub'); ?></li>
                <li><?php esc_html_e('Your content will automatically include affiliate links for matching keywords', 'affiliate-hub'); ?></li>
            </ol>
            
            <h4><?php esc_html_e('Tips:', 'affiliate-hub'); ?></h4>
            <ul>
                <li><?php esc_html_e('Use specific, unique keywords to avoid over-linking', 'affiliate-hub'); ?></li>
                <li><?php esc_html_e('Set reasonable limits to maintain content quality', 'affiliate-hub'); ?></li>
                <li><?php esc_html_e('Test on a staging site before going live', 'affiliate-hub'); ?></li>
                <li><?php esc_html_e('Rebuild the cache after making changes to keywords', 'affiliate-hub'); ?></li>
            </ul>
        </div>
        <?php
    }
    
    // Settings field callbacks
    public function enabled_callback() {
        $value = get_option(Constants::OPTION_AUTOLINKER_ENABLED, false);
        echo '<input type="checkbox" id="autolinker_enabled" name="' . Constants::OPTION_AUTOLINKER_ENABLED . '" value="1" ' . checked($value, true, false) . ' />';
        echo '<label for="autolinker_enabled">' . esc_html__('Enable automatic keyword linking', 'affiliate-hub') . '</label>';
        echo '<p class="description">' . esc_html__('When enabled, keywords will be automatically converted to affiliate links in your content.', 'affiliate-hub') . '</p>';
    }
    
    public function keyword_limit_callback() {
        $value = get_option(Constants::OPTION_AUTOLINKER_KEYWORD_LIMIT, 0);
        echo '<input type="number" id="autolinker_keyword_limit" name="' . Constants::OPTION_AUTOLINKER_KEYWORD_LIMIT . '" value="' . esc_attr($value) . '" min="0" max="50" class="small-text" />';
        echo '<p class="description">' . esc_html__('Maximum number of links per keyword per post (0 = unlimited). Individual links can override this setting.', 'affiliate-hub') . '</p>';
    }
    
    public function post_types_callback() {
        $enabled_types = get_option(Constants::OPTION_AUTOLINKER_POST_TYPES, array('post', 'page'));
        $post_types = get_post_types(array('public' => true), 'objects');
        
        echo '<fieldset>';
        foreach ($post_types as $post_type) {
            if ($post_type->name === Constants::POST_TYPE_AFFILIATE_LINK) {
                continue; // Skip affiliate link post type
            }
            
            $checked = in_array($post_type->name, $enabled_types) ? 'checked' : '';
            echo '<label><input type="checkbox" name="' . Constants::OPTION_AUTOLINKER_POST_TYPES . '[]" value="' . esc_attr($post_type->name) . '" ' . $checked . ' /> ' . esc_html($post_type->labels->name) . '</label><br>';
        }
        echo '</fieldset>';
        echo '<p class="description">' . esc_html__('Select post types where autolinker should be active.', 'affiliate-hub') . '</p>';
    }
    
    public function random_placement_callback() {
        $value = get_option(Constants::OPTION_AUTOLINKER_RANDOM_PLACEMENT, false);
        echo '<input type="checkbox" id="autolinker_random_placement" name="' . Constants::OPTION_AUTOLINKER_RANDOM_PLACEMENT . '" value="1" ' . checked($value, true, false) . ' />';
        echo '<label for="autolinker_random_placement">' . esc_html__('Enable random placement of autolinks', 'affiliate-hub') . '</label>';
        echo '<p class="description">' . esc_html__('When enabled, keyword occurrences will be processed in random order rather than sequential.', 'affiliate-hub') . '</p>';
    }
    
    public function link_in_headings_callback() {
        $value = get_option(Constants::OPTION_AUTOLINKER_LINK_IN_HEADINGS, false);
        echo '<input type="checkbox" id="autolinker_link_in_headings" name="' . Constants::OPTION_AUTOLINKER_LINK_IN_HEADINGS . '" value="1" ' . checked($value, true, false) . ' />';
        echo '<label for="autolinker_link_in_headings">' . esc_html__('Allow linking inside heading tags (H1, H2, H3, etc.)', 'affiliate-hub') . '</label>';
        echo '<p class="description">' . esc_html__('Note: This only affects headings in post content, not the post title.', 'affiliate-hub') . '</p>';
    }
    
    public function disable_archive_callback() {
        $value = get_option(Constants::OPTION_AUTOLINKER_DISABLE_ARCHIVE, false);
        echo '<input type="checkbox" id="autolinker_disable_archive" name="' . Constants::OPTION_AUTOLINKER_DISABLE_ARCHIVE . '" value="1" ' . checked($value, true, false) . ' />';
        echo '<label for="autolinker_disable_archive">' . esc_html__('Disable autolinker on archive pages (category, tag, date, author pages)', 'affiliate-hub') . '</label>';
    }
    
    public function disable_home_callback() {
        $value = get_option(Constants::OPTION_AUTOLINKER_DISABLE_HOME, false);
        echo '<input type="checkbox" id="autolinker_disable_home" name="' . Constants::OPTION_AUTOLINKER_DISABLE_HOME . '" value="1" ' . checked($value, true, false) . ' />';
        echo '<label for="autolinker_disable_home">' . esc_html__('Disable autolinker on home page', 'affiliate-hub') . '</label>';
        echo '<p class="description">' . esc_html__('Many sites prefer not to have affiliate links on their home page.', 'affiliate-hub') . '</p>';
    }
    
    public function enable_feeds_callback() {
        $value = get_option(Constants::OPTION_AUTOLINKER_ENABLE_FEEDS, false);
        echo '<input type="checkbox" id="autolinker_enable_feeds" name="' . Constants::OPTION_AUTOLINKER_ENABLE_FEEDS . '" value="1" ' . checked($value, true, false) . ' />';
        echo '<label for="autolinker_enable_feeds">' . esc_html__('Enable autolinker in RSS/Atom feeds', 'affiliate-hub') . '</label>';
        echo '<p class="description">' . esc_html__('By default disabled due to potential feed validation issues.', 'affiliate-hub') . '</p>';
    }
    
    /**
     * Sanitize settings input
     *
     * @param array $input Raw settings input
     * @return array Sanitized settings
     */
    public function sanitize_settings($input) {
        $sanitized = array();
        
        if (is_array($input)) {
            foreach ($input as $key => $value) {
                if (is_bool($value) || $value === '1' || $value === 1) {
                    $sanitized[$key] = true;
                } elseif (is_numeric($value)) {
                    $sanitized[$key] = intval($value);
                } else {
                    $sanitized[$key] = sanitize_text_field($value);
                }
            }
        }
        
        return $sanitized;
    }
    
    /**
     * Handle form submission
     */
    public function handle_form_submission() {
        // Check if form was submitted
        if (!isset($_POST['submit']) || !isset($_POST['affiliate_hub_autolinker_settings_nonce'])) {
            return;
        }
        
        // Verify nonce
        if (!wp_verify_nonce($_POST['affiliate_hub_autolinker_settings_nonce'], 'affiliate_hub_autolinker_settings_nonce')) {
            return;
        }
        
        // Check permissions
        if (!current_user_can('manage_options')) {
            return;
        }
        
        // Save settings
        $settings_to_save = array(
            Constants::OPTION_AUTOLINKER_ENABLED,
            Constants::OPTION_AUTOLINKER_KEYWORD_LIMIT,
            Constants::OPTION_AUTOLINKER_RANDOM_PLACEMENT,
            Constants::OPTION_AUTOLINKER_LINK_IN_HEADINGS,
            Constants::OPTION_AUTOLINKER_DISABLE_ARCHIVE,
            Constants::OPTION_AUTOLINKER_DISABLE_HOME,
            Constants::OPTION_AUTOLINKER_ENABLE_FEEDS,
            Constants::OPTION_AUTOLINKER_POST_TYPES
        );
        
        foreach ($settings_to_save as $setting) {
            $value = isset($_POST[$setting]) ? $_POST[$setting] : false;
            
            // Special handling for checkboxes
            if (in_array($setting, array(
                Constants::OPTION_AUTOLINKER_ENABLED,
                Constants::OPTION_AUTOLINKER_RANDOM_PLACEMENT,
                Constants::OPTION_AUTOLINKER_LINK_IN_HEADINGS,
                Constants::OPTION_AUTOLINKER_DISABLE_ARCHIVE,
                Constants::OPTION_AUTOLINKER_DISABLE_HOME,
                Constants::OPTION_AUTOLINKER_ENABLE_FEEDS
            ))) {
                $value = !empty($value);
            }
            
            // Special handling for arrays
            if ($setting === Constants::OPTION_AUTOLINKER_POST_TYPES && is_array($value)) {
                $value = array_map('sanitize_text_field', $value);
            }
            
            // Special handling for numbers
            if ($setting === Constants::OPTION_AUTOLINKER_KEYWORD_LIMIT) {
                $value = max(0, intval($value));
            }
            
            update_option($setting, $value);
        }
        
        // Handle cache actions
        if (isset($_POST['rebuild_cache'])) {
            $this->cache->rebuild_all_cache();
        }
        
        // Redirect to prevent resubmission
        wp_redirect(add_query_arg('settings-updated', 'true', wp_get_referer()));
        exit;
    }
}
