<?php
/**
 * Link Scanner Module
 *
 * @package AffiliateHub
 * @subpackage Modules
 */

namespace AffiliateHub\Modules;

use AffiliateHub\Core\Constants;

class LinkScanner implements ModuleInterface {

    const CRON_HOOK = 'affiliate_hub_linkscanner_process';

    public function init() {
    // Ensure DB schema is up to date
    $db = new LinkScanner\DB();
    $db->ensure_schema();

        // Admin UI
    \add_action('admin_menu', array($this, 'register_admin'));
    \add_action('admin_enqueue_scripts', array($this, 'enqueue_admin_assets'));

        // AJAX handlers
    \add_action('wp_ajax_affiliate_hub_start_scan', array($this, 'ajax_start_scan'));
    \add_action('wp_ajax_affiliate_hub_get_scan_status', array($this, 'ajax_get_scan_status'));
    \add_action('wp_ajax_affiliate_hub_export_scan', array($this, 'ajax_export_scan'));
    \add_action('wp_ajax_affiliate_hub_scan_details', array($this, 'ajax_scan_details'));
    \add_action('wp_ajax_affiliate_hub_retry_link', array($this, 'ajax_retry_link'));
    \add_action('wp_ajax_affiliate_hub_ignore_link', array($this, 'ajax_ignore_link'));
    \add_action('wp_ajax_affiliate_hub_unignore_link', array($this, 'ajax_unignore_link'));
    \add_action('wp_ajax_affiliate_hub_toggle_scanner_animation', array($this, 'ajax_toggle_animations'));
    \add_action('wp_ajax_affiliate_hub_pause_scan', array($this, 'ajax_pause_scan'));
    \add_action('wp_ajax_affiliate_hub_resume_scan', array($this, 'ajax_resume_scan'));
    \add_action('wp_ajax_affiliate_hub_cancel_scan', array($this, 'ajax_cancel_scan'));
        // BUGFIX: Add new AJAX endpoint to force finish stuck scans
        \add_action('wp_ajax_affiliate_hub_force_finish_scan', array($this, 'ajax_force_finish_scan'));
        
        // Cron processing
    \add_action(self::CRON_HOOK, array($this, 'process_batch'));
        // Bulk actions for list table (ajax)
    \add_action('wp_ajax_affiliate_hub_linkscanner_bulk_action', array($this, 'ajax_bulk_action'));
    }

    public function get_info() {
        return array(
            'name' => 'Link Scanner',
            'description' => 'Scans posts for broken/forbidden links',
            'version' => '0.1',
        );
    }

    public function is_enabled() {
        return true;
    }

    public function register_admin() {
        // Link Scanner is now registered in Core\Admin.php to maintain proper menu hierarchy
        // This method is kept for interface compatibility but doesn't register duplicate menu
    }

    public function enqueue_admin_assets($hook) {
        // The link scanner page can appear under different admin hooks (e.g. edit.php when added on CPT screens).
        // Check the hook string or fallback to the 'page' query var so the scripts are always enqueued on the scanner page.
        $is_scanner_page = false;
        if (is_string($hook) && strpos($hook, 'affiliate-hub-link-scanner') !== false) {
            $is_scanner_page = true;
        }
        if (!$is_scanner_page && isset($_GET['page']) && $_GET['page'] === 'affiliate-hub-link-scanner') {
            $is_scanner_page = true;
        }
        if (!$is_scanner_page) {
            return;
        }

        // ApexCharts for better performance and visual appearance
        \wp_enqueue_script('affiliate-hub-apexcharts', 'https://cdn.jsdelivr.net/npm/apexcharts@latest/dist/apexcharts.min.js', array(), '3.45.2', true);
        \wp_enqueue_script('affiliate-hub-link-scanner-admin', \plugins_url('assets/js/link-scanner-admin.js', AFFILIATE_HUB_FILE), array('affiliate-hub-apexcharts', 'jquery'), '0.1', true);
            // Build post types list for selective scanning
            $pts = get_post_types(array('public' => true), 'objects');
            $post_types = array();
            foreach ($pts as $key => $obj) {
                $post_types[] = array('name' => $key, 'label' => $obj->labels->singular_name);
            }

            \wp_localize_script('affiliate-hub-link-scanner-admin', 'AffiliateHubLinkScanner', array(
                    'ajax_url' => admin_url('admin-ajax.php'),
                    'nonce' => wp_create_nonce('affiliate_hub_linkscanner'),
                    'strings' => array(
                        'start_scan' => __('Start scan', 'affiliate-hub'),
                        'confirm_retry' => __('Retry this link?', 'affiliate-hub'),
                        'confirm_ignore' => __('Ignore this link?', 'affiliate-hub')
                    ),
                    'chart' => array(
                        'disable_animations' => (bool) get_option(\AffiliateHub\Core\Constants::OPTION_LINK_SCANNER_DISABLE_ANIMATIONS, false),
                        'debounce_ms' => (int) get_option(\AffiliateHub\Core\Constants::OPTION_LINK_SCANNER_DEBOUNCE_MS, 200)
                    ),
                    'post_types' => $post_types,
                    'endpoints' => array(
                        'start' => 'affiliate_hub_start_scan',
                        'status' => 'affiliate_hub_get_scan_status',
                        'details' => 'affiliate_hub_scan_details',
                        'pause' => 'affiliate_hub_pause_scan',
                        'resume' => 'affiliate_hub_resume_scan',
                        'cancel' => 'affiliate_hub_cancel_scan'
                    )
                ));
    }

    public function render_admin_page() {
        // Defensive fallback: ensure assets and localization are enqueued even if the admin_enqueue_scripts
        // hook didn't run for this specific hook name (common when the page is shown under CPT edit.php).
        if (function_exists('wp_script_is') && !wp_script_is('affiliate-hub-link-scanner-admin', 'enqueued')) {
            // call the same enqueue routine with a scanner-like hook name to force localize data
            $this->enqueue_admin_assets('affiliate-hub-link-scanner');
        }

        $admin = new LinkScanner\AdminPage();
        $admin->render();
    }

    public function ajax_start_scan() {
        try {
            if (!\current_user_can('manage_options')) {
                \wp_send_json_error(array('message' => 'Forbidden', 'reason' => 'insufficient_capabilities'), 403);
            }
            if (!\check_ajax_referer('affiliate_hub_linkscanner', 'nonce', false)) {
                \wp_send_json_error(array('message' => 'Forbidden', 'reason' => 'invalid_nonce'), 403);
            }

            $scanner = new LinkScanner\Scanner();
            $db = new LinkScanner\DB();

            // Get scan type (content or affiliates)
            $scan_type = isset($_POST['scan_type']) ? sanitize_text_field($_POST['scan_type']) : 'content';

            // Accept optional post_types array from request (only used for content scan)
            $post_types = array();
            if ($scan_type === 'content' && isset($_POST['post_types']) && is_array($_POST['post_types'])) {
                $post_types = array_map('sanitize_text_field', $_POST['post_types']);
            }

            $scan_options = array(
                'scan_type' => $scan_type,
                'post_types' => maybe_serialize($post_types)
            );

            $scan_id = $db->create_scan(array(
                'started_at' => \current_time('mysql'), 
                'status' => 'pending', 
                'total_urls' => 0, 
                'processed' => 0, 
                'options' => maybe_serialize($scan_options)
            ));

            // Extract links based on scan type
            if ($scan_type === 'affiliates') {
                // Scan affiliate links destinations
                $links_count = $scanner->queue_affiliate_links($scan_id);
            } else {
                // Default: scan content links (respecting selected post types)
                $links_count = $scanner->queue_all_posts($scan_id, $post_types);
            }
            
            $db->update_scan($scan_id, array('total_urls' => $links_count, 'status' => 'running'));

            // Schedule immediate processing via WP-Cron (no Action Scheduler)
            \wp_schedule_single_event(time() + 1, self::CRON_HOOK, array($scan_id));
            
            // OPTIMIZED: For development environments where cron might not work
            if (defined('WP_DEBUG') && WP_DEBUG && defined('DISABLE_WP_CRON') && DISABLE_WP_CRON) {
                \spawn_cron();
            }

            wp_send_json_success(array('scan_id' => $scan_id));
        } catch (\Throwable $e) {
            \wp_send_json_error(array('message' => 'Server error', 'exception' => $e->getMessage()), 500);
        }
    }

    public function ajax_pause_scan() {
        try {
            if (!\current_user_can('manage_options')) {
                \wp_send_json_error(array('message' => 'Forbidden', 'reason' => 'insufficient_capabilities'), 403);
            }
            if (!\check_ajax_referer('affiliate_hub_linkscanner', 'nonce', false)) {
                \wp_send_json_error(array('message' => 'Forbidden', 'reason' => 'invalid_nonce'), 403);
            }
            $scan_id = isset($_POST['scan_id']) ? intval($_POST['scan_id']) : 0;
            if (!$scan_id) \wp_send_json_error(array('message' => 'Invalid scan id'), 400);
            $db = new LinkScanner\DB();
            $db->set_scan_paused($scan_id, true);
            \wp_send_json_success(array('message' => 'paused'));
        } catch (\Throwable $e) {
            \wp_send_json_error(array('message' => 'Server error', 'exception' => $e->getMessage()), 500);
        }
    }

    public function ajax_resume_scan() {
        try {
            if (!\current_user_can('manage_options')) {
                \wp_send_json_error(array('message' => 'Forbidden', 'reason' => 'insufficient_capabilities'), 403);
            }
            if (!\check_ajax_referer('affiliate_hub_linkscanner', 'nonce', false)) {
                \wp_send_json_error(array('message' => 'Forbidden', 'reason' => 'invalid_nonce'), 403);
            }
            $scan_id = isset($_POST['scan_id']) ? intval($_POST['scan_id']) : 0;
            if (!$scan_id) \wp_send_json_error(array('message' => 'Invalid scan id'), 400);
            
            $db = new LinkScanner\DB();
            
            // BUGFIX: Clear any existing scheduled events first to prevent duplicates
            wp_clear_scheduled_hook(self::CRON_HOOK, array($scan_id));
            
            $db->set_scan_paused($scan_id, false);
            
            // Check if scan should be finished immediately
            if ($db->finish_scan_if_complete($scan_id)) {
                \wp_send_json_success(array('message' => 'resumed', 'status' => 'finished'));
                return;
            }
            
            // Schedule processing immediately if there are pending links
            \wp_schedule_single_event(time() + 3, self::CRON_HOOK, array($scan_id));
            \wp_send_json_success(array('message' => 'resumed'));
        } catch (\Throwable $e) {
            \wp_send_json_error(array('message' => 'Server error', 'exception' => $e->getMessage()), 500);
        }
    }

    public function ajax_cancel_scan() {
        try {
            if (!\current_user_can('manage_options')) {
                \wp_send_json_error(array('message' => 'Forbidden', 'reason' => 'insufficient_capabilities'), 403);
            }
            if (!\check_ajax_referer('affiliate_hub_linkscanner', 'nonce', false)) {
                \wp_send_json_error(array('message' => 'Forbidden', 'reason' => 'invalid_nonce'), 403);
            }
            $scan_id = isset($_POST['scan_id']) ? intval($_POST['scan_id']) : 0;
            if (!$scan_id) \wp_send_json_error(array('message' => 'Invalid scan id'), 400);
            $db = new LinkScanner\DB();
            $db->set_scan_canceled($scan_id, true);
            \wp_send_json_success(array('message' => 'canceled'));
        } catch (\Throwable $e) {
            \wp_send_json_error(array('message' => 'Server error', 'exception' => $e->getMessage()), 500);
        }
    }

    /**
     * BUGFIX: Force finish a scan that got stuck (e.g., after multiple pause/resume)
     */
    public function ajax_force_finish_scan() {
        try {
            if (!\current_user_can('manage_options')) {
                \wp_send_json_error(array('message' => 'Forbidden', 'reason' => 'insufficient_capabilities'), 403);
            }
            if (!\check_ajax_referer('affiliate_hub_linkscanner', 'nonce', false)) {
                \wp_send_json_error(array('message' => 'Forbidden', 'reason' => 'invalid_nonce'), 403);
            }
            $scan_id = isset($_POST['scan_id']) ? intval($_POST['scan_id']) : 0;
            if (!$scan_id) \wp_send_json_error(array('message' => 'Invalid scan id'), 400);
            
            $db = new LinkScanner\DB();
            
            // Clear any remaining cron jobs
            wp_clear_scheduled_hook(self::CRON_HOOK, array($scan_id));
            
            // Force scan to finish
            $db->update_scan($scan_id, array('status' => 'finished', 'finished_at' => \current_time('mysql')));
            
            \wp_send_json_success(array('message' => 'Scan force-finished', 'status' => 'finished'));
        } catch (\Throwable $e) {
            \wp_send_json_error(array('message' => 'Server error', 'exception' => $e->getMessage()), 500);
        }
    }

    public function ajax_get_scan_status() {
        try {
            if (!\current_user_can('manage_options')) {
                \wp_send_json_error(array('message' => 'Forbidden', 'reason' => 'insufficient_capabilities'), 403);
            }
            if (!\check_ajax_referer('affiliate_hub_linkscanner', 'nonce', false)) {
                \wp_send_json_error(array('message' => 'Forbidden', 'reason' => 'invalid_nonce'), 403);
            }

            $scan_id = isset($_GET['scan_id']) ? intval($_GET['scan_id']) : 0;
            $db = new LinkScanner\DB();
            $status = $db->get_scan($scan_id);
            wp_send_json_success($status);
        } catch (\Throwable $e) {
            \wp_send_json_error(array('message' => 'Server error', 'exception' => $e->getMessage()), 500);
        }
    }

    public function ajax_scan_details() {
        try {
            if (!\current_user_can('manage_options')) {
                \wp_send_json_error(array('message' => 'Forbidden', 'reason' => 'insufficient_capabilities'), 403);
            }
            if (!\check_ajax_referer('affiliate_hub_linkscanner', 'nonce', false)) {
                \wp_send_json_error(array('message' => 'Forbidden', 'reason' => 'invalid_nonce'), 403);
            }
            $scan_id = isset($_GET['scan_id']) ? intval($_GET['scan_id']) : 0;
            if (!$scan_id) {
                \wp_send_json_error(array('message' => 'Invalid scan id'), 400);
            }
            $page = isset($_GET['page']) ? max(1, intval($_GET['page'])) : 1;
            $per_page = isset($_GET['per_page']) ? max(1, intval($_GET['per_page'])) : 50;
            $include_ignored = isset($_GET['include_ignored']) ? boolval($_GET['include_ignored']) : false;
            $search = isset($_GET['search']) ? sanitize_text_field($_GET['search']) : '';
            $status = isset($_GET['status']) ? sanitize_text_field($_GET['status']) : '';
            $post_type = isset($_GET['post_type']) ? sanitize_text_field($_GET['post_type']) : '';
            $since_id = isset($_GET['since_id']) ? intval($_GET['since_id']) : 0;
            $offset = ($page - 1) * $per_page;
            $db = new LinkScanner\DB();
            $links = $db->get_links_for_scan_filtered($scan_id, $per_page, $offset, $include_ignored, $search, $status, $post_type, $since_id);
            // compute total for these filters using DB helper
            // post_type may be a comma-separated list from the client; keep as-is to allow multi-type filtering
            $total = $db->count_links_for_scan_filtered($scan_id, $include_ignored, $search, $status, $post_type, $since_id);
            // compute max id returned for incremental polling
            $max_id = 0;
            foreach ($links as $l) { if (isset($l->id) && intval($l->id) > $max_id) $max_id = intval($l->id); }
            \wp_send_json_success(array('links' => $links, 'total' => $total, 'page' => $page, 'per_page' => $per_page, 'include_ignored' => $include_ignored, 'search' => $search, 'status' => $status, 'post_type' => $post_type, 'max_id' => $max_id));
        } catch (\Throwable $e) {
            error_log('[AffiliateHub] LinkScanner ajax_scan_details exception: ' . $e->getMessage() . '\n' . $e->getTraceAsString());
            \wp_send_json_error(array('message' => 'Server error', 'exception' => $e->getMessage()), 500);
        }
    }

    public function ajax_unignore_link() {
        try {
            if (!\current_user_can('manage_options')) {
                \wp_send_json_error(array('message' => 'Forbidden', 'reason' => 'insufficient_capabilities'), 403);
            }
            if (!\check_ajax_referer('affiliate_hub_linkscanner', 'nonce', false)) {
                error_log('[AffiliateHub] LinkScanner: invalid nonce on ajax_unignore_link for user_id=' . get_current_user_id());
                \wp_send_json_error(array('message' => 'Forbidden', 'reason' => 'invalid_nonce'), 403);
            }
            $link_id = isset($_POST['link_id']) ? intval($_POST['link_id']) : 0;
            if (!$link_id) \wp_send_json_error(array('message' => 'Invalid link id'), 400);
            $db = new LinkScanner\DB();
            $db->set_link_ignored($link_id, false);
            \wp_send_json_success(array('message' => 'unignored'));
        } catch (\Throwable $e) {
            error_log('[AffiliateHub] LinkScanner ajax_unignore_link exception: ' . $e->getMessage() . '\n' . $e->getTraceAsString());
            \wp_send_json_error(array('message' => 'Server error', 'exception' => $e->getMessage()), 500);
        }
    }

    public function ajax_retry_link() {
        try {
            if (!\current_user_can('manage_options')) {
                \wp_send_json_error(array('message' => 'Forbidden', 'reason' => 'insufficient_capabilities'), 403);
            }
            if (!\check_ajax_referer('affiliate_hub_linkscanner', 'nonce', false)) {
                error_log('[AffiliateHub] LinkScanner: invalid nonce on ajax_retry_link for user_id=' . get_current_user_id());
                \wp_send_json_error(array('message' => 'Forbidden', 'reason' => 'invalid_nonce'), 403);
            }
            $link_id = isset($_POST['link_id']) ? intval($_POST['link_id']) : 0;
            if (!$link_id) \wp_send_json_error(array('message' => 'Invalid link id'), 400);
            $db = new LinkScanner\DB();
            $db->requeue_link($link_id);
            \wp_send_json_success(array('message' => 'requeued'));
        } catch (\Throwable $e) {
            error_log('[AffiliateHub] LinkScanner ajax_retry_link exception: ' . $e->getMessage() . '\n' . $e->getTraceAsString());
            \wp_send_json_error(array('message' => 'Server error', 'exception' => $e->getMessage()), 500);
        }
    }

    public function ajax_ignore_link() {
        try {
            if (!\current_user_can('manage_options')) {
                \wp_send_json_error(array('message' => 'Forbidden', 'reason' => 'insufficient_capabilities'), 403);
            }
            if (!\check_ajax_referer('affiliate_hub_linkscanner', 'nonce', false)) {
                error_log('[AffiliateHub] LinkScanner: invalid nonce on ajax_ignore_link for user_id=' . get_current_user_id());
                \wp_send_json_error(array('message' => 'Forbidden', 'reason' => 'invalid_nonce'), 403);
            }
            $link_id = isset($_POST['link_id']) ? intval($_POST['link_id']) : 0;
            $ignore = isset($_POST['ignore']) ? boolval($_POST['ignore']) : true;
            if (!$link_id) \wp_send_json_error(array('message' => 'Invalid link id'), 400);
            $db = new LinkScanner\DB();
            $db->set_link_ignored($link_id, $ignore);
            \wp_send_json_success(array('message' => 'updated'));
        } catch (\Throwable $e) {
            error_log('[AffiliateHub] LinkScanner ajax_ignore_link exception: ' . $e->getMessage() . '\n' . $e->getTraceAsString());
            \wp_send_json_error(array('message' => 'Server error', 'exception' => $e->getMessage()), 500);
        }
    }

    public function ajax_bulk_action() {
        if (!\current_user_can('manage_options')) {
            \wp_send_json_error(array('message' => 'Forbidden'), 403);
        }
        if (!\check_ajax_referer('affiliate_hub_linkscanner', 'nonce', false)) {
            \wp_send_json_error(array('message' => 'Invalid nonce'), 403);
        }
        $action = isset($_POST['bulk_action']) ? sanitize_text_field($_POST['bulk_action']) : '';
        $items = isset($_POST['items']) && is_array($_POST['items']) ? array_map('intval', $_POST['items']) : array();
        $db = new LinkScanner\DB();
        if (empty($items) || empty($action)) {
            \wp_send_json_error(array('message' => 'No items'), 400);
        }
        foreach ($items as $id) {
            if ($action === 'retry') {
                $db->requeue_link($id);
            } elseif ($action === 'ignore') {
                $db->set_link_ignored($id, true);
            } elseif ($action === 'unignore') {
                $db->set_link_ignored($id, false);
            }
        }
        \wp_send_json_success(array('processed' => count($items)));
    }

    public function ajax_export_scan() {
        try {
            if (!\current_user_can('manage_options')) {
                \wp_send_json_error(array('message' => 'Forbidden', 'reason' => 'insufficient_capabilities'), 403);
            }
            if (!\check_ajax_referer('affiliate_hub_linkscanner', 'nonce', false)) {
                error_log('[AffiliateHub] LinkScanner: invalid nonce on ajax_export_scan for user_id=' . get_current_user_id());
                \wp_send_json_error(array('message' => 'Forbidden', 'reason' => 'invalid_nonce'), 403);
            }
            $scan_id = isset($_GET['scan_id']) ? intval($_GET['scan_id']) : 0;
            $db = new LinkScanner\DB();
            $links = $db->get_links_for_scan($scan_id, 0, 0); // 0 => no limit

            // Build CSV
            $csv = "url,post_id,anchor_text,status,status_code,final_url,note\n";
            foreach ($links as $l) {
                $csv .= '"' . str_replace('"','""',$l->url) . '",' . intval($l->post_id) . ',"' . str_replace('"','""',isset($l->anchor_text) ? $l->anchor_text : '') . '",' . (isset($l->status) ? $l->status : '') . ',' . intval(isset($l->status_code) ? $l->status_code : 0) . ',"' . str_replace('"','""',isset($l->final_url) ? $l->final_url : '') . '",' . '"' . str_replace('"','""', isset($l->note) ? $l->note : '') . '"' . "\n";
            }

            header('Content-Type: text/csv');
            header('Content-Disposition: attachment; filename="affiliatehub_scan_' . $scan_id . '.csv"');
            echo $csv;
            exit;
        } catch (\Throwable $e) {
            error_log('[AffiliateHub] LinkScanner ajax_export_scan exception: ' . $e->getMessage() . '\n' . $e->getTraceAsString());
            // If headers already sent this will still fail; forward a small JSON error as fallback
            if (!headers_sent()) {
                header('Content-Type: application/json', true, 500);
                echo json_encode(array('success' => false, 'message' => 'Server error', 'exception' => $e->getMessage()));
            }
            exit;
        }
    }

    /**
     * AJAX: toggle animations setting (temporary control from admin page)
     */
    public function ajax_toggle_animations() {
        if (!\current_user_can('manage_options')) {
            \wp_send_json_error(array('message' => 'Forbidden', 'reason' => 'insufficient_capabilities'), 403);
        }
        if (!\check_ajax_referer('affiliate_hub_linkscanner', 'nonce', false)) {
            \wp_send_json_error(array('message' => 'Forbidden', 'reason' => 'invalid_nonce'), 403);
        }
        $val = isset($_POST['disable']) && ($_POST['disable'] == '1' || $_POST['disable'] === 'true');
        update_option(\AffiliateHub\Core\Constants::OPTION_LINK_SCANNER_DISABLE_ANIMATIONS, $val);
        \wp_send_json_success(array('disabled' => $val));
    }

    /**
     * Process a batch of pending links for a given scan.
     * WP-Cron passes the scan_id as first arg.
     */
    public function process_batch($scan_id = 0) {
        $db = new LinkScanner\DB();
        $checker = new LinkScanner\HttpChecker();
        // Check scan state first
        $scan = $db->get_scan($scan_id);
        if ($scan && isset($scan->canceled) && $scan->canceled) {
            // Do not process canceled scans
            return;
        }
        if ($scan && isset($scan->paused) && $scan->paused) {
            // If paused, do not schedule further work now; resumption will schedule
            return;
        }

        $batch = $db->get_pending_links(50, $scan_id); // OPTIMIZED: Increased from 20 to 50
        if (empty($batch)) {
            // Clear any remaining cron jobs and mark scan finished
            wp_clear_scheduled_hook(self::CRON_HOOK, array($scan_id));
            $db->finish_scan_if_complete($scan_id);
            return;
        }

        foreach ($batch as $link) {
            try {
                $attempts = $link->attempts + 1;
                $res = $checker->check($link->url);
                $status = $this->map_http_code_to_status($res['code'], $res['note']);

                // Jeśli po 3 próbach nadal nie działa, oznacz jako failed
                if ($attempts >= 3 && $status !== 'active') {
                    $status = 'failed';
                }

                $update = array(
                    'status_code' => $res['code'],
                    'status' => $status,
                    'final_url' => $res['final_url'],
                    'last_checked' => \current_time('mysql'),
                    'attempts' => $attempts
                );
                if (!empty($res['note'])) {
                    $update['note'] = $res['note'];
                }
                $db->update_link($link->id, $update);
            } catch (\Throwable $e) {
                // Jeśli wystąpi błąd, oznacz link jako failed i przejdź dalej
                $db->update_link($link->id, array(
                    'status' => 'failed',
                    'note' => 'fatal_error: ' . $e->getMessage(),
                    'last_checked' => \current_time('mysql'),
                    'attempts' => $link->attempts + 1
                ));
            }
            // Wymuś inkrementację processed nawet jeśli link nie był pending
            $db->increment_processed($scan_id, 1);
        }

        // OPTIMIZED: Schedule next batch with minimal delay for faster processing
        $rate = intval(\get_option(\AffiliateHub\Core\Constants::OPTION_LINK_SCANNER_RATE, 20)); // Increased from 5 to 20
        if ($rate <= 0) $rate = 20;
        // OPTIMIZED: Use smaller, fixed delay instead of calculated delay
        $delay = 1; // Fixed 1 second delay instead of calculated
        
        // Check if scan should finish before scheduling next batch
        if ($db->finish_scan_if_complete($scan_id)) {
            // Scan completed, don't schedule another batch
            wp_clear_scheduled_hook(self::CRON_HOOK, array($scan_id));
            return;
        }
        
        \wp_schedule_single_event(time() + $delay, self::CRON_HOOK, array($scan_id));
    }

    /**
     * OPTIMIZED: Map HTTP response code to status efficiently
     * @param int $code HTTP status code
     * @param string $note Additional context note
     * @return string Status string
     */
    private function map_http_code_to_status($code, $note = '') {
        // Handle network/WP errors first (code = 0)
        if ($code == 0) {
            return 'network_error';
        }
        
        // Success range (2xx)
        if ($code >= 200 && $code < 300) {
            return 'active';
        }
        
        // Client errors (4xx)
        if ($code >= 400 && $code < 500) {
            switch ($code) {
                case 403:
                    return 'forbidden';
                case 404:
                    return 'not_found';
                case 405:
                    // Method Not Allowed - usually HEAD blocked but GET might work
                    // If we have a note indicating GET was successful, treat as active
                    if (strpos($note, 'head_not_allowed_used_get') !== false) {
                        return 'active';
                    }
                    return 'broken'; // If GET also failed with 405
                default:
                    return 'broken'; // 401, 408, 429, etc.
            }
        }
        
        // Server errors (5xx)
        if ($code >= 500 && $code < 600) {
            // Special handling for challenge pages
            if ($code == 503 && strpos($note, 'challenge') !== false) {
                return 'forbidden'; // Treat challenges as access restriction
            }
            return 'server_error';
        }
        
        // Redirects that shouldn't happen (3xx) - consider as issues
        if ($code >= 300 && $code < 400) {
            return 'broken'; // Shouldn't reach here due to redirection handling
        }
        
        // Unknown codes
        return 'broken';
    }
}
