<?php
namespace AffiliateHub\Modules\LinkScanner;

use AffiliateHub\Core\Constants;

class DB {
    protected $wpdb;
    protected $table_scans;
    protected $table_links;

    public function __construct() {
        global $wpdb;
        $this->wpdb = $wpdb;
        $this->table_scans = $wpdb->prefix . Constants::TABLE_LINK_SCANS;
        $this->table_links = $wpdb->prefix . Constants::TABLE_SCANNED_LINKS;
    }

    /**
     * Ensure database schema updates for scanned links table are applied.
     * Adds the `ignored` column if missing.
     */
    public function ensure_schema() {
        // If the scanned links table doesn't exist, create both scanner tables via dbDelta
        $table_exists = $this->wpdb->get_var($this->wpdb->prepare("SHOW TABLES LIKE %s", $this->table_links));
        if (empty($table_exists)) {
            // Create tables (same schema as Activator)
            $charset_collate = $this->wpdb->get_charset_collate();

            $table_scans = $this->wpdb->prefix . Constants::TABLE_LINK_SCANS;
            $sql_scans = "CREATE TABLE $table_scans (
                id bigint(20) NOT NULL AUTO_INCREMENT,
                started_at datetime DEFAULT NULL,
                finished_at datetime DEFAULT NULL,
                status varchar(20) DEFAULT 'pending',
                total_urls int DEFAULT 0,
                processed int DEFAULT 0,
                created_by bigint(20) DEFAULT 0,
                paused tinyint(1) DEFAULT 0,
                canceled tinyint(1) DEFAULT 0,
                options longtext,
                PRIMARY KEY (id),
                KEY status (status)
            ) $charset_collate;";

            $table_scanned = $this->table_links;
            $sql_scanned = "CREATE TABLE $table_scanned (
                id bigint(20) NOT NULL AUTO_INCREMENT,
                scan_id bigint(20) NOT NULL,
                post_id bigint(20) DEFAULT 0,
                url text NOT NULL,
                anchor_text varchar(255) DEFAULT '',
                context_snippet text,
                status_code int DEFAULT 0,
                status varchar(20) DEFAULT 'pending',
                ignored tinyint(1) DEFAULT 0,
                note text DEFAULT NULL,
                final_url text,
                attempts int DEFAULT 0,
                last_checked datetime DEFAULT NULL,
                created_at datetime DEFAULT CURRENT_TIMESTAMP,
                PRIMARY KEY (id),
                KEY scan_id (scan_id),
                KEY status (status)
            ) $charset_collate;";

            require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
            dbDelta($sql_scans);
            dbDelta($sql_scanned);
            return;
        }

        // Table exists — add missing columns if needed
    $col_ignored = $this->wpdb->get_var($this->wpdb->prepare("SHOW COLUMNS FROM {$this->table_links} LIKE %s", 'ignored'));
        if (empty($col_ignored)) {
            $sql = "ALTER TABLE {$this->table_links} ADD COLUMN ignored tinyint(1) DEFAULT 0";
            $this->wpdb->query($sql);
        }

        $col_note = $this->wpdb->get_var($this->wpdb->prepare("SHOW COLUMNS FROM {$this->table_links} LIKE %s", 'note'));
        if (empty($col_note)) {
            $sql = "ALTER TABLE {$this->table_links} ADD COLUMN note text NULL";
            $this->wpdb->query($sql);
        }

        // Ensure scans table has paused/canceled columns
        $col_paused = $this->wpdb->get_var($this->wpdb->prepare("SHOW COLUMNS FROM {$this->table_scans} LIKE %s", 'paused'));
        if (empty($col_paused)) {
            $sql = "ALTER TABLE {$this->table_scans} ADD COLUMN paused tinyint(1) DEFAULT 0";
            $this->wpdb->query($sql);
        }
        $col_canceled = $this->wpdb->get_var($this->wpdb->prepare("SHOW COLUMNS FROM {$this->table_scans} LIKE %s", 'canceled'));
        if (empty($col_canceled)) {
            $sql = "ALTER TABLE {$this->table_scans} ADD COLUMN canceled tinyint(1) DEFAULT 0";
            $this->wpdb->query($sql);
        }
    }

    public function set_scan_paused($scan_id, $paused = true) {
        $val = $paused ? 1 : 0;
        $this->wpdb->update($this->table_scans, array('paused' => $val, 'status' => $paused ? 'paused' : 'running'), array('id' => $scan_id));
    }

    public function set_scan_canceled($scan_id, $canceled = true) {
        $val = $canceled ? 1 : 0;
        $data = array('canceled' => $val);
        if ($canceled) {
            $data['status'] = 'canceled';
            $data['finished_at'] = \current_time('mysql');
        }
        $this->wpdb->update($this->table_scans, $data, array('id' => $scan_id));
    }

    public function create_scan($data) {
        $this->wpdb->insert($this->table_scans, $data);
        return (int) $this->wpdb->insert_id;
    }

    public function update_scan($id, $data) {
        $this->wpdb->update($this->table_scans, $data, array('id' => $id));
    }

    public function get_scan($id) {
        $scan = $this->wpdb->get_row($this->wpdb->prepare("SELECT * FROM {$this->table_scans} WHERE id = %d", $id));
        if (!$scan) return null;

        // Add simple counts for admin UI - OPTIMIZED: Single query instead of 6
        $count_query = $this->wpdb->prepare("
            SELECT 
                status,
                COUNT(*) as count 
            FROM {$this->table_links} 
            WHERE scan_id = %d 
            GROUP BY status", 
            $id
        );
        
        $status_results = $this->wpdb->get_results($count_query);
        
        // Initialize all counts to 0
        $counts = (object) array(
            'active' => 0,
            'broken' => 0, 
            'forbidden' => 0,
            'not_found' => 0,
            'server_error' => 0,
            'network_error' => 0,
            'pending' => 0,
            'failed' => 0,
            'skipped' => 0
        );
        
        // Fill in actual counts
        foreach ($status_results as $row) {
            if (property_exists($counts, $row->status)) {
                $counts->{$row->status} = (int) $row->count;
            }
        }

    $scan->counts = $counts;
    $scan->processed = isset($scan->processed) ? intval($scan->processed) : 0;
    $scan->total_urls = isset($scan->total_urls) ? intval($scan->total_urls) : 0;
    $scan->paused = isset($scan->paused) ? intval($scan->paused) : 0;
    $scan->canceled = isset($scan->canceled) ? intval($scan->canceled) : 0;
    // options column may be serialized
    if (isset($scan->options) && !empty($scan->options)) {
        $opts = maybe_unserialize($scan->options);
        $scan->options = is_array($opts) ? $opts : $opts;
    }
    return $scan;
    }

    /**
     * Insert a scanned link row. Accepts optional note, status and ignored flag.
     * Backwards-compatible defaults keep behavior unchanged.
     */
    public function insert_link($scan_id, $post_id, $url, $anchor_text = '', $snippet = '', $note = '', $status = 'pending', $ignored = 0) {
        $data = array(
            'scan_id' => $scan_id,
            'post_id' => $post_id,
            'url' => $url,
            'anchor_text' => $anchor_text,
            'context_snippet' => $snippet,
            'status' => $status,
            'attempts' => 0,
            'ignored' => $ignored,
            'created_at' => \current_time('mysql')
        );

        if (!empty($note)) {
            $data['note'] = $note;
        }

        $this->wpdb->insert($this->table_links, $data);
        return (int) $this->wpdb->insert_id;
    }

    public function get_pending_links($limit = 20, $scan_id = 0) {
        // Do not return links that are marked ignored
        if ($scan_id) {
            $sql = $this->wpdb->prepare("SELECT * FROM {$this->table_links} WHERE status = %s AND ignored = 0 AND scan_id = %d LIMIT %d", 'pending', $scan_id, $limit);
        } else {
            $sql = $this->wpdb->prepare("SELECT * FROM {$this->table_links} WHERE status = %s AND ignored = 0 LIMIT %d", 'pending', $limit);
        }
        return $this->wpdb->get_results($sql);
    }

    public function update_link($id, $data) {
        $this->wpdb->update($this->table_links, $data, array('id' => $id));
    }

    public function increment_processed($scan_id, $by = 1) {
        $this->wpdb->query($this->wpdb->prepare("UPDATE {$this->table_scans} SET processed = processed + %d WHERE id = %d", $by, $scan_id));
    }

    /**
     * Fetch links for a scan with optional ignored filter
     * @param int $scan_id
     * @param int $limit
     * @param int $offset
     * @param bool $include_ignored
     * @return array
     */
    public function get_links_for_scan($scan_id, $limit = 100, $offset = 0, $include_ignored = false) {
        $ignore_clause = $include_ignored ? '' : " AND ignored = 0";
        if ($limit <= 0) {
            $sql = $this->wpdb->prepare("SELECT * FROM {$this->table_links} WHERE scan_id = %d $ignore_clause ORDER BY id ASC", $scan_id);
        } else {
            $sql = $this->wpdb->prepare("SELECT * FROM {$this->table_links} WHERE scan_id = %d $ignore_clause ORDER BY id ASC LIMIT %d OFFSET %d", $scan_id, $limit, $offset);
        }
        return $this->wpdb->get_results($sql);
    }

    /**
     * Filtered fetch for admin details: supports search (url, anchor_text, note), status filter, post_type filter
     */
    public function get_links_for_scan_filtered($scan_id, $limit = 100, $offset = 0, $include_ignored = false, $search = '', $status = '', $post_type = '', $since_id = 0, $orderby = 'l.id', $order = 'ASC') {
        $clauses = array();
        $clauses[] = $this->wpdb->prepare('scan_id = %d', $scan_id);
        if (!$include_ignored) {
            $clauses[] = 'ignored = 0';
        }
        if (!empty($since_id) && intval($since_id) > 0) {
            $clauses[] = $this->wpdb->prepare('l.id > %d', intval($since_id));
        }
        if (!empty($status)) {
            $clauses[] = $this->wpdb->prepare('status = %s', $status);
        }
        if (!empty($search)) {
            $like = '%' . $this->wpdb->esc_like($search) . '%';
            $clauses[] = $this->wpdb->prepare('(url LIKE %s OR anchor_text LIKE %s OR note LIKE %s)', $like, $like, $like);
        }
        // Always join post to get post_type for display
        $join_post = true;
        $sql = 'SELECT l.*, p.post_type as post_type FROM ' . $this->table_links . ' l LEFT JOIN ' . $this->wpdb->posts . ' p ON p.ID = l.post_id';
        
        // post_type filtering (if requested)
        if (!empty($post_type)) {
            // support comma-separated list of post types
            $types = array_map('trim', explode(',', $post_type));
            $types = array_filter($types);
            if (count($types) === 1) {
                $clauses[] = $this->wpdb->prepare('p.post_type = %s', $types[0]);
            } elseif (count($types) > 1) {
                // build a prepared IN clause safely
                $placeholders = implode(',', array_fill(0, count($types), '%s'));
                // prepare expects SQL then individual args; use argument unpacking
                $in_sql = $this->wpdb->prepare('(' . $placeholders . ')', ...array_values($types));
                $clauses[] = 'p.post_type IN ' . $in_sql;
            }
        }
        $where = ' WHERE ' . implode(' AND ', $clauses);
    // allow passed-in order column and direction (caller must validate allowed columns)
    $order = ' ORDER BY ' . $orderby . ' ' . ($order === 'DESC' ? 'DESC' : 'ASC');
        if ($limit <= 0) {
            $final = $sql . $where . $order;
        } else {
            $final = $sql . $where . $order . $this->wpdb->prepare(' LIMIT %d OFFSET %d', $limit, $offset);
        }
        return $this->wpdb->get_results($final);
    }

    /**
     * Count links matching the same filters as get_links_for_scan_filtered
     */
    public function count_links_for_scan_filtered($scan_id, $include_ignored = false, $search = '', $status = '', $post_type = '', $since_id = 0) {
        $clauses = array();
        $clauses[] = $this->wpdb->prepare('l.scan_id = %d', $scan_id);
        if (!$include_ignored) {
            $clauses[] = 'l.ignored = 0';
        }
        if (!empty($since_id) && intval($since_id) > 0) {
            $clauses[] = $this->wpdb->prepare('l.id > %d', intval($since_id));
        }
        if (!empty($status)) {
            $clauses[] = $this->wpdb->prepare('l.status = %s', $status);
        }
        if (!empty($search)) {
            $like = '%' . $this->wpdb->esc_like($search) . '%';
            $clauses[] = $this->wpdb->prepare('(l.url LIKE %s OR l.anchor_text LIKE %s OR l.note LIKE %s)', $like, $like, $like);
        }

        $from = $this->table_links . ' l';
        $where = ' WHERE ' . implode(' AND ', $clauses);
        if (!empty($post_type)) {
            // join posts table for post_type filtering
            $from = $this->table_links . ' l LEFT JOIN ' . $this->wpdb->posts . ' p ON p.ID = l.post_id';
            // support comma-separated list of post types
            $types = array_map('trim', explode(',', $post_type));
            $types = array_filter($types);
            if (count($types) === 1) {
                $where .= ' AND ' . $this->wpdb->prepare('p.post_type = %s', $types[0]);
            } elseif (count($types) > 1) {
                $placeholders = implode(',', array_fill(0, count($types), '%s'));
                $in_sql = $this->wpdb->prepare('(' . $placeholders . ')', ...array_values($types));
                $where .= ' AND p.post_type IN ' . $in_sql;
            }
        }
        $sql = 'SELECT COUNT(*) FROM ' . $from . $where;
        return (int) $this->wpdb->get_var($sql);
    }

    public function get_links_count($scan_id) {
        return (int) $this->wpdb->get_var($this->wpdb->prepare("SELECT COUNT(*) FROM {$this->table_links} WHERE scan_id = %d", $scan_id));
    }

    /**
     * Return most recent scan row (or null)
     */
    public function get_latest_scan() {
        $row = $this->wpdb->get_row("SELECT id FROM {$this->table_scans} ORDER BY id DESC LIMIT 1");
        if (!$row) return null;
        return $this->get_scan(intval($row->id));
    }

    public function set_link_ignored($link_id, $ignored = true) {
        $val = $ignored ? 1 : 0;
        $this->wpdb->update($this->table_links, array('ignored' => $val), array('id' => $link_id));
    }

    public function requeue_link($link_id) {
        // Reset attempts and set status to pending
        $this->wpdb->update($this->table_links, array('status' => 'pending', 'attempts' => 0), array('id' => $link_id));
    }

    public function finish_scan_if_complete($scan_id) {
    $remaining = (int) $this->wpdb->get_var($this->wpdb->prepare("SELECT COUNT(*) FROM {$this->table_links} WHERE scan_id = %d AND status = %s", $scan_id, 'pending'));
        if ($remaining == 0) {
            $this->update_scan($scan_id, array('status' => 'finished', 'finished_at' => \current_time('mysql')));
            return true;
        }
        return false;
    }
}
