<?php
namespace AffiliateHub\Modules\LinkScanner;

class HttpChecker {
    public function check($url) {
        $result = array('code' => 0, 'final_url' => '', 'note' => '');
        $args = array(
            'timeout' => 5,  // OPTIMIZED: Reduced from 10 to 5 seconds
            'redirection' => 3,  // OPTIMIZED: Reduced from 5 to 3 redirects
            'sslverify' => false,
            'user-agent' => 'AffiliateHub-Scanner/1.0'
        );

        // IMPROVED: Add common sites that often block HEAD requests
        $sites_that_block_head = array(
            'amazon.', 'ebay.', 'aliexpress.', 'walmart.', 'target.com',
            'bestbuy.', 'etsy.', 'shopify.', 'facebook.com', 'instagram.com',
            'linkedin.com', 'pinterest.com', 'youtube.com'
        );
        
        $force_get = false;
        foreach ($sites_that_block_head as $site) {
            if (strpos(strtolower($url), $site) !== false) {
                $force_get = true;
                break;
            }
        }
        
        if ($force_get) {
            // Skip HEAD entirely for known problematic sites
            $res = \wp_remote_get($url, $args);
            $result['note'] = 'known_head_blocker_used_get';
        } else {
            // Try HEAD first for efficiency
            $res = \wp_remote_head($url, $args);
            
            // BUGFIX: If HEAD fails, returns empty, OR returns 405 (Method Not Allowed), try GET
            $should_retry_with_get = (
                \is_wp_error($res) || 
                empty($res['response']['code']) ||
                (isset($res['response']['code']) && $res['response']['code'] == 405)
            );
            
            if ($should_retry_with_get) {
                $res = \wp_remote_get($url, $args);
                if (!empty($res['response']['code']) && $res['response']['code'] != 405) {
                    // Mark that we had to use GET due to HEAD not being allowed
                    $result['note'] = 'head_not_allowed_used_get';
                }
            }
        }

        // IMPROVED: Better error handling with specific error types
        if (\is_wp_error($res)) {
            $error_code = $res->get_error_code();
            $result['note'] = $this->map_wp_error_to_note($error_code, $res->get_error_message());
            return $result; // code remains 0
        }

        $code = intval($res['response']['code']);
        $final = '';
        if (!empty($res['headers'])) {
            if (isset($res['headers']['location'])) {
                $final = $res['headers']['location'];
            } elseif (isset($res['headers']['x-final-url'])) {
                $final = $res['headers']['x-final-url'];
            }
        }

        // Heuristics for common anti-bot/challenge pages
        if ($code == 503) {
            // Cloudflare/anti-bot sometimes returns 503 with challenge
            $result['note'] = 'service_unavailable_or_challenge';
        }

        // Inspect body for challenge tokens if GET response available
        if (!empty($res['body']) && is_string($res['body'])) {
            $body = strtolower($res['body']);
            if (strpos($body, 'attention required') !== false || strpos($body, 'cloudflare') !== false && strpos($body, 'checking your browser') !== false) {
                $result['note'] = 'cloudflare_challenge';
            } elseif (strpos($body, 'captcha') !== false) {
                $result['note'] = 'captcha_challenge';
            } elseif (strpos($body, 'access denied') !== false) {
                $result['note'] = 'access_denied';
            }
        }

        $result['code'] = $code;
        $result['final_url'] = $final;
        return $result;
    }

    /**
     * OPTIMIZED: Map WordPress HTTP errors to meaningful notes
     * @param string $error_code WP error code
     * @param string $error_message WP error message  
     * @return string Descriptive note
     */
    private function map_wp_error_to_note($error_code, $error_message) {
        $error_map = array(
            'http_request_failed' => 'connection_failed',
            'http_request_timeout' => 'timeout',
            'resolve_host' => 'dns_error', 
            'connection_timeout' => 'timeout',
            'ssl_verify_failed' => 'ssl_error',
            'ssl_connect_error' => 'ssl_error',
            'too_many_redirects' => 'redirect_loop',
            'http_404' => 'not_found_redirect'
        );
        
        // Enhanced note mapping based on error message content
        $message_lower = strtolower($error_message);
        if (strpos($message_lower, 'ssl') !== false || strpos($message_lower, 'certificate') !== false) {
            return 'ssl_certificate_error';
        }
        if (strpos($message_lower, 'timeout') !== false) {
            return 'timeout';
        }
        if (strpos($message_lower, 'dns') !== false || strpos($message_lower, 'resolve') !== false) {
            return 'dns_error';
        }
        if (strpos($message_lower, '405') !== false || strpos($message_lower, 'method not allowed') !== false) {
            return 'method_not_allowed';
        }
        
        return isset($error_map[$error_code]) ? $error_map[$error_code] : 'network_error';
    }
}
