<?php
namespace AffiliateHub\Modules\LinkScanner;

if (!class_exists('\WP_List_Table')) {
    require_once(ABSPATH . 'wp-admin/includes/class-wp-list-table.php');
}

/**
 * Custom List Table for Link Scanner results
 */
class LinksListTable extends \WP_List_Table {
    
    private $db;
    private $scan_id;
    
    public function __construct($args = array()) {
        parent::__construct(array(
            'singular' => 'link',
            'plural'   => 'links',
            'ajax'     => false
        ));
        
        $this->db = new DB();
        $this->scan_id = isset($args['scan_id']) ? $args['scan_id'] : 0;
    }
    
    /**
     * Define table columns
     */
    public function get_columns() {
        return array(
            'cb'        => '<input type="checkbox" />',
            'url'       => __('Link', 'affiliate-hub'),
            'anchor_text' => __('Link Text', 'affiliate-hub'),
            'post_id'   => __('Post', 'affiliate-hub'),
            'post_type' => __('Post Type', 'affiliate-hub'),
            'status'    => __('Status', 'affiliate-hub'),
            'status_code' => __('Code', 'affiliate-hub'),
            'note'      => __('Note', 'affiliate-hub'),
            'actions'   => __('Actions', 'affiliate-hub')
        );
    }
    
    /**
     * Define sortable columns
     */
    public function get_sortable_columns() {
        return array(
            'url'         => array('url', false),
            'post_id'     => array('post_id', false),
            'status'      => array('status', false),
            'status_code' => array('status_code', false)
        );
    }
    
    /**
     * Handle checkbox column
     */
    public function column_cb($item) {
        return sprintf('<input type="checkbox" name="scanned[]" value="%s" />', $item->id);
    }
    
    /**
     * Handle URL column
     */
    public function column_url($item) {
        $url_display = strlen($item->url) > 50 ? substr($item->url, 0, 50) . '...' : $item->url;
        
        $output = sprintf(
            '<a href="%s" target="_blank" rel="noopener" title="%s">%s</a>',
            esc_url($item->url),
            esc_attr($item->url),
            esc_html($url_display)
        );
        
        $output .= sprintf(
            ' <button type="button" class="button-link copy-url-btn" data-url="%s" title="%s">
                <span class="dashicons dashicons-clipboard"></span>
            </button>',
            esc_attr($item->url),
            esc_attr__('Copy URL', 'affiliate-hub')
        );
        
        return $output;
    }
    
    /**
     * Handle anchor text column
     */
    public function column_anchor_text($item) {
        // Check if this is an affiliate link by looking at post_type
        $post_type = get_post_type($item->post_id);
        $is_affiliate = ($post_type === \AffiliateHub\Core\Constants::POST_TYPE_AFFILIATE_LINK);
        
        $output = '';
        if ($is_affiliate) {
            $output .= '<span class="dashicons dashicons-external" style="color: #0073aa;" title="' . esc_attr__('Affiliate Link Destination', 'affiliate-hub') . '"></span> ';
        }
        
        $output .= esc_html($item->anchor_text);
        
        if ($item->anchor_text && $item->post_id) {
            $output .= sprintf(
                ' <a href="%s" target="_blank" class="open-link-btn" title="%s">
                    <span class="dashicons dashicons-external"></span>
                </a>',
                esc_url(admin_url('post.php?post=' . $item->post_id . '&action=edit')),
                esc_attr__('Open post in editor', 'affiliate-hub')
            );
        }
        
        return $output;
    }
    
    /**
     * Handle post ID column
     */
    public function column_post_id($item) {
        if ($item->post_id) {
            $output = sprintf(
                '<a href="%s" target="_blank">#%s</a>',
                esc_url(admin_url('post.php?post=' . $item->post_id . '&action=edit')),
                esc_html($item->post_id)
            );
            
            $output .= sprintf(
                ' <a href="%s" target="_blank" class="open-link-btn" title="%s">
                    <span class="dashicons dashicons-external"></span>
                </a>',
                esc_url(get_permalink($item->post_id)),
                esc_attr__('View post on frontend', 'affiliate-hub')
            );
            
            return $output;
        } else {
            return '—';
        }
    }
    
    /**
     * Handle post type column
     */
    public function column_post_type($item) {
        if ($item->post_id) {
            $post_type = get_post_type($item->post_id);
            $post_type_names = array(
                'post' => __('Wpis', 'affiliate-hub'),
                'page' => __('Strona', 'affiliate-hub'),
                'product' => __('Produkt', 'affiliate-hub'),
                'attachment' => __('Załącznik', 'affiliate-hub'),
                'affiliate_link' => __('Link Partnerski', 'affiliate-hub')
            );
            $display_name = isset($post_type_names[$post_type]) ? $post_type_names[$post_type] : ucfirst($post_type);
            return esc_html($display_name);
        } else {
            return '—';
        }
    }
    
    /**
     * Handle status column - OPTIMIZED: Consolidated status colors
     */
    public function column_status($item) {
        // OPTIMIZED: Status color mapping
        $status_colors = array(
            'active' => '#28a745',          // Green - working
            'forbidden' => '#ffc107',       // Yellow - access restricted  
            'not_found' => '#fd7e14',       // Orange - missing resource
            'server_error' => '#dc3545',    // Red - server problem
            'network_error' => '#6f42c1',   // Purple - connectivity issue
            'broken' => '#dc3545',          // Red - other errors
            'pending' => '#6c757d',         // Gray - not checked yet
            'skipped' => '#17a2b8'          // Cyan - intentionally skipped
        );
        
        $status_color = isset($status_colors[$item->status]) ? $status_colors[$item->status] : '#6c757d';
        
        return sprintf(
            '<span class="affiliate-hub-status-badge" style="background-color: %s; color: #fff; padding: 2px 6px; border-radius: 3px; font-size: 11px;">%s</span>',
            esc_attr($status_color),
            esc_html(ucfirst(str_replace('_', ' ', $item->status)))
        );
    }
    
    /**
     * Handle status code column
     */
    public function column_status_code($item) {
        $code = intval($item->status_code);
        $code_color = '#6c757d';
        if ($code >= 200 && $code < 300) $code_color = '#28a745';
        elseif ($code >= 400 && $code < 500) $code_color = '#dc3545';
        elseif ($code >= 500) $code_color = '#b02a37';
        
        return sprintf(
            '<span class="affiliate-hub-redirect-type" style="background-color: %s; color: #fff; padding: 2px 6px; border-radius: 3px; font-size: 11px;">%s</span>',
            esc_attr($code_color),
            esc_html($code)
        );
    }
    
    /**
     * Handle note column
     */
    public function column_note($item) {
        // Show affiliate link context if available
        if (!empty($item->context_snippet) && strpos($item->context_snippet, 'Affiliate Link:') === 0) {
            return '<em style="color: #0073aa;">' . esc_html($item->context_snippet) . '</em>';
        } else {
            return esc_html($item->note ?? ''); 
        }
    }
    
    /**
     * Handle actions column
     */
    public function column_actions($item) {
        if (isset($item->ignored) && intval($item->ignored) === 1) {
            return sprintf(
                '<button class="button ah-unignore-link" data-id="%s">%s</button>',
                esc_attr($item->id),
                __('Un-ignore', 'affiliate-hub')
            );
        } else {
            return sprintf(
                '<button class="button ah-retry-link" data-id="%s">%s</button> 
                <button class="button ah-ignore-link" data-id="%s">%s</button>',
                esc_attr($item->id),
                __('Retry', 'affiliate-hub'),
                esc_attr($item->id),
                __('Ignore', 'affiliate-hub')
            );
        }
    }
    
    /**
     * Default column handler
     */
    public function column_default($item, $column_name) {
        switch ($column_name) {
            default:
                return isset($item->$column_name) ? $item->$column_name : '';
        }
    }
    
    /**
     * Get bulk actions
     */
    public function get_bulk_actions() {
        return array(
            'retry'  => __('Retry', 'affiliate-hub'),
            'ignore' => __('Ignore', 'affiliate-hub'),
            'unignore' => __('Un-ignore', 'affiliate-hub')
        );
    }
    
    /**
     * Prepare table items
     */
    public function prepare_items() {
        // Get current page
        $per_page = 20;
        $current_page = $this->get_pagenum();
        $offset = ($current_page - 1) * $per_page;
        
        // Get search and filters
        $search = isset($_REQUEST['s']) ? sanitize_text_field($_REQUEST['s']) : '';
        $status_filter = isset($_REQUEST['status']) ? sanitize_text_field($_REQUEST['status']) : '';
        $post_type_filter = isset($_REQUEST['post_type_filter']) ? sanitize_text_field($_REQUEST['post_type_filter']) : '';
        $include_ignored = isset($_REQUEST['include_ignored']) ? (bool)$_REQUEST['include_ignored'] : false;
        
        // Get sorting
        $orderby = isset($_REQUEST['orderby']) ? sanitize_text_field($_REQUEST['orderby']) : 'id';
        $order = isset($_REQUEST['order']) && $_REQUEST['order'] === 'desc' ? 'DESC' : 'ASC';
        
        // Map orderby to actual database columns
        $sql_orderby_map = array(
            'url' => 'l.url',
            'post_id' => 'l.post_id', 
            'status' => 'l.status',
            'status_code' => 'l.status_code'
        );
        $sql_orderby = isset($sql_orderby_map[$orderby]) ? $sql_orderby_map[$orderby] : 'l.id';
        
        // Get data
        $this->items = $this->db->get_links_for_scan_filtered(
            $this->scan_id, 
            $per_page, 
            $offset, 
            $include_ignored, 
            $search, 
            $status_filter, 
            $post_type_filter, 
            0, 
            $sql_orderby, 
            $order
        );
        
        $total_items = $this->db->count_links_for_scan_filtered(
            $this->scan_id, 
            $include_ignored, 
            $search, 
            $status_filter, 
            $post_type_filter, 
            0
        );
        
        // Set pagination
        $this->set_pagination_args(array(
            'total_items' => $total_items,
            'per_page'    => $per_page,
            'total_pages' => ceil($total_items / $per_page)
        ));
        
        // Set column headers
        $this->_column_headers = array(
            $this->get_columns(),
            array(),
            $this->get_sortable_columns()
        );
    }
    
    /**
     * Display when no items
     */
    public function no_items() {
        _e('No links found.', 'affiliate-hub');
    }
}
