<?php
namespace AffiliateHub\Modules\LinkScanner;

if (!class_exists('WP_List_Table')) {
    require_once ABSPATH . 'wp-admin/includes/class-wp-list-table.php';
}

class LinkScanner_ListTable extends \WP_List_Table {
    protected $db;
    protected $scan_id;

    public function __construct($scan_id = 0) {
        parent::__construct(array(
            'singular' => 'scanned_link',
            'plural'   => 'scanned_links',
            // disable built-in WP_List_Table ajax to avoid WordPress firing admin-ajax requests
            // for this table; we perform our own incremental AJAX updates instead
            'ajax'     => false
        ));
        $this->db = new DB();
        $this->scan_id = intval($scan_id);
    }

    public function get_columns() {
        $cols = array(
            'cb' => '<input type="checkbox" />',
            'link' => __('Link', 'affiliate-hub'),
            'label' => __('Link Label', 'affiliate-hub'),
            'post' => __('Post ID', 'affiliate-hub'),
            'post_type' => __('Post Type', 'affiliate-hub'),
            'status' => __('Status', 'affiliate-hub'),
            'code' => __('Code', 'affiliate-hub'),
            'note' => __('Note', 'affiliate-hub'),
            'actions' => __('Actions', 'affiliate-hub')
        );
        return $cols;
    }

    public function column_cb($item) {
        return sprintf('<input type="checkbox" name="scanned[]" value="%d" />', $item->id);
    }

    public function column_link($item) {
        $url = esc_url($item->url);
        $open = '<a href="' . $url . '" target="_blank" rel="noopener noreferrer">';
        $open .= esc_html($item->url);
        $open .= '</a>';
        return $open;
    }

    public function column_label($item) {
        return esc_html($item->anchor_text);
    }

    public function column_post($item) {
        if ($item->post_id) {
            $edit = admin_url('post.php?post=' . intval($item->post_id) . '&action=edit');
            return sprintf('<a href="%s" target="_blank">#%d</a>', esc_url($edit), intval($item->post_id));
        }
        return '';
    }

    public function column_post_type($item) {
        if (isset($item->post_type) && !empty($item->post_type)) {
            return esc_html($item->post_type);
        }
        // fallback: try to derive from post_id
        if (!empty($item->post_id)) {
            $pt = get_post_type(intval($item->post_id));
            return $pt ? esc_html($pt) : '';
        }
        return '';
    }

    public function column_status($item) {
        $s = esc_html($item->status);
        $color = '#6c757d';
        if ($item->status === 'active') $color = '#28a745';
        if ($item->status === 'broken') $color = '#dc3545';
        if ($item->status === 'forbidden') $color = '#ffc107';
        return '<span class="ah-status-badge" style="background:'.$color.';color:#fff;padding:4px 8px;border-radius:12px;">'. $s .'</span>';
    }

    public function column_code($item) {
        $c = intval($item->status_code);
        $color = '#6c757d';
        if ($c >= 200 && $c < 300) $color = '#28a745';
        else if ($c >= 400 && $c < 500) $color = '#dc3545';
        else if ($c >= 500) $color = '#b02a37';
        return '<span class="ah-code-badge" style="background:'.$color.';color:#fff;padding:3px 7px;border-radius:8px;">'.esc_html($c).'</span>';
    }

    public function column_note($item) {
        return esc_html(isset($item->note) ? $item->note : '');
    }

    public function column_actions($item) {
        if (isset($item->ignored) && intval($item->ignored) === 1) {
            return '<button class="button ah-unignore-link" data-id="'.intval($item->id).'">Un-ignore</button>';
        }
        return '<button class="button ah-retry-link" data-id="'.intval($item->id).'">Retry</button> <button class="button ah-ignore-link" data-id="'.intval($item->id).'">Ignore</button>';
    }

    public function get_sortable_columns() {
        return array(
            'post' => array('post', true),          // maps to l.post_id
            'status' => array('status', false),     // maps to l.status
            'code' => array('status_code', false),  // maps to l.status_code
            'url' => array('url', false),           // maps to l.url
            'label' => array('label', false)        // maps to l.anchor_text
        );
    }

    protected function get_bulk_actions() {
        return array('retry' => __('Retry', 'affiliate-hub'), 'ignore' => __('Ignore', 'affiliate-hub'), 'unignore' => __('Un-ignore', 'affiliate-hub'));
    }

    public function prepare_items() {
        $per_page = $this->get_items_per_page('ah_scan_per_page', 20);
        $current_page = $this->get_pagenum();
        $offset = ($current_page - 1) * $per_page;

        // register columns with WP_List_Table so sortable links are generated
        $columns = $this->get_columns();
        $hidden = array();
        $sortable = $this->get_sortable_columns();
        $this->_column_headers = array($columns, $hidden, $sortable);

        // Filters
        $search = isset($_REQUEST['s']) ? sanitize_text_field($_REQUEST['s']) : '';
        $status = isset($_REQUEST['status']) ? sanitize_text_field($_REQUEST['status']) : '';
        $include_ignored = isset($_REQUEST['include_ignored']) ? boolval($_REQUEST['include_ignored']) : false;

        // Sorting: map incoming orderby to safe column names
        $raw_orderby = isset($_REQUEST['orderby']) ? sanitize_text_field($_REQUEST['orderby']) : '';
        $raw_order = isset($_REQUEST['order']) ? strtoupper(sanitize_text_field($_REQUEST['order'])) : 'ASC';
        $order = ($raw_order === 'DESC') ? 'DESC' : 'ASC';
        $allowed = array(
            'id' => 'l.id',
            'post' => 'l.post_id',
            'status' => 'l.status',
            'status_code' => 'l.status_code',
            'code' => 'l.status_code',
            'url' => 'l.url',
            'label' => 'l.anchor_text'
        );
        $orderby = 'l.id';
        if (!empty($raw_orderby) && isset($allowed[$raw_orderby])) {
            $orderby = $allowed[$raw_orderby];
        }

        // Support multi-select post_type filter coming from the admin UI
        $post_type_param = '';
        if (isset($_REQUEST['post_type'])) {
            if (is_array($_REQUEST['post_type'])) {
                $post_type_param = implode(',', array_map('sanitize_text_field', $_REQUEST['post_type']));
            } else {
                $post_type_param = sanitize_text_field($_REQUEST['post_type']);
            }
        }

        $items = $this->db->get_links_for_scan_filtered($this->scan_id, $per_page, $offset, $include_ignored, $search, $status, $post_type_param, 0, $orderby, $order);
    $total = $this->db->count_links_for_scan_filtered($this->scan_id, $include_ignored, $search, $status, '', 0);

        $this->items = $items;
        $this->set_pagination_args(array('total_items' => $total, 'per_page' => $per_page));
    }
}
