<?php
/**
 * Link Shortener Module
 *
 * @package AffiliateHub
 * @subpackage Modules
 */

namespace AffiliateHub\Modules;

use AffiliateHub\Core\Constants;

/**
 * Link Shortener Module
 */
class LinkShortener implements ModuleInterface {
    
    /**
     * Initialize the module
     */
    public function init() {
        // Add shortcode for affiliate links
        add_shortcode('affiliate_link', array($this, 'affiliate_link_shortcode'));
        
        // Add TinyMCE button
        if (is_admin()) {
            add_action('init', array($this, 'add_tinymce_button'));
        }
        
        // Add quick insert functionality to post editor
        add_action('media_buttons', array($this, 'add_media_button'));
        add_action('wp_ajax_get_affiliate_links_list', array($this, 'ajax_get_links_list'));
    }
    
    /**
     * Get module information
     *
     * @return array
     */
    public function get_info() {
        return array(
            'name' => __('Link Shortener', 'affiliate-hub'),
            'description' => __('Provides shortcode functionality and editor integration for affiliate links.', 'affiliate-hub'),
            'version' => '1.0.0',
            'author' => 'Affiliate Hub',
            'required' => true
        );
    }
    
    /**
     * Check if module is enabled
     *
     * @return bool
     */
    public function is_enabled() {
        return true; // Always enabled as it's a core module
    }
    
    /**
     * Affiliate link shortcode
     *
     * @param array $atts Shortcode attributes
     * @param string $content Shortcode content
     * @return string
     */
    public function affiliate_link_shortcode($atts, $content = '') {
        $atts = shortcode_atts(array(
            'id' => '',
            'slug' => '',
            'title' => '',
            'text' => '',
            'class' => '',
            'target' => '',
            'rel' => ''
        ), $atts, 'affiliate_link');
        
        // Get link by ID, slug, or title
        $post = null;
        
        if (!empty($atts['id'])) {
            // Get by ID
            $post = get_post($atts['id']);
        } elseif (!empty($atts['slug'])) {
            // Get by slug - search in post_name or custom slug meta
            $posts = get_posts(array(
                'post_type' => Constants::POST_TYPE_AFFILIATE_LINK,
                'name' => $atts['slug'],
                'posts_per_page' => 1,
                'post_status' => 'publish'
            ));
            
            // If not found by post_name, try custom slug meta
            if (empty($posts)) {
                global $wpdb;
                // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching -- Read-only lookup for published affiliate link by custom slug; limited to single row
                $post_id = $wpdb->get_var($wpdb->prepare("
                    SELECT p.ID 
                    FROM {$wpdb->posts} p
                    INNER JOIN {$wpdb->postmeta} pm ON p.ID = pm.post_id
                    WHERE p.post_type = %s 
                    AND p.post_status = 'publish'
                    AND pm.meta_key = '_affiliate_slug'
                    AND pm.meta_value = %s
                    LIMIT 1
                ", Constants::POST_TYPE_AFFILIATE_LINK, $atts['slug']));
                
                if ($post_id) {
                    $post = get_post($post_id);
                }
            } else {
                $post = $posts[0];
            }
        } elseif (!empty($atts['title'])) {
            // Get by title
            $posts = get_posts(array(
                'post_type' => Constants::POST_TYPE_AFFILIATE_LINK,
                'title' => $atts['title'],
                'posts_per_page' => 1,
                'post_status' => 'publish'
            ));
            $post = !empty($posts) ? $posts[0] : null;
        } else {
            return '<!-- Affiliate Hub: No link ID, slug, or title specified -->';
        }
        
        if (!$post || $post->post_type !== Constants::POST_TYPE_AFFILIATE_LINK) {
            return '<!-- Affiliate Hub: Link not found -->';
        }
        
        $affiliate_link = new \AffiliateHub\Models\AffiliateLink($post->ID);
        $cloaked_url = $affiliate_link->get_cloaked_url();
        
        // Build link attributes
        $link_atts = array(
            'href' => $cloaked_url,
            'data-affiliate-id' => $post->ID
        );
        
        // Add CSS class
        $classes = array('affiliate-link');
        if (!empty($atts['class'])) {
            $classes[] = $atts['class'];
        }
        $link_atts['class'] = implode(' ', $classes);
        
        // Add target
        if (!empty($atts['target']) || $affiliate_link->is_new_window()) {
            $link_atts['target'] = !empty($atts['target']) ? $atts['target'] : '_blank';
            if ($link_atts['target'] === '_blank') {
                // Preserve existing rel attributes and add noopener noreferrer
                $existing_rel = !empty($link_atts['rel']) ? $link_atts['rel'] : '';
                $security_rel = 'noopener noreferrer';
                $link_atts['rel'] = trim($existing_rel . ' ' . $security_rel);
            }
        }
        
        // Add rel attribute
        $rel_parts = array();
        if (!empty($atts['rel'])) {
            $rel_parts[] = $atts['rel'];
        }
        if ($affiliate_link->is_nofollow()) {
            $rel_parts[] = 'nofollow';
        }
        if ($affiliate_link->is_sponsored()) {
            $rel_parts[] = 'sponsored';
        }
        if ($affiliate_link->is_tracking()) {
            $rel_parts[] = 'ugc';
        }
        // If target="_blank" was set above, preserve those rel attributes
        if (!empty($link_atts['rel']) && $link_atts['target'] === '_blank') {
            $rel_parts[] = $link_atts['rel'];
        }
        if (!empty($rel_parts)) {
            $link_atts['rel'] = implode(' ', array_unique(array_filter($rel_parts)));
        }
        
        // Build link HTML
        $link_html = '<a';
        foreach ($link_atts as $attr => $value) {
            $link_html .= ' ' . $attr . '="' . esc_attr($value) . '"';
        }
        $link_html .= '>';
        
        // Determine link text priority: shortcode text > content > post title
        $link_text = '';
        if (!empty($atts['text'])) {
            $link_text = $atts['text'];
        } elseif (!empty($content)) {
            $link_text = $content;
        } else {
            $link_text = $post->post_title;
        }
        
        $link_html .= esc_html($link_text);
        $link_html .= '</a>';
        
        return $link_html;
    }
    
    /**
     * Add TinyMCE button
     */
    public function add_tinymce_button() {
        if (!current_user_can('edit_posts') || !current_user_can('edit_pages')) {
            return;
        }
        
        if (get_user_option('rich_editing') === 'true') {
            add_filter('mce_external_plugins', array($this, 'add_tinymce_plugin'));
            add_filter('mce_buttons', array($this, 'register_tinymce_button'));
        }
    }
    
    /**
     * Add TinyMCE plugin
     */
    public function add_tinymce_plugin($plugin_array) {
        $plugin_array['affiliate_hub'] = AFFILIATE_HUB_URL . 'assets/js/tinymce-plugin.js';
        return $plugin_array;
    }
    
    /**
     * Register TinyMCE button
     */
    public function register_tinymce_button($buttons) {
        array_push($buttons, 'affiliate_hub_button');
        return $buttons;
    }
    
    /**
     * Add media button for affiliate links
     */
    public function add_media_button() {
        if (!current_user_can('edit_posts') || !current_user_can('edit_pages')) {
            return;
        }
        
    echo '<button type="button" class="button affiliate-hub-media-button" data-editor="content">';
    echo '<span class="wp-media-buttons-icon dashicons dashicons-admin-links"></span>';
    echo esc_html__('Add Affiliate Link', 'affiliate-hub');
    echo '</button>';
    }
    
    /**
     * AJAX handler to get affiliate links list
     */
    public function ajax_get_links_list() {
        check_ajax_referer('affiliate_hub_nonce', 'nonce');
        
        $search = isset($_GET['search']) ? sanitize_text_field($_GET['search']) : '';
        $page = isset($_GET['page']) ? max(1, intval($_GET['page'])) : 1;
        $per_page = 20;
        
        $args = array(
            'post_type' => Constants::POST_TYPE_AFFILIATE_LINK,
            'post_status' => 'publish',
            'posts_per_page' => $per_page,
            'paged' => $page,
            'orderby' => 'title',
            'order' => 'ASC'
        );
        
        if (!empty($search)) {
            $args['s'] = $search;
        }
        
        $query = new \WP_Query($args);
        $links = array();
        
        foreach ($query->posts as $post) {
            $affiliate_link = new \AffiliateHub\Models\AffiliateLink($post->ID);
            $links[] = array(
                'id' => $post->ID,
                'title' => $post->post_title,
                'url' => $affiliate_link->get_cloaked_url(),
                'destination' => $affiliate_link->get_destination_url(),
                'clicks' => $affiliate_link->get_click_count()
            );
        }
        
        wp_send_json_success(array(
            'links' => $links,
            'total' => $query->found_posts,
            'pages' => $query->max_num_pages,
            'current_page' => $page
        ));
    }
}
