<?php
/**
 * MaxMind GeoLite2 Database Manager
 * 
 * Downloads and maintains MaxMind GeoLite2 database for local lookups
 * This is for PHASE 3 implementation - advanced local database
 *
 * @package AffiliateHub
 * @subpackage Modules
 */

namespace AffiliateHub\Modules;

class MaxMindManager {
    
    private $database_url = 'https://download.maxmind.com/app/geoip_download?edition_id=GeoLite2-City&license_key={LICENSE_KEY}&suffix=tar.gz';
    
    private $upload_dir;
    private $geoip_dir;
    private $database_file;
    
    public function __construct() {
        $upload_dir = wp_upload_dir();
        $this->upload_dir = $upload_dir['basedir'];
        $this->geoip_dir = $this->upload_dir . '/affiliate-hub/geoip';
        $this->database_file = $this->geoip_dir . '/GeoLite2-City.mmdb';
        
        // Schedule weekly updates
        if (!wp_next_scheduled('affiliate_hub_update_geoip')) {
            wp_schedule_event(time(), 'weekly', 'affiliate_hub_update_geoip');
        }
        
        add_action('affiliate_hub_update_geoip', array($this, 'update_database'));
        add_action('wp_ajax_affiliate_hub_download_geoip', array($this, 'ajax_download_database'));
    }
    
    /**
     * Check if MaxMind database exists and is recent
     */
    public function is_database_available() {
        if (!file_exists($this->database_file)) {
            return false;
        }
        
        $file_time = filemtime($this->database_file);
        $week_ago = time() - (7 * 24 * 60 * 60);
        
        return $file_time > $week_ago;
    }
    
    /**
     * Download and install MaxMind database
     */
    public function download_database($license_key = '') {
        if (empty($license_key)) {
            $license_key = \get_option('affiliate_hub_maxmind_license_key', '');
        }
        
        if (empty($license_key)) {
            return new \WP_Error('no_license', __('MaxMind license key required', 'affiliate-hub'));
        }
        
        // Create directory if it doesn't exist
        if (!file_exists($this->geoip_dir)) {
            \wp_mkdir_p($this->geoip_dir);
            // Add .htaccess for security
            file_put_contents($this->geoip_dir . '/.htaccess', "deny from all\n");
        }
        
        $download_url = str_replace('{LICENSE_KEY}', $license_key, $this->database_url);
        $temp_file = $this->geoip_dir . '/geoip_temp.tar.gz';
        
        // Download the file
        $response = \wp_remote_get($download_url, array(
            'timeout' => 300, // 5 minutes
            'stream' => true,
            'filename' => $temp_file
        ));
        
        if (\is_wp_error($response)) {
            return $response;
        }
        
        if (\wp_remote_retrieve_response_code($response) !== 200) {
            @unlink($temp_file);
            return new \WP_Error('download_failed', __('Failed to download MaxMind database', 'affiliate-hub'));
        }
        
        // Extract the .mmdb file from tar.gz
        if (!$this->extract_database($temp_file)) {
            @unlink($temp_file);
            return new \WP_Error('extract_failed', __('Failed to extract MaxMind database', 'affiliate-hub'));
        }
        
        // Cleanup temp file
        @unlink($temp_file);
        
        // Update last download timestamp
        \update_option('affiliate_hub_geoip_last_update', time());
        
        return true;
    }
    
    /**
     * Extract .mmdb file from downloaded tar.gz
     */
    private function extract_database($tar_file) {
        error_log('MaxMindManager: Starting database extraction from: ' . $tar_file);
        error_log('MaxMindManager: Target database file: ' . $this->database_file);
        
        // Check if PharData is available for extraction
        if (!class_exists('PharData')) {
            error_log('MaxMindManager: PharData not available, trying system tar command');
            
            // Fallback: look for system tar command
            if ($this->is_command_available('tar')) {
                error_log('MaxMindManager: System tar command available');
                $extract_dir = $this->geoip_dir . '/temp_extract';
                \wp_mkdir_p($extract_dir);
                
                $command = sprintf('tar -xzf %s -C %s --strip-components=1 --wildcards "*/GeoLite2-City.mmdb"', 
                    escapeshellarg($tar_file), 
                    escapeshellarg($extract_dir)
                );
                
                error_log('MaxMindManager: Running tar command: ' . $command);
                exec($command, $output, $return_code);
                error_log('MaxMindManager: Tar return code: ' . $return_code);
                error_log('MaxMindManager: Tar output: ' . print_r($output, true));
                
                if ($return_code === 0 && file_exists($extract_dir . '/GeoLite2-City.mmdb')) {
                    rename($extract_dir . '/GeoLite2-City.mmdb', $this->database_file);
                    $this->cleanup_directory($extract_dir);
                    error_log('MaxMindManager: Database successfully extracted using tar');
                    return true;
                }
                
                $this->cleanup_directory($extract_dir);
                error_log('MaxMindManager: Tar extraction failed');
                return false;
            }
            
            error_log('MaxMindManager: Neither PharData nor tar command available');
            return false;
        }
        
        try {
            error_log('MaxMindManager: Using PharData for extraction');
            
            // Use PharData to extract
            $phar = new \PharData($tar_file);
            error_log('MaxMindManager: PharData object created successfully');
            
            // Use RecursiveIteratorIterator to search through all files and subdirectories
            $iterator = new \RecursiveIteratorIterator($phar);
            foreach ($iterator as $file) {
                $filename = $file->getFilename();
                error_log('MaxMindManager: Found file: ' . $filename);
                
                if (strpos($filename, '.mmdb') !== false) {
                    error_log('MaxMindManager: Found .mmdb file: ' . $filename);
                    $content = file_get_contents($file->getPathname());
                    file_put_contents($this->database_file, $content);
                    error_log('MaxMindManager: Database successfully extracted using PharData');
                    return true;
                }
            }
            
            error_log('MaxMindManager: No .mmdb file found in entire archive');
            return false;
            
        } catch (Exception $e) {
            error_log('MaxMind extraction error: ' . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Check if system command is available
     */
    private function is_command_available($command) {
        $test = shell_exec("which $command 2>/dev/null");
        return !empty($test);
    }
    
    /**
     * Recursively remove directory
     */
    private function cleanup_directory($dir) {
        if (!is_dir($dir)) {
            return;
        }
        
        $files = array_diff(scandir($dir), array('.', '..'));
        foreach ($files as $file) {
            $path = $dir . '/' . $file;
            is_dir($path) ? $this->cleanup_directory($path) : unlink($path);
        }
        rmdir($dir);
    }
    
    /**
     * Scheduled database update
     */
    public function update_database() {
        $license_key = \get_option('affiliate_hub_maxmind_license_key', '');
        
        if (empty($license_key)) {
            return; // Skip if no license key configured
        }
        
        $result = $this->download_database($license_key);
        
        if (is_wp_error($result)) {
            // Log error but don't throw exception
            error_log('MaxMind database update failed: ' . $result->get_error_message());
        }
    }
    
    /**
     * AJAX handler for manual database download
     */
    public function ajax_download_database() {
        check_ajax_referer('affiliate_hub_admin_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Insufficient permissions');
        }
        
        $license_key = sanitize_text_field($_POST['license_key'] ?? '');
        
        if (empty($license_key)) {
            wp_send_json_error(__('License key is required', 'affiliate-hub'));
        }
        
        $result = $this->download_database($license_key);
        
        if (is_wp_error($result)) {
            wp_send_json_error($result->get_error_message());
        }
        
        // Save license key for future updates
        \update_option('affiliate_hub_maxmind_license_key', $license_key);
        
        wp_send_json_success(array(
            'message' => __('MaxMind database downloaded and installed successfully', 'affiliate-hub'),
            'file_size' => $this->get_database_info()
        ));
    }
    
    /**
     * Get database file information
     */
    public function get_database_info() {
        if (!file_exists($this->database_file)) {
            return null;
        }
        
        return array(
            'size' => filesize($this->database_file),
            'size_formatted' => size_format(filesize($this->database_file)),
            'modified' => date('Y-m-d H:i:s', filemtime($this->database_file)),
            'path' => $this->database_file
        );
    }
    
    /**
     * Check MaxMind Reader dependency
     */
    public function check_dependencies() {
        $checks = array(
            'phar_data' => class_exists('PharData'),
            'geoip2_reader' => class_exists('GeoIp2\Database\Reader'),
            'writable_dir' => is_writable($this->upload_dir),
            'tar_command' => $this->is_command_available('tar')
        );
        
        return $checks;
    }
    
    /**
     * Get database path
     */
    public function get_database_path() {
        return $this->database_file;
    }
    
    /**
     * Get last update timestamp
     */
    public function get_last_update() {
        if (!file_exists($this->database_file)) {
            return false;
        }
        return filemtime($this->database_file);
    }
}
