<?php
/**
 * Module Manager
 *
 * @package AffiliateHub
 * @subpackage Modules
 */

namespace AffiliateHub\Modules;

/**
 * Module Manager class
 */
class ModuleManager {
    
    /**
     * Registered modules
     *
     * @var array
     */
    private $modules = array();
    
    /**
     * Constructor
     */
    public function __construct() {
        $this->load_modules();
        add_action('init', array($this, 'init_modules'));
    }
    
    /**
     * Load available modules
     */
    private function load_modules() {
        // Core modules that are always loaded
        $core_modules = array(
            'LinkShortener' => '\\AffiliateHub\\Modules\\LinkShortener',
            'ClickTracker' => '\\AffiliateHub\\Modules\\ClickTracker',
            'LinkScanner' => '\\AffiliateHub\\Modules\\LinkScanner',
            'KeywordsAutolinker' => '\\AffiliateHub\\Modules\\KeywordsAutolinker',
            'GeoLocation' => '\\AffiliateHub\\Modules\\GeoLocation',
        );
        
        foreach ($core_modules as $name => $class) {
            if (class_exists($class)) {
                $this->register_module($name, new $class());
            }
        }
        
        // Load optional modules based on settings
        $this->load_optional_modules();
        
        // Allow other plugins/themes to register modules
    \do_action('affiliate_hub_register_modules', $this);
    }
    
    /**
     * Load optional modules
     */
    private function load_optional_modules() {
        // Example: Load advanced statistics module if enabled
    if (\get_option('affiliate_hub_enable_advanced_stats', false)) {
            // Load advanced stats module
        }
        
        // Example: Load import/export module if needed
    if (\get_option('affiliate_hub_enable_import_export', false)) {
            // Load import/export module
        }
    }
    
    /**
     * Register a module
     *
     * @param string $name Module name
     * @param ModuleInterface $module Module instance
     */
    public function register_module($name, $module) {
        if ($module instanceof ModuleInterface) {
            $this->modules[$name] = $module;
        }
    }
    
    /**
     * Get a module by name
     *
     * @param string $name Module name
     * @return ModuleInterface|null
     */
    public function get_module($name) {
        return isset($this->modules[$name]) ? $this->modules[$name] : null;
    }
    
    /**
     * Get all modules
     *
     * @return array
     */
    public function get_modules() {
        return $this->modules;
    }
    
    /**
     * Check if a module is registered
     *
     * @param string $name Module name
     * @return bool
     */
    public function has_module($name) {
        return isset($this->modules[$name]);
    }
    
    /**
     * Initialize all modules
     */
    public function init_modules() {
        foreach ($this->modules as $module) {
            if (method_exists($module, 'init')) {
                $module->init();
            }
        }
        
    \do_action('affiliate_hub_modules_initialized', $this->modules);
    }
    
    /**
     * Enable a module
     *
     * @param string $name Module name
     * @return bool
     */
    public function enable_module($name) {
        if (isset($this->modules[$name])) {
            $module = $this->modules[$name];
            if (method_exists($module, 'enable')) {
                return $module->enable();
            }
        }
        return false;
    }
    
    /**
     * Disable a module
     *
     * @param string $name Module name
     * @return bool
     */
    public function disable_module($name) {
        if (isset($this->modules[$name])) {
            $module = $this->modules[$name];
            if (method_exists($module, 'disable')) {
                return $module->disable();
            }
        }
        return false;
    }
    
    /**
     * Get module information
     *
     * @param string $name Module name
     * @return array|null
     */
    public function get_module_info($name) {
        if (isset($this->modules[$name])) {
            $module = $this->modules[$name];
            if (method_exists($module, 'get_info')) {
                return $module->get_info();
            }
        }
        return null;
    }
}
