<?php
namespace AffiliateHub\Modules;

/**
 * Lightweight User-Agent Parser
 * 
 * Detects browser, OS, and device type without external dependencies
 * Focused on performance and essential analytics data
 */
class UserAgentParser {
    
    /**
     * Browser detection patterns
     */
    private $browsers = [
        'Edge' => '/Edg\/([0-9\.]+)/',  // New Chromium-based Edge
        'Edge Legacy' => '/Edge\/([0-9\.]+)/',  // Old Edge
        'Chrome' => '/Chrome\/([0-9\.]+)/',
        'Firefox' => '/Firefox\/([0-9\.]+)/',
        'Safari' => '/Safari\/[0-9\.]+ Version\/([0-9\.]+)/',
        'Opera' => '/(?:Opera|OPR)\/([0-9\.]+)/',
        'Internet Explorer' => '/MSIE ([0-9\.]+)/',
        'Internet Explorer 11' => '/Trident\/.*rv:([0-9\.]+)/'
    ];
    
    /**
     * Operating system detection patterns
     */
    private $operating_systems = [
        'Windows 11' => '/Windows NT 10\.0.*Windows 11/',
        'Windows 10' => '/Windows NT 10\.0/',
        'Windows 8.1' => '/Windows NT 6\.3/',
        'Windows 8' => '/Windows NT 6\.2/',
        'Windows 7' => '/Windows NT 6\.1/',
        'Windows Vista' => '/Windows NT 6\.0/',
        'Windows XP' => '/Windows NT 5\.1/',
        'macOS' => '/Mac OS X ([0-9_\.]+)/',
        'iOS' => '/(?:iPhone|iPad).*OS ([0-9_\.]+)/',
        'Android' => '/Android ([0-9\.]+)/',
        'Linux' => '/Linux/',
        'Ubuntu' => '/Ubuntu/',
        'Chrome OS' => '/CrOS/'
    ];
    
    /**
     * Device type detection patterns
     */
    private $devices = [
        'Mobile' => '/Mobile|Android|iPhone|iPod|BlackBerry|Opera Mini/',
        'Tablet' => '/iPad|Android.*Tablet|Kindle|Silk/',
        'Bot' => '/bot|crawler|spider|scraper|facebook|twitter|linkedin/i'
    ];
    
    /**
     * Parse user agent string
     * 
     * @param string $user_agent
     * @return array
     */
    public function parse($user_agent = '') {
        if (empty($user_agent)) {
            $user_agent = $_SERVER['HTTP_USER_AGENT'] ?? '';
        }
        
        if (empty($user_agent)) {
            return $this->get_empty_result();
        }
        
        return [
            'browser' => $this->detect_browser($user_agent),
            'browser_version' => $this->detect_browser_version($user_agent),
            'os' => $this->detect_os($user_agent),
            'os_version' => $this->detect_os_version($user_agent),
            'device_type' => $this->detect_device_type($user_agent),
            'is_bot' => $this->is_bot($user_agent),
            'raw_user_agent' => $user_agent
        ];
    }
    
    /**
     * Detect browser name
     * 
     * @param string $user_agent
     * @return string
     */
    private function detect_browser($user_agent) {
        // Special case for Internet Explorer 11
        if (strpos($user_agent, 'Trident/') !== false && strpos($user_agent, 'rv:') !== false) {
            return 'Internet Explorer';
        }
        
        // Check other browsers
        foreach ($this->browsers as $browser => $pattern) {
            if (preg_match($pattern, $user_agent)) {
                return $browser === 'Internet Explorer 11' ? 'Internet Explorer' : $browser;
            }
        }
        
        return 'Unknown';
    }
    
    /**
     * Detect browser version
     * 
     * @param string $user_agent
     * @return string
     */
    private function detect_browser_version($user_agent) {
        // Special case for Internet Explorer 11
        if (strpos($user_agent, 'Trident/') !== false && preg_match('/rv:([0-9\.]+)/', $user_agent, $matches)) {
            return $matches[1];
        }
        
        // Check other browsers
        foreach ($this->browsers as $browser => $pattern) {
            if (preg_match($pattern, $user_agent, $matches)) {
                return $matches[1] ?? '';
            }
        }
        
        return '';
    }
    
    /**
     * Detect operating system
     * 
     * @param string $user_agent
     * @return string
     */
    private function detect_os($user_agent) {
        foreach ($this->operating_systems as $os => $pattern) {
            if (preg_match($pattern, $user_agent)) {
                return $os;
            }
        }
        
        return 'Unknown';
    }
    
    /**
     * Detect operating system version
     * 
     * @param string $user_agent
     * @return string
     */
    private function detect_os_version($user_agent) {
        // macOS version
        if (preg_match('/Mac OS X ([0-9_\.]+)/', $user_agent, $matches)) {
            return str_replace('_', '.', $matches[1]);
        }
        
        // iOS version
        if (preg_match('/(?:iPhone|iPad).*OS ([0-9_\.]+)/', $user_agent, $matches)) {
            return str_replace('_', '.', $matches[1]);
        }
        
        // Android version
        if (preg_match('/Android ([0-9\.]+)/', $user_agent, $matches)) {
            return $matches[1];
        }
        
        // Windows versions are detected by OS name patterns
        return '';
    }
    
    /**
     * Detect device type
     * 
     * @param string $user_agent
     * @return string
     */
    private function detect_device_type($user_agent) {
        // Check for bot first
        if ($this->is_bot($user_agent)) {
            return 'Bot';
        }
        
        // Check for tablet (before mobile, as some tablets contain "Mobile")
        if (preg_match($this->devices['Tablet'], $user_agent)) {
            return 'Tablet';
        }
        
        // Check for mobile
        if (preg_match($this->devices['Mobile'], $user_agent)) {
            return 'Mobile';
        }
        
        return 'Desktop';
    }
    
    /**
     * Check if user agent is a bot/crawler
     * 
     * @param string $user_agent
     * @return bool
     */
    private function is_bot($user_agent) {
        return preg_match($this->devices['Bot'], $user_agent) === 1;
    }
    
    /**
     * Get empty result structure
     * 
     * @return array
     */
    private function get_empty_result() {
        return [
            'browser' => 'Unknown',
            'browser_version' => '',
            'os' => 'Unknown',
            'os_version' => '',
            'device_type' => 'Unknown',
            'is_bot' => false,
            'raw_user_agent' => ''
        ];
    }
    
    /**
     * Get browser statistics for analytics
     * 
     * @param array $user_agents Array of user agent strings
     * @return array
     */
    public function get_browser_stats($user_agents) {
        $browser_counts = [];
        $os_counts = [];
        $device_counts = [];
        
        foreach ($user_agents as $ua) {
            $parsed = $this->parse($ua);
            
            // Count browsers
            $browser = $parsed['browser'];
            $browser_counts[$browser] = ($browser_counts[$browser] ?? 0) + 1;
            
            // Count operating systems
            $os = $parsed['os'];
            $os_counts[$os] = ($os_counts[$os] ?? 0) + 1;
            
            // Count device types
            $device = $parsed['device_type'];
            $device_counts[$device] = ($device_counts[$device] ?? 0) + 1;
        }
        
        // Sort by count descending
        arsort($browser_counts);
        arsort($os_counts);
        arsort($device_counts);
        
        return [
            'browsers' => $browser_counts,
            'operating_systems' => $os_counts,
            'device_types' => $device_counts,
            'total_parsed' => count($user_agents)
        ];
    }
    
    /**
     * Get simplified browser name for grouping
     * 
     * @param string $browser_name
     * @return string
     */
    public function get_browser_family($browser_name) {
        $families = [
            'Chrome' => 'Chrome',
            'Firefox' => 'Firefox',
            'Safari' => 'Safari',
            'Edge' => 'Edge',
            'Opera' => 'Opera',
            'Internet Explorer' => 'Internet Explorer'
        ];
        
        return $families[$browser_name] ?? 'Other';
    }
    
    /**
     * Get simplified OS name for grouping
     * 
     * @param string $os_name
     * @return string
     */
    public function get_os_family($os_name) {
        if (strpos($os_name, 'Windows') !== false) {
            return 'Windows';
        }
        
        if ($os_name === 'macOS') {
            return 'macOS';
        }
        
        if ($os_name === 'iOS') {
            return 'iOS';
        }
        
        if (strpos($os_name, 'Android') !== false) {
            return 'Android';
        }
        
        if (strpos($os_name, 'Linux') !== false || strpos($os_name, 'Ubuntu') !== false) {
            return 'Linux';
        }
        
        return 'Other';
    }
}
