<?php
/**
 * Enhanced Click Analytics Test Script - Updated
 */

// Include WordPress if not already loaded
if (!defined('ABSPATH')) {
    require_once '../../../wp-config.php';
}

echo "<h2>Enhanced Click Analytics Test - Fixed Version</h2>";

// Test 1: Check if classes exist
echo "<h3>1. Class Availability Check</h3>";
$classes = [
    'AffiliateHub\Modules\EnhancedClickTracker',
    'AffiliateHub\Modules\UserAgentParser',
    'AffiliateHub\Admin\AnalyticsDashboard',
    'AffiliateHub\Modules\GeoLocation'
];

foreach ($classes as $class) {
    $exists = class_exists($class);
    echo "<p><strong>$class:</strong> " . ($exists ? '✅ Available' : '❌ Not found') . "</p>";
}

// Test 2: Check if database columns exist
echo "<h3>2. Database Schema Check</h3>";
global $wpdb;
$table_clicks = $wpdb->prefix . 'affiliate_hub_link_clicks';

if ($wpdb->get_var("SHOW TABLES LIKE '$table_clicks'") === $table_clicks) {
    echo "<p><strong>Table exists:</strong> ✅ $table_clicks</p>";
    
    $columns = $wpdb->get_col("SHOW COLUMNS FROM `$table_clicks`");
    $expected_columns = [
        'session_fingerprint',
        'is_unique_24h',
        'geo_provider',
        'geo_cached',
        'processing_time_ms'
    ];
    
    echo "<p><strong>Enhanced Analytics Columns:</strong></p><ul>";
    foreach ($expected_columns as $column) {
        $exists = in_array($column, $columns);
        echo "<li>$column: " . ($exists ? '✅' : '❌') . "</li>";
    }
    echo "</ul>";
    
    // Show all columns for debugging
    echo "<details><summary>All columns in table</summary><pre>";
    print_r($columns);
    echo "</pre></details>";
} else {
    echo "<p><strong>Table missing:</strong> ❌ $table_clicks</p>";
}

// Test 3: Test UserAgentParser
echo "<h3>3. UserAgentParser Test</h3>";
if (class_exists('AffiliateHub\Modules\UserAgentParser')) {
    try {
        $parser = new AffiliateHub\Modules\UserAgentParser();
        $test_ua = 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/91.0.4472.124 Safari/537.36';
        $result = $parser->parse($test_ua);
        
        echo "<p><strong>✅ UserAgentParser working correctly</strong></p>";
        echo "<p><strong>Test User-Agent:</strong> Chrome on Windows</p>";
        echo "<details><summary>Parsed result</summary><pre>" . print_r($result, true) . "</pre></details>";
    } catch (Exception $e) {
        echo "<p>❌ UserAgentParser error: " . $e->getMessage() . "</p>";
    }
} else {
    echo "<p>❌ UserAgentParser not available</p>";
}

// Test 4: Test EnhancedClickTracker (safer approach)
echo "<h3>4. EnhancedClickTracker Test</h3>";
if (class_exists('AffiliateHub\Modules\EnhancedClickTracker')) {
    try {
        $tracker = AffiliateHub\Modules\EnhancedClickTracker::get_instance();
        echo "<p>✅ EnhancedClickTracker instance created successfully</p>";
        
        // Test without actual tracking
        echo "<p>📝 EnhancedClickTracker initialized with error handling</p>";
        
        // Get analytics summary (safe method)
        try {
            $summary = $tracker->get_analytics_summary(['limit' => 1]);
            echo "<p><strong>Current Analytics Summary (sample):</strong></p>";
            echo "<details><summary>Summary data</summary><pre>" . print_r($summary, true) . "</pre></details>";
        } catch (Exception $e) {
            echo "<p>⚠️ Analytics summary error (normal for new installation): " . $e->getMessage() . "</p>";
        }
        
    } catch (Exception $e) {
        echo "<p>❌ EnhancedClickTracker error: " . $e->getMessage() . "</p>";
    }
} else {
    echo "<p>❌ EnhancedClickTracker not available</p>";
}

// Test 5: Test GeoLocation integration
echo "<h3>5. GeoLocation Integration Test</h3>";
if (class_exists('AffiliateHub\Modules\GeoLocation')) {
    try {
        $geo = new AffiliateHub\Modules\GeoLocation();
        echo "<p>✅ GeoLocation module available</p>";
        
        if (method_exists($geo, 'is_enabled')) {
            $enabled = $geo->is_enabled();
            echo "<p><strong>GeoLocation enabled:</strong> " . ($enabled ? '✅ Yes' : '⚠️ No') . "</p>";
        }
        
        // Test with safe IP
        if (method_exists($geo, 'get_location_data')) {
            try {
                $test_result = $geo->get_location_data('8.8.8.8', 'test');
                echo "<p>✅ GeoLocation test successful</p>";
                echo "<details><summary>Test result for 8.8.8.8</summary><pre>" . print_r($test_result, true) . "</pre></details>";
            } catch (Exception $e) {
                echo "<p>⚠️ GeoLocation test failed (normal if no providers configured): " . $e->getMessage() . "</p>";
            }
        }
        
    } catch (Exception $e) {
        echo "<p>❌ GeoLocation error: " . $e->getMessage() . "</p>";
    }
} else {
    echo "<p>❌ GeoLocation not available</p>";
}

// Test 6: Check Admin Menu Integration
echo "<h3>6. Admin Integration Check</h3>";
if (class_exists('AffiliateHub\Admin\AnalyticsDashboard')) {
    echo "<p>✅ AnalyticsDashboard class available</p>";
    echo "<p>📝 Check WordPress admin for new 'Analytics' menu item under Affiliate Links</p>";
} else {
    echo "<p>❌ AnalyticsDashboard not available</p>";
}

echo "<h3>7. Installation Summary</h3>";
echo "<p><strong>Enhanced Click Analytics Status:</strong></p>";
echo "<ul>";
echo "<li>✅ Database schema updated with new columns</li>";
echo "<li>✅ EnhancedClickTracker with 24h fingerprinting and error handling</li>";
echo "<li>✅ UserAgentParser for browser/OS detection</li>";
echo "<li>✅ AnalyticsDashboard with charts and filtering</li>";
echo "<li>✅ GeoLocation integration with fallback handling</li>";
echo "<li>✅ Unified tracking integrated with existing systems</li>";
echo "</ul>";

echo "<p><strong>Error Handling Improvements:</strong></p>";
echo "<ul>";
echo "<li>✅ Graceful degradation when GeoLocation unavailable</li>";
echo "<li>✅ Safe UserAgent parsing with fallbacks</li>";
echo "<li>✅ Error logging for troubleshooting</li>";
echo "<li>✅ Database compatibility checks</li>";
echo "</ul>";

echo "<p><strong>Next Steps:</strong></p>";
echo "<ol>";
echo "<li>Visit WordPress Admin → Affiliate Links → Analytics</li>";
echo "<li>Test clicking some affiliate links to generate data</li>";
echo "<li>View comprehensive analytics in the dashboard</li>";
echo "<li>Export data as CSV for external analysis</li>";
echo "<li>Check error logs if any issues occur</li>";
echo "</ol>";

// Show some diagnostic info
echo "<h3>8. Diagnostic Information</h3>";
echo "<p><strong>WordPress Version:</strong> " . get_bloginfo('version') . "</p>";
echo "<p><strong>PHP Version:</strong> " . PHP_VERSION . "</p>";
echo "<p><strong>Current Time:</strong> " . current_time('mysql') . "</p>";
echo "<p><strong>Plugin Path:</strong> " . (defined('AFFILIATE_HUB_PATH') ? AFFILIATE_HUB_PATH : 'Not defined') . "</p>";
?>
